/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

import { HttpClient, HttpContext, HttpResponse } from '@angular/common/http';
import { inject, Injectable } from '@angular/core';
import { Observable } from 'rxjs';
import { SharedDatalakeRestService } from './shared-dashboard.service';
import {
    CompositeDashboard,
    Dashboard,
} from '../model/dashboard/dashboard.model';
import { NGX_LOADING_BAR_IGNORED } from '@ngx-loading-bar/http-client';

@Injectable({
    providedIn: 'root',
})
export class DashboardService {
    private http = inject(HttpClient);
    private sharedDatalakeRestService = inject(SharedDatalakeRestService);

    getDashboards(): Observable<Dashboard[]> {
        return this.sharedDatalakeRestService.getDashboards(this.dashboardUrl);
    }

    getDashboard(dashboardId: string): Observable<Dashboard> {
        return this.http.get<Dashboard>(`${this.dashboardUrl}/${dashboardId}`);
    }

    getCompositeDashboard(
        dashboardId: string,
        eTag = undefined,
    ): Observable<HttpResponse<any>> {
        const headers = eTag ? { 'If-None-Match': eTag } : {};
        return this.http.get<CompositeDashboard>(
            `${this.dashboardUrl}/${dashboardId}/composite`,
            {
                headers,
                observe: 'response',
                context: new HttpContext().set(NGX_LOADING_BAR_IGNORED, true),
            },
        );
    }

    updateDashboard(dashboard: Dashboard): Observable<Dashboard> {
        return this.sharedDatalakeRestService.updateDashboard(
            this.dashboardUrl,
            dashboard,
        );
    }

    deleteDashboard(dashboard: Dashboard): Observable<any> {
        return this.sharedDatalakeRestService.deleteDashboard(
            this.dashboardUrl,
            dashboard,
        );
    }

    saveDashboard(dashboard: Dashboard): Observable<any> {
        return this.sharedDatalakeRestService.saveDashboard(
            this.dashboardUrl,
            dashboard,
        );
    }

    private get baseUrl() {
        return '/streampipes-backend';
    }

    private get dashboardUrl() {
        return `${this.baseUrl}/api/v3/datalake/dashboard`;
    }
}
