﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/AWSMigrationHub/MigrationHubRequest.h>
#include <aws/AWSMigrationHub/MigrationHub_EXPORTS.h>
#include <aws/AWSMigrationHub/model/ApplicationStatus.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace MigrationHub {
namespace Model {

/**
 */
class NotifyApplicationStateRequest : public MigrationHubRequest {
 public:
  AWS_MIGRATIONHUB_API NotifyApplicationStateRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "NotifyApplicationState"; }

  AWS_MIGRATIONHUB_API Aws::String SerializePayload() const override;

  AWS_MIGRATIONHUB_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The configurationId in Application Discovery Service that uniquely identifies
   * the grouped application.</p>
   */
  inline const Aws::String& GetApplicationId() const { return m_applicationId; }
  inline bool ApplicationIdHasBeenSet() const { return m_applicationIdHasBeenSet; }
  template <typename ApplicationIdT = Aws::String>
  void SetApplicationId(ApplicationIdT&& value) {
    m_applicationIdHasBeenSet = true;
    m_applicationId = std::forward<ApplicationIdT>(value);
  }
  template <typename ApplicationIdT = Aws::String>
  NotifyApplicationStateRequest& WithApplicationId(ApplicationIdT&& value) {
    SetApplicationId(std::forward<ApplicationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Status of the application - Not Started, In-Progress, Complete.</p>
   */
  inline ApplicationStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(ApplicationStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline NotifyApplicationStateRequest& WithStatus(ApplicationStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp when the application state changed.</p>
   */
  inline const Aws::Utils::DateTime& GetUpdateDateTime() const { return m_updateDateTime; }
  inline bool UpdateDateTimeHasBeenSet() const { return m_updateDateTimeHasBeenSet; }
  template <typename UpdateDateTimeT = Aws::Utils::DateTime>
  void SetUpdateDateTime(UpdateDateTimeT&& value) {
    m_updateDateTimeHasBeenSet = true;
    m_updateDateTime = std::forward<UpdateDateTimeT>(value);
  }
  template <typename UpdateDateTimeT = Aws::Utils::DateTime>
  NotifyApplicationStateRequest& WithUpdateDateTime(UpdateDateTimeT&& value) {
    SetUpdateDateTime(std::forward<UpdateDateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional boolean flag to indicate whether any effect should take place. Used
   * to test if the caller has permission to make the call.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline NotifyApplicationStateRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationId;

  ApplicationStatus m_status{ApplicationStatus::NOT_SET};

  Aws::Utils::DateTime m_updateDateTime{};

  bool m_dryRun{false};
  bool m_applicationIdHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_updateDateTimeHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace MigrationHub
}  // namespace Aws
