﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/kendra/Kendra_EXPORTS.h>
#include <aws/kendra/model/DataSourceToIndexFieldMapping.h>
#include <aws/kendra/model/DataSourceVpcConfiguration.h>
#include <aws/kendra/model/FsxFileSystemType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace kendra {
namespace Model {

/**
 * <p>Provides the configuration information to connect to Amazon FSx as your data
 * source.</p>  <p>Amazon Kendra now supports an upgraded Amazon FSx Windows
 * connector.</p> <p>You must now use the <a
 * href="https://docs.aws.amazon.com/kendra/latest/APIReference/API_TemplateConfiguration.html">TemplateConfiguration</a>
 * object instead of the <code>FsxConfiguration</code> object to configure your
 * connector.</p> <p>Connectors configured using the older console and API
 * architecture will continue to function as configured. However, you won't be able
 * to edit or update them. If you want to edit or update your connector
 * configuration, you must create a new connector.</p> <p>We recommended migrating
 * your connector workflow to the upgraded version. Support for connectors
 * configured using the older architecture is scheduled to end by June 2024.</p>
 * <p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/FsxConfiguration">AWS
 * API Reference</a></p>
 */
class FsxConfiguration {
 public:
  AWS_KENDRA_API FsxConfiguration() = default;
  AWS_KENDRA_API FsxConfiguration(Aws::Utils::Json::JsonView jsonValue);
  AWS_KENDRA_API FsxConfiguration& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_KENDRA_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The identifier of the Amazon FSx file system.</p> <p>You can find your file
   * system ID on the file system dashboard in the Amazon FSx console. For
   * information on how to create a file system in Amazon FSx console, using Windows
   * File Server as an example, see <a
   * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/getting-started-step1.html">Amazon
   * FSx Getting started guide</a>.</p>
   */
  inline const Aws::String& GetFileSystemId() const { return m_fileSystemId; }
  inline bool FileSystemIdHasBeenSet() const { return m_fileSystemIdHasBeenSet; }
  template <typename FileSystemIdT = Aws::String>
  void SetFileSystemId(FileSystemIdT&& value) {
    m_fileSystemIdHasBeenSet = true;
    m_fileSystemId = std::forward<FileSystemIdT>(value);
  }
  template <typename FileSystemIdT = Aws::String>
  FsxConfiguration& WithFileSystemId(FileSystemIdT&& value) {
    SetFileSystemId(std::forward<FileSystemIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon FSx file system type. Windows is currently the only supported
   * type.</p>
   */
  inline FsxFileSystemType GetFileSystemType() const { return m_fileSystemType; }
  inline bool FileSystemTypeHasBeenSet() const { return m_fileSystemTypeHasBeenSet; }
  inline void SetFileSystemType(FsxFileSystemType value) {
    m_fileSystemTypeHasBeenSet = true;
    m_fileSystemType = value;
  }
  inline FsxConfiguration& WithFileSystemType(FsxFileSystemType value) {
    SetFileSystemType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Configuration information for an Amazon Virtual Private Cloud to connect to
   * your Amazon FSx. Your Amazon FSx instance must reside inside your VPC.</p>
   */
  inline const DataSourceVpcConfiguration& GetVpcConfiguration() const { return m_vpcConfiguration; }
  inline bool VpcConfigurationHasBeenSet() const { return m_vpcConfigurationHasBeenSet; }
  template <typename VpcConfigurationT = DataSourceVpcConfiguration>
  void SetVpcConfiguration(VpcConfigurationT&& value) {
    m_vpcConfigurationHasBeenSet = true;
    m_vpcConfiguration = std::forward<VpcConfigurationT>(value);
  }
  template <typename VpcConfigurationT = DataSourceVpcConfiguration>
  FsxConfiguration& WithVpcConfiguration(VpcConfigurationT&& value) {
    SetVpcConfiguration(std::forward<VpcConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the
   * key-value pairs required to connect to your Amazon FSx file system. Windows is
   * currently the only supported type. The secret must contain a JSON structure with
   * the following keys:</p> <ul> <li> <p>username—The Active Directory user name,
   * along with the Domain Name System (DNS) domain name. For example,
   * <i>user@corp.example.com</i>. The Active Directory user account must have read
   * and mounting access to the Amazon FSx file system for Windows.</p> </li> <li>
   * <p>password—The password of the Active Directory user account with read and
   * mounting access to the Amazon FSx Windows file system.</p> </li> </ul>
   */
  inline const Aws::String& GetSecretArn() const { return m_secretArn; }
  inline bool SecretArnHasBeenSet() const { return m_secretArnHasBeenSet; }
  template <typename SecretArnT = Aws::String>
  void SetSecretArn(SecretArnT&& value) {
    m_secretArnHasBeenSet = true;
    m_secretArn = std::forward<SecretArnT>(value);
  }
  template <typename SecretArnT = Aws::String>
  FsxConfiguration& WithSecretArn(SecretArnT&& value) {
    SetSecretArn(std::forward<SecretArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of regular expression patterns to include certain files in your Amazon
   * FSx file system. Files that match the patterns are included in the index. Files
   * that don't match the patterns are excluded from the index. If a file matches
   * both an inclusion and exclusion pattern, the exclusion pattern takes precedence
   * and the file isn't included in the index.</p>
   */
  inline const Aws::Vector<Aws::String>& GetInclusionPatterns() const { return m_inclusionPatterns; }
  inline bool InclusionPatternsHasBeenSet() const { return m_inclusionPatternsHasBeenSet; }
  template <typename InclusionPatternsT = Aws::Vector<Aws::String>>
  void SetInclusionPatterns(InclusionPatternsT&& value) {
    m_inclusionPatternsHasBeenSet = true;
    m_inclusionPatterns = std::forward<InclusionPatternsT>(value);
  }
  template <typename InclusionPatternsT = Aws::Vector<Aws::String>>
  FsxConfiguration& WithInclusionPatterns(InclusionPatternsT&& value) {
    SetInclusionPatterns(std::forward<InclusionPatternsT>(value));
    return *this;
  }
  template <typename InclusionPatternsT = Aws::String>
  FsxConfiguration& AddInclusionPatterns(InclusionPatternsT&& value) {
    m_inclusionPatternsHasBeenSet = true;
    m_inclusionPatterns.emplace_back(std::forward<InclusionPatternsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of regular expression patterns to exclude certain files in your Amazon
   * FSx file system. Files that match the patterns are excluded from the index.
   * Files that don't match the patterns are included in the index. If a file matches
   * both an inclusion and exclusion pattern, the exclusion pattern takes precedence
   * and the file isn't included in the index.</p>
   */
  inline const Aws::Vector<Aws::String>& GetExclusionPatterns() const { return m_exclusionPatterns; }
  inline bool ExclusionPatternsHasBeenSet() const { return m_exclusionPatternsHasBeenSet; }
  template <typename ExclusionPatternsT = Aws::Vector<Aws::String>>
  void SetExclusionPatterns(ExclusionPatternsT&& value) {
    m_exclusionPatternsHasBeenSet = true;
    m_exclusionPatterns = std::forward<ExclusionPatternsT>(value);
  }
  template <typename ExclusionPatternsT = Aws::Vector<Aws::String>>
  FsxConfiguration& WithExclusionPatterns(ExclusionPatternsT&& value) {
    SetExclusionPatterns(std::forward<ExclusionPatternsT>(value));
    return *this;
  }
  template <typename ExclusionPatternsT = Aws::String>
  FsxConfiguration& AddExclusionPatterns(ExclusionPatternsT&& value) {
    m_exclusionPatternsHasBeenSet = true;
    m_exclusionPatterns.emplace_back(std::forward<ExclusionPatternsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of <code>DataSourceToIndexFieldMapping</code> objects that map Amazon
   * FSx data source attributes or field names to Amazon Kendra index field names. To
   * create custom fields, use the <code>UpdateIndex</code> API before you map to
   * Amazon FSx fields. For more information, see <a
   * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping
   * data source fields</a>. The Amazon FSx data source field names must exist in
   * your Amazon FSx custom metadata.</p>
   */
  inline const Aws::Vector<DataSourceToIndexFieldMapping>& GetFieldMappings() const { return m_fieldMappings; }
  inline bool FieldMappingsHasBeenSet() const { return m_fieldMappingsHasBeenSet; }
  template <typename FieldMappingsT = Aws::Vector<DataSourceToIndexFieldMapping>>
  void SetFieldMappings(FieldMappingsT&& value) {
    m_fieldMappingsHasBeenSet = true;
    m_fieldMappings = std::forward<FieldMappingsT>(value);
  }
  template <typename FieldMappingsT = Aws::Vector<DataSourceToIndexFieldMapping>>
  FsxConfiguration& WithFieldMappings(FieldMappingsT&& value) {
    SetFieldMappings(std::forward<FieldMappingsT>(value));
    return *this;
  }
  template <typename FieldMappingsT = DataSourceToIndexFieldMapping>
  FsxConfiguration& AddFieldMappings(FieldMappingsT&& value) {
    m_fieldMappingsHasBeenSet = true;
    m_fieldMappings.emplace_back(std::forward<FieldMappingsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_fileSystemId;

  FsxFileSystemType m_fileSystemType{FsxFileSystemType::NOT_SET};

  DataSourceVpcConfiguration m_vpcConfiguration;

  Aws::String m_secretArn;

  Aws::Vector<Aws::String> m_inclusionPatterns;

  Aws::Vector<Aws::String> m_exclusionPatterns;

  Aws::Vector<DataSourceToIndexFieldMapping> m_fieldMappings;
  bool m_fileSystemIdHasBeenSet = false;
  bool m_fileSystemTypeHasBeenSet = false;
  bool m_vpcConfigurationHasBeenSet = false;
  bool m_secretArnHasBeenSet = false;
  bool m_inclusionPatternsHasBeenSet = false;
  bool m_exclusionPatternsHasBeenSet = false;
  bool m_fieldMappingsHasBeenSet = false;
};

}  // namespace Model
}  // namespace kendra
}  // namespace Aws
