// SPDX-License-Identifier: BSD-3-Clause
package org.islandoftex.albatross

import com.github.ajalt.mordant.rendering.BorderStyle
import com.github.ajalt.mordant.rendering.TextAlign
import com.github.ajalt.mordant.table.ColumnWidth
import com.github.ajalt.mordant.table.table
import com.github.ajalt.mordant.terminal.AnsiLevel
import com.github.ajalt.mordant.terminal.Terminal

data class Configuration(
    val showStyles: Boolean,
    val detailed: Boolean,
    val matchAny: Boolean,
    val ansiLevel: AnsiLevel,
    val borderStyle: BorderStyle
)

class Viewer(
    private val configuration: Configuration,
    private val codepoints: List<String>
) {
    private val terminal = Terminal(configuration.ansiLevel)

    fun view() {
        when {
            !FCList.isAvailable -> viewNotFound()
            configuration.matchAny -> {
                codepoints.forEach {
                    val fonts = FCList.getFonts(it)
                    if (fonts.isEmpty()) {
                        viewNoEntries(listOf(it))
                    } else {
                        viewEntries(listOf(it), fonts)
                    }
                }
            }
            else -> {
                val fonts = codepoints.flatMap { FCList.getFonts(it) }.distinct()
                if (fonts.isEmpty()) {
                    viewNoEntries(codepoints)
                } else {
                    viewEntries(codepoints, fonts.toList())
                }
            }
        }
    }

    private fun viewNoEntries(codepoint: List<String>) {
        terminal.println(table {
            captionTop(
                if (codepoint.size == 1)
                    "Unicode code point $codepoint mapping to ${toGlyph(codepoint.first())}"
                else
                    "Unicode code points $codepoint mapping to ${codepoint.map { toGlyph(it) }}",
                style = Settings.cliTitleStyle
            )
            borderStyle = configuration.borderStyle
            borderTextStyle = Settings.borderColour
            column(0) {
                width = ColumnWidth.Expand(Settings.noEntriesColumnWidthPercentage)
                align = TextAlign.CENTER
                style = Settings.cliTextStyle
            }
            body {
                row("No fonts with support for this Unicode code point were found.")
            }
        })
    }

    private fun viewEntries(codepoint: List<String>, fonts: List<FontEntry>) {
        val entries = fonts.groupBy { it.name }

        val detailString = ", font details,".takeIf { configuration.detailed } ?: ""
        val caption = if (codepoint.size == 1)
            "Unicode code point $codepoint$detailString mapping to ${toGlyph(codepoint.first())}"
        else
            "Unicode code points $codepoint$detailString mapping to ${codepoint.map { toGlyph(it) }}"

        if (!configuration.detailed) {
            terminal.println(table {
                captionTop(caption, style = Settings.cliTitleStyle)
                borderStyle = configuration.borderStyle
                borderTextStyle = Settings.borderColour
                column(0) {
                    width =
                        ColumnWidth.Expand(
                            if (configuration.showStyles) Settings.fontNameColumnWidthPercentage
                            else Settings.fontNameColumnFullWidthPercentage
                        )
                }
                if (configuration.showStyles) {
                    column(1) {
                        width = ColumnWidth.Expand(Settings.fontStylesColumnWidthPercentage)
                    }
                }
                header {
                    if (configuration.showStyles)
                        row(Settings.cliHeaderStyle("Font name"), Settings.cliHeaderStyle("Available styles"))
                    else
                        row(Settings.cliHeaderStyle("Font name"))
                }
                body {
                    style = Settings.cliTextStyle
                    if (configuration.showStyles) {
                        entries.forEach { (name, font) ->
                            row(name, font.map { it.styles }.flatten().distinct().joinToString(", "))
                        }
                    } else {
                        entries.keys.forEach { row(it) }
                    }
                }
            })
        } else {
            entries.forEach {
                terminal.println(table {
                    captionTop(caption, style = Settings.cliTitleStyle)
                    borderStyle = configuration.borderStyle
                    column(0) {
                        width = ColumnWidth.Expand(Settings.detailsKeyColumnWidthPercentage)
                    }
                    column(1) {
                        width = ColumnWidth.Expand(Settings.detailsValueColumnWidthPercentage)
                    }
                    body {
                        row(Settings.cliHeaderStyle("Name"), Settings.cliTextStyle(it.key))
                        row(
                            Settings.cliHeaderStyle("Type"),
                            Settings.cliTextStyle(it.value.map { it.description }.distinct().joinToString(", "))
                        )
                        row(Settings.cliHeaderStyle("Files"), table {
                            outerBorder = false
                            borderStyle = configuration.borderStyle
                            column(0) {
                                width = ColumnWidth.Expand(Settings.detailsPathsColumnWidthPercentage)
                            }
                            body {
                                it.value.forEach { row(Settings.cliPathStyle(it.path.toString())) }
                            }
                        })
                        if (configuration.showStyles) {
                            row(
                                Settings.cliHeaderStyle("Styles"),
                                Settings.cliTextStyle(
                                    it.value.map { it.styles }.flatten().distinct().joinToString(", ")
                                )
                            )
                        }
                    }
                })
            }
        }
    }

    private fun viewNotFound() {
        terminal.println(table {
            captionTop("Attention!", style = Settings.cliTitleStyle)
            borderStyle = configuration.borderStyle
            column(0) {
                width = ColumnWidth.Expand(Settings.notFoundColumnWidthPercentage)
                align = TextAlign.CENTER
                style = Settings.cliTextStyle
            }
            body {
                row("This program requires fc-list to be available in your system path.")
            }
        })
    }

    @Suppress("MagicNumber")
    private fun toGlyph(codepoint: String) = Character.toChars(codepoint.toInt(16)).concatToString()
}
