/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2025 Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, 38000 Grenoble, France
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef INTERFACEFRAME_H
#define INTERFACEFRAME_H

#include <QString>
#include <vtkAxesActor.h>
#include <memory>

namespace camitk {

class FrameOfReference;
class Transformation;

/**
 * @ingroup group_sdk_libraries_core_component
 *
 * @brief
 * This class describes the methods to implement in order to manage a Component position in space.
 *
 * Each Component has a frame of reference which is used to define its relation to other objects.
 * You can define new frames and transformations between frames as required, but
 * all FrameOfReference and Transformation objects must be managed by the TransformationManager.
 *
 * @see TransformationManager
 *
 */
class InterfaceFrame {

public:
    /// empty virtual destructor, to avoid memory leak
    virtual ~InterfaceFrame() = default;

    /// Set the FrameOfReference of this object.
    /// Note that this methods will take ownership of the given frame thanks to the shared_ptr.
    virtual void setFrame(const std::shared_ptr<FrameOfReference>& frame) = 0;

    /// Get the pointer to this object's FrameOfReference.
    /// \note Please use TransformationManager::getFrameOfReferenceOwnership(FrameOfReference*)
    // in conjunction with this method if you need to share the ownership of this frame
    virtual const FrameOfReference* getFrame() const = 0;

    /// Get all FrameOfReference owned by this object
    /// @arg includeChildrenFrames Include the frames of this object's children along with its own
    ////@return A multimap that associates each FrameOfReference to the objects that own it
    virtual QMultiMap<const FrameOfReference*, Component*> getAllFrames(bool includeChildrenFrames = true) = 0;

    /// Get all Transformation owned by this object
    /// @arg includeChildrenTransformations Include the Transformation of this object's children along with its own
    ////@return A multimap that associates each Transformation to the objects that own it
    virtual QMultiMap<const Transformation*, Component*> getAllTransformations(bool includeChildrenTransformations = true) = 0;

    /// Modify this object's frame using the given object's frame.
    /// \note you can reimplement this method if you need to manage more than this frame of reference (@see ImageComponent::setFrameFrom())
    virtual void setFrameFrom(const InterfaceFrame*) = 0;

    /// Reset this object's FrameOfReference, that is call setFrame with a newly created frame of reference.
    /// \note you can reimplement this method if you need to manage more than this frame of reference (@see ImageComponent::setFrameFrom())
    virtual void resetFrame() = 0;

    /// add a 3D Actor representing the Frame
    /// FIXME \note this is temporary, the frame axis actor should be managed otherwise
    /// as the component might not be the sole owner of its frame
    virtual vtkSmartPointer<vtkAxesActor> getFrameAxisActor(QString) = 0;

    /// get the visibility of the Frame axis actor
    /// FIXME \note this is temporary, the frame axis actor should be managed otherwise
    /// as the component might not be the sole owner of its frame
    virtual bool getFrameVisibility(QString) const = 0;

    /// set the visibility of the Frame axis actor
    /// FIXME \note this is temporary, the frame axis actor should be managed otherwise
    /// as the component might not be the sole owner of its frame
    virtual void setFrameVisibility(QString, bool) = 0;

};

}

#endif // INTERFACEFRAME_H
