/*
 *                            COPYRIGHT
 *
 *  sch-rnd - modular/flexible schematics editor - Osmond netlist export
 *  Copyright (C) 2025 Tibor 'Igor2' Palinkas
 *  Copyright (C) 2025 Aron Barath
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 31 Milk Street, # 960789 Boston, MA 02196 USA.
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <librnd/core/compat_misc.h>
#include <librnd/core/safe_fs.h>
#include <librnd/core/plugins.h>
#include <librnd/core/error.h>
#include <libcschem/config.h>
#include <libcschem/plug_io.h>

#include <plugins/lib_netlist_exp/lib_netlist_exp.h>

static csch_plug_io_t eosmond_net;

static int osmond_export_prio(const char *fn, const char *fmt, csch_plug_io_type_t type)
{
	if (type != CSCH_IOTYP_NETLIST)
		return 0;
	if (rnd_strcasecmp(fmt, "osmond") == 0)
		return 100;
	return 0;
}

static void osmond_print_footprint(FILE* const f, const char* const str)
{
	if(str)
	{
		/* spaces are allowed in footprint */
		if(strpbrk(str, "{}"))
		{
			rnd_message(RND_MSG_ERROR, "osmond: Broken output! Footprint contains illegal character, but it cannot be escaped: %s\n", str);
		}

		fputs(str, f);
	}
	else
	{
		rnd_message(RND_MSG_ERROR, "osmond: Broken output! Footprint is empty!\n");
	}
}

/* print a name that is part of a net ("Signal") */
static void osmond_print_netpart__(FILE* const f, const char* const str)
{
	if(str)
	{
		if(strpbrk(str, "\" {}-"))
		{
			rnd_message(RND_MSG_ERROR, "osmond: Broken output! Name contains illegal character, but it cannot be escaped: %s\n", str);
		}

		fputs(str, f);
	}
	else
	{
		rnd_message(RND_MSG_ERROR, "osmond: Broken output! Name is empty!\n");
	}
}

static void osmond_print_refdes(FILE* const f, const char* const str)
{
	osmond_print_netpart__(f, str);
}

static void osmond_print_pinnum(FILE* const f, const char* const str)
{
	osmond_print_netpart__(f, str);
}

/* print a quoted name with quote symbols */
static void osmond_print_quoted(FILE* const f, const char* const str)
{
	fputc('\"', f);

	if(str)
	{
		if(strchr(str, '\"'))
		{
			rnd_message(RND_MSG_ERROR, "osmond: Broken output! String contains double-qute (\"), but it cannot be escaped: %s\n", str);
		}

		fputs(str, f);
	}

	fputc('\"', f);
}

/* Export abstract components; exports refdes and footprint */
static void osmond_export_comps(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;
	for(e = htsp_first(&abs->comps); e != NULL; e = htsp_next(&abs->comps, e)) {
		csch_acomp_t *comp = e->value;
		const char *refdes = sch_nle_get_refdes(comp);
		const char *fp;

		if (refdes == NULL)
			continue;

		if (comp->hdr.omit)
			continue;

		/* Get main symbol attributes the safe way, considering alternate names;
		   these should be always exported and are usually hardwiared in the
		   netlist format. */
		fp = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_FOOTPRINT);

		fputs("Part ", f);
		osmond_print_footprint(f, fp);
		fputs(" { Name ", f);
		osmond_print_refdes(f, refdes);
		fputs(" }\n", f);
	}
}

/* Export abstract nets; exports net's name and a list of refdes-pinnum */
 /*pairs connected */
static void osmond_export_nets(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;
	long n;

	for(e = htsp_first(&abs->nets); e != NULL; e = htsp_next(&abs->nets, e)) {
		csch_anet_t *net = e->value;
		const char *netname = sch_nle_get_netname(net);
		int net_exported = 0; /* net had at least one connection so it was written */

		if (net->hdr.omit) continue;

		for(n = 0; n < net->conns.used; n++) {
			csch_aport_t *port = net->conns.array[n];
			const char *refdes = NULL, *pinnums;

			if (port->hdr.type != CSCH_ATYPE_PORT) {
				rnd_message(RND_MSG_ERROR, "osmond: invalid connection (object type)\n");
				continue;
			}

			pinnums = sch_nle_get_pinnum(port);
			if (pinnums == NULL) {
				rnd_message(RND_MSG_ERROR, "osmond: can't determine port's pin number\n");
				continue;
			}

			if (port->parent != NULL) {
				refdes = sch_nle_get_refdes(port->parent);
				if (port->parent->hdr.omit)
					continue; /* omit component */
			}
			if (refdes == NULL) {
				/* This is not an error: no refdes means: do not export (e.g. gnd) */
/*				rnd_message(RND_MSG_ERROR, "osmond: can't determine port's parent component refdes\n");*/
				continue;
			}

			/* split up pinnum at space and create one or more conn lines connecting
			   each pin to the given net */
			SCH_NLE_FOREACH_PINNUM(pinnums, my_num,
				{
					if(!net_exported)
					{
						fputs("Signal ", f);
						osmond_print_quoted(f, netname);
						fputs("\n  {", f);

						net_exported = 1;
					}

					fputc(' ', f);
					osmond_print_refdes(f, refdes);
					fputc('-', f);
					osmond_print_pinnum(f, my_num);
				}
			);
		}

		/* If the net got exported, close the net export */
		if(net_exported)
		{
			fputs(" }\n", f);
		}
	}
}


/* Export netlist from the abstract model */
static int osmond_export_project_abst(const char *fn, const char *fmt, csch_abstract_t *abs, rnd_hid_attr_val_t *options)
{
	TODO("get hidlib as an arg")
	rnd_design_t *hidlib = NULL;
	FILE *f = rnd_fopen(hidlib, fn, "w");
	if (f == NULL)
		return -1;

	osmond_export_comps(f, abs);
	osmond_export_nets(f, abs);

	fclose(f);
	return 0;
}

#include "hid_impl.c"

int pplg_check_ver_export_osmond(int ver_needed) { return 0; }

void pplg_uninit_export_osmond(void)
{
	csch_plug_io_unregister(&eosmond_net);
	rnd_export_remove_opts_by_cookie(osmond_cookie);
	rnd_hid_remove_hid(&osmond_hid);
}

int pplg_init_export_osmond(void)
{
	RND_API_CHK_VER;

	eosmond_net.name = "export to Osmond";
	eosmond_net.export_prio = osmond_export_prio;
	eosmond_net.export_project_abst = osmond_export_project_abst;
	eosmond_net.ext_export_project = ".net";
	csch_plug_io_register(&eosmond_net);


	rnd_hid_nogui_init(&osmond_hid);

	osmond_hid.struct_size = sizeof(rnd_hid_t);
	osmond_hid.name = "osmond";
	osmond_hid.description = "Exports project's Osmond netlist";
	osmond_hid.exporter = 1;

	osmond_hid.get_export_options = osmond_get_export_options;
	osmond_hid.do_export = osmond_do_export;
	osmond_hid.parse_arguments = osmond_parse_arguments;
	osmond_hid.argument_array = osmond_values;

	osmond_hid.usage = osmond_usage;

	rnd_hid_register_hid(&osmond_hid);
	rnd_hid_load_defaults(&osmond_hid, osmond_options, NUM_OPTIONS);


	return 0;
}

