/*
 * Copyright (C) 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "java/ClassDefinition.h"

#include "androidfw/StringPiece.h"

using android::StringPiece;

namespace aapt {

void ClassMember::WriteToStream(const StringPiece& prefix, bool final, std::ostream* out) const {
  processor_.WriteToStream(out, prefix);
}

void MethodDefinition::AppendStatement(const StringPiece& statement) {
  statements_.push_back(statement.to_string());
}

void MethodDefinition::WriteToStream(const StringPiece& prefix, bool final,
                                     std::ostream* out) const {
  *out << prefix << signature_ << " {\n";
  for (const auto& statement : statements_) {
    *out << prefix << "  " << statement << "\n";
  }
  *out << prefix << "}";
}

ClassDefinition::Result ClassDefinition::AddMember(std::unique_ptr<ClassMember> member) {
  Result result = Result::kAdded;
  auto iter = indexed_members_.find(member->GetName());
  if (iter != indexed_members_.end()) {
    // Overwrite the entry.
    ordered_members_[iter->second].reset();
    result = Result::kOverridden;
  }

  indexed_members_[member->GetName()] = ordered_members_.size();
  ordered_members_.push_back(std::move(member));
  return result;
}

bool ClassDefinition::empty() const {
  for (const std::unique_ptr<ClassMember>& member : ordered_members_) {
    if (member != nullptr && !member->empty()) {
      return false;
    }
  }
  return true;
}

void ClassDefinition::WriteToStream(const StringPiece& prefix, bool final,
                                    std::ostream* out) const {
  if (empty() && !create_if_empty_) {
    return;
  }

  ClassMember::WriteToStream(prefix, final, out);

  *out << prefix << "public ";
  if (qualifier_ == ClassQualifier::kStatic) {
    *out << "static ";
  }
  *out << "final class " << name_ << " {\n";

  std::string new_prefix = prefix.to_string();
  new_prefix.append(kIndent);

  for (const std::unique_ptr<ClassMember>& member : ordered_members_) {
    // There can be nullptr members when a member is added to the ClassDefinition
    // and takes precedence over a previous member with the same name. The overridden member is
    // set to nullptr.
    if (member != nullptr) {
      member->WriteToStream(new_prefix, final, out);
      *out << "\n";
    }
  }

  *out << prefix << "}";
}

constexpr static const char* sWarningHeader =
    "/* AUTO-GENERATED FILE. DO NOT MODIFY.\n"
    " *\n"
    " * This class was automatically generated by the\n"
    " * aapt tool from the resource data it found. It\n"
    " * should not be modified by hand.\n"
    " */\n\n";

bool ClassDefinition::WriteJavaFile(const ClassDefinition* def,
                                    const StringPiece& package, bool final,
                                    std::ostream* out) {
  *out << sWarningHeader << "package " << package << ";\n\n";
  def->WriteToStream("", final, out);
  return bool(*out);
}

}  // namespace aapt
