package deepinid

import (
	"fmt"
	"time"

	"pkg.deepin.io/daemon/sync/infrastructure/hardware"
	"pkg.deepin.io/daemon/sync/infrastructure/log"

	"github.com/fatih/structs"
	"github.com/godbus/dbus"
	"github.com/mitchellh/mapstructure"
	"pkg.deepin.io/daemon/sync/infrastructure/helper"
	"pkg.deepin.io/daemon/sync/infrastructure/utils"
)

const (
	dbusPropUserInfo    = "UserInfo"
	dbusPropMachineName = "MachineName"
)

// SetToken check token verify
func (d *DeepinID) SetToken(tokenInfo map[string]interface{}) *dbus.Error {
	log.Info("call", tokenInfo)

	token := fmt.Sprint(tokenInfo["token"])
	accessToken := fmt.Sprint(tokenInfo["access_token"])
	expiry, _ := time.Parse(time.RFC3339, fmt.Sprint(tokenInfo["expiry"]))

	method := d.GetDBusInterface() + ".SetToken"
	if len(token) == 0 || len(accessToken) == 0 {
		return dbus.NewError(method, []interface{}{"empty token"})
	}

	err := d.updateToken(token, accessToken, expiry.Unix())
	if err != nil {
		return dbus.NewError(method, []interface{}{err.Error()})
	}

	return nil
}

// GetHardware return hardware info
func (d *DeepinID) GetHardware() (*hardware.Hardware, *dbus.Error) {
	return d.hardware, nil
}

func (d *DeepinID) HasConfirmPrivacy(id string) (bool, *dbus.Error) {
	ret, err := helper.HasConfirmPrivacy(id)
	if nil != err {
		return false, dbus.NewError(dbusIfc+".HasConfirmPrivacy", []interface{}{err.Error()})
	}
	return ret, nil
}

func (d *DeepinID) ConfirmPrivacy(id string) ( *dbus.Error) {
	err := helper.ConfirmPrivacy(id)
	if nil != err {
		return  dbus.NewError(dbusIfc+".ConfirmPrivacy", []interface{}{err.Error()})
	}
	return  nil
}

// GetToken return hardware info
func (d *DeepinID) GetToken() (string, *dbus.Error) {
	td := &tokenData{}
	mapstructure.Decode(d.userInfo, td)
	if td.isExpired() {
		//refresh
		log.Info("token expired, get new")
		d.refreshToken(td.Token, td.AccessToken, td.Expiry)
		mapstructure.Decode(d.userInfo, td)
		return td.AccessToken, nil
	}
	return td.AccessToken, nil

}

func (d *DeepinID) isLogin() bool {
	td := d.getTokenData()
	return td.IsLoggedIn
}

// Login call dialog to login
func (d *DeepinID) Login() *dbus.Error {
	method := d.GetDBusInterface() + ".Login"

	if d.isLogin() {
		return dbus.NewError(method, []interface{}{fmt.Sprint("user has login")})
	}

	err := launchLoginApp()
	if err != nil {
		return dbus.NewError(method, []interface{}{err.Error()})
	}
	return nil
}

// Logout remove login info
func (d *DeepinID) Logout() *dbus.Error {
	method := d.GetDBusInterface() + ".Logout"

	tk := d.getTokenData()
	if !tk.IsLoggedIn {
		return dbus.NewError(method, []interface{}{fmt.Sprint("user has NOT login")})
	}

	err := doLogout(tk.AccessToken)
	if err != nil {
		log.Error("Failed to do logout:", err)
	}

	// remove token
	err = removeToken()
	if err != nil {
		return dbus.NewError(method, []interface{}{err.Error()})
	}

	// drop sync cache
	uid := d.userInfo["UserID"]
	if v, ok := uid.(int64); ok {
		err := utils.DropLocalCache(v)
		if err != nil {
			log.Warning("Failed to drop user sync cache:", err)
		}
	}

	d.SetMapProp(&d.userInfo, dbusPropUserInfo, structs.Map(&tokenData{}))
	return nil
}

// RevokeAccess remove login info
func (d *DeepinID) RevokeAccess(accessToken string) *dbus.Error {
	method := d.GetDBusInterface() + ".RevokeAccess"

	if !d.isLogin() {
		return dbus.NewError(method, []interface{}{fmt.Sprint("user has NOT login")})
	}

	td := d.getTokenData()
	if td.AccessToken != accessToken {
		return dbus.NewError(method, []interface{}{fmt.Sprint("user accessToken not match")})
	}

	td.Expiry = time.Now().Unix()

	m := structs.Map(td)
	d.SetMapProp(&d.userInfo, dbusPropUserInfo, m)

	td.AccessToken = ""
	saveToken(td)
	return nil
}

// func (d *DeepinID) Debug() *dbus.Error {
// 	td := d.getTokenData()
// 	userInfo := structs.Map(td)
// 	userInfo["tttttt"] = time.Now().Unix()
// 	d.SetMapProp(&d.userInfo, dbusPropUserInfo, userInfo)
// 	return nil

// }
