@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
perl -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/usr/bin/perl -w
#line 15
#
# ecaccess-ectrans-request: Request a new ECtrans transfer
#
# Laurent.Gougeon@ecmwf.int - 2010-10-15

use ECMWF::ECaccess;
use Getopt::Long;
use Pod::Usage;

my %opt = (
	gateway         => undef,
	target          => undef,
	mailTo          => undef,
	retryCount      => 144,
	retryFrequency  => 600,
	priority        => 99,
	lifeTime        => '7d',
	onSuccess       => 0,
	onFailure       => 0,
	onRetry         => 0,
	deleteSource    => 0,
	keep            => 0,
	reject          => 0,
	append          => 0,
	resume          => 0,
	overwrite       => 0,
	maximumDuration => undef,
	minimumDuration => undef,
	minimumRate     => undef,
	version         => 0,
	help            => 0,
	manual          => 0,
	retry			=> 0,
	debug           => 0
);

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if !GetOptions(
	\%opt,
	qw(
	  gateway=s
	  target=s
	  mailTo=s
	  retryCount=i
	  retryFrequency=i
	  priority=i
	  lifeTime=s
	  onSuccess
	  onFailure
	  onRetry
	  deleteSource
	  keep
	  reject
	  append
	  resume
	  overwrite
	  maximumDuration=s
	  minimumDuration=s
	  minimumRate=i
	  version
	  help|?
	  manual
	  retry=i
	  debug
	  )
);

# Display version if requested
die ECMWF::ECaccess->VERSION . "\n" if ( $opt{version} );

my $remote = $ARGV[0];
my $source = $ARGV[1];

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if ( $opt{help} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 2 ) if ( $opt{manual} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "No remote specified!\n" )                    if not($remote);
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "No source specified!\n" )                    if not($source);
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "Invalid -retryCount specified (>0)!\n" )     if not( $opt{retryCount} >= 0 );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "Invalid -retryFrequency specified (>0)!\n" ) if not( $opt{retryFrequency} > 0 );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "Invalid -priority specified (0-99)!\n" ) if not( $opt{priority} >= 0 && $opt{priority} <= 99 );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "Incompatible options (-append,-resume)!\n" )    if ( $opt{append} && $opt{resume} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "Incompatible options (-append,-overwrite)!\n" ) if ( $opt{append} && $opt{overwrite} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "Incompatible options (-append,-reject)!\n" )    if ( $opt{append} && $opt{reject} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "Incompatible options!(-resume,-overwrite)\n" )  if ( $opt{resume} && $opt{overwrite} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "Incompatible options!(-resume,-reject)\n" )     if ( $opt{resume} && $opt{reject} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "Incompatible options!(-reject,-overwrite)\n" )  if ( $opt{reject} && $opt{overwrite} );

# Create the ECaccess Controler
my $ecaccess = ECMWF::ECaccess->new( $opt{retry}, $opt{debug});

# Get the Token (using the Certificate in $HOME)
my $token = $ecaccess->getToken();

# Get the Control Channel
my $controlChannel = $ecaccess->getControlChannel();

# Value for "ifTargetFileExists" option
my $ifTargetFileExists = 'reject';
$ifTargetFileExists = 'append'    if ( $opt{append} );
$ifTargetFileExists = 'resume'    if ( $opt{resume} );
$ifTargetFileExists = 'overwrite' if ( $opt{overwrite} );

# Request for the new ECTrans
$ectransId = $controlChannel->requestTransfer(
	$token,
	SOAP::Data->name(
		"request" => \SOAP::Data->value(
			SOAP::Data->name( 'gatewayName'        => $opt{gateway} ),
			SOAP::Data->name( 'remoteLocation'     => $remote ),
			SOAP::Data->name( 'sourceFileName'     => $source ),
			SOAP::Data->name( 'targetFileName'     => $opt{target} ),
			SOAP::Data->name( 'retryCount'         => $opt{retryCount} ),
			SOAP::Data->name( 'retryFrequency'     => $opt{retryFrequency} ),
			SOAP::Data->name( 'priority'           => $opt{priority} ),
			SOAP::Data->name( 'lifeTime'           => $opt{lifeTime} ),
			SOAP::Data->name( 'maximumDuration'    => $opt{maximumDuration} ),
			SOAP::Data->name( 'minimumDuration'    => $opt{minimumDuration} ),
			SOAP::Data->name( 'minimumRate'        => $opt{minimumRate} ),
			SOAP::Data->name( 'deleteSource'       => $opt{deleteSource} ? 'true' : 'false' )->type('xsd:boolean'),
			SOAP::Data->name( 'keepInSpool'        => $opt{keep} ? 'true' : 'false' )->type('xsd:boolean'),
			SOAP::Data->name( 'userMailAddress'    => $opt{mailTo} ),
			SOAP::Data->name( 'sendMailOnSuccess'  => $opt{onSuccess} ? 'true' : 'false' )->type('xsd:boolean'),
			SOAP::Data->name( 'sendMailOnFailure'  => $opt{onFailure} ? 'true' : 'false' )->type('xsd:boolean'),
			SOAP::Data->name( 'sendMailOnRetry'    => $opt{onRetry} ? 'true' : 'false' )->type('xsd:boolean'),
			SOAP::Data->name( 'ifTargetFileExists' => $ifTargetFileExists )
		)
	)
)->result;
print $ectransId. "\n";

# Logout
$ecaccess->releaseToken($token);

__END__

=head1 NAME

ecaccess-ectrans-request - Request a new ECtrans transfer

=head1 SYNOPSIS

B<ecaccess-ectrans-request -version|-help|-manual>

B<ecaccess-ectrans-request [-debug] [-gateway> I<name>B<] [-target> I<file-name>B<] [-retryCount> I<number>B<] [-retryFrequency> I<frequency>B<]
[-priority> I<priority>B<] [-lifeTime> I<duration>B<] [-mailTo> I<email>B<] [-onSuccess] [-onFailure] [-onRetry] [-deleteSource] [-keep]
[-reject>|B<-append>|B<-resume>|B<-overwrite] [-maximumDuration> I<duration>B<] [-minimumDuration> I<duration>B<] [-minimumRate> I<rate>B<]>
I<association-name>B<[>I<@protocol>B<]> I<source>

=head1 DESCRIPTION

Allow Member State users to initiate file transfers between ECMWF and Member State servers. The ECaccess Server
will spool the ECMWF file specified by the I<source> parameter in the user's ECtrans transfer queue: if the connection
between the ECMWF and Member State gateways is down or if any error occurs, the file will be kept in the spool area at
ECMWF and you can resume the transfer through the web interface or with the B<ecaccess-ectrans-restart> command. If
required the transfer can be retried automatically (see the B<-retryCount> and B<-retryFrequency> options to tailor the
retry mechanism).

The B<-reject>, B<-append>, B<-resume> and B<-overwrite> options are mutually exclusive and determine what to do if there is an
existing target file. The B<-mailTo> option specifies an I<email> address to be notified in case of a successful (option
B<-onSuccess>) and/or a failed transfer (option B<-onFailure>).

When a request has been spooled successfully, an I<ectrans-id> is returned immediately. The I<ectrans-id> can be used to reference
the transfer, using the web interface or with the B<ecaccess-ectrans-delete>, B<ecaccess-ectrans-list> or B<ecaccess-ectrans-restart>
commands.

=head1 ARGUMENTS

=over 8

=item I<association-name>B<[>I<@protocol>B<]>

The ECtrans Association with optionaly the name of the protocol to use.

=item I<source>

Name of the file at ECMWF to spool in ECtrans.

=back

=head1 OPTIONS

=over 8

=item B<-gateway> I<name>

This is the I<name> of the ECaccess Gateway where the ECtrans Association I<association-name>
is defined. It is by default the Gateway you are connected to. In order to get the name
of your current Gateway you can use the B<ecaccess-gateway-name> command. When using
the commands at ECMWF the default Gateway is always "ecaccess.ecmwf.int".

=item B<-target> I<file-name>

Defines the target I<file-name> (default: same as I<source>).

=item B<-retryCount> I<number>

Defines the I<number> of retries (default: 144).

=item B<-retryFrequency> I<frequency>

Defines the I<frequency> of retries in seconds (default: 600 seconds).

=item B<-priority> I<priority>

Defines the transmission I<priority> 0-99 (default: 99).

=item B<-lifeTime> I<duration>

Defines the lifetime of the request (default: 7 days). The I<duration> is specified
in [w]eeks, [d]ays, [h]ours, [m]inutes or [s]econds (e.g. I<1w> or I<2d>).

=item B<-mailTo> I<email>

Defines the target I<email> address (default: <user-id>@ecmwf.int).

=item B<-onSuccess>

Allow sending a mail when the transfer is successful.

=item B<-onFailure>

Allow sending a mail when the transfer has failed.

=item B<-onRetry>

Allow sending a mail when the transfer is retried.

=item B<-deleteSource>

Allow deleting the original I<source> file once it is safe in the spool.

=item B<-keep>

Allow keeping the request in the spool until the duration specified in the
B<-lifeTime> option has passed. By default a successful transfer is removed
from the spool.

=item B<-reject>

Allow failing if there is an existing target file (default).

=item B<-append>

Allow appending if there is an existing target file.

=item B<-resume>

Allow resuming if there is an existing target file.

=item B<-overwrite>

Allow overwriting if there is an existing target file.

=item B<-maximumDuration> I<duration>

Define the maximum transfer I<duration>. The I<duration> is specified in [w]eeks,
[d]ays, [h]ours, [m]inutes or [s]econds (e.g. I<10m> or I<1h>). A value <=0 will
deactivate the feature (default: 12h).

=item B<-minimumDuration> I<duration>

Allow setting the I<duration> before to control a transmission (e.g. for the first 10 minutes
ECtrans don't try to check the transfer rate, which allow not to enforce some minimum transfer
rates for very small files). The I<duration> is specified in [w]eeks, [d]ays, [h]ours,
[m]inutes or [s]econds (e.g. I<30s> or I<10m>). A value <=0 will deactivate the feature
(default: 10m).

=item B<-minimumRate> I<rate>

Allow setting a minimum transfer I<rate> for a transmission (expressed in bytes/s). This is an
average value, not the transmission I<rate> at one particular moment in time (e.g. if a big file
is stuck for 5 minutes but then transmit much faster later on then it will not be interrupted
as long as the overall transfer rate is above the minimum). A value <=0 will deactivate the
feature (default: 10240).

=item B<-version>

Display version number and exits.

=item B<-help>

Print a brief help message and exits.

=item B<-manual>

Prints the manual page and exits.

=item B<-retry> I<count>

Number of SSL connection retries per 5s to ECMWF. This parameter only apply to the
initial SSL connection initiated by the command to the ECMWF server. It does not
apply to all the subsequent requests made afteward as it is mainly targeting errors
that can happen from time to time during the SSL handshake. Default is no retry.

=item B<-debug>

Display the SOAP and SSL messages exchanged.

=back

=head1 EXAMPLES

B<ecaccess-ectrans-request -lifeTime> I<2d> B<-overwrite -onFailure -onRetry> I<test@genericFtp> I<ec:test.txt>

Request a file transfer of the I<ec:test.txt> file to the default gateway with the association I<test> and the
protocol I<genericFtp>. If the target file already exists then the original file is deleted. If the transfer is retried
or fail then an email is sent to the default email address (<user-id>@ecmwf.int). After 2 days the file is removed from
the spool (whatever the status of the transmission is).

B<ecaccess-ectrans-request -minimumDuration> I<10m> B<-maximumDuration> I<35m> B<-minimumRate> I<8192> I<test@genericFtp>
I<ec:test.txt>

After 10 minutes ECtrans will start to control the transfer rate. If after 35 minutes the file is still not transmitted or
if at a certain point in time the average transmission rate is less than 8Kbytes/s then the transmission will be canceled.
If canceled, the transmission will be retried 144 times (with a 10 minutes interval).

=head1 SEE ALSO

B<ecaccess-ectrans-delete>, B<ecaccess-ectrans-list>, B<ecaccess-ectrans-restart> and B<ecaccess>.

=cut

__END__
:endofperl
