/*
 * Copyright (C) 2013 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef UniStdExtras_h
#define UniStdExtras_h

#include <errno.h>
#include <unistd.h>

namespace PurCWTF {

WTF_EXPORT_PRIVATE bool setCloseOnExec(int fileDescriptor);
WTF_EXPORT_PRIVATE int dupCloseOnExec(int fileDescriptor);

inline int closeWithRetry(int fileDescriptor)
{
    int ret;
#if OS(LINUX)
    // Workaround for the Linux behavior of closing the descriptor
    // unconditionally, even if the close() call is interrupted.
    // See https://bugs.webkit.org/show_bug.cgi?id=117266 for more
    // details.
    if ((ret = close(fileDescriptor)) == -1 && errno == EINTR)
        return 0;
#else
    while ((ret = close(fileDescriptor)) == -1 && errno == EINTR) { }
#endif
    return ret;
}

WTF_EXPORT_PRIVATE bool setNonBlock(int fileDescriptor);

} // namespace PurCWTF

using PurCWTF::closeWithRetry;
using PurCWTF::dupCloseOnExec;
using PurCWTF::setCloseOnExec;
using PurCWTF::setNonBlock;

#endif // UniStdExtras_h
