/*
 *  preferences.h  -  program preference settings
 *  Program:  kalarm
 *  Copyright © 2001-2018 by David Jarvie <djarvie@kde.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef PREFERENCES_H
#define PREFERENCES_H

#include "kalarm.h"

#include "kalarmconfig.h"
#include <kalarmcal/kadatetime.h>

#include <QObject>
#include <QDateTime>
#include <QTimeZone>

using namespace KAlarmCal;

class QTimeZone;
namespace KHolidays { class HolidayRegion; }


// Settings configured in the Preferences dialog
class Preferences : public PreferencesBase
{
        Q_OBJECT
    public:
        enum MailFrom   { MAIL_FROM_KMAIL, MAIL_FROM_SYS_SETTINGS, MAIL_FROM_ADDR };

        static Preferences*     self();
        static void             connect(const char* signal, const QObject* receiver, const char* member);
        static int              autoHideSystemTray();
        static void             setAutoHideSystemTray(int timeout);
        static bool             autoStartChangedByUser()         { return mAutoStartChangedByUser; }
        static void             setAutoStartChangedByUser(bool c){ mAutoStartChangedByUser = c; }

        // Access to settings
        static QString          previousVersion()                { return mPreviousVersion; }
        static Backend          previousBackend()                { return mPreviousBackend; }
        static void             setAskAutoStart(bool yes);
        static bool             noAutoStart()                    { return self()->base_NoAutoStart(); }
        static void             setNoAutoStart(bool yes);
        static KADateTime::Spec timeSpec();
        static QTimeZone        timeSpecAsZone();
        static void             setTimeSpec(const KADateTime::Spec&);
        static const KHolidays::HolidayRegion& holidays();
        static void             setHolidayRegion(const QString& regionCode);
        static QTime            startOfDay()                     { return self()->mBase_StartOfDay.time(); }
        static void             setStartOfDay(const QTime&);
        static QTime            workDayStart()                   { return self()->mBase_WorkDayStart.time(); }
        static QTime            workDayEnd()                     { return self()->mBase_WorkDayEnd.time(); }
        static QBitArray        workDays();
        static void             setWorkDayStart(const QTime& t)  { self()->setBase_WorkDayStart(QDateTime(QDate(1900,1,1), t)); }
        static void             setWorkDayEnd(const QTime& t)    { self()->setBase_WorkDayEnd(QDateTime(QDate(1900,1,1), t)); }
        static void             setWorkDays(const QBitArray&);
        static bool             quitWarn()                       { return mUsingDefaults ? self()->base_QuitWarn() : notifying(QUIT_WARN); }
        static void             setQuitWarn(bool yes)            { setNotify(QUIT_WARN, yes); }
        static bool             confirmAlarmDeletion()           { return mUsingDefaults ? self()->base_ConfirmAlarmDeletion() : notifying(CONFIRM_ALARM_DELETION); }
        static void             setConfirmAlarmDeletion(bool yes){ setNotify(CONFIRM_ALARM_DELETION, yes); }
        static bool             emailCopyToKMail()               { return self()->mBase_EmailCopyToKMail  &&  self()->mEmailClient == sendmail; }
        static void             setEmailCopyToKMail(bool yes)    { self()->setBase_EmailCopyToKMail(yes); }
        static bool             emailQueuedNotify()              { return mUsingDefaults ? self()->base_EmailQueuedNotify() : notifying(EMAIL_QUEUED_NOTIFY); }
        static void             setEmailQueuedNotify(bool yes)   { setNotify(EMAIL_QUEUED_NOTIFY, yes); }
        static MailFrom         emailFrom();
        static QString          emailAddress();
        static void             setEmailAddress(MailFrom, const QString& address);
        static MailFrom         emailBccFrom();
        static QString          emailBccAddress();
        static void             setEmailBccAddress(bool useSystemSettings, const QString& address);
        static bool             emailBccUseSystemSettings();
        static QString          cmdXTermCommand();
        static void             setCmdXTermCommand(const QString& cmd);
        static float            defaultSoundVolume()             { int vol = self()->mBase_DefaultSoundVolume; return (vol < 0) ? -1 : static_cast<float>(vol) / 100; }
        static void             setDefaultSoundVolume(float v)   { self()->setBase_DefaultSoundVolume(v < 0 ? -1 : static_cast<int>(v * 100)); }

        // Config file entry names for notification messages
        static const QLatin1String QUIT_WARN;
        static const QLatin1String ASK_AUTO_START;
        static const QLatin1String CONFIRM_ALARM_DELETION;
        static const QLatin1String EMAIL_QUEUED_NOTIFY;

        bool                    useDefaults(bool def) override   { mUsingDefaults = def;  return PreferencesBase::useDefaults(def); }

    Q_SIGNALS:
        void                    timeZoneChanged(const KADateTime::Spec& newTz);
        void                    holidaysChanged(const KHolidays::HolidayRegion& newHolidays);
        void                    startOfDayChanged(const QTime& newStartOfDay);
        void                    workTimeChanged(const QTime& startTime, const QTime& endTime, const QBitArray& workDays);

    private Q_SLOTS:
        void                    timeZoneChange(const QString&);
        void                    holidaysChange(const QString& regionCode);
        void                    startDayChange(const QDateTime&);
        void                    workTimeChange(const QDateTime&, const QDateTime&, int days);

    private:
        Preferences();         // only one instance allowed
        static int              startOfDayCheck(const QTime&);
        static void             setNotify(const QString& messageID, bool notify);
        static bool             notifying(const QString& messageID);

        static Preferences*     mInstance;
        static bool             mUsingDefaults;
        static KHolidays::HolidayRegion* mHolidays;
        static QString          mPreviousVersion;  // last KAlarm version which wrote the config file
        static Backend          mPreviousBackend;  // backend used by last used version of KAlarm

        // All the following members are accessed by the Preferences dialog classes
        static int              mMessageButtonDelay;  // 0 = scatter; -1 = no delay, no scatter; >0 = delay, no scatter

        // Change tracking
        static bool             mAutoStartChangedByUser; // AutoStart has been changed by the user
};

#endif // PREFERENCES_H

// vim: et sw=4:
