import unittest
import mock
import os
import shutil

import koji
import kojihub
from koji import GenericError


class TestCGImporter(unittest.TestCase):
    TMP_PATH = os.path.join(os.path.dirname(__file__), 'tmptest')

    def setUp(self):
        if not os.path.exists(self.TMP_PATH):
            os.mkdir(self.TMP_PATH)

    def tearDown(self):
        if os.path.exists(self.TMP_PATH):
            shutil.rmtree(self.TMP_PATH)

    def test_basic_instantiation(self):
        kojihub.CG_Importer()  # No exception!

    def test_get_metadata_is_instance(self):
        mock_input_val = {'foo': 'bar'}
        x = kojihub.CG_Importer()
        x.get_metadata(mock_input_val, '')
        assert x.raw_metadata
        assert isinstance(x.raw_metadata, str)

    def test_get_metadata_is_not_instance(self):
        x = kojihub.CG_Importer()
        with self.assertRaises(GenericError):
            x.get_metadata(42, '')

    def test_get_metadata_is_none(self):
        x = kojihub.CG_Importer()
        with self.assertRaises(GenericError):
            x.get_metadata(None, '')

    @mock.patch("koji.pathinfo.work")
    def test_get_metadata_missing_json_file(self, work):
        work.return_value = os.path.dirname(__file__)
        x = kojihub.CG_Importer()
        with self.assertRaises(GenericError):
            x.get_metadata('missing.json', 'cg_importer_json')

    @mock.patch("koji.pathinfo.work")
    def test_get_metadata_is_json_file(self, work):
        work.return_value = os.path.dirname(__file__)
        x = kojihub.CG_Importer()
        x.get_metadata('default.json', 'cg_importer_json')
        assert x.raw_metadata
        assert isinstance(x.raw_metadata, str)

    @mock.patch('kojihub.context')
    @mock.patch("koji.pathinfo.work")
    def test_assert_cg_access(self, work, context):
        work.return_value = os.path.dirname(__file__)
        cursor = mock.MagicMock()
        context.session.user_id = 42
        context.cnx.cursor.return_value = cursor
        cursor.fetchall.return_value = [(1, 'foo'), (2, 'bar')]
        x = kojihub.CG_Importer()
        x.get_metadata('default.json', 'cg_importer_json')
        x.assert_cg_access()
        assert x.cgs
        assert isinstance(x.cgs, set)

    @mock.patch("kojihub.get_user")
    @mock.patch('kojihub.context')
    @mock.patch("koji.pathinfo.work")
    def test_prep_build(self, work, context, get_user):
        work.return_value = os.path.dirname(__file__)
        cursor = mock.MagicMock()
        context.cnx.cursor.return_value = cursor
        get_user.return_value = {'id': 123}
        x = kojihub.CG_Importer()
        x.get_metadata('default.json', 'cg_importer_json')
        x.prep_build()
        assert x.buildinfo
        assert isinstance(x.buildinfo, dict)

    @mock.patch('koji.pathinfo.build')
    @mock.patch('os.path.lexists')
    @mock.patch('koji.util.rmtree')
    def test_check_build_dir(self, rmtree, lexists, build):
        path = '/random_path/random_dir'
        build.return_value = path

        x = kojihub.CG_Importer()

        # directory exists
        lexists.return_value = True
        with self.assertRaises(koji.GenericError):
            x.check_build_dir(delete=False)
        rmtree.assert_not_called()

        # directory exists + delete
        lexists.return_value = True
        x.check_build_dir(delete=True)
        rmtree.assert_called_once_with(path)

        # directory doesn't exist
        rmtree.reset_mock()
        lexists.return_value = False
        x.check_build_dir()
        rmtree.assert_not_called()

    @mock.patch('kojihub.get_build')
    @mock.patch("koji.pathinfo.work")
    def test_prep_build_exists(self, work, get_build):
        work.return_value = os.path.dirname(__file__)
        get_build.return_value = True
        x = kojihub.CG_Importer()
        x.get_metadata('default.json', 'cg_importer_json')
        with self.assertRaises(GenericError):
            x.prep_build()

    @mock.patch("kojihub.get_user")
    @mock.patch('kojihub.get_build')
    @mock.patch('kojihub.new_build')
    @mock.patch('kojihub.context')
    @mock.patch("koji.pathinfo.work")
    def test_get_build(self, work, context, new_build_id, get_build, get_user):
        work.return_value = os.path.dirname(__file__)
        cursor = mock.MagicMock()
        context.cnx.cursor.return_value = cursor
        new_build_id.return_value = 42
        get_build.return_value = False
        get_user.return_value = {'id': 123}
        x = kojihub.CG_Importer()
        x.get_metadata('default.json', 'cg_importer_json')
        x.prep_build()
        x.prepped_outputs = []
        get_build.return_value = {'id': 43, 'package_id': 1,
                                  'package_name': 'testpkg',
                                  'name': 'testpkg', 'version': '1.0.1e',
                                  'release': '42.el7', 'epoch': None,
                                  'nvr': 'testpkg-1.0.1-1.fc24',
                                  'state': 'complete', 'task_id': 1,
                                  'owner_id': 1, 'owner_name': 'jvasallo',
                                  'volume_id': 'id-1212', 'volume_name': 'testvolume',
                                  'creation_event_id': '', 'creation_time': '',
                                  'creation_ts': 424242424242,
                                  'start_time': None, 'start_ts': None,
                                  'completion_time': None, 'completion_ts': None,
                                  'source': 'https://example.com', 'extra': {}
                                 }
        new_build_id.return_value = 43
        x.get_build()
        assert x.buildinfo
        assert isinstance(x.buildinfo, dict)

    @mock.patch("koji.pathinfo.build")
    @mock.patch("koji.pathinfo.work")
    def test_import_metadata(self, work, build):
        work.return_value = os.path.dirname(__file__)
        build.return_value = self.TMP_PATH
        x = kojihub.CG_Importer()
        x.get_metadata('default.json', 'cg_importer_json')
        x.import_metadata()


class TestMatchKojiFile(unittest.TestCase):

    def setUp(self):
        self.importer = kojihub.CG_Importer()
        self.archive1 = {
                'id': 99,
                'build_id': 42,
                'checksum': 'e1f95555eae04b8e1ebdc5555c5555f0',
                'checksum_type': 0,
                'filename': 'foo-bar-3.0.jar',
                'size': 42710,
                }
        self.build1 = {
                'id': 79218,
                'nvr': 'foo-3.0-1',
                }
        self.comp1 = {
                'type': 'kojifile',
                'archive_id': self.archive1['id'],
                'nvr': self.build1['nvr'],
                'filename': self.archive1['filename'],
                }
        self.get_archive = mock.patch('kojihub.get_archive').start()
        self.get_build = mock.patch('kojihub.get_build').start()

    def tearDown(self):
        mock.patch.stopall()

    def test_match_kojifile_basic(self):
        comp = self.comp1
        self.get_archive.return_value = self.archive1
        self.get_build.return_value = self.build1
        match = self.importer.match_kojifile(comp)
        self.assertEqual(match, self.archive1)
        self.get_build.assert_called_once()

    def test_match_kojifile_missing_fields(self):
        self.get_archive.return_value = self.archive1
        self.get_build.return_value = self.build1
        # nvr missing
        comp = self.comp1.copy()
        del comp['nvr']
        with self.assertRaises(koji.GenericError):
            self.importer.match_kojifile(comp)
        # filename missing
        comp = self.comp1.copy()
        del comp['filename']
        with self.assertRaises(koji.GenericError):
            self.importer.match_kojifile(comp)

    def test_match_kojifile_mismatch(self):
        comp = self.comp1
        comp['filesize'] = 1
        self.get_archive.return_value = self.archive1
        self.get_build.return_value = self.build1
        with self.assertRaises(koji.GenericError):
            self.importer.match_kojifile(comp)
