/* mingw32 host-specific hook definitions.
   Copyright (C) 2004 Free Software Foundation, Inc.

   This file is part of GCC.

   GCC is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published
   by the Free Software Foundation; either version 2, or (at your
   option) any later version.

   GCC is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
   License for more details.

   You should have received a copy of the GNU General Public License
   along with GCC; see the file COPYING.  If not, write to the
   Free Software Foundation, 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.  */

#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "hosthooks.h"
#include "hosthooks-def.h"
#include "toplev.h"
#include "diagnostic.h"

#define WIN32_LEAN_AND_MEAN
#include <windows.h>

static void * mingw32_gt_pch_get_address (size_t);
static bool mingw32_gt_pch_use_address (void *, size_t);

#undef HOST_HOOKS_GT_PCH_GET_ADDRESS
#define HOST_HOOKS_GT_PCH_GET_ADDRESS mingw32_gt_pch_get_address
#undef HOST_HOOKS_GT_PCH_USE_ADDRESS
#define HOST_HOOKS_GT_PCH_USE_ADDRESS mingw32_gt_pch_use_address

/* Get a few diagnostics.  */
static inline void w32_perror(const char*, const char*, int, const char*);
static inline void dump_mbi (FILE*, void *);

/* FIXME: should this be set using physmem heuristics?  */
static const size_t pch_VA_max_size  = 128 * 1024 * 1024;

/* Pagesize for allocating memory.  */
static const size_t pagesize = 0x1000;

/* Granularity for reserving address space */
static const size_t granularity = 0x10000;


/* Print out the GetLastError() translation.  */ 
static inline void
w32_perror(const char* function, const char* file, int line,
	   const char* my_msg)
{
  LPSTR w32_msgbuf;
  FormatMessageA (FORMAT_MESSAGE_ALLOCATE_BUFFER
		  | FORMAT_MESSAGE_FROM_SYSTEM
		  | FORMAT_MESSAGE_IGNORE_INSERTS
		  | FORMAT_MESSAGE_MAX_WIDTH_MASK,
    		  NULL, GetLastError(),
		  MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
		  (LPSTR) &w32_msgbuf, 0, NULL);
  fprintf(stderr, "internal error in %s, at %s:%d: %s: %s\n",
	  function, trim_filename (file), line, my_msg, w32_msgbuf);
  LocalFree ((HLOCAL)w32_msgbuf);
}

static inline void dump_mbi (FILE* log, void * loc)
{
  MEMORY_BASIC_INFORMATION mbi;
  VirtualQuery (loc, &mbi, sizeof (mbi));
  fprintf (log,
	   "BaseAddress: 0x%x\n"
	   "AllocationBase: 0x%x\n"
	   "RegionSize: 0x%x\n"
	   "State: 0x%x\n"
	   "Protect: 0x%x\n"
	   "Type: 0x%x\n\n",
  	   (unsigned) mbi.BaseAddress,
	   (unsigned) mbi.AllocationBase,
	   (unsigned) mbi.RegionSize,
	   (unsigned) mbi.State,
	   (unsigned) mbi.Protect,
	   (unsigned) mbi.Type);
}


/* Return the address of the PCH address space, if the PCH will fit in it.  */
static void *
mingw32_gt_pch_get_address (size_t sz)
{
  void* res;
  size_t rounded_sz = (sz + pagesize - 1) & ~(pagesize - 1);
  if (rounded_sz > pch_VA_max_size)
    return NULL;

  /* FIXME: We let system determine base by setting first arg to NULL.
     Allocating at top of available address space avoids unnecessary
     fragmentation of "ordinary" (malloc's)  address space but may not be safe
     with delayed load of system dll's. Preferred addresses for NT system
     dlls is in 0x70000000 to 0x78000000 range.  */

  res = VirtualAlloc (NULL, pch_VA_max_size,
		      MEM_RESERVE | MEM_TOP_DOWN,
		      PAGE_NOACCESS);
  if (!res)
    w32_perror (__FUNCTION__, __FILE__, __LINE__, "VirtualAlloc (reserve)");
  else
    /* We do not need the address space for now, so free it.  */
    VirtualFree (res, 0, MEM_RELEASE);

#ifdef PCH_DEBUG
  fprintf (stderr,"'%s' setting address space for 0x%x bytes at base %p\n",
	  __FUNCTION__, pch_VA_max_size, res);  
#endif

  return res; 
}

/* Check ADDR and SZ for validity and allocate  */

static bool
mingw32_gt_pch_use_address (void *addr, size_t sz)
{
  void * va_addr;

  if (sz == 0)
    return false;
  

   /* Round the size up to a whole page size.  Normally this is a no-op.  */
  sz = (sz + pagesize - 1) & ~(pagesize - 1);


  va_addr =  VirtualAlloc (addr, sz, MEM_RESERVE | MEM_COMMIT,
			   PAGE_READWRITE);

#ifdef PCH_DEBUG
  fprintf (stderr, "'%s' committing 0x%x bytes at base %p\n",
	   __FUNCTION__, sz, addr);
  dump_mbi (stderr, addr);
#endif


  if (addr != va_addr)
    {
      w32_perror (__FUNCTION__, __FILE__, __LINE__, "VirtualAlloc (commit)");
      return false;
    }
   
   return true;   
}

const struct host_hooks host_hooks = HOST_HOOKS_INITIALIZER;

