/*
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *  
 *  Copyright (C) 2015 Lukasz Komsta
 */

#include <libnova/lunar.h>
#include <libnova/julian_day.h>
#include <libnova/rise_set.h>
#include <libnova/transform.h>

/*! \fn const char* ln_get_constellation(struct ln_equ_posn *position)
* \param position Equitorial position
*
* \return Name of the the constellation at the given position, or "---" if constellation
* cannot be found.
*
* Returns name of the constellation based on boundaries found at:
* http://vizier.u-strasbg.fr/viz-bin/VizieR?-source=6042
*/
const char* ln_get_constellation(struct ln_equ_posn *position)
{
	static const float	ra_lower[357] =
	{
		0,
		120,
		315,
		270,
		0,
		137.5,
		0,
		160,
		262.5,
		302.5,
		0,
		172.5,
		248,
		302.5,
		119.5,
		137.5,
		195,
		46.5,
		306.25,
		170,
		0,
		210,
		353.75,
		180,
		202.5,
		347.5,
		91.5,
		300,
		308,
		105,
		119.5,
		296.5,
		300,
		343,
		0,
		291.25,
		25.5,
		36.5,
		46.5,
		334.75,
		75,
		210.5,
		216.25,
		47.5,
		332,
		309,
		0,
		91.5,
		181.25,
		228.75,
		329.5,
		50,
		343,
		236.25,
		30.625,
		255,
		0,
		20.5,
		97.5,
		350,
		202.5,
		0,
		353.75,
		272.625,
		273.5,
		286.25,
		25,
		126.25,
		2.5,
		180,
		102,
		328.625,
		328.125,
		287.5,
		137.5,
		152.5,
		231.5,
		236.25,
		138.75,
		0,
		37.75,
		290.375,
		67.5,
		326,
		328.125,
		98,
		110.5,
		0,
		330,
		342.25,
		343,
		38.5,
		161.75,
		180,
		116.25,
		138.75,
		10.75,
		227.75,
		352.5,
		185,
		356.25,
		209.375,
		36.25,
		40.75,
		67.5,
		272.625,
		165,
		295,
		71.25,
		148.25,
		198.75,
		0,
		21.125,
		88.25,
		118.25,
		313.75,
		288.875,
		28.75,
		242.5,
		226.25,
		227.75,
		275.5,
		161.25,
		283,
		25,
		10.75,
		157.5,
		318.75,
		85.5,
		1,
		238.75,
		88.25,
		297.5,
		283,
		2.125,
		303.75,
		117.125,
		308.5,
		288.75,
		49.25,
		283,
		85.5,
		93.25,
		285,
		74.5,
		238.75,
		297.5,
		69.25,
		80,
		192.5,
		258.75,
		178,
		112.5,
		251.25,
		0,
		84,
		105,
		316.75,
		94.625,
		273.75,
		313.125,
		315.75,
		172.75,
		93.625,
		104,
		117.125,
		357.5,
		25,
		302.125,
		202.5,
		341.25,
		118.875,
		138.75,
		273.75,
		279.875,
		312.5,
		105,
		273.75,
		241.25,
		273.75,
		322,
		0,
		278.75,
		304.5,
		312.5,
		320,
		330,
		325,
		105.25,
		53.75,
		69.25,
		108,
		220,
		267.5,
		39.75,
		49.25,
		226.25,
		70,
		87.5,
		267.5,
		273.75,
		278.75,
		341.25,
		161.25,
		172.75,
		0,
		357.5,
		213.75,
		238.75,
		300,
		320,
		257.5,
		87.5,
		73.75,
		76.25,
		121.25,
		143.75,
		177.5,
		263.75,
		283,
		72.5,
		308,
		257.5,
		273.75,
		125.5,
		244,
		128.75,
		161.25,
		244,
		235,
		188.75,
		192.5,
		136.25,
		25,
		39.75,
		162.5,
		177.5,
		213.75,
		244,
		0,
		320,
		328,
		357.5,
		146.25,
		70.5,
		72.5,
		300,
		153.75,
		188.75,
		223.75,
		235,
		68.75,
		251.25,
		264,
		158.75,
		91.75,
		183.75,
		162.5,
		52.5,
		125.5,
		64,
		267.5,
		320,
		345,
		45,
		140.5,
		0,
		25,
		58,
		350,
		212.5,
		235,
		240,
		72.5,
		75,
		120,
		51.25,
		246.25,
		267.5,
		287.5,
		305,
		45,
		67.5,
		230,
		0,
		40,
		61.25,
		64,
		320,
		90,
		120,
		36.25,
		57.5,
		0,
		90,
		122.5,
		52.5,
		57.5,
		0,
		32.5,
		67.5,
		225.75,
		126.75,
		92.5,
		177.5,
		212.5,
		225.75,
		60,
		132.5,
		165,
		262.5,
		270,
		330,
		48,
		75,
		97.5,
		0,
		20,
		350,
		65,
		230,
		305,
		82.5,
		227.5,
		246.25,
		223.75,
		248.75,
		90,
		102.5,
		168.75,
		177.5,
		192.5,
		202.5,
		251.25,
		32.5,
		48,
		221.25,
		252.5,
		262.5,
		330,
		68.75,
		205,
		221.25,
		0,
		52.5,
		98.75,
		135.5,
		168.75,
		270,
		320,
		350,
		11.25,
		0,
		115,
		205,
		52.5,
		0
	};

	static const float	ra_upper[357] =
	{
		360,
		217.5,
		345,
		315,
		120,
		160,
		75,
		217.5,
		270,
		315,
		52.625,
		203.75,
		262.5,
		310,
		137.5,
		170,
		248,
		51.25,
		310,
		180,
		5,
		235,
		360,
		202.5,
		216.25,
		353.75,
		105,
		306.25,
		309,
		119.5,
		126.25,
		300,
		308,
		347.5,
		36.5,
		296.5,
		28.625,
		46.5,
		47.5,
		343,
		91.5,
		216.25,
		291.25,
		50,
		334.75,
		329.5,
		25.5,
		97.5,
		202.5,
		236.25,
		332,
		75,
		350,
		255,
		37.75,
		273.5,
		20.5,
		25,
		102,
		360,
		210.5,
		16.75,
		360,
		273.5,
		286.25,
		287.5,
		30.625,
		137.5,
		13,
		181.25,
		110.5,
		329.5,
		328.625,
		291,
		152.5,
		161.75,
		236.25,
		245,
		143.75,
		37.75,
		38.5,
		291,
		70.375,
		328.125,
		330,
		110.5,
		116.25,
		30,
		342.25,
		343,
		352.5,
		40.75,
		165,
		185,
		138.75,
		148.25,
		21.125,
		231.5,
		356.25,
		198.75,
		360,
		210.5,
		40.75,
		67.5,
		71.25,
		290.375,
		180,
		313.75,
		88.25,
		157.5,
		209.375,
		1,
		25,
		98,
		120,
		326,
		295,
		36.25,
		245,
		227.75,
		242.5,
		283,
		165,
		288.875,
		28.75,
		12.75,
		161.25,
		321.25,
		88.25,
		2.125,
		240.5,
		93.25,
		303.75,
		288.75,
		12.75,
		308.5,
		118.25,
		318.75,
		297.5,
		50.5,
		285,
		86.5,
		94.625,
		297.5,
		80,
		241.25,
		303.75,
		74.5,
		84,
		202.5,
		273.75,
		192.5,
		117.125,
		258.75,
		2.125,
		86.5,
		112.5,
		320,
		104,
		283,
		315.75,
		316.75,
		178,
		94.625,
		105,
		118.875,
		360,
		49.25,
		304.5,
		226.25,
		357.5,
		138.75,
		161.25,
		279.875,
		283,
		313.125,
		105.25,
		276.375,
		251.25,
		276.375,
		325,
		30,
		283,
		312.5,
		320,
		322,
		341.25,
		330,
		108,
		69.25,
		70,
		121.25,
		226.25,
		273.75,
		49.25,
		53.75,
		244,
		76.25,
		93.625,
		269.5,
		278.75,
		283,
		357.5,
		172.75,
		177.5,
		5,
		360,
		220,
		244,
		308,
		328,
		269.5,
		121.25,
		76.25,
		87.5,
		125.5,
		161.25,
		192.5,
		265,
		300,
		73.75,
		320,
		273.75,
		283,
		128.75,
		245.625,
		136.25,
		162.5,
		245.625,
		238.75,
		192.5,
		213.75,
		146.25,
		39.75,
		56.25,
		177.5,
		188.75,
		223.75,
		251.25,
		25,
		328,
		357.5,
		360,
		153.75,
		72.5,
		91.75,
		320,
		158.75,
		223.75,
		235,
		240,
		70.5,
		264,
		267.5,
		162.5,
		110.5,
		188.75,
		183.75,
		56.25,
		140.5,
		68.75,
		287.5,
		345,
		350,
		52.5,
		165,
		25,
		45,
		64,
		360,
		223.75,
		240,
		246.25,
		75,
		98.75,
		125.5,
		58,
		267.5,
		287.5,
		305,
		320,
		51.25,
		72.5,
		235,
		35,
		45,
		64,
		67.5,
		330,
		120,
		122.5,
		40,
		61.25,
		27.5,
		92.5,
		126.75,
		57.5,
		60,
		23.75,
		36.25,
		75,
		230,
		132.5,
		97.5,
		192.5,
		225.75,
		230,
		65,
		165,
		168.75,
		270,
		305,
		350,
		52.5,
		82.5,
		102.5,
		20,
		32.5,
		360,
		68.75,
		246.25,
		320,
		90,
		230,
		248.75,
		227.5,
		251.25,
		102.5,
		135.5,
		177.5,
		192.5,
		218,
		205,
		252.5,
		48,
		68.75,
		223.75,
		262.5,
		270,
		350,
		98.75,
		221.25,
		255,
		20,
		68.75,
		135.5,
		168.75,
		205,
		320,
		350,
		360,
		20,
		52.5,
		205,
		270,
		115,
		360
	};

	static const float	dec_lower[357] =
	{
		88,
		86.5,
		86.1667,
		86,
		85,
		82,
		80,
		80,
		80,
		80,
		77,
		77,
		75,
		75,
		73.5,
		73.5,
		70,
		68,
		67,
		66.5,
		66,
		66,
		66,
		64,
		63,
		63,
		62,
		61.5,
		60.9167,
		60,
		60,
		59.5,
		59.5,
		59.0833,
		58.5,
		58,
		57.5,
		57,
		57,
		56.25,
		56,
		55.5,
		55.5,
		55,
		55,
		54.8333,
		54,
		54,
		53,
		53,
		52.75,
		52.5,
		52.5,
		51.5,
		50.5,
		50.5,
		50,
		50,
		50,
		50,
		48.5,
		48,
		48,
		47.5,
		47.5,
		47.5,
		47,
		47,
		46,
		45,
		44.5,
		44,
		43.75,
		43.5,
		42,
		40,
		40,
		40,
		39.75,
		36.75,
		36.75,
		36.5,
		36,
		36,
		36,
		35.5,
		35.5,
		35,
		35,
		34.5,
		34.5,
		34,
		34,
		34,
		33.5,
		33.5,
		33,
		33,
		32.0833,
		32,
		31.3333,
		30.75,
		30.6667,
		30.6667,
		30,
		30,
		29,
		29,
		28.5,
		28.5,
		28.5,
		28,
		28,
		28,
		28,
		28,
		27.5,
		27.25,
		27,
		26,
		26,
		26,
		25.5,
		25.5,
		25,
		23.75,
		23.5,
		23.5,
		22.8333,
		22,
		22,
		21.5,
		21.25,
		21.0833,
		21,
		20.5,
		20,
		19.5,
		19.1667,
		19,
		18.5,
		18,
		17.5,
		16.1667,
		16,
		16,
		15.75,
		15.5,
		15.5,
		15,
		14.3333,
		14,
		13.5,
		12.8333,
		12.5,
		12.5,
		12.5,
		12.5,
		12,
		12,
		11.8333,
		11.8333,
		11,
		10,
		10,
		10,
		10,
		9.9167,
		8.5,
		8,
		7.5,
		7,
		7,
		6.25,
		6.25,
		6,
		5.5,
		4.5,
		4,
		3,
		2.75,
		2,
		2,
		2,
		2,
		2,
		2,
		1.75,
		1.5,
		0,
		0,
		0,
		0,
		0,
		-1.75,
		-1.75,
		-3.25,
		-4,
		-4,
		-4,
		-4,
		-4,
		-4,
		-6,
		-6,
		-7,
		-7,
		-8,
		-8,
		-9,
		-9,
		-10,
		-11,
		-11,
		-11,
		-11,
		-11,
		-11,
		-11.6667,
		-12.0333,
		-14.5,
		-15,
		-16,
		-16,
		-17,
		-18.25,
		-19,
		-19,
		-19.25,
		-20,
		-22,
		-22,
		-24,
		-24.3833,
		-24.3833,
		-24.5,
		-24.5,
		-24.5,
		-24.5833,
		-25.5,
		-25.5,
		-25.5,
		-25.5,
		-26.5,
		-27.25,
		-27.25,
		-28,
		-29.1667,
		-29.5,
		-29.5,
		-29.5,
		-30,
		-30,
		-30,
		-31.1667,
		-33,
		-33,
		-35,
		-36,
		-36.75,
		-37,
		-37,
		-37,
		-37,
		-39.5833,
		-39.75,
		-40,
		-40,
		-40,
		-40,
		-42,
		-42,
		-42,
		-43,
		-43,
		-43,
		-44,
		-45.5,
		-45.5,
		-45.5,
		-45.5,
		-46,
		-46.5,
		-48,
		-48.1667,
		-49,
		-49,
		-49,
		-50,
		-50.75,
		-50.75,
		-51,
		-51,
		-51.5,
		-52.5,
		-53,
		-53.1667,
		-53.1667,
		-53.5,
		-54,
		-54,
		-54,
		-54.5,
		-55,
		-55,
		-55,
		-55,
		-56.5,
		-56.5,
		-56.5,
		-57,
		-57,
		-57,
		-57.5,
		-57.5,
		-58,
		-58.5,
		-58.5,
		-58.5,
		-59,
		-60,
		-60,
		-61,
		-61,
		-61,
		-63.5833,
		-63.5833,
		-64,
		-64,
		-64,
		-64,
		-64,
		-65,
		-65,
		-67.5,
		-67.5,
		-67.5,
		-67.5,
		-67.5,
		-67.5,
		-70,
		-70,
		-70,
		-75,
		-75,
		-75,
		-75,
		-75,
		-75,
		-75,
		-75,
		-76,
		-82.5,
		-82.5,
		-82.5,
		-85,
		-90
	};

	static const int	constel_numbers[357] =
	{
		84,
		84,
		84,
		84,
		16,
		11,
		16,
		11,
		84,
		34,
		16,
		11,
		84,
		16,
		11,
		34,
		84,
		14,
		34,
		34,
		16,
		84,
		16,
		34,
		34,
		16,
		11,
		34,
		16,
		11,
		83,
		34,
		16,
		16,
		14,
		34,
		14,
		14,
		11,
		16,
		11,
		83,
		34,
		11,
		16,
		16,
		14,
		51,
		83,
		34,
		16,
		11,
		14,
		34,
		63,
		34,
		14,
		63,
		51,
		14,
		83,
		14,
		14,
		40,
		34,
		31,
		63,
		83,
		14,
		83,
		51,
		31,
		31,
		31,
		83,
		83,
		9,
		40,
		51,
		1,
		63,
		52,
		63,
		31,
		45,
		8,
		51,
		1,
		45,
		45,
		1,
		63,
		83,
		30,
		51,
		49,
		1,
		9,
		1,
		30,
		1,
		30,
		81,
		63,
		8,
		52,
		83,
		31,
		8,
		49,
		30,
		1,
		81,
		8,
		38,
		31,
		31,
		81,
		26,
		9,
		26,
		52,
		49,
		52,
		81,
		67,
		49,
		88,
		78,
		1,
		74,
		38,
		88,
		88,
		1,
		88,
		38,
		88,
		88,
		7,
		76,
		60,
		38,
		76,
		78,
		40,
		76,
		78,
		78,
		24,
		40,
		24,
		38,
		40,
		62,
		78,
		38,
		62,
		38,
		40,
		32,
		62,
		46,
		60,
		38,
		22,
		62,
		7,
		32,
		9,
		62,
		22,
		46,
		59,
		4,
		32,
		21,
		74,
		40,
		59,
		62,
		67,
		74,
		32,
		35,
		62,
		62,
		62,
		21,
		78,
		60,
		21,
		86,
		59,
		17,
		78,
		74,
		60,
		60,
		74,
		74,
		4,
		67,
		46,
		86,
		67,
		67,
		86,
		59,
		4,
		5,
		59,
		55,
		36,
		60,
		42,
		75,
		86,
		59,
		4,
		36,
		5,
		74,
		73,
		42,
		59,
		42,
		27,
		72,
		48,
		29,
		86,
		42,
		17,
		36,
		27,
		29,
		48,
		59,
		17,
		12,
		5,
		17,
		42,
		36,
		47,
		12,
		42,
		42,
		48,
		72,
		36,
		59,
		77,
		42,
		20,
		42,
		42,
		37,
		69,
		36,
		77,
		66,
		71,
		37,
		2,
		71,
		37,
		36,
		71,
		15,
		50,
		72,
		10,
		23,
		68,
		36,
		72,
		25,
		77,
		54,
		36,
		10,
		50,
		64,
		36,
		41,
		10,
		39,
		68,
		85,
		36,
		41,
		64,
		13,
		85,
		41,
		33,
		64,
		36,
		65,
		50,
		85,
		13,
		15,
		50,
		57,
		33,
		85,
		15,
		6,
		79,
		39,
		41,
		65,
		13,
		64,
		36,
		64,
		33,
		57,
		44,
		65,
		19,
		6,
		19,
		6,
		65,
		13,
		15,
		28,
		15,
		19,
		6,
		41,
		70,
		19,
		6,
		61,
		82,
		33,
		19,
		80,
		82,
		43,
		87,
		13,
		56,
		61,
		44,
		82,
		82,
		43,
		18,
		3,
		53,
		58
	};

	static const char *constel_names[88] =
	{
		"And",
		"Ant",
		"Aps",
		"Aql",
		"Aqr",
		"Ara",
		"Ari",
		"Aur",
		"Boo",
		"Cae",
		"Cam",
		"Cap",
		"Car",
		"Cas",
		"Cen",
		"Cep",
		"Cet",
		"Cha",
		"Cir",
		"CMa",
		"CMi",
		"Cnc",
		"Col",
		"Com",
		"CrA",
		"CrB",
		"Crt",
		"Cru",
		"Crv",
		"CVn",
		"Cyg",
		"Del",
		"Dor",
		"Dra",
		"Equ",
		"Eri",
		"For",
		"Gem",
		"Gru",
		"Her",
		"Hor",
		"Hya",
		"Hyi",
		"Ind",
		"Lac",
		"Leo",
		"Lep",
		"Lib",
		"LMi",
		"Lup",
		"Lyn",
		"Lyr",
		"Men",
		"Mic",
		"Mon",
		"Mus",
		"Nor",
		"Oct",
		"Oph",
		"Ori",
		"Pav",
		"Peg",
		"Per",
		"Phe",
		"Pic",
		"PsA",
		"Psc",
		"Pup",
		"Pyx",
		"Ret",
		"Scl",
		"Sco",
		"Sct",
		"Ser",
		"Sex",
		"Sge",
		"Sgr",
		"Tau",
		"Tel",
		"TrA",
		"Tri",
		"Tuc",
		"UMa",
		"UMi",
		"Vel",
		"Vir",
		"Vol",
		"Vul"
	};

	int i = 0;

	if((position->ra >= 0) || (position->ra < 360) || (position->dec >= -90) || (position->dec <= 90))
	{
		while((position->ra < ra_lower[i]) || (position->ra >= ra_upper[i]) || (position->dec < dec_lower[i]))
			i++;

		return constel_names[constel_numbers[i] - 1];
	}
	else
	{
		return "---";
	}
}
