# -*- coding: utf-8 -*-
# emacs: -*- mode: python; py-indent-offset: 4; indent-tabs-mode: nil -*-
# vi: set ft=python sts=4 ts=4 sw=4 et:
"""
The workbench module provides classes for interfacing with `connectome workbench
<https://www.humanconnectome.org/software/connectome-workbench>`_ tools.

`Connectome Workbench is an open source, freely available visualization and
 discovery tool used to map neuroimaging data, especially data generated by the
 Human Connectome Project.
"""

from __future__ import (print_function, division, unicode_literals,
                        absolute_import)
import os
import re

from ... import logging
from ...utils.filemanip import split_filename
from ..base import CommandLine, PackageInfo

iflogger = logging.getLogger('nipype.interface')


class Info(PackageInfo):
    """
    Handle `wb_command` version information.
    """

    version_cmd = 'wb_command -version'

    @staticmethod
    def parse_version(raw_info):
        m = re.search(r'\nVersion (\S+)', raw_info)
        return m.groups()[0] if m else None


class WBCommand(CommandLine):
    """Base support for workbench commands."""

    @property
    def version(self):
        return Info.version()

    def _gen_filename(self, name, outdir=None, suffix='', ext=None):
        """Generate a filename based on the given parameters.
        The filename will take the form: <basename><suffix><ext>.
        Parameters
        ----------
        name : str
            Filename to base the new filename on.
        suffix : str
            Suffix to add to the `basename`.  (defaults is '' )
        ext : str
            Extension to use for the new filename.
        Returns
        -------
        fname : str
            New filename based on given parameters.
        """
        if not name:
            raise ValueError("Cannot generate filename - filename not set")

        _, fname, fext = split_filename(name)
        if ext is None:
            ext = fext
        if outdir is None:
            outdir = '.'
        return os.path.join(outdir, fname + suffix + ext)
