///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// mayavi vtk visualization
//
// author: Pierre.Saramito@imag.fr
//
// date: 12 may 1997  update: 23 oct 2011
//
#include "rheolef/field.h"
#include "rheolef/piola.h"
#include "rheolef/rheostream.h"
#include "rheolef/iorheo.h"
#include "rheolef/field_evaluate.h"
#include "rheolef/space_component.h"
#include "rheolef/interpolate.h"

namespace rheolef { 
using namespace std;

// extern:
template <class T> odiststream& geo_put_vtk (odiststream& ods, const geo_basic<T,sequential>& omega,
	const numbering<T,sequential>& my_numb, const disarray<point_basic<T>,sequential>& my_node, bool append_data);
template <class T> odiststream& geo_put_vtk (odiststream& ods, const geo_basic<T,sequential>& omega,
	size_t my_order, const disarray<point_basic<T>,sequential>& my_node, bool append_data);

// when uh is P0 on a P1 mesh, compute uh on a P1d field, such that
// xdofs coincide locally on each element with mesh nodes
template <class T>
static
field_basic<T,sequential>
interpolate_scalar (const space_basic<T,sequential>& Vh, const field_basic<T,sequential>& uh)
{
  typedef typename field_basic<T,sequential>::size_type size_type;
  check_macro (Vh.get_geo() == uh.get_geo(), "interpolate: incompatible geos");
  field_basic<T,sequential> vh(Vh);
  const space_basic<T,sequential>& Uh = uh.get_space();
  const geo_basic<T,sequential>& omega = uh.get_geo();
  std::vector<bool> dof_done (Vh.ndof(), false);
  std::vector<size_type> u_dis_idof;
  std::vector<size_type> v_dis_idof;
  basis_basic<T> fem_basis    = Uh.get_numbering().get_basis();
  basis_basic<T> pointset     = Vh.get_numbering().get_basis();
  basis_on_pointset<T> fem_basis_on_geo_nodes (pointset, fem_basis);
  for (size_type ie = 0, ne = omega.size(); ie < ne; ie++) {
    const geo_element& K = omega[ie];
    const std::vector<point_basic<T> >& hat_x = Vh.get_numbering().get_basis().hat_node (K.variant());
    Uh.dis_idof (K, u_dis_idof);
    Vh.dis_idof (K, v_dis_idof);
    for (size_type v_loc_idof = 0, v_loc_ndof = v_dis_idof.size(); v_loc_idof < v_loc_ndof; v_loc_idof++) {
      if (dof_done[v_dis_idof[v_loc_idof]]) continue;
      dof_done[v_dis_idof[v_loc_idof]] = true;
      vh.dof (v_dis_idof[v_loc_idof]) = field_evaluate (uh, fem_basis_on_geo_nodes, K, u_dis_idof, v_loc_idof);
    }
  }
  return vh;
}
template <class T>
static
field_basic<T,sequential>
interpolate (const space_basic<T,sequential>& Vh, const field_basic<T,sequential>& uh)
{
  if (uh.size() == 0) {
    return interpolate_scalar (Vh, uh);
  }
  typedef typename field_basic<T,sequential>::size_type size_type;
  field_basic<T,sequential> vh(Vh);
  for (size_type i_comp = 0, n_comp = uh.size(); i_comp < n_comp; i_comp++) {
    vh [i_comp] = interpolate_scalar (space_basic<T,sequential>(Vh[i_comp]), field_basic<T,sequential>(uh[i_comp]));
  }
  return vh;
}
template <class T>
odiststream&
put_vtk_scalar_values (odiststream& ods, const field_basic<T,sequential>& uh, std::string name, bool put_header)
{
  typedef typename field_basic<T,sequential>::size_type size_type;
  ostream& vtk = ods.os();
  size_type degree = uh.get_space().get_numbering().get_basis().degree();
  vtk << setprecision(numeric_limits<T>::digits10);
  if (put_header) {
    std::string data_type = (degree == 0) ? "CELL_DATA" : "POINT_DATA";
    vtk << data_type << " " << uh.ndof() << endl;
  }
  vtk << "SCALARS " << name << " float" << endl
      << "LOOKUP_TABLE default"  << endl;
  for (size_type idof = 0, ndof = uh.ndof(); idof < ndof; idof++)
    vtk << uh.dof(idof) << endl;
  vtk << endl;
  return ods;
}
template <class T>
odiststream&
put_vtk_vector_values (odiststream& ods, const field_basic<T,sequential>& uh, std::string name, bool put_header)
{
  typedef typename field_basic<T,sequential>::size_type size_type;
  ostream& vtk = ods.os();
  // without memory allocation: use field_component proxy directly:
  size_type n_comp = uh.size();
  std::vector<field_component_const<T,sequential> > uh_comp (n_comp);
  for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
    uh_comp[i_comp].proxy_assign (uh[i_comp]);
  }
  space_basic<T,sequential> Xh (uh.get_geo(), uh.get_approx());
  field_basic<T,sequential> norm_uh = interpolate (Xh, norm(uh));
  put_vtk_scalar_values (ods, norm_uh, name+"_norm", put_header);
  vtk << setprecision(numeric_limits<T>::digits10)
      << "VECTORS " << name << " float" << endl;
  std::vector<T> u_dof (n_comp);
  for (size_type idof = 0, ndof = uh_comp[0].ndof(); idof < ndof; idof++) {
    for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
      vtk << uh_comp[i_comp].dof (idof);
      if (i_comp != 2) vtk << " ";
    }
    for (size_type i_comp = n_comp; i_comp < 3; i_comp++) {
      vtk << "0";
      if (i_comp != 2) vtk << " ";
    }
    vtk << endl;
  }
  vtk << endl;
  return ods;
}
template <class T>
odiststream&
put_vtk_tensor_values (odiststream& ods, const field_basic<T,sequential>& tau_h, std::string name, bool put_header)
{
  typedef typename field_basic<T,sequential>::size_type size_type;
  ostream& vtk = ods.os();
  space_basic<T,sequential> Xh (tau_h.get_geo(), tau_h.get_approx());
  field_basic<T,sequential> norm_tau_h = interpolate (Xh, norm(tau_h));
  put_vtk_scalar_values (ods, norm_tau_h, name+"_norm", put_header);
  vtk << setprecision(numeric_limits<T>::digits10)
      << "TENSORS " << name << " float" << endl;
  size_type d = tau_h.get_geo().dimension();
  switch (d) {
    case 1: {
      field_basic<T,sequential> t00 = tau_h(0,0);
      for (size_type idof = 0, ndof = t00.ndof(); idof < ndof; idof++) {
        vtk << t00.dof(idof) << " 0 0" << endl
            << "0 0 0" << endl
            << "0 0 0" << endl;
      }
      break;
    }
    case 2: {
      field_basic<T,sequential> t00 = tau_h(0,0);
      field_basic<T,sequential> t01 = tau_h(0,1);
      field_basic<T,sequential> t11 = tau_h(1,1);
      for (size_type idof = 0, ndof = t00.ndof(); idof < ndof; idof++) {
        vtk << t00.dof(idof) << " " << t01.dof(idof) << " 0" << endl
            << t01.dof(idof) << " " << t11.dof(idof) << " 0" << endl
            << "0 0 0" << endl;
      }
      break;
    }
    default: {
      field_basic<T,sequential> t00 = tau_h(0,0);
      field_basic<T,sequential> t01 = tau_h(0,1);
      field_basic<T,sequential> t11 = tau_h(1,1);
      field_basic<T,sequential> t02 = tau_h(0,2);
      field_basic<T,sequential> t12 = tau_h(1,2);
      field_basic<T,sequential> t22 = tau_h(2,2);
      for (size_type idof = 0, ndof = t00.ndof(); idof < ndof; idof++) {
        vtk << t00.dof(idof) << " " << t01.dof(idof) << " " << t02.dof(idof) << endl
            << t01.dof(idof) << " " << t11.dof(idof) << " " << t12.dof(idof) << endl
            << t02.dof(idof) << " " << t12.dof(idof) << " " << t22.dof(idof) << endl;
      }
    }
  }
  return ods;
}
template <class T>
odiststream&
field_put_vtk (odiststream& ods, const field_basic<T,sequential>& uh, std::string name, bool put_geo)
{
  typedef typename field_basic<T,sequential>::size_type size_type;
  // 1) output geo
  size_type degree = uh.get_space().get_numbering().get_basis().degree();
  size_type order  = uh.get_space().get_geo().order();
  field_basic<T,sequential> vh;
  if (degree == 0) {
    vh = uh;
    if (put_geo) {
      geo_put_vtk (ods, vh.get_geo(), order, vh.get_geo().get_nodes(), false);
    }
  } else if (order <= degree && ! uh.get_space().get_numbering().has_compact_support_inside_element()) {
    vh = uh;
    if (put_geo) {
      geo_put_vtk (ods, vh.get_geo(), degree, vh.get_space().get_xdofs(), false);
    }
  } else { // order > degree or discontinuous
    // re-interpolate uh on the finer mesh lattice (eg P1d field on a P2 mesh)
    std::string approx = "P" + itos(order);
    if (uh.get_space().get_numbering().has_compact_support_inside_element()) approx += "d";
    space_basic<T,sequential> Vh (uh.get_geo(), approx, uh.get_space().valued());
    vh = interpolate (Vh, uh);
    if (put_geo) {
      geo_put_vtk (ods, vh.get_geo(), vh.get_space().get_numbering(), vh.get_space().get_xdofs(), false);
    }
  }
  // 2) output values
  bool put_header = put_geo;
  if (name == "") { name = vh.get_space().valued(); }
  switch (vh.get_space().valued_tag()) {
    case space_constant::scalar: put_vtk_scalar_values (ods, vh, name, put_header); break;
    case space_constant::vector: put_vtk_vector_values (ods, vh, name, put_header); break;
    case space_constant::tensor: put_vtk_tensor_values (ods, vh, name, put_header); break;
    default: error_macro ("put_vtk: do not known how to print " << vh.valued() << "-valued field");
  }
  return ods;
}
template <class T>
odiststream&
field_put_vtk (odiststream& ods, const field_basic<T,sequential>& uh)
{
  return field_put_vtk (ods, uh, "", true);
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template odiststream& field_put_vtk<Float>  (odiststream&, const field_basic<Float,sequential>&, std::string, bool);
template odiststream& field_put_vtk<Float>  (odiststream&, const field_basic<Float,sequential>&);

}// namespace rheolef
