///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
#include "rheolef/form_vf_assembly.h"
#include "rheolef/ublas-invert.h"

namespace rheolef { 
using namespace std;

// ------------------------------------------------------------------------------
// local lump
// ------------------------------------------------------------------------------
template <class T>
bool
is_symmetric (ublas::matrix<T>& m, const T& tol) {
  if (m.size1() != m.size2()) return false; // non-square matrix
  T m_max = 0;
  for (size_t i = 0; i < m.size1(); i++) {
    for (size_t j = 0; j < m.size2(); j++) {
      m_max = std::max (m_max, m(i,j));
    }
  }
  if (m_max < tol) return true; // zero matrix
  for (size_t i = 0; i < m.size1(); i++) {
    for (size_t j = i+1; j < m.size2(); j++) {
      if (abs(m(i,j) - m(j,i)) > tol*m_max) return false;
    }
  }
  return true;
}
// ------------------------------------------------------------------------------
// local lump
// ------------------------------------------------------------------------------
template <class T>
void
local_lump (ublas::matrix<T>& m) {
  check_macro (m.size1() == m.size2(), "unexpected rectangular matrix for lumped mass");
  for (size_t i = 0; i < m.size1(); i++) {
    T sum = 0;
    for (size_t j = 0; j < m.size2(); j++) {
      sum += m(i,j);
      m(i,j) = T(0);
    }
    m(i,i) = sum;
  }
}
// ------------------------------------------------------------------------------
// local inversion
// ------------------------------------------------------------------------------
template <class T>
void
local_invert (ublas::matrix<T>& m, bool is_diag) {
    check_macro (m.size1() == m.size2(), "unexpected rectangular matrix for local invert");
    if (is_diag) {
      // matrix has been lumped: easy diagonal inversion
      for (size_t i = 0; i < m.size1(); i++) {
        m(i,i) = 1./m(i,i);
      }
      return;
    }
    // general inversion
    bool singular;
    ublas::matrix<T> inv_m = invert (m, singular);
    check_macro (!singular, "singular element matrix founded");
#undef DO_CHECK_INV
#ifdef  DO_CHECK_INV
    ublas::matrix<T> id;
    id.resize (m.size1(), m.size2());
    id.clear();
    for (size_t i = 0; i < id.size1(); i++) id(i,i)=1;
    ublas::matrix<T> check1 = prod(m,inv_m) - id;
    T err = 0;
    for (size_t i = 0; i < id.size1(); i++)
    for (size_t j = 0; j < id.size2(); j++)
      err += sqr(check1(i,j));
    if (err > 1e-16) {
	warning_macro ("K" << K.dis_ie() << ": invert error = " << err);
        cerr << setprecision(16);
        cerr << "M" << K.dis_ie() << " = [";
        for (size_t i = 0; i < id.size1(); i++) {
          for (size_t j = 0; j < id.size2(); j++) {
	    if (j == 0) cerr << "      ";
            cerr << m(i,j);
	    if (j != id.size2()-1) cerr << ",";
	    else if (i != id.size1()-1) cerr << ";" << endl;
	    else cerr << "];" << endl;
          }
        }
        cerr << "cond(M" << K.dis_ie() << ")" << endl;
    }
#endif //  DO_CHECK_INV
    m = inv_m;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciate(T)					\
template bool is_symmetric (ublas::matrix<T>&, const T&);	\
template void local_lump   (ublas::matrix<T>&);			\
template void local_invert (ublas::matrix<T>&, bool);

_RHEOLEF_instanciate(Float)

}// namespace rheolef
