/**
 * Stores a table of RGB colors with depth cueing flag and index to the CLUT
 * created by the GUI module.
 * The table is automatically released from memory when the program terminates.
 * 
 * Note that the GUI module does not store the exact RGB value but only the
 * index to the closest matching entry in the CLUT, while Alib needs the RGB
 * value and the depth cueing flag to do the haze effect.
 * 
 * @file
 * @author Umberto Salsi <salsi@icosaedro.it>
 * @version $Date: 2017/08/04 10:22:46 $
 */

#ifndef VCOLOR_H
#define VCOLOR_H

#ifdef VColor_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

typedef struct VColor_Type VColor_Type;

/**
 * Returns the color specified by RGB value.
 * @param red [0,255].
 * @param green [0,255].
 * @param blue [0,255].
 * @param is_depth_cueing True if depth cueing is enabled on this color.
 * @return Pointer to new or cached color.
 */
EXTERN VColor_Type * VColor_getByRGB(int red, int green, int blue, int is_depth_cueing);

/**
 * Returns the color specified by name.
 * @param name Descriptive string of the color name (example: "red") or
 * hexadecimal RGB value (examples: "#f00", "#ff0000").
 * @param is_depth_cueing
 * @return 
 */
EXTERN VColor_Type * VColor_getByName(char *name, int is_depth_cueing);

/**
 * Returns the index of this color in the CLUT of the GUI module.
 * @param c Subject color.
 * @return Index of this color in the CLUT of the GUI module.
 */
EXTERN int VColor_getIndex(VColor_Type *c);

/**
 * Tells if this color has to be depth cued.
 * @param c Subject color.
 * @return True if this color has to be depth cued.
 */
EXTERN int VColor_isDepthCueing(VColor_Type *c);

EXTERN int VColor_getRed(VColor_Type *c);
EXTERN int VColor_getGreen(VColor_Type *c);
EXTERN int VColor_getBlue(VColor_Type *c);

/**
 * Formats and then returns the color name in the hexadecimal notation.
 * @param c Subject color.
 * @return Pointer to a statically allocated string where the hexadecimal RGB
 * value of the long form "#HHHHHH" is retrieved.
 */
EXTERN char * VColor_getName(VColor_Type *c);

#undef EXTERN
#endif
