/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */





/**
   \file cdw_fs_browser.c

   File implements a widget that allows browsing native file system.

   It can be used to implement a more complex structures that are used
   to select group of files from native file system to be burned to
   CD, or to select single file, e.g. an ISO9660 image file.

   This widget handles following special keys:
   Enter: enter current dir,
   Backspace: go to parent directory
   dot: toggle displaying hidden files
   tilde: go to user's home dir
   ?: display help

   If Lynx-like navigation is enabled in configuration, two more
   special keys are handled by this widget:
   Key Left: go up in file system hierarchy
   Key Right: enter current dir

   This widget inherits some special keys from underlying list display
   widget.
*/





#define _BSD_SOURCE /* strdup() */
#define _GNU_SOURCE /* strndup() */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "cdw_fs_browser.h"
#include "gettext.h"
#include "cdw_file.h"
#include "cdw_fs.h"
#include "cdw_file_picker.h"
#include "cdw_string.h"
#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_window.h"
#include "cdw_debug.h"
#include "canonicalize.h"
#include "cdw_config.h"





/* TODO: get rid of this extern. */
extern cdw_config_t global_config;


static cdw_fs_browser_t *cdw_fs_browser_new_base(WINDOW *window);
static size_t            cdw_fs_browser_read_dir_content(cdw_dll_item_t **list, char const * dirpath);
static void              cdw_fs_browser_internal_error_dialog(void);
static cdw_rv_t          cdw_fs_browser_change_dir_on_enter(cdw_fs_browser_t *browser);

static bool cdw_fs_browser_is_ascend_key(int key);
static bool cdw_fs_browser_is_descend_key(int key);

static void cdw_fs_browser_add_return_key(cdw_widget_t *widget, int key);





/**
   \brief Create new file system browser

   \date Function's top-level comment reviewed on 2016-02-06
   \date Function's body reviewed on 2016-02-06

   Create new fs browser widget. Use \p window as a main window for
   the browser. Set initial location in file system to \p
   initial_fullpath

   \param window - initialized ncurses window to be used as main browser window
   \param initial_fullpath - initial location in file system

   \return pointer to correctly initialized file system browser on success
   \return null pointer on failure
*/
cdw_fs_browser_t *cdw_fs_browser_new(WINDOW *window, char const * initial_fullpath)
{
	cdw_fs_browser_t *browser = cdw_fs_browser_new_base(window);
	if (!browser) {
		cdw_vdm ("ERROR: failed to create browser's base\n");
		return (cdw_fs_browser_t *) NULL;
	}

	/* make several attempts to browse to initial location */

	/* 1. first try to browse to location specified by caller */
	int e = cdw_fs_browser_browse_to(browser, initial_fullpath);
	if (e == 0) {
		/* success, browsed to given location and refreshed the view */
		return browser;
	}

	/* 2. try to browse to a dir that is one level up from location specified by caller */
	ssize_t i = cdw_fs_get_filename_start(initial_fullpath);
	if (i == -1) {
		cdw_vdm ("WARNING: failed to get file name start in \"%s\"\n", initial_fullpath);
	} else {
		char *parent = strndup(initial_fullpath, (size_t) i);
		cdw_assert (parent, "ERROR: failed to strndup fullpath \"%s\"\n", initial_fullpath);
		e = cdw_fs_browser_browse_to(browser, parent);
		free(parent);
		parent = (char *) NULL;
		if (e == 0) {
			/* success, browsed to given 'parent' location and refreshed the view */
			return browser;
		}
	}

	/* 3. try to browse to "safe" location provided by fs module.
	   cdw_fs_get_initial_dirpath() tries to return one of following:
	   current working dir, "home" dir, "/tmp" dir, or "/" (root) dir */
	char *tmp = cdw_fs_get_initial_dirpath();
	if (tmp) {
		e = cdw_fs_browser_browse_to(browser, tmp);
		free(tmp);
		tmp = (char *) NULL;
		if (e == 0) {
			/* success, browsed to given location and refreshed the view */
			return browser;
		} else {
			cdw_vdm ("ERROR: can't browse to any safe initial dirpath for fs browser, this is bad\n");
		}
	} else {
		cdw_vdm ("ERROR: can't get any safe initial dirpath for fs browser, this is bad\n");
	}

	cdw_fs_browser_internal_error_dialog();
	cdw_fs_browser_delete(&browser);
	return (cdw_fs_browser_t *) NULL;
}





/**
   \brief Wrapper for some initial code creating file system browser

   \date Function's top-level comment reviewed on 2016-02-06
   \date Function's body reviewed on 2016-02-06

   Function that hides some boring details of creating new file system
   browser like allocating memory for new fs browser data structure,
   setting widget size, initializing empty full paths etc.

   \param window - non-NULL parent window, in which a fs browser will be displayed

   \return pointer to new, partially initialized fs browser on success
   \return NULL on problems
*/
cdw_fs_browser_t *cdw_fs_browser_new_base(WINDOW *window)
{
	cdw_assert (window, "ERROR: \"window\" argument is NULL\n");

	cdw_fs_browser_t *browser = (cdw_fs_browser_t *) malloc(sizeof (cdw_fs_browser_t));
	if (!browser) {
		cdw_vdm ("ERROR: failed to allocate memory for fs browser\n");
		return (cdw_fs_browser_t *) NULL;
	}

	browser->parent_n_lines = getmaxy(window);
	browser->parent_n_cols = getmaxx(window);
	browser->parent_begin_y = getbegy(window);
	browser->parent_begin_x = getbegx(window);

	int n_lines = browser->parent_n_lines;
	cdw_assert (n_lines > 2, "ERROR: number of lines of window is too small: %d\n", n_lines);
	n_lines -= 2;
	int n_cols = browser->parent_n_cols;
	cdw_assert (n_cols > 2, "ERROR: number of cols of window is too small: %d\n", n_cols);
	n_cols -= 2;

	cdw_widget_init(&browser->widget);
	browser->widget.type_id = CDW_WIDGET_ID_INIT; /* No ID for fs browser (yet?). */
	browser->widget.add_return_key = cdw_fs_browser_add_return_key; /* Custom function for adding return keys by parent cdw_form. */
	browser->widget.self = browser;

	/* 1, 1 - begin_y, begin_x */
	browser->display = cdw_list_display_new(window, n_lines, n_cols, 1, 1, CDW_COLORS_DIALOG);
	if (browser->display == (CDW_LIST_DISPLAY *) NULL) {
		cdw_vdm ("ERROR: failed to create display for fs browser\n");
		cdw_fs_browser_internal_error_dialog();

		free(browser);
		browser = (cdw_fs_browser_t *) NULL;

		return (cdw_fs_browser_t *) NULL;
	}

	browser->display->display_item = cdw_file_display_file;

	/* These keys change position in file system or change current
	   highlight on list of files. Parent of this widget needs to
	   be notified that such change has been made (the parent may
	   want to read current ->fullpath). These keys become fs
	   browser's return keys and are also added to browser->display. */
	cdw_widget_add_return_keys_pointer(&browser->widget,
					   CDW_KEY_ENTER,  /* Enter current dir. */
					   KEY_BACKSPACE,  /* Go to parent directory. */
					 '.',            /* Toggle displaying hidden files. */
					   '~',            /* Go to home dir. */
				   
					   /* fs browser's driver won't return on these keys if lynx-like navigation is disabled. */
					   KEY_LEFT,       /* Go up in file system hierarchy. */
					   KEY_RIGHT,      /* Enter current dir. */
					   
					   0);             /* Guard. */
	
	/* Usual suspects. 'q' stands for "Quit". */
	cdw_widget_add_return_keys_pointer(&browser->widget, CDW_KEY_ESCAPE, 'Q', 'q', 0);

	/* fs browser needs to handle these keys pressed in list
	   display, so the keys should be display's (but not
	   browser's) return keys. */
	cdw_list_display_add_return_keys(browser->display,
					 '?',            /* Display help. */
					 0);             /* Guard. */

	return browser;
}





/**
   \brief Deallocate all resources related to given browser widget

   \date Function's top-level comment reviewed on 2016-02-06
   \date Function's body reviewed on 2016-02-06

   Function frees all resources related to given \p browser (including
   browser itself) or calls functions that do their job to free
   some resources. \p browser is set to NULL on return.

   \param browser - browser to be freed/destroyed
*/
void cdw_fs_browser_delete(cdw_fs_browser_t **browser)
{
	cdw_assert (browser != (cdw_fs_browser_t **) NULL, "ERROR: you passed null pointer\n");

	if (!browser || !*browser) {
		cdw_vdm ("WARNING: you passed pointer to null browser\n");
		return;
	}

	cdw_file_dealloc_files_from_list((*browser)->display->list);

	cdw_list_display_delete(&((*browser)->display));
	(*browser)->display = (CDW_LIST_DISPLAY *) NULL;

	free(*browser);
	*browser = (cdw_fs_browser_t *) NULL;

	return;
}





/**
   \brief Read information about files in given directory, store the information on a list

   \date Function's top-level comment reviewed on 2016-02-06
   \date Function's body reviewed on 2016-02-06

   Function scans directory specified by \p dirpath, checks information about
   all files (files and directories)  in the directory, and saves the
   information on \p list - one node of 'cdw_file_t *' per file. Function
   returns number of items added to \p list. Reference to current directory (".")
   is not put on the list.

   If displaying of hidden files is disabled in configuration, the function won't
   put them on the \p list, and returned value won't include hidden files.
   Therefore the function returns number of items on the list - not number
   of items (files+dirs) in directory.

   There will be always at least one item in the \p list: reference to
   parent dir (".."). Its presence is independent of "display hidden
   files" setting. So in normal conditions value returned by the
   function is at least 1.

   In case of errors the \p list is cleaned, \p list is set to null, and
   function returns 0.

   You can pass \p list that already carries some information, but the
   list will be emptied and then filled with new data.

   \param list - list to be filled with information about files
   \param dirpath - path to directory which you want to scan

   \return number of files read from given dir into list
   \return 0 on errors
*/
size_t cdw_fs_browser_read_dir_content(cdw_dll_item_t **list, char const * dirpath)
{
	if (*list != (cdw_dll_item_t *) NULL) {
		cdw_file_dealloc_files_from_list(*list);
		cdw_dll_clean(*list);
		*list = (cdw_dll_item_t *) NULL;
	}

	struct dirent **eps = (struct dirent **) NULL;
	/* By passing one() to scandir() we tell scandir() to omit "." item;
	   caution: scandir mallocs() space for eps, don't forget to free() it. */
	ssize_t n_files = scandir(dirpath, &eps, cdw_scandir_filter_one, alphasort);

	if (n_files == -1) {
		/* this happened to me once when after suspending to RAM and
		   then bringing system up I ran cdw and browsed to secondary
		   disc - the disc wasn't reconnected (?) after coming from
		   suspension to RAM */
		cdw_vdm ("ERROR: scandir() returns -1 for dirpath \"%s\"\n", dirpath);
		cdw_fs_browser_internal_error_dialog();
		return 0;
	} else if (n_files == 0) {
		/* not going to happen, with scandir() called with one(),
		   there will be at least '..' item in eps */
		cdw_assert (n_files != 0, "ERROR: scandir() called with one() returns 0 for dirpath \"%s\"\n", dirpath);;
		return 0;
	} else {
		size_t n = (size_t) n_files;
		size_t cp = cdw_fs_copy_dirent_to_list(list, dirpath, eps, n, global_config.general.display_hidden_files);

		/* eps is no longer needed, all interesting information about
		   directory content is now in display->list */
		for (size_t i = 0; i < n; i++) {
			free(eps[i]);
			eps[i] = (struct dirent *) NULL;
		}
		free(eps);
		eps = (struct dirent **) NULL;

		if (cp == 0) {
			if (*list != (cdw_dll_item_t *) NULL) {
				cdw_file_dealloc_files_from_list(*list);
				cdw_dll_clean(*list);
				*list = (cdw_dll_item_t *) NULL;
			}
			cdw_vdm ("ERROR: failed to copy dirent to list for dirpath \"%s\"\n", dirpath);
			return 0;
		} else {
#if 0
#ifndef NDEBUG
			cdw_vdm ("INFO: Content of directory \"%s\":\n", dirpath);
			for (size_t i = 0; i < cp; i++) {
				cdw_dll_item_t *item = cdw_dll_ith_item(*list, (size_t) i);
			        cdw_file_t *file = (cdw_file_t *) item->data;
				ssize_t f = cdw_fs_get_filename_start(file->fullpath);
				cdw_vdm ("INFO:       file #%6lld = \"%s\"\n", i, file->fullpath + f);
			}
#endif
#endif
			return cp;
		}
	}
}





/**
   \brief Visit new directory in given display

   \date Function's top-level comment reviewed on 2016-02-06
   \date Function's body reviewed on 2016-02-06

   Function should be called when user presses Enter on any file in
   fs browser. The function checks if current item is a directory
   and if it is, then the function tries to enter the directory.
   If current item is not a dir then nothing happens, nothing is
   changed, and function returns CDW_NO.

   If function enters the directory then content of new directory is read
   to list associated with \p browser, the content is displayed by the
   browser's display, and browser's dirpath is updated to new current
   directory.

   If resolving new dir path fails because of an error, or new dir content
   cannot be read because of an error, browser's dirpath is not modified
   and list associated with browser's display is reset to empty list.

   \param browser - fs browser with associated display that should be updated with content of new dir

   \return CDW_OK if directory was successfully changed and displayed
   \return CDW_NO if directory was not changed, but not because of error
   \return CDW_ERROR if other error occurred and dir was not changed because of that error
*/
cdw_rv_t cdw_fs_browser_change_dir_on_enter(cdw_fs_browser_t *browser)
{
	cdw_assert (browser->display, "ERROR: display is NULL\n");
	cdw_assert (browser->display->n_items > browser->display->current_item_ind,
		    "ERROR, item index (0-based) is %zd but number of items is %zd\n",
		    browser->display->current_item_ind, browser->display->n_items);

	cdw_file_t * const file = cdw_fs_browser_get_current_file(browser);
	if (file->type == CDW_FS_FILE || file->type != CDW_FS_DIR || file->invalid) {
		/* Enter on file has no meaning, on invalid file has no
		   result, nothing needs to be changed */
		return CDW_NO;
	} else if (file->is_ref_to_parent_dir && !strcmp(file->fullpath, "/../")) {
		/* we are in root dir, nowhere else to go up to */
		return CDW_OK;
	} else {
		;
	}

	int e = 0; /* errno returned by some cdw functions. */
	char *new_dirpath = (char *) NULL;
	if (file->is_ref_to_parent_dir) {
		/* "/some/dir/../", remove last dir from the path */
		new_dirpath = cdw_file_shorten_fullpath(file);
		if (!new_dirpath) {
			cdw_vdm ("ERROR: failed to shorten fullpath \"%s\"\n", file->fullpath);
			return CDW_ERROR;
		} else if (!strcmp(new_dirpath, "/..") || !strcmp(new_dirpath, "/")) {
			/* Don't attempt to re-read root directory
			   again. This may fail because the code will
			   try to find a (null) file to browse to. */
			free(new_dirpath);
			new_dirpath = (char *) NULL;
			return CDW_OK;
		}

		/* browse_to_file() moves cursor (focus) to directory item from
		   which we have just ascended instead of going to 0-th item
		   in newly visited dir */
		e = cdw_fs_browser_browse_to_file(browser, new_dirpath);
	} else {
		/* canonicalize because current item may be a link */
		new_dirpath = canonicalize_filename_mode(file->fullpath, CAN_MISSING);
		if (!new_dirpath) {
			/* canonicalization of dirpath failed */
			cdw_vdm ("ERROR: failed to canonicalize path \"%s\"\n", file->fullpath);
			return CDW_ERROR;
		}
		e = cdw_fs_browser_browse_into_dir(browser, new_dirpath);
	}

	if (e) {
		cdw_vdm ("ERROR: failed to browse to new dirpath \"%s\": \"%s\"\n", new_dirpath, strerror(e));
		cdw_fs_errno_handler(e);

		/* Refresh the list after displaying dialog window. */
		cdw_list_display_refresh(browser->display);
	}
	free(new_dirpath);
	new_dirpath = (char *) NULL;

	if (e == 0) {
		return CDW_OK;
	} else if (e == EACCES) {
		/* Wrong permissions. Keep fs browser window open, but
		   just don't browse into the dir. */
		return CDW_NO;
	} else {
		return CDW_ERROR;
	}
}





/**
   \brief Does user want to navigate down in file system hierarchy into selected directory?

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   In cdw, pressing Enter on a directory usually means "enter this
   directory". This either leads us up (ascending) in file system
   hierarchy if the dir is "..", or down (descending) for any other
   dir. Pressing Backspace key means ascending.

   Some programs provide possibility to navigate file system with left
   and right arrows. Midnight Commander calls this "Lynx-like motion".

   Up until 0.8.0, cdw used left and right arrows exclusively to shift
   a directory listing left and right in file system browser.

   Simply changing behaviour of left and right arrows is impossible,
   but there can be an option for this :)

   This function decides if - based on current configuration - a \p
   key is a file system navigation key.

   \param key - key to be checked

   \return true if the key means "go down in file system hierarchy / go into directory"
   \return false otherwise
*/
bool cdw_fs_browser_is_descend_key(int key)
{
	if (key == CDW_KEY_ENTER) {
		/* This case is always true, independent from any
		   configuration. */
		return true;
	}

	if (key == KEY_RIGHT
	    && cdw_config_fs_lynx_like_motion()) {

		return true;
	}

	return false;
}




/**
   \brief Does user want to navigate up in file system hierarchy?

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   See comment for cdw_fs_browser_is_descend_key().

   This function decides if - based on current configuration - a \p
   key is a file system navigation key.

   \param key - key to be checked

   \return true if the key means "go up in file system hierarchy"
   \return false otherwise
*/
bool cdw_fs_browser_is_ascend_key(int key)
{
	if (key == KEY_BACKSPACE) {
		/* This case is always true, independent from any
		   configuration. */
		return true;
	}

	if (key == KEY_LEFT
	    && cdw_config_fs_lynx_like_motion()) {

		return true;
	}

	return false;
}





/**
   \brief Is given key a "navigate in file system" key?

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   Does given \p key be treated as "change location in file system" command?

   \param key - key to be checked

   \return true if key is a navigation key
   \return false otherwise
*/
bool cdw_fs_browser_is_fs_navigation_key(int key)
{
	if (cdw_fs_browser_is_ascend_key(key)
	    || cdw_fs_browser_is_descend_key(key)
	    || key == '~') {

		return true;
	} else {
		return false;
	}
}





/**
   \brief Function that controls file system browser's behaviour

   \date Function's top-level comment reviewed on 2016-02-06
   \date Function's body reviewed on 2016-02-06

   \p browser has associated list display widget, and list display
   widget has a driver. This function controls the driver by calling it
   in a loop and checking it's return values. The function reacts to
   some of the return values in a way that you would expect from file
   system browser: the function calls another function to change working
   directory (when user pressed Enter in display).

   The function returns when a key defined as return key for fs
   browser has been pressed, or when error occurred.

   \param browser - file system browser that should be controlled with this function

   \return fs browser's return key when such key has been pressed
   \return -1 on errors
*/
int cdw_fs_browser_driver(cdw_fs_browser_t *browser)
{
	int key = 'a'; /* Safe initial value. */

	while (!cdw_fs_browser_is_return_key(browser, key)) {

		key = cdw_list_display_driver(browser->display);

		if (cdw_fs_browser_is_descend_key(key)
		    || cdw_fs_browser_is_ascend_key(key)) {

			if (cdw_fs_browser_is_ascend_key(key)) {
				/* Scroll to ".." first. After that we
				   will "press" Enter on that dir. */
				cdw_list_display_scroll_to(browser->display, 0, 0, false);
			}

			cdw_rv_t crv = cdw_fs_browser_change_dir_on_enter(browser);
			if (crv == CDW_OK) {
				cdw_fs_browser_debug_report_current_fullpath(browser, "Enter key pressed");
			} else if (crv == CDW_ERROR) {
				cdw_vdm ("ERROR: failed to change dir in fs browser\n");
				return -1;
			} else { /* CDW_NO - not a dir or no perms to change dir, continue */
				;
			}
#ifndef NDEBUG
#if 0
		} else if (key == ' ') {
			cdw_fs_browser_debug_report_current_fullpath(browser, "Space key pressed");

			/* Space key may be used by higher level code for selecting
			   files (if Space is defined as return key). */
			cdw_assert (browser->display->n_items > browser->display->current_item_ind,
				    "ERROR: index of current file is larger than number of files\n");
#endif
#endif
		} else if (key == '.') {
			if (global_config.general.display_hidden_files) {
				global_config.general.display_hidden_files = false;
			} else {
				global_config.general.display_hidden_files = true;
			}
			char *fullpath = cdw_fs_browser_get_current_fullpath(browser);
			/* FIXME: check return values */
			int e = cdw_fs_browser_browse_to_file(browser, fullpath);
			cdw_assert (e == 0, "Failed to browse to file \"%s\", error = \"%s\"\n", fullpath, strerror(e));
			CDW_STRING_DELETE (fullpath);

		} else if (key == '~') {
			char const * const dir = cdw_fs_get_home_dir_fullpath();
			if (dir) {
				cdw_fs_browser_browse_into_dir(browser, dir);
			}
		} else if (key == '?') {
			cdw_fs_browser_help_window(browser);
		} else {
			;
		}
	}

	cdw_vdm ("INFO: returning key %s\n", cdw_ncurses_key_label(key));
	return key;
}





/**
   \brief Display dialog window with generic error message

   \date Function's top-level comment reviewed on 2016-02-06
   \date Function's body reviewed on 2016-02-06

   Function displays dialog window with generic error message, to be
   used where cdw_fs_errno_handler() would be too specific.
*/
void cdw_fs_browser_internal_error_dialog(void)
{
	/* 2TRANS: this is title of dialog window */
	cdw_buttons_dialog(_("Error"),
			   /* 2TRANS: this is message in dialog
			      window. Some unspecified error
			      occurred. user can press OK button. */
			   _("Unexpected error, perhaps cdw can't find some file. Please close this window and open it again."),
			   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

	return;
}





/**
   Returned pointer is owned by \p browser
*/
cdw_file_t *cdw_fs_browser_get_current_file(cdw_fs_browser_t const * browser)
{
	cdw_assert (browser, "ERROR: browser pointer is NULL\n");

	void *data = cdw_list_display_get_current_item_data(browser->display);
	if (!data) {
		cdw_assert (data,
			    "ERROR: failed to fetch current file #%zd from fs browser->display\n",
			    browser->display->current_item_ind);
		return (cdw_file_t *) NULL;
	} else {
		cdw_file_t *file = (cdw_file_t *) data;
		cdw_vdm ("INFO: fetching file #%zd = \"%s\"\n", browser->display->current_item_ind, file->fullpath);
		return file;
	}
}





/**
   \return 0 on success
   \return errno on failure
*/
int cdw_fs_browser_browse_to(cdw_fs_browser_t *browser, char const * fullpath)
{
	struct stat finfo;
	if (stat(fullpath, &finfo) == -1) {
		int e = errno;
		/* This is not an error. It may be a perfectly normal
		   situation that a file has existed at some point,
		   but it no longer does. */
		cdw_vdm ("WARNING: stat() results in errno \"%s\" for fullpath \"%s\"\n", strerror(e), fullpath);
		return e;
	}

	if (S_ISDIR(finfo.st_mode)) {
		int e = cdw_fs_browser_browse_into_dir(browser, fullpath);
		if (e) {
			cdw_vdm ("ERROR: errno \"%s\" for dir fullpath \"%s\"\n", strerror(e), fullpath);
		}
		return e;

	} else if (S_ISREG(finfo.st_mode)) { /* file */
		int e = cdw_fs_browser_browse_to_file(browser, fullpath);
		if (e) {
			cdw_vdm ("ERROR: errno \"%s\" for file fullpath \"%s\"\n", strerror(e), fullpath);
		}
		return e;
	} else {
		cdw_vdm ("INFO: neither dir nor file: %s\n", fullpath);
		return ENONET;
	}
}





/**

   \return 0 on success
   \return errno otherwise
*/
int cdw_fs_browser_browse_into_dir(cdw_fs_browser_t *browser, char const * dirpath)
{
	int e = cdw_fs_check_existing_path(dirpath, X_OK | R_OK, CDW_FS_DIR);
	if (e) {
		/* not a dir, or can't visit dir or can't read dir's content */
		return e;
	}

	/* n is a number of files on list, not number of files in
	   directory. The two may be different if displaying of hidden
	   files is turned off. */
	size_t n = cdw_fs_browser_read_dir_content(&(browser->display->list), dirpath);
	if (n == 0) {
		cdw_assert (browser->display->list == (cdw_dll_item_t *) NULL,
			    "ERROR: dir not read properly, but list of files is not null\n");

		browser->display->n_items = 0;
		cdw_vdm ("ERROR: failed to read dir content for new dirpath \"%s\"\n", dirpath);

		return ENOENT;
	} else {
#ifndef NDEBUG
		size_t len = cdw_dll_length(browser->display->list);
		cdw_assert (n == len, "ERROR: lengths mismatch: dir content count (n) = %zd, len of browser display list (len) = %zd\n", n, len);
#endif
		browser->display->n_items = n;
		/* first 0 - position cursor on first item in new dir,
		   second 0 - horizontal offset,
		   true - highlight selected item */
		cdw_list_display_scroll_to(browser->display, 0, 0, true);

		cdw_fs_browser_debug_report_current_fullpath(browser, "successfully read new dir");
		return 0;
	}
}





/**
   \brief Browse to file specified by given fullpath

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   If you know that \p fullpath specifies a regular file, you can use
   this function to make the \b browser go to file's parent dir and
   then scroll list display to the file. If you want to browse into a
   directory, use cdw_fs_browser_browse_into_dir().

   \param browser
   \param fullpath - full path to a regular file

   \return ENONET if there is no directory specified in fullpath
                  (or there are no access rights to the directory)
   \return ENOENT if there is a dir specified in fullpath, but there
                  is no file specified by fullpath
   \return 0 on success
*/
int cdw_fs_browser_browse_to_file(cdw_fs_browser_t *browser, char const * fullpath)
{
	ssize_t f = cdw_fs_get_filename_start(fullpath);
	cdw_assert (f >= 0, "ERROR: can't get start of file name in \"%s\"\n", fullpath);
	char *dirpath = strndup(fullpath, (size_t) f);
	cdw_assert (dirpath, "ERROR: failed to strndup() dirpath from \"%s\"\n", fullpath);

	int e = cdw_fs_browser_browse_into_dir(browser, dirpath);
	if (e) {
		cdw_vdm ("ERROR: failed to browse to new dir \"%s\"\n", dirpath);
		CDW_STRING_DELETE (dirpath);
		return ENONET;
	}


	/* We are in directory specified by dirpath, now we need to
	   move cursor (highlight) to file in this directory,
	   specified by fullpath; this search may not always succeed,
	   e.g. because the file has been removed from native file
	   system, or never existed. */
	bool found = false;
	size_t item_ind = 0;
	cdw_sdm ("INFO: ==== searching for file \"%s\"\n", fullpath + f);
	for (item_ind = 0; item_ind < browser->display->n_items; item_ind++) {

		/* TODO: optimize this loop by traversing list directly. */

		cdw_dll_item_t *list_item = cdw_dll_ith_item(browser->display->list, item_ind);
		cdw_file_t *file_from_list = (cdw_file_t *) list_item->data;
		cdw_sdm ("INFO: ==== matching against file \"%s\"\n", file_from_list->fullpath + file_from_list->name_start);

		if (!strcmp(fullpath + f, file_from_list->fullpath + file_from_list->name_start)) {
			found = true;
			break;
		}
	}

	if (found) {
		cdw_list_display_scroll_to(browser->display, item_ind, 0, true);
		CDW_STRING_DELETE (dirpath);

		return 0;
	} else {
		/* File may be not available for one reason: it is a
		   hidden file, but displaying of files has been
		   turned off in the middle of browsing; this is not
		   an error, handle this gracefully. */
		cdw_vdm ("WARNING: file \"%s\" not found in dir \"%s\"\n", fullpath + f, dirpath);

		CDW_STRING_DELETE (dirpath);

		ssize_t start = cdw_fs_get_filename_start(fullpath);
		if (cdw_fs_is_hidden(fullpath + start) && !global_config.general.display_hidden_files) {
			cdw_vdm ("WARNING: it may be because displaying hidden files has been disabled\n");
			cdw_list_display_scroll_to(browser->display, 0, 0, true);
			return 0;
		} else {
			return ENOENT;
		}
	}
}





char *cdw_fs_browser_get_current_dirpath(cdw_fs_browser_t const * browser)
{
	cdw_file_t * const file = cdw_fs_browser_get_current_file(browser);
	cdw_assert (file, "ERROR: can't get current file\n");
	ssize_t i = cdw_fs_get_filename_start(file->fullpath);
	cdw_assert (i > 0, "ERROR: can't get file name start\n");
	char *dirpath = strndup(file->fullpath, (size_t) i);
	cdw_assert (dirpath, "ERROR: can't strndup dirpath\n");
	return dirpath;
}





char *cdw_fs_browser_get_current_printable_dirpath(cdw_fs_browser_t *browser)
{
	cdw_file_t * const file = cdw_fs_browser_get_current_file(browser);
	cdw_assert (file, "ERROR: can't get current file\n");
	if (file->printable_fullpath) {
		ssize_t i = cdw_fs_get_filename_start(file->printable_fullpath);
		cdw_assert (i > 0, "ERROR: can't get file name start\n");
		char *dirpath = strndup(file->printable_fullpath, (size_t) i);
		cdw_assert (dirpath, "ERROR: can't strndup dirpath\n");
		return dirpath;
	} else {
		return cdw_fs_browser_get_current_dirpath(browser);
	}
}





char *cdw_fs_browser_get_current_fullpath(cdw_fs_browser_t *browser)
{
	cdw_file_t * const file = cdw_fs_browser_get_current_file(browser);
	cdw_assert (file, "ERROR: can't get current file\n");
	cdw_assert (file->fullpath, "ERROR: current file has no fullpath\n");
	/* file->fullpath is a valid pointer at this point, but if
	   you will pass it to function that re-reads (and thus reallocates)
	   list with current dir listing, then the pointer becomes
	   invalid an you are in trouble; so let's just allocate
	   new string */
	char *fullpath = strdup(file->fullpath);
	cdw_assert (fullpath, "ERROR: can't strndup fullpath\n");
	return fullpath;
}





/**
   Vertical scroll key.
*/
bool cdw_fs_browser_is_vscroll_key(int key)
{
	if (key == KEY_DOWN
	    || key == KEY_UP
	    || key == KEY_NPAGE
	    || key == KEY_PPAGE
	    || key == KEY_HOME
	    || key == KEY_END) {

		return true;
	} else {
		return false;
	}
}





/**
   \brief Display window with information about available hotkeys

   \date Function's top-level comment reviewed on 2016-02-06
   \date Function's body reviewed on 2016-02-06

   Information is displayed in window that is no larger than
   given \p browser, and \p browser window is automatically refreshed
   after closing help window.

   \p browser - fs browser, on top of which to display a help window
*/
void cdw_fs_browser_help_window(cdw_fs_browser_t *browser)
{
	int n_lines = 12;
	int n_cols = browser->display->n_cols < 50 ? browser->display->n_cols : 50;
	int begin_y = browser->parent_begin_y + browser->display->begin_y;
	int begin_x = browser->parent_begin_x + browser->display->begin_x;

	WINDOW *window = cdw_window_new(NULL,
					n_lines, n_cols,
					begin_y, begin_x,
					CDW_COLORS_DIALOG,
					/* 2TRANS: this is title of help window */
					_("Help"),
					/* 2TRANS: this is helper text at the bottom of help window */
					_("Press any key to close"));
	if (!window) {
		cdw_vdm ("ERROR: failed to create new window\n");
		return;
	}
	mvwprintw(window, 1, 2, _("Keys available in this window:"));
	mvwprintw(window, 3, 2, _(" Enter      - enter current dir"));
	mvwprintw(window, 4, 2, _(" Backspace  - go to parent directory"));
	mvwprintw(window, 5, 2, _(" .          - toggle displaying hidden files"));
	mvwprintw(window, 6, 2, _(" ~          - go to home directory"));
	mvwprintw(window, 7, 2, _(" < >        - horizontal scroll"));
	if (cdw_config_fs_lynx_like_motion()) {
		mvwprintw(window, 8, 2, _(" LEFT/RIGHT - file system navigation"));
	} else {
		mvwprintw(window, 8, 2, _(" LEFT/RIGHT - horizontal scroll"));
	}
	mvwprintw(window, 9, 2, _(" ?          - display this help"));

	wgetch(window);
	delwin(window);
	window = (WINDOW *) NULL;

	cdw_list_display_refresh(browser->display);

	return;
}





/**
   Normally we could just use cdw_widget_is_return_key(), but we
   need to take into consideration the "lynx-like motion" configuration
   option.
*/
bool cdw_fs_browser_is_return_key(cdw_fs_browser_t *browser, int key)
{
	if (!cdw_widget_is_return_key(&browser->widget, key)) {
		cdw_vdm ("INFO: is return key \"%s\": false\n", cdw_ncurses_key_label(key));
		return false;
	}

	if (key == KEY_RIGHT || key == KEY_LEFT) {
		if (!cdw_config_fs_lynx_like_motion()) {
			cdw_vdm ("INFO: is return key \"%s\": false\n", cdw_ncurses_key_label(key));
			return false;
		}
	}

	cdw_vdm ("INFO: is return key \"%s\": true\n", cdw_ncurses_key_label(key));
	return true;
}





/**
	\brief Add return key to self and internal widgets that given widget builds on
*/
void cdw_fs_browser_add_return_key(cdw_widget_t *widget, int key)
{
	cdw_assert (widget, "ERROR: \"widget\" argument is NULL\n");
	cdw_assert (widget->n_return_keys < CDW_WIDGET_N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return keys in the widget, can't add another one\n",
		    widget->n_return_keys, CDW_WIDGET_N_RETURN_KEYS_MAX);
	cdw_assert (key != 0, "ERROR: trying to add key == 0, but 0 is an initializer value\n");

	cdw_widget_add_return_key(widget, key);
	
	if (widget->self) {
		cdw_fs_browser_t *fs_browser = (cdw_fs_browser_t *) widget->self;
		cdw_list_display_add_return_key(fs_browser->display, key);
	}

	return;
}
