package COD::AtomProperties;

use strict;
use warnings;

require Exporter;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw( ok );

# The hash %atoms has been created using data from the following sources:
# http://www.ccdc.cam.ac.uk/products/csd/radii/table.php4 2008-07-16
#     (also provided in Meng, E. C. & Lewis, R. A.
#      Determination of molecular topology and atomic hybridization
#      states from heavy atom coordinates
#      Journal of Computational Chemistry, Wiley-Blackwell,
#      1991, 12, 891-898)
# http://www.economicexpert.com/a/Valency.html 2008-07-16
# %atoms = (
#           H => { #(chemical_type)
#                     name => Hydrogen,
#                     period => 1,
#                     group => 1,
#                     block => s,
#                     atomic_number => "1",
#                     atomic_weight => 1.008,
#                     covalent_radius => 0.23,
#                     vdw_radius => 1.09,
#                     valency => [1],
#                     },
#          );

%COD::AtomProperties::atoms = (
     "." => {
             "name" => "Dummy",
             "period" => "0",
             "group" => "0",
             "block" => "0",
             "atomic_number" => "0",
             "atomic_weight" => "0",
             # covalent and Van der Waals radii copied from carbon
             "covalent_radius" => 0.68,
             "vdw_radius" => 1.70,
             "valency" => [0],
     },
     "H" => {
             "name" => "Hydrogen",
             "period" => "1",
             "group" => "1",
             "block" => "s",
             "atomic_number" => "1",
             "atomic_weight" => "1.008",
             # Note: the vdW radius is taken from Wikipedia as more
             # realistic (also confirmed by Alvarez2012 and LANL). The
             # covalent radius, however, is left as it was in the old
             # table of the COD AtomProperties, to avoid breaking
             # existing computations.
             # (S.G. 2019-11-06)
             #
             # http://en.wikipedia.org/wiki/Hydrogen [2019-11-06]
             # gives covalent radius 37 pm (0.37A), and vdW radius
             # 120 pm (1.25A)
             "covalent_radius" => 0.23,
             "vdw_radius" => 1.2,
             "valency" => [1],
           },
     "D" => {
             "name" => "Deuterium",
             "period" => "1",
             "group" => "1",
             "block" => "s",
             "atomic_number" => "1",
             "atomic_weight" => "2.014",
                 # http://en.wikipedia.org/wiki/Deuterium 2009-06-15
             "covalent_radius" => 0.23, # Taken from H entry
             "vdw_radius" => 1.09,      # Taken from H entry
             "valency" => [1],
           },
     "He" => {
             "name" => "Helium",
             "period" => "1",
             "group" => "18",
             "block" => "p",
             "atomic_number" => "2",
             "atomic_weight" => "4.003",
             "covalent_radius" => 1.50,
             "vdw_radius" => 1.40,
             "valency" => [0],
           },
     "Li" => {
             "name" => "Lithium",
             "period" => "2",
             "group" => "1",
             "block" => "s",
             "atomic_number" => "3",
             "atomic_weight" => "6.941",
             "covalent_radius" => 0.68,
             "vdw_radius" => 1.82,
             "valency" => [1],
           },
     "Be" => {
             "name" => "Beryllium",
             "period" => "2",
             "group" => "2",
             "block" => "s",
             "atomic_number" => "4",
             "atomic_weight" => "9.012",
             "covalent_radius" => 0.35,
             "vdw_radius" => 2.00,
             "valency" => [2],
           },
     "B" => {
             "name" => "Boron",
             "period" => "2",
             "group" => "13",
             "block" => "p",
             "atomic_number" => "5",
             "atomic_weight" => "10.811",
             "covalent_radius" => 0.83,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "C" => {
             "name" => "Carbon",
             "period" => "2",
             "group" => "14",
             "block" => "p",
             "atomic_number" => "6",
             "atomic_weight" => "12.011",
             "covalent_radius" => 0.68,
             "vdw_radius" => 1.70,
             "valency" => [2,4],
           },
     "N" => {
             "name" => "Nitrogen",
             "period" => "2",
             "group" => "15",
             "block" => "p",
             "atomic_number" => "7",
             "atomic_weight" => "14.007",
             "covalent_radius" => 0.68,
             "vdw_radius" => 1.55,
             "valency" => [3,5],
           },
     "O" => {
             "name" => "Oxygen",
             "period" => "2",
             "group" => "16",
             "block" => "p",
             "atomic_number" => "8",
             "atomic_weight" => "15.999",
             "covalent_radius" => 0.68,
             "vdw_radius" => 1.52,
             "valency" => [2],
           },
     "F" => {
             "name" => "Fluorine",
             "period" => "2",
             "group" => "17",
             "block" => "p",
             "atomic_number" => "9",
             "atomic_weight" => "18.998",
             "covalent_radius" => 0.64,
             "vdw_radius" => 1.47,
             "valency" => [1],
           },
     "Ne" => {
             "name" => "Neon",
             "period" => "2",
             "group" => "18",
             "block" => "p",
             "atomic_number" => "10",
             "atomic_weight" => "20.180",
             "covalent_radius" => 1.50,
             "vdw_radius" => 1.54,
             "valency" => [0],
           },
     "Na" => {
             "name" => "Sodium",
             "period" => "3",
             "group" => "1",
             "block" => "s",
             "atomic_number" => "11",
             "atomic_weight" => "22.991",
             "covalent_radius" => 0.97,
             "vdw_radius" => 2.27,
             "valency" => [1],
           },
     "Mg" => {
             "name" => "Magnesium",
             "period" => "3",
             "group" => "2",
             "block" => "s",
             "atomic_number" => "12",
             "atomic_weight" => "24.305",
             "covalent_radius" => 1.10,
             "vdw_radius" => 1.73,
             "valency" => [2],
           },
     "Al" => {
             "name" => "Aluminium",
             "period" => "3",
             "group" => "13",
             "block" => "p",
             "atomic_number" => "13",
             "atomic_weight" => "26.982",
             "covalent_radius" => 1.35,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "Si" => {
             "name" => "Silicon",
             "period" => "3",
             "group" => "14",
             "block" => "p",
             "atomic_number" => "14",
             "atomic_weight" => "28.086",
             "covalent_radius" => 1.20,
             "vdw_radius" => 2.10,
             "valency" => [4],
           },
     "P" => {
             "name" => "Phosphorus",
             "period" => "3",
             "group" => "15",
             "block" => "p",
             "atomic_number" => "15",
             "atomic_weight" => "30.974",
             "covalent_radius" => 1.05,
             "vdw_radius" => 1.80,
             "valency" => [3,5],
           },
     "S" => {
             "name" => "Sulphur",
             "period" => "3",
             "group" => "16",
             "block" => "p",
             "atomic_number" => "16",
             "atomic_weight" => "32.066",
             "covalent_radius" => 1.02,
             "vdw_radius" => 1.80,
             "valency" => [2,4,6],
           },
     "Cl" => {
             "name" => "Chlorine",
             "period" => "3",
             "group" => "17",
             "block" => "p",
             "atomic_number" => "17",
             "atomic_weight" => "35.453",
             "covalent_radius" => 0.99,
             "vdw_radius" => 1.75,
             "valency" => [1,3,5,7],
           },
     "Ar" => {
             "name" => "Argon",
             "period" => "3",
             "group" => "18",
             "block" => "p",
             "atomic_number" => "18",
             "atomic_weight" => "39.948",
             "covalent_radius" => 1.51,
             "vdw_radius" => 1.88,
             "valency" => [0],
           },
     "K" => {
             "name" => "Potassium",
             "period" => "4",
             "group" => "1",
             "block" => "s",
             "atomic_number" => "19",
             "atomic_weight" => "39.098",
             "covalent_radius" => 1.33,
             "vdw_radius" => 2.75,
             "valency" => [1],
           },
     "Ca" => {
             "name" => "Calcium",
             "period" => "4",
             "group" => "2",
             "block" => "s",
             "atomic_number" => "20",
             "atomic_weight" => "40.078",
             "covalent_radius" => 0.99,
             "vdw_radius" => 2.00,
             "valency" => [3,5],
           },
     "Sc" => {
             "name" => "Scandium",
             "period" => "4",
             "group" => "3",
             "block" => "d",
             "atomic_number" => "21",
             "atomic_weight" => "44.956",
             "covalent_radius" => 1.44,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "Ti" => {
             "name" => "Titanium",
             "period" => "4",
             "group" => "4",
             "block" => "d",
             "atomic_number" => "22",
             "atomic_weight" => "47.867",
             "covalent_radius" => 1.47,
             "vdw_radius" => 2.00,
             "valency" => [3,4],
           },
     "V" => {
             "name" => "Vanadium",
             "period" => "4",
             "group" => "5",
             "block" => "d",
             "atomic_number" => "23",
             "atomic_weight" => "50.942",
             "covalent_radius" => 1.33,
             "vdw_radius" => 2.00,
             "valency" => [2,3,4,5],
           },
     "Cr" => {
             "name" => "Chromium",
             "period" => "4",
             "group" => "6",
             "block" => "d",
             "atomic_number" => "24",
             "atomic_weight" => "51.996",
             "covalent_radius" => 1.35,
             "vdw_radius" => 2.00,
             "valency" => [2,3,6],
           },
     "Mn" => {
             "name" => "Manganese",
             "period" => "4",
             "group" => "7",
             "block" => "d",
             "atomic_number" => "25",
             "atomic_weight" => "54.938",
             "covalent_radius" => 1.35,
             "vdw_radius" => 2.00,
             "valency" => [2,3,4,6,7],
           },
     "Fe" => {
             "name" => "Iron",
             "period" => "4",
             "group" => "8",
             "block" => "d",
             "atomic_number" => "26",
             "atomic_weight" => "55.845",
             "covalent_radius" => 1.34,
             "vdw_radius" => 2.00,
             "valency" => [2,3],
           },
     "Co" => {
             "name" => "Cobalt",
             "period" => "4",
             "group" => "9",
             "block" => "d",
             "atomic_number" => "27",
             "atomic_weight" => "58.933",
             "covalent_radius" => 1.33,
             "vdw_radius" => 2.00,
             "valency" => [2,3],
           },
     "Ni" => {
             "name" => "Nickel",
             "period" => "4",
             "group" => "10",
             "block" => "d",
             "atomic_number" => "28",
             "atomic_weight" => "58.693",
             "covalent_radius" => 1.50,
             "vdw_radius" => 1.63,
             "valency" => [2,3],
           },
     "Cu" => {
             "name" => "Copper",
             "period" => "4",
             "group" => "11",
             "block" => "d",
             "atomic_number" => "29",
             "atomic_weight" => "63.546",
             "covalent_radius" => 1.52,
             "vdw_radius" => 1.40,
             "valency" => [1,2],
           },
     "Zn" => {
             "name" => "Zinc",
             "period" => "4",
             "group" => "12",
             "block" => "d",
             "atomic_number" => "30",
             "atomic_weight" => "65.390",
             "covalent_radius" => 1.45,
             "vdw_radius" => 1.39,
             "valency" => [2],
           },
     "Ga" => {
             "name" => "Gallium",
             "period" => "4",
             "group" => "13",
             "block" => "p",
             "atomic_number" => "31",
             "atomic_weight" => "69.723",
             "covalent_radius" => 1.22,
             "vdw_radius" => 1.87,
             "valency" => [2,3],
           },
     "Ge" => {
             "name" => "Germanium",
             "period" => "4",
             "group" => "14",
             "block" => "p",
             "atomic_number" => "32",
             "atomic_weight" => "72.610",
             "covalent_radius" => 1.17,
             "vdw_radius" => 2.00,
             "valency" => [4],
           },
     "As" => {
             "name" => "Arsenic",
             "period" => "4",
             "group" => "15",
             "block" => "p",
             "atomic_number" => "33",
             "atomic_weight" => "74.922",
             "covalent_radius" => 1.21,
             "vdw_radius" => 1.85,
             "valency" => [3,5],
           },
     "Se" => {
             "name" => "Selenium",
             "period" => "4",
             "group" => "16",
             "block" => "p",
             "atomic_number" => "34",
             "atomic_weight" => "78.960",
             "covalent_radius" => 1.22,
             "vdw_radius" => 1.90,
             "valency" => [2,4,6],
           },
     "Br" => {
             "name" => "Bromine",
             "period" => "4",
             "group" => "17",
             "block" => "p",
             "atomic_number" => "35",
             "atomic_weight" => "79.904",
             "covalent_radius" => 1.21,
             "vdw_radius" => 1.85,
             "valency" => [1,3,5,7],
           },
     "Kr" => {
             "name" => "Krypton",
             "period" => "4",
             "group" => "18",
             "block" => "p",
             "atomic_number" => "36",
             "atomic_weight" => "83.800",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.02,
             "valency" => [0],
           },
     "Rb" => {
             "name" => "Rubidium",
             "period" => "5",
             "group" => "1",
             "block" => "s",
             "atomic_number" => "37",
             "atomic_weight" => "85.468",
             "covalent_radius" => 1.47,
             "vdw_radius" => 2.00,
             "valency" => [1],
           },
     "Sr" => {
             "name" => "Strontium",
             "period" => "5",
             "group" => "2",
             "block" => "s",
             "atomic_number" => "38",
             "atomic_weight" => "87.620",
             "covalent_radius" => 1.12,
             "vdw_radius" => 2.00,
             "valency" => [2],
           },
     "Y" => {
             "name" => "Yttrium",
             "period" => "5",
             "group" => "3",
             "block" => "d",
             "atomic_number" => "39",
             "atomic_weight" => "88.906",
             "covalent_radius" => 1.78,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "Zr" => {
             "name" => "Zirconium",
             "period" => "5",
             "group" => "4",
             "block" => "d",
             "atomic_number" => "40",
             "atomic_weight" => "91.224",
             "covalent_radius" => 1.56,
             "vdw_radius" => 2.00,
             "valency" => [4],
           },
     "Nb" => {
             "name" => "Niobium",
             "period" => "5",
             "group" => "5",
             "block" => "d",
             "atomic_number" => "41",
             "atomic_weight" => "92.906",
             "covalent_radius" => 1.48,
             "vdw_radius" => 2.00,
             "valency" => [3,5],
           },
     "Mo" => {
             "name" => "Molybdenum",
             "period" => "5",
             "group" => "6",
             "block" => "d",
             "atomic_number" => "42",
             "atomic_weight" => "95.940",
             "covalent_radius" => 1.47,
             "vdw_radius" => 2.00,
             "valency" => [2,3,4,5,6],
           },
     "Tc" => {
             "name" => "Technetium",
             "period" => "5",
             "group" => "7",
             "block" => "d",
             "atomic_number" => "43",
             "atomic_weight" => "[98]",
             "covalent_radius" => 1.35,
             "vdw_radius" => 2.00,
             "valency" => [2,3,4,6,7],
           },
     "Ru" => {
             "name" => "Ruthenium",
             "period" => "5",
             "group" => "8",
             "block" => "d",
             "atomic_number" => "44",
             "atomic_weight" => "101.070",
             "covalent_radius" => 1.40,
             "vdw_radius" => 2.00,
             "valency" => [3,4,6,8],
           },
     "Rh" => {
             "name" => "Rhodium",
             "period" => "5",
             "group" => "9",
             "block" => "d",
             "atomic_number" => "45",
             "atomic_weight" => "102.906",
             "covalent_radius" => 1.45,
             "vdw_radius" => 2.00,
             "valency" => [3,4],
           },
     "Pd" => {
             "name" => "Palladium",
             "period" => "5",
             "group" => "10",
             "block" => "d",
             "atomic_number" => "46",
             "atomic_weight" => "106.420",
             "covalent_radius" => 1.50,
             "vdw_radius" => 1.63,
             "valency" => [2,4],
           },
     "Ag" => {
             "name" => "Silver",
             "period" => "5",
             "group" => "11",
             "block" => "d",
             "atomic_number" => "47",
             "atomic_weight" => "107.868",
             "covalent_radius" => 1.59,
             "vdw_radius" => 1.72,
             "valency" => [1],
           },
     "Cd" => {
             "name" => "Cadmium",
             "period" => "5",
             "group" => "12",
             "block" => "d",
             "atomic_number" => "48",
             "atomic_weight" => "112.411",
             "covalent_radius" => 1.69,
             "vdw_radius" => 1.58,
             "valency" => [2],
           },
     "In" => {
             "name" => "Indium",
             "period" => "5",
             "group" => "13",
             "block" => "p",
             "atomic_number" => "49",
             "atomic_weight" => "114.818",
             "covalent_radius" => 1.63,
             "vdw_radius" => 1.93,
             "valency" => [1,3],
           },
     "Sn" => {
             "name" => "Tin",
             "period" => "5",
             "group" => "14",
             "block" => "p",
             "atomic_number" => "50",
             "atomic_weight" => "118.71",
             "covalent_radius" => 1.46,
             "vdw_radius" => 2.17,
             "valency" => [2,4],
           },
     "Sb" => {
             "name" => "Antimony",
             "period" => "5",
             "group" => "15",
             "block" => "p",
             "atomic_number" => "51",
             "atomic_weight" => "121.760",
             "covalent_radius" => 1.46,
             "vdw_radius" => 2.00,
             "valency" => [3,5],
           },
     "Te" => {
             "name" => "Tellurium",
             "period" => "5",
             "group" => "16",
             "block" => "p",
             "atomic_number" => "52",
             "atomic_weight" => "127.600",
             "covalent_radius" => 1.47,
             "vdw_radius" => 2.06,
             "valency" => [2,4,6],
           },
     "I" => {
             "name" => "Iodine",
             "period" => "5",
             "group" => "17",
             "block" => "p",
             "atomic_number" => "53",
             "atomic_weight" => "126.904",
             "covalent_radius" => 1.40,
             "vdw_radius" => 1.98,
             "valency" => [1,3,5,7],
           },
     "Xe" => {
             "name" => "Xenon",
             "period" => "5",
             "group" => "18",
             "block" => "p",
             "atomic_number" => "54",
             "atomic_weight" => "131.290",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.16,
             "valency" => [0],
           },
     "Cs" => {
             "name" => "Caesium",
             "period" => "6",
             "group" => "1",
             "block" => "s",
             "atomic_number" => "55",
             "atomic_weight" => "132.905",
             "covalent_radius" => 1.67,
             "vdw_radius" => 2.00,
             "valency" => [1],
           },
     "Ba" => {
             "name" => "Barium",
             "period" => "6",
             "group" => "2",
             "block" => "s",
             "atomic_number" => "56",
             "atomic_weight" => "137.327",
             "covalent_radius" => 1.34,
             "vdw_radius" => 2.00,
             "valency" => [2],
           },
     "Lu" => {
             "name" => "Lutetium",
             "period" => "6",
             "group" => "3",
             "block" => "d",
             "atomic_number" => "71",
             "atomic_weight" => "174.967",
             "covalent_radius" => 1.72,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "Hf" => {
             "name" => "Hafnium",
             "period" => "6",
             "group" => "4",
             "block" => "d",
             "atomic_number" => "72",
             "atomic_weight" => "178.490",
             "covalent_radius" => 1.57,
             "vdw_radius" => 2.00,
             "valency" => [4],
           },
     "Ta" => {
             "name" => "Tantalum",
             "period" => "6",
             "group" => "5",
             "block" => "d",
             "atomic_number" => "73",
             "atomic_weight" => "180.948",
             "covalent_radius" => 1.43,
             "vdw_radius" => 2.00,
             "valency" => [3,5],
           },
     "W" => {
             "name" => "Tungsten",
             "period" => "6",
             "group" => "6",
             "block" => "d",
             "atomic_number" => "74",
             "atomic_weight" => "183.840",
             "covalent_radius" => 1.37,
             "vdw_radius" => 2.00,
             "valency" => [2,4,5,6],
           },
     "Re" => {
             "name" => "Rhenium",
             "period" => "6",
             "group" => "7",
             "block" => "d",
             "atomic_number" => "75",
             "atomic_weight" => "186.207",
             "covalent_radius" => 1.35,
             "vdw_radius" => 2.00,
             "valency" => [1,4,7],
           },
     "Os" => {
             "name" => "Osmium",
             "period" => "6",
             "group" => "8",
             "block" => "d",
             "atomic_number" => "76",
             "atomic_weight" => "190.230",
             "covalent_radius" => 1.37,
             "vdw_radius" => 2.00,
             "valency" => [2,3,4,6,8],
           },
     "Ir" => {
             "name" => "Iridium",
             "period" => "6",
             "group" => "9",
             "block" => "d",
             "atomic_number" => "77",
             "atomic_weight" => "192.217",
             "covalent_radius" => 1.32,
             "vdw_radius" => 2.00,
             "valency" => [3,4],
           },
     "Pt" => {
             "name" => "Platinum",
             "period" => "6",
             "group" => "10",
             "block" => "d",
             "atomic_number" => "78",
             "atomic_weight" => "195.078",
             "covalent_radius" => 1.50,
             "vdw_radius" => 1.72,
             "valency" => [2,4],
           },
     "Au" => {
             "name" => "Gold",
             "period" => "6",
             "group" => "11",
             "block" => "d",
             "atomic_number" => "79",
             "atomic_weight" => "196.967",
             "covalent_radius" => 1.50,
             "vdw_radius" => 1.66,
             "valency" => [1,3],
           },
     "Hg" => {
             "name" => "Mercury",
             "period" => "6",
             "group" => "12",
             "block" => "d",
             "atomic_number" => "80",
             "atomic_weight" => "200.590",
             "covalent_radius" => 1.70,
             "vdw_radius" => 1.55,
             "valency" => [1,2],
           },
     "Tl" => {
             "name" => "Thallium",
             "period" => "6",
             "group" => "13",
             "block" => "p",
             "atomic_number" => "81",
             "atomic_weight" => "204.383",
             "covalent_radius" => 1.55,
             "vdw_radius" => 1.96,
             "valency" => [1,3],
           },
     "Pb" => {
             "name" => "Lead",
             "period" => "6",
             "group" => "14",
             "block" => "p",
             "atomic_number" => "82",
             "atomic_weight" => "207.200",
             "covalent_radius" => 1.54,
             "vdw_radius" => 2.02,
             "valency" => [2,4],
           },
     "Bi" => {
             "name" => "Bismuth",
             "period" => "6",
             "group" => "15",
             "block" => "p",
             "atomic_number" => "83",
             "atomic_weight" => "208.980",
             "covalent_radius" => 1.54,
             "vdw_radius" => 2.00,
             "valency" => [3,5],
           },
     "Po" => {
             "name" => "Polonium",
             "period" => "6",
             "group" => "16",
             "block" => "p",
             "atomic_number" => "84",
             "atomic_weight" => "[210]",
             "covalent_radius" => 1.68,
             "vdw_radius" => 2.00,
             "valency" => [2,3,4],
           },
     "At" => {
             "name" => "Astatine",
             "period" => "6",
             "group" => "17",
             "block" => "p",
             "atomic_number" => "85",
             "atomic_weight" => "[210]",
             "covalent_radius" => 1.21,
             "vdw_radius" => 2.00,
             "valency" => [1,3,5,7],
           },
     "Rn" => {
             "name" => "Radon",
             "period" => "6",
             "group" => "18",
             "block" => "p",
             "atomic_number" => "86",
             "atomic_weight" => "[222]",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.00,
             "valency" => [0],
           },
     "Ce" => {
             "name" => "Cerium",
             "period" => "6Lan",
             "group" => "19",
             "block" => "f",
             "atomic_number" => "58",
             "atomic_weight" => "140.116",
             "covalent_radius" => 1.83,
             "vdw_radius" => 2.00,
             "valency" => [3,4],
           },
     "Dy" => {
             "name" => "Dysprosium",
             "period" => "6Lan",
             "group" => "19",
             "block" => "f",
             "atomic_number" => "66",
             "atomic_weight" => "162.500",
             "covalent_radius" => 1.75,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "Er" => {
             "name" => "Erbium",
             "period" => "6Lan",
             "group" => "19",
             "block" => "f",
             "atomic_number" => "68",
             "atomic_weight" => "167.260",
             "covalent_radius" => 1.73,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "Eu" => {
             "name" => "Europium",
             "period" => "6Lan",
             "group" => "19",
             "block" => "f",
             "atomic_number" => "63",
             "atomic_weight" => "151.964",
             "covalent_radius" => 1.99,
             "vdw_radius" => 2.00,
             "valency" => [2,3],
           },
     "Gd" => {
             "name" => "Gadolinium",
             "period" => "6Lan",
             "group" => "19",
             "block" => "f",
             "atomic_number" => "64",
             "atomic_weight" => "157.250",
             "covalent_radius" => 1.79,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "Ho" => {
             "name" => "Holmium",
             "period" => "6Lan",
             "group" => "19",
             "block" => "f",
             "atomic_number" => "67",
             "atomic_weight" => "164.930",
             "covalent_radius" => 1.74,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "La" => {
             "name" => "Lanthanum",
             "period" => "6Lan",
             "group" => "19",
             "block" => "f",
             "atomic_number" => "57",
             "atomic_weight" => "138.906",
             "covalent_radius" => 1.87,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "Nd" => {
             "name" => "Neodymium",
             "period" => "6Lan",
             "group" => "19",
             "block" => "f",
             "atomic_number" => "60",
             "atomic_weight" => "144.240",
             "covalent_radius" => 1.81,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "Pm" => {
             "name" => "Promethium",
             "period" => "6Lan",
             "group" => "19",
             "block" => "f",
             "atomic_number" => "61",
             "atomic_weight" => "[145]",
             "covalent_radius" => 1.80,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "Pr" => {
             "name" => "Praseodymium",
             "period" => "6Lan",
             "group" => "19",
             "block" => "f",
             "atomic_number" => "59",
             "atomic_weight" => "140.908",
             "covalent_radius" => 1.82,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "Sm" => {
             "name" => "Samarium",
             "period" => "6Lan",
             "group" => "19",
             "block" => "f",
             "atomic_number" => "62",
             "atomic_weight" => "150.360",
             "covalent_radius" => 1.80,
             "vdw_radius" => 2.00,
             "valency" => [2,3],
           },
     "Tb" => {
             "name" => "Terbium",
             "period" => "6Lan",
             "group" => "19",
             "block" => "f",
             "atomic_number" => "65",
             "atomic_weight" => "158.925",
             "covalent_radius" => 1.76,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "Tm" => {
             "name" => "Thulium",
             "period" => "6Lan",
             "group" => "19",
             "block" => "f",
             "atomic_number" => "69",
             "atomic_weight" => "168.934",
             "covalent_radius" => 1.72,
             "vdw_radius" => 2.00,
             "valency" => [2,3],
           },
     "Yb" => {
             "name" => "Ytterbium",
             "period" => "6Lan",
             "group" => "19",
             "block" => "f",
             "atomic_number" => "70",
             "atomic_weight" => "173.040",
             "covalent_radius" => 1.94,
             "vdw_radius" => 2.00,
             "valency" => [2,3],
           },
     "Fr" => {
             "name" => "Francium",
             "period" => "7",
             "group" => "1",
             "block" => "s",
             "atomic_number" => "87",
             "atomic_weight" => "[223]",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.00,
             "valency" => [1],
           },
     "Ra" => {
             "name" => "Radium",
             "period" => "7",
             "group" => "2",
             "block" => "s",
             "atomic_number" => "88",
             "atomic_weight" => "[226]",
             "covalent_radius" => 1.90,
             "vdw_radius" => 2.00,
             "valency" => [2],
           },
     "Lr" => {
             "name" => "Lawrencium",
             "period" => "7",
             "group" => "3",
             "block" => "d",
             "atomic_number" => "103",
             "atomic_weight" => "[262]",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "Rf" => {
             "name" => "Rutherfordium",
             "period" => "7",
             "group" => "4",
             "block" => "d",
             "atomic_number" => "104",
             "atomic_weight" => "[261]",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.00,
             "valency" => [],
           },
     "Db" => {
             "name" => "Dubnium",
             "period" => "7",
             "group" => "5",
             "block" => "d",
             "atomic_number" => "105",
             "atomic_weight" => "[262]",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.00,
             "valency" => [],
           },
     "Sg" => {
             "name" => "Seaborgium",
             "period" => "7",
             "group" => "6",
             "block" => "d",
             "atomic_number" => "106",
             "atomic_weight" => "[266]",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.00,
             "valency" => [],
           },
     "Bh" => {
             "name" => "Bohrium",
             "period" => "7",
             "group" => "7",
             "block" => "d",
             "atomic_number" => "107",
             "atomic_weight" => "[264]",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.00,
             "valency" => [],
           },
     "Hs" => {
             "name" => "Hassium",
             "period" => "7",
             "group" => "8",
             "block" => "d",
             "atomic_number" => "108",
             "atomic_weight" => "[269]",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.00,
             "valency" => [],
           },
     "Mt" => {
             "name" => "Meitnerium",
             "period" => "7",
             "group" => "9",
             "block" => "d",
             "atomic_number" => "109",
             "atomic_weight" => "[268]",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.00,
             "valency" => [],
           },
     "Ds" => {
             "name" => "Darmstadtium",
             "period" => "7",
             "group" => "10",
             "block" => "d",
             "atomic_number" => "110",
             "atomic_weight" => "[271]",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.00,
             "valency" => [],
           },
     "Ac" => {
             "name" => "Actinium",
             "period" => "7Act",
             "group" => "20",
             "block" => "f",
             "atomic_number" => "89",
             "atomic_weight" => "[227]",
             "covalent_radius" => 1.88,
             "vdw_radius" => 2.00,
             "valency" => [3],
           },
     "Am" => {
             "name" => "Americium",
             "period" => "7Act",
             "group" => "20",
             "block" => "f",
             "atomic_number" => "95",
             "atomic_weight" => "[243]",
             "covalent_radius" => 1.51,
             "vdw_radius" => 2.00,
             "valency" => [2,3,4,5,6],
           },
     "Bk" => {
             "name" => "Berkelium",
             "period" => "7Act",
             "group" => "20",
             "block" => "f",
             "atomic_number" => "97",
             "atomic_weight" => "[247]",
             "covalent_radius" => 1.54,
             "vdw_radius" => 2.00,
             "valency" => [2,3,4],
           },
     "Cf" => {
             "name" => "Californium",
             "period" => "7Act",
             "group" => "20",
             "block" => "f",
             "atomic_number" => "98",
             "atomic_weight" => "[251]",
             "covalent_radius" => 1.83,
             "vdw_radius" => 2.00,
             "valency" => [2,3,4],
           },
     "Cm" => {
             "name" => "Curium",
             "period" => "7Act",
             "group" => "20",
             "block" => "f",
             "atomic_number" => "96",
             "atomic_weight" => "[247]",
             "covalent_radius" => 0.99,
             "vdw_radius" => 2.00,
             "valency" => [2,3,4],
           },
     "Es" => {
             "name" => "Einsteinium",
             "period" => "7Act",
             "group" => "20",
             "block" => "f",
             "atomic_number" => "99",
             "atomic_weight" => "[252]",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.00,
             "valency" => [2,3],
           },
     "Fm" => {
             "name" => "Fermium",
             "period" => "7Act",
             "group" => "20",
             "block" => "f",
             "atomic_number" => "100",
             "atomic_weight" => "[257]",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.00,
             "valency" => [2,3],
           },
     "Md" => {
             "name" => "Mendelevium",
             "period" => "7Act",
             "group" => "20",
             "block" => "f",
             "atomic_number" => "101",
             "atomic_weight" => "[258]",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.00,
             "valency" => [2,3],
           },
     "No" => {
             "name" => "Nobelium",
             "period" => "7Act",
             "group" => "20",
             "block" => "f",
             "atomic_number" => "102",
             "atomic_weight" => "[259]",
             "covalent_radius" => 1.50,
             "vdw_radius" => 2.00,
             "valency" => [2,3],
           },
     "Np" => {
             "name" => "Neptunium",
             "period" => "7Act",
             "group" => "20",
             "block" => "f",
             "atomic_number" => "93",
             "atomic_weight" => "[237]",
             "covalent_radius" => 1.55,
             "vdw_radius" => 2.00,
             "valency" => [2,3,4,5,6],
           },
     "Pa" => {
             "name" => "Protactinium",
             "period" => "7Act",
             "group" => "20",
             "block" => "f",
             "atomic_number" => "91",
             "atomic_weight" => "231.036",
             "covalent_radius" => 1.61,
             "vdw_radius" => 2.00,
             "valency" => [4,5],
           },
     "Pu" => {
             "name" => "Plutonium",
             "period" => "7Act",
             "group" => "20",
             "block" => "f",
             "atomic_number" => "94",
             "atomic_weight" => "[244]",
             "covalent_radius" => 1.53,
             "vdw_radius" => 2.00,
             "valency" => [2,3,4,5,6],
           },
     "Th" => {
             "name" => "Thorium",
             "period" => "7Act",
             "group" => "20",
             "block" => "f",
             "atomic_number" => "90",
             "atomic_weight" => "232.038",
             "covalent_radius" => 1.79,
             "vdw_radius" => 2.00,
             "valency" => [4],
           },
     "U" => {
             "name" => "Uranium",
             "period" => "7Act",
             "group" => "20",
             "block" => "f",
             "atomic_number" => "92",
             "atomic_weight" => "238.029",
             "covalent_radius" => 1.58,
             "vdw_radius" => 1.86,
             "valency" => [3,4,5,6],
           },
);

1;
