@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
perl -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/usr/bin/perl -w
#line 15
#
# ecaccess-file-dir: List ECaccess Directory Contents
#
# Laurent.Gougeon@ecmwf.int - 2010-10-15

use ECMWF::ECaccess;
use Getopt::Long;
use Pod::Usage;
use Term::ReadKey;

my %opt = ( long => 0, version => 0, help => 0, manual => 0, retry => 0, debug => 0 );

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if !GetOptions(
	\%opt,
	qw(
	  long
	  version
	  help|?
	  manual
	  retry=i
	  debug
	  )
);

# Display version if requested
die ECMWF::ECaccess->VERSION . "\n" if ( $opt{version} );

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if ( $opt{help} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 2 ) if ( $opt{manual} );

my $ecaccessFile = $ARGV[0];

# If no file specified then list home directory
$ecaccessFile = '' if ( not($ecaccessFile) );

# Create the ECaccess Controler
my $ecaccess = ECMWF::ECaccess->new( $opt{retry}, $opt{debug});

# Get the Token (using the Certificate in $HOME)
my $token = $ecaccess->getToken();

# Get the Control Channel
my $controlChannel = $ecaccess->getControlChannel();

# List the files
my $ecaccessFiles = $controlChannel->getDirList( $token, $ecaccessFile );

# Display the information for each file
foreach my $ecaccessFile ( $ecaccessFiles->valueof('//getDirListResponse/return') ) {
	if ( $opt{long} ) {
		printf "%s   1 %-8s %-8s %-11s %s %s", $ecaccessFile->{permissions}, $ecaccessFile->{user}, $ecaccessFile->{group}, $ecaccessFile->{size},
		  $ecaccessFile->{time}, $ecaccessFile->{name};
		print " -> " . $ecaccessFile->{link} if ( $ecaccessFile->{link} );
		print "\n";
	}
	else {
		print $ecaccessFile->{name} . "\n";
	}
}

# Logout
$ecaccess->releaseToken($token);

__END__

=head1 NAME

ecaccess-file-dir - List ECaccess Directory Contents

=head1 SYNOPSIS

B<ecaccess-file-dir -version|-help|-manual>

B<ecaccess-file-dir [-debug] [-long]> B<[>I<ecaccess-file>B<]>

=head1 DESCRIPTION

List information about I<ecaccess-file> (the $HOME directory by default). Sort entries alphabetically.

The I<ecaccess-file> is in the form [domain:][/user-id/]path. Please read the "Shell commands -> File Management"
section of the "ecaccess" guide for more information on the ECaccess File System.

=head1 ARGUMENTS

=over 8

=item I<ecaccess-file> (optional)

The name of the ECaccess File/Directory to retrieve the details.

=back

=head1 OPTIONS

=over 8

=item B<-long>

Print detailed information about each file.

=item B<-version>

Display version number and exits.

=item B<-help>

Print a brief help message and exits.

=item B<-manual>

Prints the manual page and exits.

=item B<-retry> I<count>

Number of SSL connection retries per 5s to ECMWF. This parameter only apply to the
initial SSL connection initiated by the command to the ECMWF server. It does not
apply to all the subsequent requests made afteward as it is mainly targeting errors
that can happen from time to time during the SSL handshake. Default is no retry.

=item B<-debug>

Display the SOAP and SSL messages exchanged.

=back

=head1 EXAMPLES

B<ecaccess-file-dir>

List information about each ECaccess File in the $HOME directory of the authenticated user.

B<ecaccess-file-dir> I<home:/xyz/bin>

List information about each ECaccess File in the $HOME/bin directory of the user xyz.

B<ecaccess-file-dir> I<'home:/xyz/*.txt'>

List information about each ECaccess File ending with ".txt" in the $HOME/bin directory
of the user xyz.

=head1 SEE ALSO

B<ecaccess-file-delete>, B<ecaccess-file-get>, B<ecaccess-file-mget>, B<ecaccess-file-modtime>, B<ecaccess-file-mput>,
B<ecaccess-file-rmdir>, B<ecaccess-file-copy>, B<ecaccess-file-chmod>, B<ecaccess-file-mdelete>, B<ecaccess-file-mkdir>,
B<ecaccess-file-move>, B<ecaccess-file-put>, B<ecaccess-file-size> and B<ecaccess>.

=cut

__END__
:endofperl
