# 🪄 go-safecast: safe numbers conversion

[![Go Report Card](https://goreportcard.com/badge/github.com/ccoveille/go-safecast)](https://goreportcard.com/report/github.com/ccoveille/go-safecast)
[![GoDoc](https://godoc.org/github.com/ccoVeille/go-safecast?status.svg)](https://godoc.org/github.com/ccoVeille/go-safecast)
[![codecov](https://codecov.io/gh/ccoVeille/go-safecast/graph/badge.svg?token=VW0VO503U6)](https://codecov.io/gh/ccoVeille/go-safecast)
[![Code Climate](https://codeclimate.com/github/ccoVeille/go-safecast.png)](https://codeclimate.com/github/ccoVeille/go-safecast)
[![Sourcegraph](https://sourcegraph.com/github.com/ccoveille/go-safecast/-/badge.svg)](https://sourcegraph.com/github.com/ccoveille/go-safecast?badge)

go-safecast solves the type conversion issues in Go

In Go, integer type conversion can lead to a silent and unexpected behavior and errors if not handled carefully.

This package helps to convert any number to another, and report an error when if there would be a [loss or overflow in the conversion](#conversion-issues)

## Usage

```go
package main

import (
  "fmt"
  "math"

  "github.com/ccoveille/go-safecast"
)

func main() {
  var a int

  a = 42
  b, err := safecast.ToUint8(a) // everything is fine
  if err != nil {
    fmt.Println(err)
  }
  fmt.Println(b)
  // Output: 42

  a = 255 + 1
  _, err = safecast.ToUint8(a) // 256 is greater than uint8 maximum value
  if err != nil {
    fmt.Println(err)
    // Output: conversion issue: 256 (int) is greater than 255 (uint8): maximum value for this type exceeded
  }

  a = -1
  _, err = safecast.ToUint8(a) // -1 cannot fit in uint8
  if err != nil {
    fmt.Println(err)
    // Output: conversion issue: -1 (int) is less than 0 (uint8): minimum value for this type exceeded
  }

  str := "\x99" // ASCII code 153 for Trademark symbol
  e := str[0]
  _, err = safecast.ToInt8(e)
  if err != nil {
    fmt.Println(err)
    // Output: conversion issue: 153 (uint8) is greater than 127 (int8): maximum value for this type exceeded
  }
}
```

[Go Playground](https://go.dev/play/p/nelJshulOnj)

## Conversion issues

Issues can happen when converting between signed and unsigned integers, or when converting to a smaller integer type.

```go
package main

import "fmt"

func main() {
  var a int64
  a = 42
  b := uint8(a)
  fmt.Println(b) // 42

  a = 255 // this is the math.MaxUint8
  b = uint8(a)
  fmt.Println(b) // 255

  a = 255 + 1
  b = uint8(a)
  fmt.Println(b) // 0 conversion overflow

  a = -1
  b = uint8(a)
  fmt.Println(b) // 255 conversion overflow
}
```

[Go Playground](https://go.dev/play/p/DHfNUcZBvVn)

So you need to adapt your code to write something like this.

```go
package main

import "fmt"

func main() {
  var a int64
  a = 42
  if a < 0 || a > math.MaxUint8 {
    log.Println("overflow") // Output: overflow
  }
  fmt.Println(b) // 42

  a = 255 // this is the math.MaxUint8
  b = uint8(a)
  fmt.Println(b) // 255

  a = 255 + 1
  b = uint8(a)
  if a < 0 || a > math.MaxUint8 {
    log.Println("overflow") // Output: overflow
  }
  fmt.Println(b) // Output: 0

  a = -1
  b = uint8(a)
  if a < 0 || a > math.MaxUint8 {
    log.Println("overflow") // Output: overflow
  }
  fmt.Println(b) // Output:255
}
```

[Go Playground](https://go.dev/play/p/qAHGyy4NCLP)

`go-safecast` is there to avoid boilerplate copy pasta.

## Motivation

The gosec project raised this to my attention when the gosec [G115 rule was added](https://github.com/securego/gosec/pull/1149)

> G115: Potential overflow when converting between integer types.

This issue was way more complex than expected, and required multiple fixes.

[CWE-190](https://cwe.mitre.org/data/definitions/190.html) explains in detail.

But to sum it up, you can face:

- infinite loop
- access to wrong resource by id
- grant access to someone who exhausted their quota

The gosec G115 will now report issues in a lot of project.

## Alternatives

Some libraries existed, but they were not able to cover all the use cases.

- [github.com/rung/go-safecast](https://github.com/rung/go-safecast):
  Unmaintained, not architecture agnostic, do not support `uint` -> `int` conversion

- [github.com/cybergarage/go-safecast](https://github.com/cybergarage/go-safecast)
  Work with pointer like `json.Marshall`
