#include "basic/math.h"

#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <vector>

using namespace std;

const int unit_size = 1000;
const double Pi = 3.1415926535;
extern double normal_cdf_table[];

double NormalRand()
{
    double x = rand()*1.0 / RAND_MAX;
    double y = rand()*1.0 / RAND_MAX;
    return sqrt(-2*log(x)) * cos(2*Pi*y);
}

double NormalCDF(double x)
{
    if (x < 0)
        return 1 - NormalCDF(-x);
    else
    {
        int index = int(x * unit_size + 0.5);
        if (index >= 5 * unit_size)
            return 1;
        else
            return normal_cdf_table[index];
    }
}

void GenerateNormalCDFTable(int times, int unit_size, vector<double> &table)
{
    vector<double> rand_numbers;
    for (int i = 0; i < times; ++i)
        rand_numbers.push_back(NormalRand());
    sort(rand_numbers.begin(), rand_numbers.end());

    table.resize(5 * unit_size);
    int count = 0;
    int index = 0;
    for (int i = 0; i < (int)table.size(); ++i)
    {
        double x = 1.0 * i / unit_size;
        while (index < (int)rand_numbers.size() && rand_numbers[index] <= x)
        {
            ++count;
            ++index;
        }
        table[i] = 1.0 * count / rand_numbers.size();
    }
}

double normal_cdf_table[] = 
{
    0.500029, 0.500418, 0.500831, 0.501230, 0.501626, 0.502021, 0.502435, 0.502827, 0.503224, 0.503621, 
    0.504022, 0.504422, 0.504815, 0.505225, 0.505629, 0.506028, 0.506422, 0.506823, 0.507222, 0.507623, 
    0.508028, 0.508428, 0.508839, 0.509237, 0.509625, 0.510016, 0.510417, 0.510825, 0.511224, 0.511629, 
    0.512030, 0.512424, 0.512820, 0.513213, 0.513605, 0.513996, 0.514397, 0.514802, 0.515200, 0.515587, 
    0.515989, 0.516381, 0.516780, 0.517175, 0.517579, 0.517970, 0.518364, 0.518762, 0.519155, 0.519551, 
    0.519949, 0.520352, 0.520748, 0.521152, 0.521546, 0.521947, 0.522343, 0.522748, 0.523144, 0.523539, 
    0.523928, 0.524324, 0.524727, 0.525121, 0.525507, 0.525909, 0.526316, 0.526712, 0.527114, 0.527523, 
    0.527921, 0.528312, 0.528714, 0.529116, 0.529507, 0.529917, 0.530318, 0.530709, 0.531101, 0.531512, 
    0.531915, 0.532310, 0.532698, 0.533101, 0.533502, 0.533902, 0.534296, 0.534687, 0.535084, 0.535479, 
    0.535879, 0.536275, 0.536671, 0.537070, 0.537455, 0.537855, 0.538241, 0.538628, 0.539027, 0.539428, 
    0.539822, 0.540225, 0.540626, 0.541024, 0.541412, 0.541805, 0.542212, 0.542605, 0.542994, 0.543391, 
    0.543780, 0.544182, 0.544580, 0.544980, 0.545370, 0.545762, 0.546164, 0.546564, 0.546959, 0.547358, 
    0.547752, 0.548153, 0.548537, 0.548938, 0.549331, 0.549728, 0.550134, 0.550535, 0.550939, 0.551336, 
    0.551741, 0.552147, 0.552551, 0.552940, 0.553339, 0.553751, 0.554141, 0.554535, 0.554932, 0.555328, 
    0.555725, 0.556118, 0.556505, 0.556899, 0.557303, 0.557697, 0.558101, 0.558490, 0.558888, 0.559291, 
    0.559692, 0.560082, 0.560471, 0.560867, 0.561265, 0.561656, 0.562042, 0.562425, 0.562815, 0.563212, 
    0.563601, 0.563996, 0.564385, 0.564788, 0.565187, 0.565576, 0.565976, 0.566372, 0.566765, 0.567170, 
    0.567562, 0.567974, 0.568363, 0.568768, 0.569151, 0.569543, 0.569932, 0.570329, 0.570728, 0.571115, 
    0.571513, 0.571905, 0.572301, 0.572688, 0.573080, 0.573471, 0.573866, 0.574252, 0.574645, 0.575043, 
    0.575435, 0.575820, 0.576205, 0.576594, 0.576993, 0.577386, 0.577776, 0.578175, 0.578560, 0.578948, 
    0.579343, 0.579733, 0.580131, 0.580534, 0.580921, 0.581302, 0.581695, 0.582079, 0.582471, 0.582866, 
    0.583259, 0.583649, 0.584029, 0.584430, 0.584821, 0.585216, 0.585604, 0.585989, 0.586374, 0.586770, 
    0.587157, 0.587544, 0.587940, 0.588323, 0.588725, 0.589113, 0.589504, 0.589898, 0.590287, 0.590676, 
    0.591062, 0.591457, 0.591856, 0.592242, 0.592622, 0.593001, 0.593393, 0.593781, 0.594182, 0.594588, 
    0.594977, 0.595358, 0.595747, 0.596132, 0.596525, 0.596912, 0.597296, 0.597677, 0.598064, 0.598457, 
    0.598842, 0.599232, 0.599617, 0.599998, 0.600379, 0.600765, 0.601151, 0.601532, 0.601918, 0.602297, 
    0.602661, 0.603051, 0.603436, 0.603816, 0.604199, 0.604581, 0.604962, 0.605335, 0.605723, 0.606115, 
    0.606496, 0.606877, 0.607258, 0.607643, 0.608024, 0.608409, 0.608781, 0.609173, 0.609559, 0.609944, 
    0.610322, 0.610703, 0.611081, 0.611464, 0.611836, 0.612242, 0.612623, 0.613017, 0.613400, 0.613790, 
    0.614172, 0.614553, 0.614944, 0.615319, 0.615694, 0.616079, 0.616467, 0.616851, 0.617232, 0.617614, 
    0.617991, 0.618370, 0.618751, 0.619138, 0.619511, 0.619890, 0.620283, 0.620662, 0.621045, 0.621433, 
    0.621811, 0.622190, 0.622576, 0.622952, 0.623328, 0.623711, 0.624090, 0.624474, 0.624855, 0.625229, 
    0.625609, 0.625981, 0.626365, 0.626742, 0.627126, 0.627510, 0.627900, 0.628274, 0.628657, 0.629041, 
    0.629422, 0.629803, 0.630175, 0.630555, 0.630926, 0.631304, 0.631680, 0.632054, 0.632444, 0.632821, 
    0.633202, 0.633573, 0.633952, 0.634330, 0.634704, 0.635076, 0.635454, 0.635825, 0.636198, 0.636580, 
    0.636957, 0.637337, 0.637712, 0.638100, 0.638476, 0.638842, 0.639206, 0.639579, 0.639943, 0.640321, 
    0.640702, 0.641076, 0.641462, 0.641833, 0.642209, 0.642586, 0.642958, 0.643332, 0.643703, 0.644083, 
    0.644452, 0.644829, 0.645210, 0.645590, 0.645968, 0.646339, 0.646710, 0.647080, 0.647451, 0.647817, 
    0.648190, 0.648567, 0.648937, 0.649302, 0.649675, 0.650035, 0.650400, 0.650779, 0.651154, 0.651529, 
    0.651888, 0.652254, 0.652617, 0.652982, 0.653338, 0.653709, 0.654074, 0.654442, 0.654803, 0.655167, 
    0.655535, 0.655910, 0.656284, 0.656639, 0.657009, 0.657375, 0.657740, 0.658110, 0.658470, 0.658833, 
    0.659203, 0.659575, 0.659946, 0.660309, 0.660671, 0.661026, 0.661398, 0.661767, 0.662127, 0.662503, 
    0.662868, 0.663239, 0.663593, 0.663964, 0.664327, 0.664687, 0.665063, 0.665423, 0.665783, 0.666144, 
    0.666507, 0.666877, 0.667244, 0.667607, 0.667975, 0.668339, 0.668697, 0.669059, 0.669419, 0.669779, 
    0.670150, 0.670518, 0.670879, 0.671239, 0.671609, 0.671961, 0.672316, 0.672679, 0.673037, 0.673400, 
    0.673761, 0.674123, 0.674472, 0.674821, 0.675170, 0.675524, 0.675882, 0.676245, 0.676594, 0.676946, 
    0.677309, 0.677661, 0.678032, 0.678391, 0.678740, 0.679091, 0.679442, 0.679802, 0.680155, 0.680519, 
    0.680877, 0.681238, 0.681603, 0.681968, 0.682321, 0.682680, 0.683038, 0.683401, 0.683755, 0.684110, 
    0.684475, 0.684830, 0.685185, 0.685535, 0.685887, 0.686244, 0.686603, 0.686952, 0.687312, 0.687661, 
    0.688010, 0.688359, 0.688712, 0.689068, 0.689426, 0.689776, 0.690137, 0.690479, 0.690826, 0.691178, 
    0.691534, 0.691873, 0.692221, 0.692567, 0.692918, 0.693280, 0.693627, 0.693974, 0.694326, 0.694677, 
    0.695032, 0.695389, 0.695743, 0.696094, 0.696454, 0.696802, 0.697146, 0.697498, 0.697854, 0.698209, 
    0.698562, 0.698908, 0.699251, 0.699596, 0.699941, 0.700280, 0.700623, 0.700982, 0.701328, 0.701662, 
    0.702006, 0.702345, 0.702703, 0.703051, 0.703407, 0.703744, 0.704092, 0.704448, 0.704790, 0.705141, 
    0.705492, 0.705830, 0.706173, 0.706510, 0.706852, 0.707191, 0.707541, 0.707890, 0.708243, 0.708584, 
    0.708928, 0.709270, 0.709616, 0.709955, 0.710291, 0.710626, 0.710980, 0.711322, 0.711672, 0.712009, 
    0.712345, 0.712687, 0.713029, 0.713363, 0.713701, 0.714045, 0.714396, 0.714739, 0.715073, 0.715412, 
    0.715758, 0.716094, 0.716437, 0.716772, 0.717114, 0.717457, 0.717787, 0.718117, 0.718446, 0.718777, 
    0.719111, 0.719448, 0.719780, 0.720119, 0.720455, 0.720794, 0.721130, 0.721467, 0.721796, 0.722125, 
    0.722463, 0.722798, 0.723139, 0.723473, 0.723809, 0.724137, 0.724476, 0.724817, 0.725151, 0.725492, 
    0.725822, 0.726158, 0.726490, 0.726815, 0.727142, 0.727472, 0.727796, 0.728129, 0.728466, 0.728806, 
    0.729134, 0.729452, 0.729772, 0.730101, 0.730431, 0.730764, 0.731099, 0.731430, 0.731764, 0.732106, 
    0.732433, 0.732765, 0.733088, 0.733407, 0.733739, 0.734079, 0.734401, 0.734733, 0.735062, 0.735396, 
    0.735719, 0.736042, 0.736377, 0.736709, 0.737045, 0.737375, 0.737697, 0.738022, 0.738341, 0.738685, 
    0.739001, 0.739325, 0.739653, 0.739976, 0.740292, 0.740611, 0.740931, 0.741264, 0.741600, 0.741917, 
    0.742245, 0.742560, 0.742894, 0.743215, 0.743538, 0.743855, 0.744186, 0.744517, 0.744829, 0.745149, 
    0.745473, 0.745799, 0.746120, 0.746447, 0.746761, 0.747087, 0.747403, 0.747721, 0.748024, 0.748340, 
    0.748660, 0.748978, 0.749290, 0.749610, 0.749922, 0.750249, 0.750564, 0.750876, 0.751200, 0.751513, 
    0.751820, 0.752129, 0.752443, 0.752760, 0.753073, 0.753389, 0.753698, 0.754013, 0.754322, 0.754629, 
    0.754944, 0.755256, 0.755574, 0.755885, 0.756201, 0.756511, 0.756826, 0.757142, 0.757452, 0.757757, 
    0.758074, 0.758393, 0.758705, 0.759010, 0.759329, 0.759630, 0.759932, 0.760241, 0.760548, 0.760869, 
    0.761175, 0.761479, 0.761772, 0.762076, 0.762385, 0.762683, 0.762992, 0.763296, 0.763611, 0.763924, 
    0.764238, 0.764542, 0.764843, 0.765160, 0.765464, 0.765772, 0.766072, 0.766388, 0.766697, 0.767007, 
    0.767315, 0.767618, 0.767922, 0.768221, 0.768518, 0.768825, 0.769123, 0.769428, 0.769737, 0.770037, 
    0.770352, 0.770651, 0.770969, 0.771265, 0.771569, 0.771865, 0.772158, 0.772457, 0.772753, 0.773052, 
    0.773350, 0.773645, 0.773945, 0.774242, 0.774546, 0.774842, 0.775136, 0.775440, 0.775738, 0.776032, 
    0.776333, 0.776630, 0.776922, 0.777215, 0.777514, 0.777812, 0.778111, 0.778402, 0.778705, 0.778991, 
    0.779286, 0.779583, 0.779870, 0.780165, 0.780458, 0.780746, 0.781045, 0.781339, 0.781633, 0.781925, 
    0.782219, 0.782518, 0.782812, 0.783104, 0.783399, 0.783694, 0.783984, 0.784280, 0.784576, 0.784861, 
    0.785159, 0.785444, 0.785743, 0.786034, 0.786319, 0.786613, 0.786902, 0.787196, 0.787482, 0.787772, 
    0.788065, 0.788356, 0.788643, 0.788931, 0.789212, 0.789505, 0.789795, 0.790089, 0.790368, 0.790659, 
    0.790947, 0.791241, 0.791521, 0.791813, 0.792110, 0.792403, 0.792687, 0.792976, 0.793254, 0.793540, 
    0.793823, 0.794107, 0.794385, 0.794657, 0.794945, 0.795228, 0.795510, 0.795791, 0.796072, 0.796353, 
    0.796635, 0.796920, 0.797199, 0.797485, 0.797766, 0.798049, 0.798333, 0.798615, 0.798896, 0.799167, 
    0.799444, 0.799728, 0.800006, 0.800281, 0.800557, 0.800832, 0.801112, 0.801388, 0.801661, 0.801936, 
    0.802212, 0.802488, 0.802766, 0.803047, 0.803330, 0.803611, 0.803889, 0.804165, 0.804434, 0.804717, 
    0.804990, 0.805258, 0.805533, 0.805814, 0.806090, 0.806370, 0.806637, 0.806908, 0.807178, 0.807456, 
    0.807727, 0.808001, 0.808284, 0.808560, 0.808833, 0.809097, 0.809375, 0.809645, 0.809919, 0.810193, 
    0.810466, 0.810741, 0.811012, 0.811277, 0.811542, 0.811810, 0.812079, 0.812361, 0.812642, 0.812913, 
    0.813177, 0.813446, 0.813710, 0.813974, 0.814245, 0.814505, 0.814773, 0.815036, 0.815305, 0.815574, 
    0.815837, 0.816093, 0.816361, 0.816626, 0.816893, 0.817160, 0.817420, 0.817679, 0.817945, 0.818198, 
    0.818465, 0.818732, 0.818995, 0.819257, 0.819511, 0.819779, 0.820045, 0.820310, 0.820573, 0.820842, 
    0.821098, 0.821360, 0.821616, 0.821887, 0.822146, 0.822401, 0.822663, 0.822925, 0.823186, 0.823448, 
    0.823711, 0.823973, 0.824241, 0.824495, 0.824748, 0.825011, 0.825265, 0.825530, 0.825795, 0.826065, 
    0.826328, 0.826574, 0.826833, 0.827085, 0.827338, 0.827599, 0.827856, 0.828112, 0.828383, 0.828637, 
    0.828895, 0.829148, 0.829398, 0.829647, 0.829905, 0.830167, 0.830418, 0.830672, 0.830928, 0.831176, 
    0.831424, 0.831676, 0.831933, 0.832182, 0.832435, 0.832685, 0.832933, 0.833187, 0.833443, 0.833695, 
    0.833944, 0.834197, 0.834452, 0.834698, 0.834949, 0.835201, 0.835450, 0.835699, 0.835958, 0.836208, 
    0.836457, 0.836700, 0.836940, 0.837182, 0.837427, 0.837675, 0.837922, 0.838168, 0.838419, 0.838669, 
    0.838906, 0.839155, 0.839400, 0.839645, 0.839885, 0.840128, 0.840370, 0.840616, 0.840856, 0.841097, 
    0.841342, 0.841574, 0.841816, 0.842048, 0.842285, 0.842532, 0.842775, 0.843020, 0.843259, 0.843496, 
    0.843743, 0.843983, 0.844219, 0.844456, 0.844702, 0.844940, 0.845175, 0.845411, 0.845641, 0.845880, 
    0.846126, 0.846366, 0.846604, 0.846838, 0.847084, 0.847317, 0.847554, 0.847787, 0.848017, 0.848254, 
    0.848490, 0.848718, 0.848954, 0.849189, 0.849427, 0.849664, 0.849896, 0.850130, 0.850370, 0.850601, 
    0.850838, 0.851069, 0.851306, 0.851542, 0.851776, 0.852001, 0.852231, 0.852465, 0.852702, 0.852929, 
    0.853165, 0.853389, 0.853616, 0.853835, 0.854057, 0.854282, 0.854511, 0.854743, 0.854973, 0.855209, 
    0.855440, 0.855661, 0.855888, 0.856120, 0.856350, 0.856572, 0.856791, 0.857021, 0.857243, 0.857468, 
    0.857700, 0.857925, 0.858147, 0.858380, 0.858612, 0.858834, 0.859059, 0.859287, 0.859512, 0.859729, 
    0.859949, 0.860171, 0.860397, 0.860622, 0.860844, 0.861058, 0.861278, 0.861502, 0.861718, 0.861934, 
    0.862173, 0.862394, 0.862613, 0.862824, 0.863047, 0.863272, 0.863494, 0.863707, 0.863928, 0.864152, 
    0.864371, 0.864595, 0.864816, 0.865035, 0.865247, 0.865471, 0.865690, 0.865913, 0.866124, 0.866344, 
    0.866558, 0.866774, 0.866993, 0.867200, 0.867412, 0.867626, 0.867837, 0.868049, 0.868271, 0.868477, 
    0.868688, 0.868905, 0.869115, 0.869324, 0.869538, 0.869748, 0.869961, 0.870174, 0.870388, 0.870594, 
    0.870804, 0.871012, 0.871219, 0.871439, 0.871649, 0.871867, 0.872081, 0.872293, 0.872503, 0.872719, 
    0.872923, 0.873133, 0.873340, 0.873545, 0.873751, 0.873948, 0.874163, 0.874365, 0.874570, 0.874772, 
    0.874976, 0.875184, 0.875396, 0.875602, 0.875813, 0.876013, 0.876217, 0.876423, 0.876628, 0.876833, 
    0.877035, 0.877238, 0.877451, 0.877661, 0.877867, 0.878059, 0.878264, 0.878467, 0.878668, 0.878869, 
    0.879067, 0.879273, 0.879471, 0.879664, 0.879860, 0.880059, 0.880254, 0.880449, 0.880652, 0.880850, 
    0.881049, 0.881247, 0.881453, 0.881647, 0.881849, 0.882046, 0.882241, 0.882449, 0.882651, 0.882841, 
    0.883036, 0.883233, 0.883431, 0.883624, 0.883812, 0.884007, 0.884204, 0.884397, 0.884593, 0.884780, 
    0.884971, 0.885166, 0.885358, 0.885553, 0.885744, 0.885934, 0.886125, 0.886317, 0.886513, 0.886707, 
    0.886894, 0.887079, 0.887272, 0.887469, 0.887662, 0.887858, 0.888049, 0.888240, 0.888428, 0.888616, 
    0.888807, 0.888996, 0.889188, 0.889376, 0.889560, 0.889751, 0.889946, 0.890140, 0.890323, 0.890513, 
    0.890704, 0.890889, 0.891081, 0.891266, 0.891452, 0.891637, 0.891818, 0.892006, 0.892192, 0.892372, 
    0.892547, 0.892727, 0.892908, 0.893086, 0.893271, 0.893453, 0.893644, 0.893829, 0.894018, 0.894201, 
    0.894383, 0.894572, 0.894756, 0.894937, 0.895122, 0.895312, 0.895496, 0.895673, 0.895852, 0.896032, 
    0.896215, 0.896397, 0.896572, 0.896751, 0.896932, 0.897110, 0.897287, 0.897466, 0.897638, 0.897817, 
    0.897998, 0.898181, 0.898355, 0.898529, 0.898702, 0.898876, 0.899051, 0.899227, 0.899405, 0.899592, 
    0.899771, 0.899946, 0.900125, 0.900304, 0.900469, 0.900642, 0.900820, 0.900989, 0.901165, 0.901340, 
    0.901516, 0.901692, 0.901869, 0.902039, 0.902217, 0.902387, 0.902561, 0.902732, 0.902896, 0.903072, 
    0.903241, 0.903412, 0.903588, 0.903762, 0.903938, 0.904114, 0.904283, 0.904454, 0.904635, 0.904807, 
    0.904982, 0.905148, 0.905315, 0.905485, 0.905651, 0.905821, 0.905983, 0.906150, 0.906315, 0.906474, 
    0.906637, 0.906800, 0.906969, 0.907138, 0.907305, 0.907472, 0.907636, 0.907799, 0.907962, 0.908128, 
    0.908290, 0.908462, 0.908621, 0.908782, 0.908946, 0.909115, 0.909273, 0.909440, 0.909603, 0.909756, 
    0.909918, 0.910077, 0.910238, 0.910401, 0.910557, 0.910721, 0.910887, 0.911052, 0.911215, 0.911378, 
    0.911544, 0.911708, 0.911867, 0.912021, 0.912178, 0.912348, 0.912511, 0.912667, 0.912821, 0.912974, 
    0.913127, 0.913287, 0.913446, 0.913605, 0.913767, 0.913929, 0.914088, 0.914240, 0.914404, 0.914564, 
    0.914722, 0.914871, 0.915023, 0.915179, 0.915328, 0.915489, 0.915643, 0.915802, 0.915958, 0.916111, 
    0.916268, 0.916429, 0.916575, 0.916727, 0.916878, 0.917032, 0.917181, 0.917342, 0.917502, 0.917655, 
    0.917812, 0.917965, 0.918119, 0.918260, 0.918415, 0.918566, 0.918723, 0.918872, 0.919019, 0.919169, 
    0.919320, 0.919461, 0.919617, 0.919769, 0.919917, 0.920061, 0.920206, 0.920353, 0.920496, 0.920636, 
    0.920784, 0.920936, 0.921081, 0.921226, 0.921371, 0.921520, 0.921671, 0.921827, 0.921975, 0.922118, 
    0.922264, 0.922409, 0.922550, 0.922696, 0.922829, 0.922972, 0.923123, 0.923266, 0.923415, 0.923556, 
    0.923703, 0.923851, 0.923998, 0.924140, 0.924281, 0.924420, 0.924568, 0.924710, 0.924847, 0.924994, 
    0.925132, 0.925276, 0.925420, 0.925561, 0.925701, 0.925845, 0.925984, 0.926123, 0.926263, 0.926408, 
    0.926547, 0.926686, 0.926829, 0.926967, 0.927109, 0.927246, 0.927381, 0.927513, 0.927653, 0.927791, 
    0.927923, 0.928063, 0.928199, 0.928337, 0.928477, 0.928616, 0.928752, 0.928887, 0.929028, 0.929160, 
    0.929291, 0.929424, 0.929561, 0.929705, 0.929843, 0.929976, 0.930112, 0.930247, 0.930380, 0.930518, 
    0.930651, 0.930778, 0.930910, 0.931037, 0.931176, 0.931311, 0.931446, 0.931573, 0.931696, 0.931826, 
    0.931962, 0.932096, 0.932230, 0.932360, 0.932492, 0.932622, 0.932754, 0.932883, 0.933009, 0.933137, 
    0.933262, 0.933390, 0.933512, 0.933636, 0.933761, 0.933889, 0.934017, 0.934141, 0.934275, 0.934406, 
    0.934533, 0.934662, 0.934787, 0.934917, 0.935046, 0.935167, 0.935297, 0.935422, 0.935548, 0.935671, 
    0.935799, 0.935923, 0.936047, 0.936175, 0.936302, 0.936435, 0.936562, 0.936685, 0.936810, 0.936934, 
    0.937053, 0.937176, 0.937307, 0.937427, 0.937548, 0.937673, 0.937797, 0.937921, 0.938041, 0.938160, 
    0.938286, 0.938406, 0.938529, 0.938646, 0.938770, 0.938895, 0.939016, 0.939140, 0.939255, 0.939377, 
    0.939494, 0.939618, 0.939738, 0.939855, 0.939973, 0.940094, 0.940210, 0.940331, 0.940442, 0.940560, 
    0.940684, 0.940800, 0.940911, 0.941029, 0.941145, 0.941272, 0.941386, 0.941504, 0.941624, 0.941740, 
    0.941850, 0.941964, 0.942080, 0.942195, 0.942310, 0.942422, 0.942535, 0.942647, 0.942764, 0.942883, 
    0.942995, 0.943110, 0.943224, 0.943344, 0.943459, 0.943571, 0.943686, 0.943798, 0.943914, 0.944027, 
    0.944143, 0.944262, 0.944382, 0.944490, 0.944605, 0.944718, 0.944829, 0.944942, 0.945057, 0.945173, 
    0.945281, 0.945390, 0.945508, 0.945624, 0.945733, 0.945847, 0.945961, 0.946068, 0.946176, 0.946281, 
    0.946390, 0.946499, 0.946606, 0.946714, 0.946828, 0.946940, 0.947048, 0.947158, 0.947264, 0.947367, 
    0.947473, 0.947584, 0.947696, 0.947804, 0.947911, 0.948020, 0.948129, 0.948233, 0.948342, 0.948451, 
    0.948554, 0.948657, 0.948761, 0.948864, 0.948968, 0.949073, 0.949182, 0.949289, 0.949388, 0.949491, 
    0.949597, 0.949698, 0.949796, 0.949903, 0.950003, 0.950103, 0.950205, 0.950314, 0.950418, 0.950520, 
    0.950628, 0.950732, 0.950829, 0.950928, 0.951035, 0.951136, 0.951238, 0.951343, 0.951444, 0.951538, 
    0.951642, 0.951748, 0.951846, 0.951949, 0.952048, 0.952147, 0.952242, 0.952343, 0.952446, 0.952546, 
    0.952651, 0.952747, 0.952840, 0.952944, 0.953037, 0.953138, 0.953236, 0.953337, 0.953434, 0.953526, 
    0.953620, 0.953716, 0.953813, 0.953912, 0.954006, 0.954101, 0.954196, 0.954292, 0.954387, 0.954485, 
    0.954580, 0.954677, 0.954776, 0.954871, 0.954965, 0.955057, 0.955155, 0.955248, 0.955348, 0.955439, 
    0.955533, 0.955625, 0.955718, 0.955812, 0.955907, 0.956002, 0.956095, 0.956186, 0.956279, 0.956369, 
    0.956458, 0.956548, 0.956639, 0.956729, 0.956820, 0.956912, 0.957002, 0.957094, 0.957186, 0.957276, 
    0.957364, 0.957457, 0.957541, 0.957637, 0.957727, 0.957817, 0.957910, 0.958000, 0.958094, 0.958182, 
    0.958269, 0.958359, 0.958445, 0.958534, 0.958619, 0.958708, 0.958798, 0.958882, 0.958969, 0.959059, 
    0.959147, 0.959232, 0.959317, 0.959407, 0.959494, 0.959578, 0.959669, 0.959761, 0.959848, 0.959932, 
    0.960020, 0.960104, 0.960190, 0.960276, 0.960360, 0.960443, 0.960525, 0.960609, 0.960689, 0.960777, 
    0.960860, 0.960943, 0.961028, 0.961115, 0.961198, 0.961288, 0.961369, 0.961450, 0.961532, 0.961611, 
    0.961692, 0.961775, 0.961862, 0.961939, 0.962019, 0.962098, 0.962186, 0.962263, 0.962346, 0.962426, 
    0.962510, 0.962590, 0.962670, 0.962760, 0.962842, 0.962924, 0.963005, 0.963087, 0.963168, 0.963249, 
    0.963330, 0.963411, 0.963496, 0.963576, 0.963658, 0.963740, 0.963816, 0.963897, 0.963974, 0.964054, 
    0.964132, 0.964214, 0.964290, 0.964371, 0.964449, 0.964523, 0.964605, 0.964681, 0.964755, 0.964834, 
    0.964911, 0.964992, 0.965068, 0.965144, 0.965220, 0.965295, 0.965373, 0.965444, 0.965520, 0.965600, 
    0.965673, 0.965747, 0.965819, 0.965898, 0.965974, 0.966053, 0.966127, 0.966200, 0.966276, 0.966353, 
    0.966427, 0.966506, 0.966580, 0.966649, 0.966730, 0.966804, 0.966877, 0.966955, 0.967028, 0.967100, 
    0.967171, 0.967249, 0.967320, 0.967392, 0.967465, 0.967541, 0.967611, 0.967681, 0.967758, 0.967832, 
    0.967902, 0.967977, 0.968048, 0.968121, 0.968192, 0.968264, 0.968337, 0.968407, 0.968483, 0.968559, 
    0.968628, 0.968695, 0.968765, 0.968832, 0.968902, 0.968969, 0.969042, 0.969109, 0.969178, 0.969253, 
    0.969327, 0.969391, 0.969459, 0.969532, 0.969599, 0.969670, 0.969740, 0.969808, 0.969875, 0.969946, 
    0.970016, 0.970082, 0.970147, 0.970217, 0.970279, 0.970350, 0.970420, 0.970487, 0.970558, 0.970626, 
    0.970694, 0.970762, 0.970827, 0.970893, 0.970956, 0.971023, 0.971086, 0.971157, 0.971223, 0.971289, 
    0.971353, 0.971421, 0.971486, 0.971551, 0.971616, 0.971682, 0.971746, 0.971813, 0.971875, 0.971943, 
    0.972013, 0.972075, 0.972140, 0.972204, 0.972264, 0.972330, 0.972389, 0.972454, 0.972514, 0.972579, 
    0.972644, 0.972704, 0.972766, 0.972830, 0.972890, 0.972955, 0.973020, 0.973081, 0.973147, 0.973209, 
    0.973269, 0.973331, 0.973394, 0.973454, 0.973514, 0.973573, 0.973632, 0.973692, 0.973755, 0.973814, 
    0.973878, 0.973938, 0.973998, 0.974060, 0.974120, 0.974182, 0.974243, 0.974303, 0.974363, 0.974423, 
    0.974479, 0.974539, 0.974600, 0.974662, 0.974721, 0.974781, 0.974839, 0.974895, 0.974952, 0.975010, 
    0.975065, 0.975125, 0.975183, 0.975240, 0.975298, 0.975356, 0.975414, 0.975475, 0.975535, 0.975591, 
    0.975647, 0.975705, 0.975761, 0.975820, 0.975876, 0.975932, 0.975986, 0.976044, 0.976103, 0.976158, 
    0.976213, 0.976267, 0.976325, 0.976385, 0.976442, 0.976500, 0.976559, 0.976617, 0.976669, 0.976720, 
    0.976773, 0.976825, 0.976878, 0.976928, 0.976982, 0.977035, 0.977087, 0.977141, 0.977196, 0.977248, 
    0.977302, 0.977353, 0.977405, 0.977457, 0.977511, 0.977564, 0.977616, 0.977668, 0.977719, 0.977774, 
    0.977826, 0.977881, 0.977932, 0.977984, 0.978036, 0.978090, 0.978146, 0.978198, 0.978251, 0.978305, 
    0.978360, 0.978415, 0.978470, 0.978525, 0.978576, 0.978623, 0.978674, 0.978723, 0.978773, 0.978826, 
    0.978878, 0.978928, 0.978981, 0.979029, 0.979076, 0.979124, 0.979174, 0.979224, 0.979274, 0.979324, 
    0.979375, 0.979424, 0.979476, 0.979525, 0.979573, 0.979624, 0.979670, 0.979722, 0.979769, 0.979815, 
    0.979867, 0.979915, 0.979963, 0.980013, 0.980065, 0.980120, 0.980169, 0.980214, 0.980261, 0.980310, 
    0.980356, 0.980402, 0.980448, 0.980492, 0.980540, 0.980590, 0.980634, 0.980680, 0.980730, 0.980781, 
    0.980829, 0.980878, 0.980922, 0.980970, 0.981016, 0.981061, 0.981105, 0.981154, 0.981198, 0.981245, 
    0.981292, 0.981336, 0.981384, 0.981433, 0.981477, 0.981524, 0.981568, 0.981612, 0.981657, 0.981699, 
    0.981745, 0.981791, 0.981835, 0.981877, 0.981918, 0.981960, 0.982003, 0.982050, 0.982094, 0.982136, 
    0.982180, 0.982224, 0.982265, 0.982307, 0.982350, 0.982392, 0.982435, 0.982478, 0.982520, 0.982566, 
    0.982609, 0.982649, 0.982691, 0.982734, 0.982776, 0.982821, 0.982862, 0.982903, 0.982942, 0.982985, 
    0.983027, 0.983067, 0.983112, 0.983152, 0.983192, 0.983235, 0.983281, 0.983322, 0.983363, 0.983407, 
    0.983447, 0.983486, 0.983527, 0.983566, 0.983608, 0.983650, 0.983689, 0.983729, 0.983770, 0.983810, 
    0.983850, 0.983892, 0.983931, 0.983972, 0.984015, 0.984056, 0.984098, 0.984137, 0.984177, 0.984217, 
    0.984257, 0.984298, 0.984337, 0.984372, 0.984410, 0.984452, 0.984489, 0.984526, 0.984565, 0.984603, 
    0.984639, 0.984677, 0.984716, 0.984756, 0.984794, 0.984833, 0.984870, 0.984908, 0.984948, 0.984983, 
    0.985019, 0.985057, 0.985094, 0.985135, 0.985171, 0.985207, 0.985244, 0.985280, 0.985315, 0.985348, 
    0.985383, 0.985421, 0.985461, 0.985498, 0.985532, 0.985566, 0.985600, 0.985639, 0.985674, 0.985706, 
    0.985744, 0.985782, 0.985821, 0.985855, 0.985892, 0.985930, 0.985962, 0.985995, 0.986034, 0.986068, 
    0.986103, 0.986138, 0.986175, 0.986208, 0.986242, 0.986280, 0.986315, 0.986352, 0.986386, 0.986421, 
    0.986458, 0.986491, 0.986526, 0.986562, 0.986594, 0.986626, 0.986660, 0.986693, 0.986726, 0.986759, 
    0.986793, 0.986823, 0.986859, 0.986891, 0.986924, 0.986958, 0.986991, 0.987022, 0.987056, 0.987093, 
    0.987128, 0.987164, 0.987198, 0.987228, 0.987263, 0.987294, 0.987326, 0.987356, 0.987391, 0.987422, 
    0.987454, 0.987488, 0.987519, 0.987553, 0.987587, 0.987618, 0.987646, 0.987680, 0.987709, 0.987739, 
    0.987770, 0.987801, 0.987831, 0.987863, 0.987892, 0.987926, 0.987957, 0.987989, 0.988020, 0.988049, 
    0.988079, 0.988107, 0.988137, 0.988167, 0.988199, 0.988229, 0.988258, 0.988287, 0.988317, 0.988348, 
    0.988379, 0.988412, 0.988445, 0.988477, 0.988508, 0.988537, 0.988566, 0.988596, 0.988627, 0.988660, 
    0.988689, 0.988716, 0.988745, 0.988773, 0.988801, 0.988830, 0.988861, 0.988894, 0.988924, 0.988954, 
    0.988983, 0.989012, 0.989042, 0.989072, 0.989100, 0.989128, 0.989159, 0.989190, 0.989219, 0.989248, 
    0.989276, 0.989303, 0.989335, 0.989363, 0.989391, 0.989419, 0.989445, 0.989474, 0.989499, 0.989529, 
    0.989558, 0.989586, 0.989615, 0.989644, 0.989672, 0.989697, 0.989725, 0.989754, 0.989783, 0.989808, 
    0.989834, 0.989861, 0.989889, 0.989915, 0.989944, 0.989970, 0.989996, 0.990020, 0.990047, 0.990075, 
    0.990105, 0.990132, 0.990156, 0.990179, 0.990205, 0.990229, 0.990254, 0.990278, 0.990302, 0.990329, 
    0.990356, 0.990381, 0.990405, 0.990432, 0.990458, 0.990485, 0.990510, 0.990536, 0.990561, 0.990585, 
    0.990609, 0.990635, 0.990660, 0.990685, 0.990708, 0.990736, 0.990757, 0.990779, 0.990803, 0.990829, 
    0.990855, 0.990883, 0.990906, 0.990929, 0.990954, 0.990981, 0.991004, 0.991030, 0.991053, 0.991077, 
    0.991101, 0.991123, 0.991149, 0.991174, 0.991197, 0.991223, 0.991248, 0.991271, 0.991294, 0.991318, 
    0.991342, 0.991364, 0.991386, 0.991408, 0.991431, 0.991454, 0.991478, 0.991499, 0.991523, 0.991546, 
    0.991569, 0.991592, 0.991614, 0.991637, 0.991658, 0.991678, 0.991701, 0.991722, 0.991745, 0.991767, 
    0.991792, 0.991818, 0.991841, 0.991866, 0.991887, 0.991912, 0.991936, 0.991957, 0.991980, 0.992001, 
    0.992025, 0.992048, 0.992069, 0.992092, 0.992114, 0.992136, 0.992158, 0.992181, 0.992202, 0.992225, 
    0.992247, 0.992269, 0.992289, 0.992312, 0.992336, 0.992358, 0.992379, 0.992402, 0.992424, 0.992442, 
    0.992462, 0.992485, 0.992506, 0.992526, 0.992547, 0.992570, 0.992589, 0.992609, 0.992630, 0.992650, 
    0.992672, 0.992689, 0.992709, 0.992730, 0.992752, 0.992769, 0.992788, 0.992809, 0.992828, 0.992849, 
    0.992869, 0.992887, 0.992907, 0.992926, 0.992946, 0.992964, 0.992981, 0.993001, 0.993019, 0.993040, 
    0.993062, 0.993078, 0.993099, 0.993116, 0.993135, 0.993153, 0.993170, 0.993192, 0.993211, 0.993231, 
    0.993250, 0.993268, 0.993286, 0.993305, 0.993323, 0.993342, 0.993363, 0.993379, 0.993398, 0.993418, 
    0.993439, 0.993457, 0.993476, 0.993495, 0.993513, 0.993531, 0.993548, 0.993566, 0.993587, 0.993602, 
    0.993622, 0.993640, 0.993659, 0.993676, 0.993694, 0.993710, 0.993730, 0.993749, 0.993767, 0.993784, 
    0.993802, 0.993819, 0.993837, 0.993853, 0.993871, 0.993888, 0.993905, 0.993923, 0.993942, 0.993960, 
    0.993975, 0.993992, 0.994007, 0.994025, 0.994042, 0.994057, 0.994074, 0.994090, 0.994108, 0.994123, 
    0.994142, 0.994158, 0.994173, 0.994192, 0.994209, 0.994226, 0.994243, 0.994259, 0.994275, 0.994290, 
    0.994305, 0.994321, 0.994336, 0.994351, 0.994370, 0.994389, 0.994405, 0.994421, 0.994438, 0.994453, 
    0.994469, 0.994485, 0.994500, 0.994517, 0.994533, 0.994548, 0.994566, 0.994581, 0.994598, 0.994615, 
    0.994632, 0.994647, 0.994661, 0.994677, 0.994692, 0.994710, 0.994725, 0.994740, 0.994756, 0.994770, 
    0.994783, 0.994800, 0.994814, 0.994829, 0.994845, 0.994860, 0.994875, 0.994889, 0.994904, 0.994920, 
    0.994932, 0.994945, 0.994960, 0.994973, 0.994988, 0.995002, 0.995019, 0.995033, 0.995046, 0.995060, 
    0.995075, 0.995091, 0.995106, 0.995122, 0.995137, 0.995151, 0.995166, 0.995180, 0.995195, 0.995209, 
    0.995221, 0.995236, 0.995252, 0.995265, 0.995279, 0.995292, 0.995306, 0.995319, 0.995331, 0.995344, 
    0.995359, 0.995373, 0.995387, 0.995402, 0.995413, 0.995427, 0.995441, 0.995454, 0.995468, 0.995479, 
    0.995493, 0.995507, 0.995519, 0.995531, 0.995546, 0.995560, 0.995572, 0.995584, 0.995599, 0.995613, 
    0.995625, 0.995640, 0.995649, 0.995663, 0.995677, 0.995688, 0.995701, 0.995717, 0.995728, 0.995739, 
    0.995752, 0.995764, 0.995775, 0.995789, 0.995800, 0.995812, 0.995824, 0.995835, 0.995847, 0.995859, 
    0.995870, 0.995883, 0.995893, 0.995905, 0.995917, 0.995929, 0.995939, 0.995950, 0.995960, 0.995972, 
    0.995985, 0.995995, 0.996009, 0.996020, 0.996034, 0.996047, 0.996057, 0.996069, 0.996080, 0.996090, 
    0.996104, 0.996118, 0.996128, 0.996140, 0.996153, 0.996165, 0.996176, 0.996188, 0.996202, 0.996214, 
    0.996225, 0.996235, 0.996246, 0.996256, 0.996268, 0.996278, 0.996290, 0.996301, 0.996313, 0.996325, 
    0.996337, 0.996348, 0.996360, 0.996371, 0.996381, 0.996391, 0.996402, 0.996414, 0.996425, 0.996436, 
    0.996446, 0.996456, 0.996465, 0.996475, 0.996486, 0.996495, 0.996506, 0.996515, 0.996526, 0.996537, 
    0.996546, 0.996558, 0.996568, 0.996577, 0.996588, 0.996599, 0.996609, 0.996619, 0.996629, 0.996638, 
    0.996647, 0.996656, 0.996665, 0.996675, 0.996686, 0.996694, 0.996703, 0.996713, 0.996723, 0.996734, 
    0.996743, 0.996751, 0.996761, 0.996771, 0.996781, 0.996792, 0.996801, 0.996812, 0.996820, 0.996830, 
    0.996840, 0.996850, 0.996859, 0.996869, 0.996878, 0.996888, 0.996897, 0.996907, 0.996916, 0.996925, 
    0.996936, 0.996944, 0.996953, 0.996962, 0.996972, 0.996980, 0.996988, 0.996997, 0.997007, 0.997016, 
    0.997025, 0.997034, 0.997043, 0.997053, 0.997062, 0.997070, 0.997079, 0.997088, 0.997097, 0.997107, 
    0.997116, 0.997126, 0.997135, 0.997145, 0.997155, 0.997161, 0.997168, 0.997177, 0.997187, 0.997195, 
    0.997205, 0.997214, 0.997222, 0.997230, 0.997237, 0.997246, 0.997256, 0.997264, 0.997274, 0.997284, 
    0.997293, 0.997302, 0.997309, 0.997316, 0.997325, 0.997334, 0.997342, 0.997351, 0.997360, 0.997368, 
    0.997377, 0.997385, 0.997393, 0.997399, 0.997406, 0.997413, 0.997422, 0.997429, 0.997437, 0.997444, 
    0.997453, 0.997460, 0.997468, 0.997475, 0.997482, 0.997489, 0.997497, 0.997504, 0.997513, 0.997519, 
    0.997529, 0.997537, 0.997547, 0.997555, 0.997562, 0.997568, 0.997576, 0.997584, 0.997592, 0.997600, 
    0.997606, 0.997614, 0.997621, 0.997628, 0.997636, 0.997643, 0.997651, 0.997658, 0.997665, 0.997672, 
    0.997678, 0.997686, 0.997693, 0.997700, 0.997706, 0.997713, 0.997719, 0.997726, 0.997732, 0.997740, 
    0.997747, 0.997754, 0.997762, 0.997768, 0.997775, 0.997782, 0.997788, 0.997795, 0.997801, 0.997807, 
    0.997814, 0.997820, 0.997826, 0.997832, 0.997838, 0.997845, 0.997852, 0.997860, 0.997868, 0.997873, 
    0.997879, 0.997886, 0.997892, 0.997898, 0.997906, 0.997912, 0.997921, 0.997927, 0.997934, 0.997942, 
    0.997948, 0.997954, 0.997961, 0.997967, 0.997973, 0.997979, 0.997984, 0.997989, 0.997996, 0.998003, 
    0.998009, 0.998013, 0.998019, 0.998026, 0.998032, 0.998039, 0.998046, 0.998052, 0.998059, 0.998065, 
    0.998071, 0.998077, 0.998082, 0.998088, 0.998094, 0.998102, 0.998109, 0.998115, 0.998121, 0.998127, 
    0.998132, 0.998138, 0.998143, 0.998149, 0.998155, 0.998163, 0.998167, 0.998173, 0.998179, 0.998185, 
    0.998191, 0.998198, 0.998204, 0.998210, 0.998216, 0.998222, 0.998228, 0.998233, 0.998239, 0.998245, 
    0.998251, 0.998256, 0.998260, 0.998266, 0.998272, 0.998277, 0.998283, 0.998289, 0.998296, 0.998300, 
    0.998305, 0.998312, 0.998316, 0.998321, 0.998327, 0.998331, 0.998336, 0.998341, 0.998346, 0.998351, 
    0.998355, 0.998361, 0.998367, 0.998372, 0.998378, 0.998386, 0.998391, 0.998397, 0.998403, 0.998409, 
    0.998413, 0.998417, 0.998423, 0.998428, 0.998433, 0.998438, 0.998443, 0.998449, 0.998455, 0.998460, 
    0.998464, 0.998469, 0.998474, 0.998478, 0.998483, 0.998487, 0.998492, 0.998497, 0.998500, 0.998504, 
    0.998509, 0.998514, 0.998518, 0.998521, 0.998526, 0.998531, 0.998536, 0.998540, 0.998544, 0.998550, 
    0.998555, 0.998559, 0.998564, 0.998569, 0.998573, 0.998577, 0.998580, 0.998585, 0.998589, 0.998594, 
    0.998599, 0.998604, 0.998609, 0.998615, 0.998620, 0.998624, 0.998627, 0.998633, 0.998636, 0.998641, 
    0.998645, 0.998649, 0.998654, 0.998659, 0.998663, 0.998668, 0.998672, 0.998676, 0.998681, 0.998685, 
    0.998689, 0.998694, 0.998698, 0.998702, 0.998705, 0.998708, 0.998712, 0.998716, 0.998721, 0.998726, 
    0.998730, 0.998734, 0.998738, 0.998742, 0.998746, 0.998750, 0.998754, 0.998758, 0.998762, 0.998766, 
    0.998770, 0.998775, 0.998779, 0.998782, 0.998785, 0.998789, 0.998795, 0.998799, 0.998803, 0.998807, 
    0.998810, 0.998814, 0.998819, 0.998822, 0.998825, 0.998828, 0.998831, 0.998835, 0.998839, 0.998845, 
    0.998848, 0.998852, 0.998856, 0.998859, 0.998863, 0.998868, 0.998871, 0.998875, 0.998879, 0.998882, 
    0.998886, 0.998890, 0.998892, 0.998896, 0.998900, 0.998904, 0.998908, 0.998911, 0.998915, 0.998919, 
    0.998923, 0.998926, 0.998930, 0.998933, 0.998936, 0.998940, 0.998944, 0.998947, 0.998950, 0.998954, 
    0.998957, 0.998961, 0.998965, 0.998968, 0.998972, 0.998974, 0.998979, 0.998982, 0.998986, 0.998989, 
    0.998993, 0.998997, 0.999000, 0.999004, 0.999007, 0.999011, 0.999014, 0.999016, 0.999020, 0.999024, 
    0.999027, 0.999030, 0.999034, 0.999037, 0.999040, 0.999043, 0.999046, 0.999050, 0.999053, 0.999056, 
    0.999059, 0.999061, 0.999063, 0.999066, 0.999070, 0.999072, 0.999076, 0.999079, 0.999082, 0.999084, 
    0.999087, 0.999090, 0.999093, 0.999096, 0.999099, 0.999102, 0.999105, 0.999109, 0.999112, 0.999114, 
    0.999118, 0.999121, 0.999124, 0.999127, 0.999131, 0.999134, 0.999137, 0.999141, 0.999144, 0.999147, 
    0.999150, 0.999154, 0.999156, 0.999158, 0.999161, 0.999165, 0.999168, 0.999171, 0.999174, 0.999176, 
    0.999178, 0.999181, 0.999184, 0.999186, 0.999189, 0.999192, 0.999195, 0.999197, 0.999201, 0.999203, 
    0.999205, 0.999208, 0.999212, 0.999214, 0.999216, 0.999219, 0.999221, 0.999225, 0.999228, 0.999230, 
    0.999233, 0.999236, 0.999239, 0.999242, 0.999245, 0.999247, 0.999249, 0.999252, 0.999255, 0.999256, 
    0.999259, 0.999261, 0.999263, 0.999266, 0.999268, 0.999270, 0.999273, 0.999275, 0.999278, 0.999279, 
    0.999282, 0.999284, 0.999285, 0.999289, 0.999291, 0.999293, 0.999296, 0.999299, 0.999302, 0.999305, 
    0.999306, 0.999309, 0.999311, 0.999314, 0.999317, 0.999320, 0.999322, 0.999325, 0.999328, 0.999329, 
    0.999332, 0.999334, 0.999336, 0.999338, 0.999341, 0.999343, 0.999345, 0.999347, 0.999349, 0.999351, 
    0.999354, 0.999356, 0.999360, 0.999361, 0.999363, 0.999364, 0.999368, 0.999370, 0.999372, 0.999374, 
    0.999377, 0.999378, 0.999380, 0.999383, 0.999386, 0.999387, 0.999390, 0.999391, 0.999394, 0.999396, 
    0.999398, 0.999400, 0.999402, 0.999404, 0.999406, 0.999407, 0.999409, 0.999411, 0.999413, 0.999415, 
    0.999417, 0.999419, 0.999420, 0.999423, 0.999424, 0.999426, 0.999428, 0.999430, 0.999433, 0.999434, 
    0.999436, 0.999439, 0.999441, 0.999443, 0.999445, 0.999447, 0.999449, 0.999451, 0.999453, 0.999456, 
    0.999458, 0.999460, 0.999462, 0.999463, 0.999464, 0.999466, 0.999468, 0.999470, 0.999472, 0.999475, 
    0.999477, 0.999479, 0.999482, 0.999483, 0.999485, 0.999487, 0.999488, 0.999490, 0.999491, 0.999494, 
    0.999496, 0.999498, 0.999500, 0.999502, 0.999503, 0.999505, 0.999506, 0.999509, 0.999510, 0.999513, 
    0.999514, 0.999516, 0.999518, 0.999520, 0.999521, 0.999524, 0.999525, 0.999526, 0.999529, 0.999531, 
    0.999533, 0.999536, 0.999537, 0.999539, 0.999541, 0.999543, 0.999544, 0.999546, 0.999547, 0.999549, 
    0.999551, 0.999553, 0.999555, 0.999556, 0.999558, 0.999559, 0.999560, 0.999562, 0.999564, 0.999565, 
    0.999566, 0.999568, 0.999569, 0.999570, 0.999572, 0.999573, 0.999575, 0.999576, 0.999578, 0.999579, 
    0.999580, 0.999582, 0.999584, 0.999586, 0.999587, 0.999588, 0.999589, 0.999591, 0.999592, 0.999593, 
    0.999595, 0.999596, 0.999598, 0.999599, 0.999601, 0.999603, 0.999603, 0.999605, 0.999606, 0.999607, 
    0.999609, 0.999610, 0.999612, 0.999613, 0.999614, 0.999616, 0.999617, 0.999618, 0.999619, 0.999621, 
    0.999622, 0.999624, 0.999625, 0.999626, 0.999628, 0.999629, 0.999630, 0.999631, 0.999633, 0.999634, 
    0.999636, 0.999637, 0.999637, 0.999639, 0.999640, 0.999641, 0.999643, 0.999643, 0.999644, 0.999645, 
    0.999646, 0.999648, 0.999650, 0.999651, 0.999652, 0.999654, 0.999655, 0.999656, 0.999656, 0.999658, 
    0.999659, 0.999661, 0.999662, 0.999664, 0.999666, 0.999667, 0.999668, 0.999669, 0.999670, 0.999671, 
    0.999672, 0.999673, 0.999675, 0.999676, 0.999677, 0.999678, 0.999679, 0.999680, 0.999682, 0.999683, 
    0.999685, 0.999686, 0.999687, 0.999688, 0.999690, 0.999691, 0.999692, 0.999693, 0.999695, 0.999696, 
    0.999697, 0.999698, 0.999699, 0.999700, 0.999701, 0.999702, 0.999702, 0.999703, 0.999704, 0.999706, 
    0.999707, 0.999708, 0.999708, 0.999710, 0.999710, 0.999711, 0.999713, 0.999714, 0.999714, 0.999716, 
    0.999717, 0.999718, 0.999719, 0.999720, 0.999721, 0.999721, 0.999723, 0.999724, 0.999725, 0.999726, 
    0.999727, 0.999728, 0.999730, 0.999731, 0.999732, 0.999733, 0.999734, 0.999735, 0.999737, 0.999738, 
    0.999739, 0.999740, 0.999741, 0.999742, 0.999743, 0.999744, 0.999744, 0.999745, 0.999746, 0.999747, 
    0.999748, 0.999749, 0.999749, 0.999750, 0.999750, 0.999751, 0.999753, 0.999754, 0.999755, 0.999755, 
    0.999757, 0.999757, 0.999757, 0.999758, 0.999759, 0.999759, 0.999760, 0.999761, 0.999762, 0.999763, 
    0.999764, 0.999765, 0.999766, 0.999767, 0.999768, 0.999769, 0.999769, 0.999770, 0.999771, 0.999771, 
    0.999772, 0.999773, 0.999774, 0.999776, 0.999776, 0.999776, 0.999777, 0.999778, 0.999779, 0.999779, 
    0.999781, 0.999781, 0.999782, 0.999782, 0.999783, 0.999784, 0.999785, 0.999785, 0.999786, 0.999787, 
    0.999788, 0.999789, 0.999791, 0.999792, 0.999792, 0.999793, 0.999794, 0.999795, 0.999796, 0.999796, 
    0.999797, 0.999798, 0.999799, 0.999799, 0.999800, 0.999800, 0.999802, 0.999802, 0.999803, 0.999803, 
    0.999804, 0.999805, 0.999806, 0.999807, 0.999807, 0.999808, 0.999808, 0.999810, 0.999811, 0.999811, 
    0.999811, 0.999812, 0.999813, 0.999813, 0.999814, 0.999814, 0.999816, 0.999816, 0.999817, 0.999818, 
    0.999819, 0.999819, 0.999820, 0.999821, 0.999821, 0.999822, 0.999823, 0.999823, 0.999824, 0.999825, 
    0.999825, 0.999826, 0.999826, 0.999827, 0.999828, 0.999829, 0.999830, 0.999830, 0.999831, 0.999832, 
    0.999833, 0.999833, 0.999834, 0.999834, 0.999835, 0.999836, 0.999837, 0.999837, 0.999838, 0.999838, 
    0.999839, 0.999839, 0.999840, 0.999841, 0.999841, 0.999842, 0.999842, 0.999843, 0.999843, 0.999844, 
    0.999844, 0.999845, 0.999845, 0.999845, 0.999846, 0.999846, 0.999846, 0.999847, 0.999848, 0.999848, 
    0.999849, 0.999849, 0.999850, 0.999850, 0.999851, 0.999852, 0.999852, 0.999853, 0.999854, 0.999854, 
    0.999854, 0.999856, 0.999856, 0.999857, 0.999858, 0.999858, 0.999859, 0.999859, 0.999860, 0.999860, 
    0.999861, 0.999861, 0.999862, 0.999862, 0.999863, 0.999863, 0.999864, 0.999864, 0.999865, 0.999865, 
    0.999865, 0.999866, 0.999866, 0.999867, 0.999867, 0.999868, 0.999869, 0.999869, 0.999869, 0.999869, 
    0.999870, 0.999871, 0.999871, 0.999872, 0.999872, 0.999873, 0.999873, 0.999873, 0.999873, 0.999874, 
    0.999874, 0.999874, 0.999875, 0.999875, 0.999876, 0.999876, 0.999877, 0.999877, 0.999878, 0.999878, 
    0.999878, 0.999879, 0.999880, 0.999880, 0.999880, 0.999881, 0.999881, 0.999882, 0.999882, 0.999882, 
    0.999883, 0.999884, 0.999884, 0.999884, 0.999885, 0.999885, 0.999885, 0.999886, 0.999886, 0.999887, 
    0.999887, 0.999888, 0.999889, 0.999890, 0.999890, 0.999891, 0.999891, 0.999891, 0.999892, 0.999892, 
    0.999892, 0.999892, 0.999893, 0.999893, 0.999894, 0.999894, 0.999894, 0.999895, 0.999895, 0.999896, 
    0.999896, 0.999897, 0.999897, 0.999897, 0.999898, 0.999898, 0.999898, 0.999899, 0.999899, 0.999899, 
    0.999899, 0.999900, 0.999900, 0.999901, 0.999901, 0.999902, 0.999902, 0.999902, 0.999903, 0.999903, 
    0.999904, 0.999904, 0.999904, 0.999905, 0.999905, 0.999905, 0.999905, 0.999906, 0.999906, 0.999906, 
    0.999907, 0.999907, 0.999908, 0.999908, 0.999908, 0.999909, 0.999909, 0.999909, 0.999910, 0.999910, 
    0.999911, 0.999911, 0.999911, 0.999912, 0.999912, 0.999913, 0.999913, 0.999913, 0.999913, 0.999914, 
    0.999914, 0.999915, 0.999915, 0.999915, 0.999915, 0.999916, 0.999916, 0.999916, 0.999917, 0.999917, 
    0.999917, 0.999917, 0.999918, 0.999918, 0.999919, 0.999919, 0.999919, 0.999920, 0.999920, 0.999920, 
    0.999920, 0.999920, 0.999920, 0.999921, 0.999921, 0.999921, 0.999922, 0.999922, 0.999922, 0.999923, 
    0.999923, 0.999923, 0.999923, 0.999923, 0.999923, 0.999924, 0.999924, 0.999924, 0.999924, 0.999925, 
    0.999925, 0.999925, 0.999926, 0.999926, 0.999927, 0.999927, 0.999927, 0.999928, 0.999928, 0.999928, 
    0.999928, 0.999929, 0.999929, 0.999929, 0.999929, 0.999930, 0.999930, 0.999930, 0.999930, 0.999931, 
    0.999932, 0.999932, 0.999932, 0.999933, 0.999933, 0.999933, 0.999933, 0.999934, 0.999934, 0.999934, 
    0.999934, 0.999934, 0.999934, 0.999935, 0.999935, 0.999936, 0.999936, 0.999936, 0.999936, 0.999937, 
    0.999937, 0.999938, 0.999938, 0.999938, 0.999938, 0.999939, 0.999939, 0.999939, 0.999939, 0.999940, 
    0.999940, 0.999941, 0.999941, 0.999941, 0.999941, 0.999941, 0.999942, 0.999942, 0.999942, 0.999942, 
    0.999943, 0.999943, 0.999943, 0.999943, 0.999944, 0.999944, 0.999944, 0.999944, 0.999944, 0.999944, 
    0.999944, 0.999944, 0.999945, 0.999945, 0.999945, 0.999946, 0.999946, 0.999946, 0.999947, 0.999947, 
    0.999947, 0.999947, 0.999947, 0.999948, 0.999948, 0.999948, 0.999948, 0.999949, 0.999949, 0.999949, 
    0.999949, 0.999949, 0.999949, 0.999950, 0.999950, 0.999950, 0.999950, 0.999950, 0.999951, 0.999951, 
    0.999951, 0.999951, 0.999951, 0.999951, 0.999952, 0.999952, 0.999952, 0.999952, 0.999952, 0.999952, 
    0.999952, 0.999953, 0.999953, 0.999953, 0.999953, 0.999953, 0.999954, 0.999954, 0.999954, 0.999954, 
    0.999954, 0.999954, 0.999954, 0.999955, 0.999955, 0.999955, 0.999955, 0.999955, 0.999956, 0.999956, 
    0.999956, 0.999956, 0.999957, 0.999957, 0.999957, 0.999957, 0.999957, 0.999958, 0.999958, 0.999958, 
    0.999958, 0.999958, 0.999958, 0.999959, 0.999959, 0.999960, 0.999960, 0.999960, 0.999960, 0.999960, 
    0.999960, 0.999960, 0.999961, 0.999961, 0.999961, 0.999961, 0.999961, 0.999961, 0.999961, 0.999962, 
    0.999962, 0.999962, 0.999962, 0.999962, 0.999962, 0.999962, 0.999963, 0.999963, 0.999963, 0.999963, 
    0.999963, 0.999963, 0.999964, 0.999964, 0.999964, 0.999964, 0.999964, 0.999964, 0.999964, 0.999965, 
    0.999965, 0.999965, 0.999965, 0.999965, 0.999966, 0.999966, 0.999966, 0.999966, 0.999966, 0.999966, 
    0.999966, 0.999966, 0.999967, 0.999967, 0.999967, 0.999967, 0.999967, 0.999967, 0.999968, 0.999968, 
    0.999968, 0.999968, 0.999968, 0.999968, 0.999969, 0.999969, 0.999969, 0.999969, 0.999969, 0.999969, 
    0.999969, 0.999970, 0.999970, 0.999970, 0.999970, 0.999970, 0.999970, 0.999970, 0.999970, 0.999971, 
    0.999971, 0.999971, 0.999971, 0.999971, 0.999971, 0.999972, 0.999972, 0.999972, 0.999972, 0.999972, 
    0.999972, 0.999972, 0.999973, 0.999973, 0.999973, 0.999973, 0.999973, 0.999973, 0.999973, 0.999974, 
    0.999974, 0.999974, 0.999974, 0.999975, 0.999975, 0.999975, 0.999975, 0.999975, 0.999975, 0.999975, 
    0.999975, 0.999975, 0.999975, 0.999975, 0.999975, 0.999975, 0.999975, 0.999976, 0.999976, 0.999976, 
    0.999976, 0.999976, 0.999976, 0.999976, 0.999976, 0.999977, 0.999977, 0.999977, 0.999977, 0.999977, 
    0.999977, 0.999977, 0.999977, 0.999977, 0.999977, 0.999977, 0.999978, 0.999978, 0.999978, 0.999978, 
    0.999978, 0.999978, 0.999978, 0.999978, 0.999978, 0.999978, 0.999978, 0.999978, 0.999978, 0.999978, 
    0.999978, 0.999979, 0.999979, 0.999979, 0.999979, 0.999979, 0.999979, 0.999979, 0.999979, 0.999979, 
    0.999979, 0.999979, 0.999979, 0.999979, 0.999979, 0.999980, 0.999980, 0.999980, 0.999980, 0.999980, 
    0.999980, 0.999980, 0.999980, 0.999980, 0.999980, 0.999980, 0.999980, 0.999980, 0.999980, 0.999980, 
    0.999981, 0.999981, 0.999981, 0.999981, 0.999981, 0.999981, 0.999981, 0.999981, 0.999981, 0.999981, 
    0.999981, 0.999981, 0.999982, 0.999982, 0.999982, 0.999982, 0.999982, 0.999982, 0.999982, 0.999982, 
    0.999982, 0.999982, 0.999983, 0.999983, 0.999983, 0.999983, 0.999983, 0.999983, 0.999983, 0.999983, 
    0.999983, 0.999983, 0.999983, 0.999983, 0.999983, 0.999983, 0.999984, 0.999984, 0.999984, 0.999984, 
    0.999984, 0.999984, 0.999984, 0.999984, 0.999984, 0.999984, 0.999984, 0.999985, 0.999985, 0.999985, 
    0.999985, 0.999985, 0.999985, 0.999985, 0.999985, 0.999985, 0.999985, 0.999985, 0.999985, 0.999985, 
    0.999985, 0.999985, 0.999986, 0.999986, 0.999986, 0.999986, 0.999986, 0.999986, 0.999986, 0.999986, 
    0.999986, 0.999986, 0.999986, 0.999986, 0.999986, 0.999986, 0.999986, 0.999986, 0.999986, 0.999986, 
    0.999986, 0.999987, 0.999987, 0.999987, 0.999987, 0.999987, 0.999987, 0.999987, 0.999987, 0.999987, 
    0.999987, 0.999987, 0.999987, 0.999987, 0.999987, 0.999987, 0.999987, 0.999987, 0.999987, 0.999987, 
    0.999987, 0.999987, 0.999987, 0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 
    0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 
    0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 0.999988, 
    0.999988, 0.999989, 0.999989, 0.999989, 0.999989, 0.999989, 0.999989, 0.999989, 0.999989, 0.999989, 
    0.999989, 0.999989, 0.999989, 0.999989, 0.999989, 0.999989, 0.999989, 0.999989, 0.999989, 0.999989, 
    0.999989, 0.999989, 0.999990, 0.999990, 0.999990, 0.999990, 0.999990, 0.999990, 0.999990, 0.999990, 
    0.999990, 0.999990, 0.999990, 0.999990, 0.999990, 0.999991, 0.999991, 0.999991, 0.999991, 0.999991, 
    0.999991, 0.999991, 0.999991, 0.999991, 0.999991, 0.999991, 0.999991, 0.999991, 0.999991, 0.999991, 
    0.999991, 0.999991, 0.999991, 0.999992, 0.999992, 0.999992, 0.999992, 0.999992, 0.999992, 0.999992, 
    0.999992, 0.999992, 0.999992, 0.999992, 0.999992, 0.999992, 0.999992, 0.999992, 0.999992, 0.999992, 
    0.999992, 0.999992, 0.999992, 0.999992, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 
    0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 
    0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 
    0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 
    0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999993, 0.999994, 0.999994, 0.999994, 0.999994, 
    0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 
    0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 
    0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 
    0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 0.999994, 
    0.999994, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 
    0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 
    0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 0.999995, 
    0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 
    0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 
    0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 
    0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 
    0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 0.999996, 
    0.999996, 0.999996, 0.999996, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 
    0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 
    0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 
    0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 
    0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 0.999997, 
    0.999997, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 
    0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 
    0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 
    0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 
    0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 
    0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 
    0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 
    0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 
    0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 
    0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 
    0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 
    0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999998, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 
    0.999999, 0.999999, 0.999999, 0.999999, 0.999999, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
};

