/*
 *  ====================================================================
 *    Licensed to the Apache Software Foundation (ASF) under one or more
 *    contributor license agreements.  See the NOTICE file distributed with
 *    this work for additional information regarding copyright ownership.
 *    The ASF licenses this file to You under the Apache License, Version 2.0
 *    (the "License"); you may not use this file except in compliance with
 *    the License.  You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 * ====================================================================
 */

package org.apache.poi.hssf.record;

import junit.framework.TestCase;
import org.apache.poi.hssf.record.common.UnicodeString;
import org.apache.poi.util.HexRead;

/**
 * @author Yegor Kozlov
 */
public final class TestExtSSTRecord extends TestCase {


    /**
     * hex dump from ISBN_UPD_PL_20100104_1525.xls atatched to Bugzilla 50867
     */
    private static final String data_50967 =
                    // "Offset=0x0008BB19(572185) recno=159 sid=0x00FF size=0x201A(8218)\n" +
                    "                           FF 00 1A 20 08 00 1A" +
                    "09 00 00 0C 00 00 00 6D 09 00 00 5F 00 00 00 B8" +
                    "09 00 00 AA 00 00 00 BC 0A 00 00 AE 01 00 00 FA" +
                    "0B 00 00 EC 02 00 00 C6 0D 00 00 B8 04 00 00 2E" +
                    "0F 00 00 20 06 00 00 5C 11 00 00 4E 08 00 00 B0" +
                    "12 00 00 A2 09 00 00 A4 13 00 00 96 0A 00 00 92" +
                    "15 00 00 84 0C 00 00 0A 17 00 00 FC 0D 00 00 32" +
                    "18 00 00 24 0F 00 00 84 19 00 00 76 10 00 00 9A" +
                    "1A 00 00 8C 11 00 00 5E 1C 00 00 50 13 00 00 24" +
                    "1D 00 00 16 14 00 00 56 1E 00 00 48 15 00 00 1C" +
                    "1F 00 00 0E 16 00 00 DA 1F 00 00 CC 16 00 00 76" +
                    "21 00 00 68 18 00 00 2E 23 00 00 20 1A 00 00 A0" +
                    "24 00 00 92 1B 00 00 FA 25 00 00 EC 1C 00 00 BA" +
                    "27 00 00 AC 1E 00 00 12 29 00 00 04 00 00 00 4E" +
                    "2A 00 00 40 01 00 00 A0 2B 00 00 92 02 00 00 C8" +
                    "2C 00 00 BA 03 00 00 62 2E 00 00 54 05 00 00 58" +
                    "2F 00 00 4A 06 00 00 9A 30 00 00 8C 07 00 00 02" +
                    "32 00 00 F4 08 00 00 36 33 00 00 28 0A 00 00 30" +
                    "34 00 00 22 0B 00 00 FC 34 00 00 EE 0B 00 00 FE" +
                    "35 00 00 F0 0C 00 00 26 37 00 00 18 0E 00 00 C6" +
                    "38 00 00 B8 0F 00 00 FA 39 00 00 EC 10 00 00 8E" +
                    "3B 00 00 80 12 00 00 2C 3D 00 00 1E 14 00 00 3C" +
                    "3E 00 00 2E 15 00 00 80 3F 00 00 72 16 00 00 74" +
                    "40 00 00 66 17 00 00 8A 41 00 00 7C 18 00 00 64" +
                    "42 00 00 56 19 00 00 96 43 00 00 88 1A 00 00 82" +
                    "44 00 00 74 1B 00 00 76 45 00 00 68 1C 00 00 80" +
                    "46 00 00 72 1D 00 00 88 47 00 00 7A 1E 00 00 80" +
                    "48 00 00 72 1F 00 00 66 49 00 00 34 00 00 00 76" +
                    "4A 00 00 44 01 00 00 D0 4B 00 00 9E 02 00 00 F2" +
                    "4C 00 00 C0 03 00 00 DA 4D 00 00 A8 04 00 00 B4" +
                    "4F 00 00 82 06 00 00 12 51 00 00 E0 07 00 00 2A" +
                    "52 00 00 F8 08 00 00 3C 53 00 00 0A 0A 00 00 6E" +
                    "54 00 00 3C 0B 00 00 BC 55 00 00 8A 0C 00 00 14" +
                    "57 00 00 E2 0D 00 00 3C 58 00 00 0A 0F 00 00 36" +
                    "59 00 00 04 10 00 00 4A 5A 00 00 18 11 00 00 4A" +
                    "5B 00 00 18 12 00 00 AC 5C 00 00 7A 13 00 00 C2" +
                    "5D 00 00 90 14 00 00 2C 5F 00 00 FA 15 00 00 74" +
                    "60 00 00 42 17 00 00 56 62 00 00 24 19 00 00 12" +
                    "64 00 00 E0 1A 00 00 E8 65 00 00 B6 1C 00 00 CA" +
                    "66 00 00 98 1D 00 00 14 68 00 00 E2 1E 00 00 56" +
                    "69 00 00 04 00 00 00 EC 6A 00 00 9A 01 00 00 DC" +
                    "6B 00 00 8A 02 00 00 EE 6C 00 00 9C 03 00 00 66" +
                    "6E 00 00 14 05 00 00 A6 6F 00 00 54 06 00 00 92" +
                    "70 00 00 40 07 00 00 E6 71 00 00 94 08 00 00 48" +
                    "73 00 00 F6 09 00 00 AC 74 00 00 5A 0B 00 00 A2" +
                    "75 00 00 50 0C 00 00 C4 76 00 00 72 0D 00 00 4E" +
                    "78 00 00 FC 0E 00 00 14 7A 00 00 C2 10 00 00 0C" +
                    "7B 00 00 BA 11 00 00 4E 7C 00 00 FC 12 00 00 5A" +
                    "7D 00 00 08 14 00 00 C4 7E 00 00 72 15 00 00 BA" +
                    "80 00 00 68 17 00 00 1C 82 00 00 CA 18 00 00 5E" +
                    "83 00 00 0C 1A 00 00 BA 84 00 00 68 1B 00 00 E6" +
                    "85 00 00 94 1C 00 00 46 87 00 00 F4 1D 00 00 94" +
                    "88 00 00 42 1F 00 00 24 8A 00 00 10 01 00 00 80" +
                    "8B 00 00 6C 02 00 00 8E 8C 00 00 7A 03 00 00 82" +
                    "8D 00 00 6E 04 00 00 54 8E 00 00 40 05 00 00 60" +
                    "8F 00 00 4C 06 00 00 68 90 00 00 54 07 00 00 92" +
                    "91 00 00 7E 08 00 00 C8 92 00 00 B4 09 00 00 2C" +
                    "94 00 00 18 0B 00 00 8E 95 00 00 7A 0C 00 00 AA" +
                    "96 00 00 96 0D 00 00 FA 97 00 00 E6 0E 00 00 3E" +
                    "99 00 00 2A 10 00 00 54 9A 00 00 40 11 00 00 D4" +
                    "9B 00 00 C0 12 00 00 C2 9C 00 00 AE 13 00 00 1E" +
                    "9E 00 00 0A 15 00 00 F4 9E 00 00 E0 15 00 00 E0" +
                    "9F 00 00 CC 16 00 00 08 A1 00 00 F4 17 00 00 0E" +
                    "A2 00 00 FA 18 00 00 8A A3 00 00 76 1A 00 00 78" +
                    "A5 00 00 64 1C 00 00 4E A6 00 00 3A 1D 00 00 42" +
                    "A7 00 00 2E 1E 00 00 08 A9 00 00 F4 1F 00 00 BE" +
                    "AA 00 00 8D 01 00 00 34 AC 00 00 03 03 00 00 62" +
                    "AD 00 00 31 04 00 00 68 AE 00 00 37 05 00 00 52" +
                    "AF 00 00 21 06 00 00 DE B0 00 00 AD 07 00 00 A8" +
                    "B1 00 00 77 08 00 00 B6 B2 00 00 85 09 00 00 DA" +
                    "B3 00 00 A9 0A 00 00 EC B4 00 00 BB 0B 00 00 EA" +
                    "B5 00 00 B9 0C 00 00 0A B7 00 00 D9 0D 00 00 0A" +
                    "B8 00 00 D9 0E 00 00 26 B9 00 00 F5 0F 00 00 72" +
                    "BA 00 00 41 11 00 00 D8 BB 00 00 A7 12 00 00 3A" +
                    "BD 00 00 09 14 00 00 6C BE 00 00 3B 15 00 00 88" +
                    "BF 00 00 57 16 00 00 D8 C0 00 00 A7 17 00 00 60" +
                    "C2 00 00 2F 19 00 00 84 C3 00 00 53 1A 00 00 A8" +
                    "C4 00 00 77 1B 00 00 CE C5 00 00 9D 1C 00 00 02" +
                    "C7 00 00 D1 1D 00 00 58 C8 00 00 27 1F 00 00 56" +
                    "C9 00 00 04 00 00 00 D0 CA 00 00 7E 01 00 00 D4" +
                    "CB 00 00 82 02 00 00 38 CD 00 00 E6 03 00 00 E4" +
                    "CF 00 00 92 06 00 00 76 D1 00 00 24 08 00 00 10" +
                    "D3 00 00 BE 09 00 00 50 D4 00 00 FE 0A 00 00 94" +
                    "D5 00 00 42 0C 00 00 14 D7 00 00 C2 0D 00 00 92" +
                    "D8 00 00 40 0F 00 00 EC D9 00 00 9A 10 00 00 20" +
                    "DC 00 00 CE 12 00 00 7E DD 00 00 2C 14 00 00 5C" +
                    "DF 00 00 0A 16 00 00 B8 E0 00 00 66 17 00 00 9C" +
                    "E1 00 00 4A 18 00 00 44 E3 00 00 F2 19 00 00 A6" +
                    "E4 00 00 54 1B 00 00 F8 E5 00 00 A6 1C 00 00 D4" +
                    "E7 00 00 82 1E 00 00 2E E9 00 00 DC 1F 00 00 C8" +
                    "EA 00 00 75 01 00 00 3C EC 00 00 E9 02 00 00 6A" +
                    "ED 00 00 17 04 00 00 BC EE 00 00 69 05 00 00 18" +
                    "F0 00 00 C5 06 00 00 D0 F1 00 00 7D 08 00 00 66" +
                    "F3 00 00 13 0A 00 00 8C F4 00 00 39 0B 00 00 EC" +
                    "F5 00 00 99 0C 00 00 2A F7 00 00 D7 0D 00 00 84" +
                    "F8 00 00 31 0F 00 00 A0 F9 00 00 4D 10 00 00 06" +
                    "FB 00 00 B3 11 00 00 E6 FB 00 00 93 12 00 00 08" +
                    "FD 00 00 B5 13 00 00 66 FE 00 00 13 15 00 00 DA" +
                    "FF 00 00 87 16 00 00 52 01 01 00 FF 17 00 00 74" +
                    "02 01 00 21 19 00 00 96 03 01 00 43 1A 00 00 BC" +
                    "04 01 00 69 1B 00 00 72 06 01 00 1F 1D 00 00 D4" +
                    "07 01 00 81 1E 00 00 56 09 01 00 04 00 00 00 FC" +
                    "0A 01 00 AA 01 00 00 5E 0C 01 00 0C 03 00 00 B6" +
                    "0D 01 00 64 04 00 00 F4 0E 01 00 A2 05 00 00 74" +
                    "10 01 00 22 07 00 00 00 12 01 00 AE 08 00 00 4A" +
                    "13 01 00 F8 09 00 00 40 14 01 00 EE 0A 00 00 66" +
                    "15 01 00 14 0C 00 00 B8 16 01 00 66 0D 00 00 3E" +
                    "18 01 00 EC 0E 00 00 58 19 01 00 06 10 00 00 72" +
                    "1A 01 00 20 11 00 00 4A 1C 01 00 F8 12 00 00 8E" +
                    "1D 01 00 3C 14 00 00 E8 1E 01 00 96 15 00 00 A8" +
                    "20 01 00 56 17 00 00 88 23 01 00 36 1A 00 00 00" +
                    "25 01 00 AE 1B 00 00 AE 26 01 00 5C 1D 00 00 88" +
                    "28 01 00 36 1F 00 00 96 29 01 00 25 00 00 00 E6" +
                    "2A 01 00 75 01 00 00 A2 2C 01 00 31 03 00 00 40" +
                    "2E 01 00 CF 04 00 00 B8 2F 01 00 47 06 00 00 64" +
                    "31 01 00 F3 07 00 00 C0 32 01 00 4F 09 00 00 1C" +
                    "34 01 00 AB 0A 00 00 A4 35 01 00 33 0C 00 00 2A" +
                    "37 01 00 B9 0D 00 00 9A 38 01 00 29 0F 00 00 94" +
                    "3A 01 00 23 11 00 00 86 3B 01 00 15 12 00 00 44" +
                    "3D 01 00 D3 13 00 00 6E 3E 01 00 FD 14 00 00 8C" +
                    "40 01 00 1B 17 00 00 44 42 01 00 D3 18 00 00 0E" +
                    "44 01 00 9D 1A 00 00 06 45 01 00 95 1B 00 00 BA" +
                    "46 01 00 49 1D 00 00 24 49 01 00 B3 1F 00 00 A2" +
                    "4A 01 00 12 01 00 00 2C 4C 01 00 9C 02 00 00 D4" +
                    "4D 01 00 44 04 00 00 5E 50 01 00 CE 06 00 00 82" +
                    "51 01 00 F2 07 00 00 9C 52 01 00 0C 09 00 00 9E" +
                    "54 01 00 0E 0B 00 00 6C 56 01 00 DC 0C 00 00 42" +
                    "57 01 00 B2 0D 00 00 70 58 01 00 E0 0E 00 00 6E" +
                    "5A 01 00 DE 10 00 00 86 5C 01 00 F6 12 00 00 F2" +
                    "5D 01 00 62 14 00 00 A0 5F 01 00 10 16 00 00 6C" +
                    "61 01 00 DC 17 00 00 6C 64 01 00 DC 1A 00 00 F0" +
                    "66 01 00 60 1D 00 00 4E 69 01 00 BE 1F 00 00 68" +
                    "6A 01 00 EA 00 00 00 6E 6C 01 00 F0 02 00 00 B8" +
                    "6D 01 00 3A 04 00 00 52 6F 01 00 D4 05 00 00 C6" +
                    "70 01 00 48 07 00 00 80 72 01 00 02 09 00 00 DC" +
                    "73 01 00 5E 0A 00 00 F2 75 01 00 74 0C 00 00 6C" +
                    "77 01 00 EE 0D 00 00 46 79 01 00 C8 0F 00 00 CC" +
                    "7A 01 00 4E 11 00 00 80 7C 01 00 02 13 00 00 08" +
                    "7E 01 00 8A 14 00 00 80 7F 01 00 02 16 00 00 46" +
                    "81 01 00 C8 17 00 00 26 83 01 00 A8 19 00 00 A4" +
                    "84 01 00 26 1B 00 00 02 86 01 00 84 1C 00 00 F4" +
                    "87 01 00 76 1E 00 00 68 89 01 00 04 00 00 00 4E" +
                    "8B 01 00 EA 01 00 00 04 8D 01 00 A0 03 00 00 FE" +
                    "8E 01 00 9A 05 00 00 A2 90 01 00 3E 07 00 00 A6" +
                    "91 01 00 42 08 00 00 76 93 01 00 12 0A 00 00 72" +
                    "95 01 00 0E 0C 00 00 36 97 01 00 D2 0D 00 00 50" +
                    "99 01 00 EC 0F 00 00 E2 9B 01 00 7E 12 00 00 6A" +
                    "9D 01 00 06 14 00 00 28 9F 01 00 C4 15 00 00 F0" +
                    "A1 01 00 8C 18 00 00 B8 A3 01 00 54 1A 00 00 14" +
                    "A6 01 00 B0 1C 00 00 7C A7 01 00 18 1E 00 00 E2" +
                    "A8 01 00 7E 1F 00 00 8C AA 01 00 2A 01 00 00 8C" +
                    "AC 01 00 2A 03 00 00 A6 AE 01 00 44 05 00 00 76" +
                    "B0 01 00 14 07 00 00 A0 B2 01 00 3E 09 00 00 12" +
                    "B4 01 00 B0 0A 00 00 78 B5 01 00 16 0C 00 00 A4" +
                    "B6 01 00 42 0D 00 00 5C B7 01 00 FA 0D 00 00 08" +
                    "B8 01 00 A6 0E 00 00 D0 B9 01 00 6E 10 00 00 D2" +
                    "BB 01 00 70 12 00 00 6C BD 01 00 0A 14 00 00 78" +
                    "BF 01 00 16 16 00 00 AE C0 01 00 4C 17 00 00 BA" +
                    "C1 01 00 58 18 00 00 02 C3 01 00 A0 19 00 00 BA" +
                    "C3 01 00 58 1A 00 00 F6 C4 01 00 94 1B 00 00 70" +
                    "C7 01 00 0E 1E 00 00 E8 C8 01 00 86 1F 00 00 AC" +
                    "CA 01 00 88 01 00 00 88 CC 01 00 64 03 00 00 90" +
                    "CE 01 00 6C 05 00 00 5C D0 01 00 38 07 00 00 66" +
                    "D2 01 00 42 09 00 00 F6 D3 01 00 D2 0A 00 00 4E" +
                    "D5 01 00 2A 0C 00 00 A2 D7 01 00 7E 0E 00 00 BE" +
                    "D8 01 00 9A 0F 00 00 AE DA 01 00 8A 11 00 00 7E" +
                    "DC 01 00 5A 13 00 00 F4 DD 01 00 D0 14 00 00 98" +
                    "DF 01 00 74 16 00 00 AA E1 01 00 86 18 00 00 24" +
                    "E3 01 00 00 1A 00 00 84 E4 01 00 60 1B 00 00 58" +
                    "E6 01 00 34 1D 00 00 DA E8 01 00 B6 1F 00 00 A6" +
                    "EA 01 00 5F 01 00 00 44 EC 01 00 FD 02 00 00 B8" +
                    "ED 01 00 71 04 00 00 18 EF 01 00 D1 05 00 00 3A" +
                    "F1 01 00 F3 07 00 00 3E F3 01 00 F7 09 00 00 9C" +
                    "F4 01 00 55 0B 00 00 74 F7 01 00 2D 0E 00 00 36" +
                    "F9 01 00 EF 0F 00 00 AA FB 01 00 63 12 00 00 60" +
                    "FD 01 00 19 14 00 00 60 FF 01 00 19 16 00 00 B2" +
                    "01 02 00 6B 18 00 00 84 03 02 00 3D 1A 00 00 42" +
                    "06 02 00 FB 1C 00 00 DE 08 02 00 97 1F 00 00 42" +
                    "0B 02 00 11 02 00 00 A0 0D 02 00 6F 04 00 00 28" +
                    "10 02 00 F7 06 00 00 FE 11 02 00 CD 08 00 00 64" +
                    "13 02 00 33 0A 00 00 4C 14 02 00 1B 0B 00 00 D8" +
                    "15 02 00 A7 0C 00 00 C6 16 02 00 95 0D 00 00 F8" +
                    "17 02 00 C7 0E 00 00 C4 19 02 00 93 10 00 00 02" +
                    "1B 02 00 D1 11 00 00 A4 1C 02 00 73 13 00 00 7A" +
                    "1D 02 00 49 14 00 00 B2 1E 02 00 81 15 00 00 66" +
                    "20 02 00 35 17 00 00 B2 21 02 00 81 18 00 00 52" +
                    "23 02 00 21 1A 00 00 B4 24 02 00 83 1B 00 00 EC" +
                    "25 02 00 BB 1C 00 00 36 27 02 00 05 1E 00 00 5E" +
                    "28 02 00 2D 1F 00 00 A8 29 02 00 62 00 00 00 4A" +
                    "2B 02 00 04 02 00 00 CA 2C 02 00 84 03 00 00 20" +
                    "2E 02 00 DA 04 00 00 C6 2F 02 00 80 06 00 00 26" +
                    "31 02 00 E0 07 00 00 3A 32 02 00 F4 08 00 00 6E" +
                    "33 02 00 28 0A 00 00 F4 34 02 00 AE 0B 00 00 DA" +
                    "35 02 00 94 0C 00 00 E4 36 02 00 9E 0D 00 00 08" +
                    "38 02 00 C2 0E 00 00 52 39 02 00 0C 10 00 00 96" +
                    "3A 02 00 50 11 00 00 3C 3C 02 00 F6 12 00 00 D2" +
                    "3D 02 00 8C 14 00 00 30 3F 02 00 EA 15 00 00 7A" +
                    "40 02 00 34 17 00 00 74 41 02 00 2E 18 00 00 F4" +
                    "42 02 00 AE 19 00 00 64 44 02 00 1E 1B 00 00 B4" +
                    "45 02 00 6E 1C 00 00 FA 46 02 00 B4 1D 00 00 4A" +
                    "48 02 00 04 1F 00 00 4C 49 02 00 04 00 00 00 58" +
                    "4A 02 00 10 01 00 00 98 4B 02 00 50 02 00 00 E0" +
                    "4C 02 00 98 03 00 00 48 4E 02 00 00 05 00 00 2E" +
                    "4F 02 00 E6 05 00 00 BE 50 02 00 76 07 00 00 18" +
                    "52 02 00 D0 08 00 00 66 53 02 00 1E 0A 00 00 BA" +
                    "54 02 00 72 0B 00 00 F2 55 02 00 AA 0C 00 00 3C" +
                    "57 02 00 F4 0D 00 00 6A 58 02 00 22 0F 00 00 D4" +
                    "59 02 00 8C 10 00 00 20 5B 02 00 D8 11 00 00 4C" +
                    "5C 02 00 04 13 00 00 FC 5C 02 00 B4 13 00 00 7A" +
                    "5E 02 00 32 15 00 00 B4 5F 02 00 6C 16 00 00 FA" +
                    "60 02 00 B2 17 00 00 7E 62 02 00 36 19 00 00 3C" +
                    "63 02 00 F4 19 00 00 CE 64 02 00 86 1B 00 00 F6" +
                    "65 02 00 AE 1C 00 00 18 67 02 00 D0 1D 00 00 82" +
                    "68 02 00 3A 1F 00 00 A0 69 02 00 3B 00 00 00 CE" +
                    "6A 02 00 69 01 00 00 CC 6B 02 00 67 02 00 00 D6" +
                    "6C 02 00 71 03 00 00 0E 6E 02 00 A9 04 00 00 24" +
                    "6F 02 00 BF 05 00 00 4A 70 02 00 E5 06 00 00 7A" +
                    "71 02 00 15 08 00 00 5A 72 02 00 F5 08 00 00 A0" +
                    "73 02 00 3B 0A 00 00 B6 74 02 00 51 0B 00 00 DA" +
                    "75 02 00 75 0C 00 00 E4 76 02 00 7F 0D 00 00 20" +
                    "78 02 00 BB 0E 00 00 00 79 02 00 9B 0F 00 00 F8" +
                    "79 02 00 93 10 00 00 16 7B 02 00 B1 11 00 00 06" +
                    "7C 02 00 A1 12 00 00 FA 7C 02 00 95 13 00 00 2A" +
                    "7E 02 00 C5 14 00 00 5C 7F 02 00 F7 15 00 00 46" +
                    "80 02 00 E1 16 00 00 54 81 02 00 EF 17 00 00 70" +
                    "82 02 00 0B 19 00 00 68 83 02 00 03 1A 00 00 40" +
                    "84 02 00 DB 1A 00 00 16 85 02 00 B1 1B 00 00 28" +
                    "86 02 00 C3 1C 00 00 DE 86 02 00 79 1D 00 00 08" +
                    "88 02 00 A3 1E 00 00 CE 88 02 00 69 1F 00 00 F2" +
                    "89 02 00 7A 00 00 00 C6 8A 02 00 4E 01 00 00 D8" +
                    "8B 02 00 60 02 00 00 B4 8C 02 00 3C 03 00 00 A2" +
                    "8D 02 00 2A 04 00 00 EE 8E 02 00 76 05 00 00 06" +
                    "90 02 00 8E 06 00 00 12 91 02 00 9A 07 00 00 F8" +
                    "91 02 00 80 08 00 00 20 93 02 00 A8 09 00 00 0A" +
                    "95 02 00 92 0B 00 00 FE 95 02 00 86 0C 00 00 C6" +
                    "96 02 00 4E 0D 00 00 92 97 02 00 1A 0E 00 00 36" +
                    "98 02 00 BE 0E 00 00 02 99 02 00 8A 0F 00 00 28" +
                    "9A 02 00 B0 10 00 00 5A 9B 02 00 E2 11 00 00 74" +
                    "9C 02 00 FC 12 00 00 6C 9D 02 00 F4 13 00 00 42" +
                    "9E 02 00 CA 14 00 00 90 9F 02 00 18 16 00 00 34" +
                    "A1 02 00 BC 17 00 00 1E A2 02 00 A6 18 00 00 42" +
                    "A3 02 00 CA 19 00 00 62 A4 02 00 EA 1A 00 00 74" +
                    "A5 02 00 FC 1B 00 00 B0 A6 02 00 38 1D 00 00 5A" +
                    "A8 02 00 E2 1E 00 00 86 A9 02 00 04 00 00 00 30" +
                    "AB 02 00 AE 01 00 00 DA AC 02 00 58 03 00 00 C4" +
                    "AD 02 00 42 04 00 00 8C AE 02 00 0A 05 00 00 C4" +
                    "AF 02 00 42 06 00 00 86 B1 02 00 04 08 00 00 54" +
                    "B2 02 00 D2 08 00 00 8A B3 02 00 08 0A 00 00 54" +
                    "B4 02 00 D2 0A 00 00 52 B5 02 00 D0 0B 00 00 58" +
                    "B6 02 00 D6 0C 00 00 68 B7 02 00 E6 0D 00 00 3E" +
                    "B8 02 00 BC 0E 00 00 2C B9 02 00 AA 0F 00 00 78" +
                    "BA 02 00 F6 10 00 00 AA BB 02 00 28 12 00 00 D6" +
                    "BC 02 00 54 13 00 00 CC BD 02 00 4A 14 00 00 F8" +
                    "BE 02 00 76 15 00 00 42 C0 02 00 C0 16 00 00 5C" +
                    "C1 02 00 DA 17 00 00 42 C2 02 00 C0 18 00 00 36" +
                    "C3 02 00 B4 19 00 00 78 C4 02 00 F6 1A 00 00 90" +
                    "C5 02 00 0E 1C 00 00 BA C6 02 00 38 1D 00 00 1A" +
                    "C8 02 00 98 1E 00 00 62 C9 02 00 E0 1F 00 00 0E" +
                    "CA 02 00 77 00 00 00 24 CB 02 00 8D 01 00 00 38" +
                    "CC 02 00 A1 02 00 00 A6 CD 02 00 0F 04 00 00 84" +
                    "CE 02 00 ED 04 00 00 50 CF 02 00 B9 05 00 00 2C" +
                    "D0 02 00 95 06 00 00 3A D1 02 00 A3 07 00 00 22" +
                    "D3 02 00 8B 09 00 00 14 D4 02 00 7D 0A 00 00 3E" +
                    "D5 02 00 A7 0B 00 00 68 D6 02 00 D1 0C 00 00 62" +
                    "D7 02 00 CB 0D 00 00 4C D8 02 00 B5 0E 00 00 6A" +
                    "D9 02 00 D3 0F 00 00 92 DA 02 00 FB 10 00 00 CC" +
                    "DB 02 00 35 12 00 00 FC DC 02 00 65 13 00 00 A6" +
                    "DE 02 00 0F 15 00 00 B2 DF 02 00 1B 16 00 00 9E" +
                    "E0 02 00 07 17 00 00 A6 E1 02 00 0F 18 00 00 AA" +
                    "E2 02 00 13 19 00 00 86 E3 02 00 EF 19 00 00 C8" +
                    "E4 02 00 31 1B 00 00 F0 E5 02 00 59 1C 00 00 C6" +
                    "E6 02 00 2F 1D 00 00 BA E7 02 00 23 1E 00 00 EC" +
                    "E8 02 00 55 1F 00 00 02 EA 02 00 6C 00 00 00 64" +
                    "EB 02 00 CE 01 00 00 F4 EC 02 00 5E 03 00 00 FE" +
                    "ED 02 00 68 04 00 00 00 EF 02 00 6A 05 00 00 F6" +
                    "EF 02 00 60 06 00 00 FA F0 02 00 64 07 00 00 CE" +
                    "F1 02 00 38 08 00 00 2E F3 02 00 98 09 00 00 36" +
                    "F4 02 00 A0 0A 00 00 46 F5 02 00 B0 0B 00 00 50" +
                    "F6 02 00 BA 0C 00 00 60 F7 02 00 CA 0D 00 00 7C" +
                    "F8 02 00 E6 0E 00 00 CC F9 02 00 36 10 00 00 DE" +
                    "FA 02 00 48 11 00 00 AA FB 02 00 14 12 00 00 2E" +
                    "FD 02 00 98 13 00 00 54 FE 02 00 BE 14 00 00 6E" +
                    "FF 02 00 D8 15 00 00 88 00 03 00 F2 16 00 00 D6" +
                    "01 03 00 40 18 00 00 4A 03 03 00 B4 19 00 00 80" +
                    "04 03 00 EA 1A 00 00 90 05 03 00 FA 1B 00 00 96" +
                    "06 03 00 00 1D 00 00 A4 07 03 00 0E 1E 00 00 12" +
                    "09 03 00 7C 1F 00 00 20 0A 03 00 8B 00 00 00 8E" +
                    "0B 03 00 F9 01 00 00 B6 0C 03 00 21 03 00 00 D6" +
                    "0D 03 00 41 04 00 00 D6 0E 03 00 41 05 00 00 28" +
                    "10 03 00 93 06 00 00 36 11 03 00 A1 07 00 00 84" +
                    "12 03 00 EF 08 00 00 F2 13 03 00 5D 0A 00 00 44" +
                    "15 03 00 AF 0B 00 00 F6 16 03 00 61 0D 00 00 26" +
                    "18 03 00 91 0E 00 00 B8 19 03 00 23 10 00 00 0A" +
                    "1B 03 00 75 11 00 00 5E 1C 03 00 C9 12 00 00 A4" +
                    "1D 03 00 0F 14 00 00 CA 1E 03 00 35 15 00 00 3A" +
                    "20 03 00 A5 16 00 00 C2 21 03 00 2D 18 00 00 A8" +
                    "22 03 00 13 19 00 00 DA 23 03 00 45 1A 00 00 3A" +
                    "25 03 00 A5 1B 00 00 2C 26 03 00 97 1C 00 00 96" +
                    "27 03 00 01 1E 00 00 A2 28 03 00 0D 1F 00 00 B0" +
                    "29 03 00 04 00 00 00 78 2B 03 00 CC 01 00 00 44" +
                    "2C 03 00 98 02 00 00 74 2D 03 00 C8 03 00 00 00" +
                    "2F 03 00 54 05 00 00 18 30 03 00 6C 06 00 00 6C" +
                    "31 03 00 C0 07 00 00 E8 32 03 00 3C 09 00 00 E6" +
                    "33 03 00 3A 0A 00 00 5E 35 03 00 B2 0B 00 00 02" +
                    "37 03 00 56 0D 00 00 56 38 03 00 AA 0E 00 00 82" +
                    "39 03 00 D6 0F 00 00 C8 3A 03 00 1C 11 00 00 08" +
                    "3C 03 00 5C 12 00 00 72 3D 03 00 C6 13 00 00 C2" +
                    "3E 03 00 16 15 00 00 28 40 03 00 7C 16 00 00 AE" +
                    "41 03 00 02 18 00 00 38 43 03 00 8C 19 00 00 7A" +
                    "44 03 00 CE 1A 00 00 C4 45 03 00 18 1C 00 00 1E" +
                    "47 03 00 72 1D 00 00 A2 48 03 00 F6 1E 00 00 1E" +
                    "4A 03 00 51 00 00 00 7A 4B 03 00 AD 01 00 00 F8" +
                    "4C 03 00 2B 03 00 00 7C 4E 03 00 AF 04 00 00 6C" +
                    "50 03 00 9F 06 00 00 B8 51 03 00 EB 07 00 00 30" +
                    "53 03 00 63 09 00 00 2A 54 03 00 5D 0A 00 00 5C" +
                    "55 03 00 8F 0B 00 00 D4 56 03 00 07 0D 00 00 DE" +
                    "57 03 00 11 0E 00 00 E8 58 03 00 1B 0F 00 00 B0" +
                    "59 03 00 E3 0F 00 00 F4 5A 03 00 27 11 00 00 68" +
                    "5C 03 00 9B 12 00 00 F6 5D 03 00 29 14 00 00 F4" +
                    "5E 03 00 27 15 00 00 64 60 03 00 97 16 00 00 9E" +
                    "61 03 00 D1 17 00 00 CA 62 03 00 FD 18 00 00 3C" +
                    "64 03 00 6F 1A 00 00 96 65 03 00 C9 1B 00 00 FC" +
                    "66 03 00 2F 1D 00 00 40 68 03 00 73 1E 00 00 AE" +
                    "69 03 00 E1 1F 00 00 3E 6B 03 00 73 01 00 00 66" +
                    "6C 03 00 9B 02 00 00 52 6D 03 00 87 03 00 00 24" +
                    "6F 03 00 59 05 00 00 AA 70 03 00 DF 06 00 00 E6" +
                    "71 03 00 1B 08 00 00 82 73 03 00 B7 09 00 00 00" +
                    "75 03 00 35 0B 00 00 9E 76 03 00 D3 0C 00 00 20" +
                    "78 03 00 55 0E 00 00 0E 79 03 00 43 0F 00 00 10" +
                    "7A 03 00 45 10 00 00 02 7B 03 00 37 11 00 00 3C" +
                    "7C 03 00 71 12 00 00 8E 7D 03 00 C3 13 00 00 F4" +
                    "7E 03 00 29 15 00 00 96 80 03 00 CB 16 00 00 44" +
                    "82 03 00 79 18 00 00 76 84 03 00 AB 1A 00 00 92" +
                    "86 03 00 C7 1C 00 00 2D 88 03 00 62 1E 00 00 19" +
                    "8A 03 00 88 00 00 00 11 8B 03 00 80 01 00 00 BB" +
                    "8C 03 00 2A 03 00 00 CB 8D 03 00 3A 04 00 00 CB" +
                    "8E 03 00 3A 05 00 00 CF 8F 03 00 3E 06 00 00 C5" +
                    "90 03 00 34 07 00 00 A7 91 03 00 16 08 00 00 C9" +
                    "92 03 00 38 09 00 00 C5 93 03 00 34 0A 00 00 CD" +
                    "94 03 00 3C 0B 00 00 C5 96 03 00 34 0D 00 00 BB" +
                    "98 03 00 2A 0F 00 00 A9 99 03 00 18 10 00 00 C3" +
                    "9A 03 00 32 11 00 00 53 9C 03 00 C2 12 00 00 51" +
                    "9D 03 00 C0 13 00 00 AF 9E 03 00 1E 15 00 00 ED" +
                    "9F 03 00 5C 16 00 00 53 A1 03 00 C2 17 00 00 17" +
                    "A3 03 00 86 19 00 00 55 A4 03 00 C4 1A 00 00 6B" +
                    "A5 03 00 DA 1B 00 00 5F A6 03 00 CE 1C 00 00 5F" +
                    "A7 03 00 CE 1D 00 00 9F A8 03 00 0E 1F 00 00 51" +
                    "AA 03 00 9E 00 00 00 1F AC 03 00 6C 02 00 00 17" +
                    "AE 03 00 64 04 00 00 B7 B0 03 00 04 07 00 00 05" +
                    "B3 03 00 52 09 00 00 97 B4 03 00 E4 0A 00 00 03" +
                    "B6 03 00 50 0C 00 00 7F B7 03 00 CC 0D 00 00 0B" +
                    "B9 03 00 58 0F 00 00 41 BB 03 00 8E 11 00 00 23" +
                    "BD 03 00 70 13 00 00 1D BF 03 00 6A 15 00 00 47" +
                    "C0 03 00 94 16 00 00 FB C1 03 00 48 18 00 00 45" +
                    "C3 03 00 92 19 00 00 7F C4 03 00 CC 1A 00 00 19" +
                    "C6 03 00 66 1C 00 00 55 C7 03 00 A2 1D 00 00 9D" +
                    "C8 03 00 EA 1E 00 00 FB C9 03 00 58 00 00 00 89" +
                    "CB 03 00 E6 01 00 00 2B CD 03 00 88 03 00 00 8D" +
                    "CE 03 00 EA 04 00 00 F9 CF 03 00 56 06 00 00 6D" +
                    "D1 03 00 CA 07 00 00 EF D3 03 00 4C 0A 00 00 1D" +
                    "D5 03 00 7A 0B 00 00 C9 D6 03 00 26 0D 00 00 C3" +
                    "D8 03 00 20 0F 00 00 1D DA 03 00 7A 10 00 00 07" +
                    "DC 03 00 64 12 00 00 61 DD 03 00 BE 13 00 00 E3" +
                    "DE 03 00 40 15 00 00 29 E0 03 00 86 16 00 00 5B" +
                    "E2 03 00 B8 18 00 00 B7 E4 03 00 14 1B 00 00 A7" +
                    "E6 03 00 04 1D 00 00 E5 E8 03 00 42 1F 00 00 AF" +
                    "EA 03 00 3D 01 00 00 AD EC 03 00 3B 03 00 00 C1" +
                    "EE 03 00 4F 05 00 00 C7 F0 03 00 55 07 00 00 0B" +
                    "F3 03 00 99 09 00 00 E9 F4 03 00 77 0B 00 00 ED" +
                    "F6 03 00 7B 0D 00 00 53 F9 03 00 E1 0F 00 00 5B" +
                    "FB 03 00 E9 11 00 00 5F FD 03 00 ED 13 00 00 EF" +
                    "FF 03 00 7D 16 00 00 AF 01 04 00 3D 18 00 00 B9" +
                    "02 04 00 47 19 00 00 AD 04 04 00 3B 1B 00 00 51" +
                    "06 04 00 DF 1C 00 00 6F 07 04 00 FD 1D 00 00 69" +
                    "08 04 00 F7 1E 00 00 E1 09 04 00 52 00 00 00 81" +
                    "0B 04 00 F2 01 00 00 51 0C 04 00 C2 02 00 00 39" +
                    "0D 04 00 AA 03 00 00 5D 0E 04 00 CE 04 00 00 BB" +
                    "0F 04 00 2C 06 00 00 91 10 04 00 02 07 00 00 1D" +
                    "12 04 00 8E 08 00 00 2D 13 04 00 9E 09 00 00 7B" +
                    "14 04 00 EC 0A 00 00 1D 16 04 00 8E 0C 00 00 8B" +
                    "17 04 00 FC 0D 00 00 B5 18 04 00 26 0F 00 00 6D" +
                    "1A 04 00 DE 10 00 00 8B 1B 04 00 FC 11 00 00 AB" +
                    "1C 04 00 1C 13 00 00 E7 1D 04 00 58 14 00 00 37" +
                    "1F 04 00 A8 15 00 00 69 20 04 00 DA 16 00 00 83" +
                    "21 04 00 F4 17 00 00 8D 22 04 00 FE 18 00 00 E7" +
                    "23 04 00 58 1A 00 00 B7 25 04 00 28 1C 00 00 29" +
                    "28 04 00 9A 1E 00 00 97 29 04 00 04 00 00 00 DB" +
                    "2A 04 00 48 01 00 00 EF 2B 04 00 5C 02 00 00 5B" +
                    "2D 04 00 C8 03 00 00 71 2E 04 00 DE 04 00 00 BF" +
                    "2F 04 00 2C 06 00 00 9D 31 04 00 0A 08 00 00 DD" +
                    "32 04 00 4A 09 00 00 FB 33 04 00 68 0A 00 00 11" +
                    "36 04 00 7E 0C 00 00 4D 37 04 00 BA 0D 00 00 01" +
                    "39 04 00 6E 0F 00 00 67 3A 04 00 D4 10 00 00 0D" +
                    "3C 04 00 7A 12 00 00 A1 3D 04 00 0E 14 00 00 F3" +
                    "3E 04 00 60 15 00 00 87 40 04 00 F4 16 00 00 83" +
                    "42 04 00 F0 18 00 00 F7 43 04 00 64 1A 00 00 F7" +
                    "45 04 00 64 1C 00 00 FB 47 04 00 68 1E 00 00 3D" +
                    "49 04 00 AA 1F 00 00 39 4B 04 00 92 01 00 00 8B" +
                    "4C 04 00 E4 02 00 00 97 4E 04 00 F0 04 00 00 5D" +
                    "50 04 00 B6 06 00 00 19 52 04 00 72 08 00 00 43" +
                    "53 04 00 9C 09 00 00 FD 54 04 00 56 0B 00 00 FD" +
                    "55 04 00 56 0C 00 00 4B 57 04 00 A4 0D 00 00 83" +
                    "58 04 00 DC 0E 00 00 E9 59 04 00 42 10 00 00 47" +
                    "5B 04 00 A0 11 00 00 B9 5C 04 00 12 13 00 00 5D" +
                    "5E 04 00 B6 14 00 00 DB 5F 04 00 34 16 00 00 F1" +
                    "61 04 00 4A 18 00 00 35 64 04 00 8E 1A 00 00 87" +
                    "66 04 00 E0 1C 00 00 B9 68 04 00 12 1F 00 00 AD" +
                    "6A 04 00 FA 00 00 00 EB 6C 04 00 38 03 00 00 0B" +
                    "6F 04 00 58 05 00 00 49 71 04 00 96 07 00 00 11" +
                    "73 04 00 5E 09 00 00 A5 75 04 00 F2 0B 00 00 89" +
                    "77 04 00 D6 0D 00 00 C9 79 04 00 16 10 00 00 C5" +
                    "7A 04 00 12 11 00 00 15 7C 04 00 62 12 00 00 7F" +
                    "7D 04 00 CC 13 00 00 83 7F 04 00 D0 15 00 00 AB" +
                    "81 04 00 F8 17 00 00 2D 83 04 00 7A 19 00 00 7B" +
                    "84 04 00 C8 1A 00 00 CD 85 04 00 1A 1C 00 00 DF" +
                    "87 04 00 2C 1E 00 00 65 89 04 00 B2 1F 00 00 BF" +
                    "8B 04 00 0B 02 00 00 25 8E 04 00 71 04 00 00 F5" +
                    "8F 04 00 41 06 00 00 79 91 04 00 C5 07 00 00 19" +
                    "93 04 00 65 09 00 00 5D 94 04 00 A9 0A 00 00 4B" +
                    "96 04 00 97 0C 00 00 B7 97 04 00 03 0E 00 00 37" +
                    "99 04 00 83 0F 00 00 EB 9A 04 00 37 11 00 00 E3" +
                    "9B 04 00 2F 12 00 00 BD 9D 04 00 09 14 00 00 C1" +
                    "9F 04 00 0D 16 00 00 AB A1 04 00 F7 17 00 00 EB" +
                    "A2 04 00 37 19 00 00 DF A3 04 00 2B 1A 00 00 CB" +
                    "A4 04 00 17 1B 00 00 93 A5 04 00 DF 1B 00 00 6F" +
                    "A6 04 00 BB 1C 00 00 59 A7 04 00 A5 1D 00 00 5B" +
                    "A8 04 00 A7 1E 00 00 47 A9 04 00 93 1F 00 00 4B" +
                    "AA 04 00 AE 00 00 00 49 AB 04 00 AC 01 00 00 53" +
                    "AC 04 00 B6 02 00 00 7B AD 04 00 DE 03 00 00 67" +
                    "AE 04 00 CA 04 00 00 73 AF 04 00 D6 05 00 00 B3" +
                    "B0 04 00 16 07 00 00 A7 B1 04 00 0A 08 00 00 7B" +
                    "B2 04 00 DE 08 00 00 C7 B3 04 00 2A 0A 00 00 B3" +
                    "B4 04 00 16 0B 00 00 EB B5 04 00 4E 0C 00 00 09" +
                    "B7 04 00 6C 0D 00 00 11 B8 04 00 74 0E 00 00 37" +
                    "B9 04 00 9A 0F 00 00 41 BA 04 00 A4 10 00 00 A7" +
                    "BB 04 00 0A 12 00 00 87 BC 04 00 EA 12 00 00 9D" +
                    "BD 04 00 00 14 00 00 DF BE 04 00 42 15 00 00 C5" +
                    "BF 04 00 28 16 00 00 A9 C0 04 00 0C 17 00 00 A1" +
                    "C1 04 00 04 18 00 00 9B C2 04 00 FE 18 00 00 DD" +
                    "C3 04 00 40 1A 00 00 AF C4 04 00 12 1B 00 00 D1" +
                    "C5 04 00 34 1C 00 00 F5 C6 04 00 58 1D 00 00 E5" +
                    "C7 04 00 48 1E 00 00 2F C9 04 00 92 1F 00 00 11" +
                    "CA 04 00 63 00 00 00 E5 CA 04 00 37 01 00 00 C3" +
                    "CB 04 00 15 02 00 00 A9 CC 04 00 FB 02 00 00 B1" +
                    "CD 04 00 03 04 00 00 A1 CE 04 00 F3 04 00 00 8B" +
                    "CF 04 00 DD 05 00 00 5D D0 04 00 AF 06 00 00 33" +
                    "D1 04 00 85 07 00 00 39 D2 04 00 8B 08 00 00 27" +
                    "D3 04 00 79 09 00 00 21 D4 04 00 73 0A 00 00 EF" +
                    "D4 04 00 41 0B 00 00 C3 D5 04 00 15 0C 00 00 7D" +
                    "D6 04 00 CF 0C 00 00 3F D7 04 00 91 0D 00 00 5F" +
                    "D8 04 00 B1 0E 00 00 3D D9 04 00 8F 0F 00 00 0D" +
                    "DA 04 00 5F 10 00 00 F7 DA 04 00 49 11 00 00 31" +
                    "DC 04 00 83 12 00 00 B7 DD 04 00 09 14 00 00 ED" +
                    "DE 04 00 3F 15 00 00 EF DF 04 00 41 16 00 00 D5" +
                    "E0 04 00 27 17 00 00 AF E1 04 00 01 18 00 00 C7" +
                    "E2 04 00 19 19 00 00 B7 E3 04 00 09 1A 00 00 A9" +
                    "E4 04 00 FB 1A 00 00 B7 E5 04 00 09 1C 00 00 C1" +
                    "E6 04 00 13 1D 00 00 A9 E7 04 00 FB 1D 00 00 AB" +
                    "E8 04 00 FD 1E 00 00 93 E9 04 00 E5 1F 00 00 75" +
                    "EA 04 00 CF 00 00 00 3F EB 04 00 99 01 00 00 63" +
                    "EC 04 00 BD 02 00 00 67 ED 04 00 C1 03 00 00 83" +
                    "EE 04 00 DD 04 00 00 8F EF 04 00 E9 05 00 00 79" +
                    "F0 04 00 D3 06 00 00 4F F1 04 00 A9 07 00 00 3F" +
                    "F2 04 00 99 08 00 00 41 F3 04 00 9B 09 00 00 11" +
                    "F4 04 00 6B 0A 00 00 27 F5 04 00 81 0B 00 00 21" +
                    "F6 04 00 7B 0C 00 00 23 F7 04 00 7D 0D 00 00 5F" +
                    "F8 04 00 B9 0E 00 00 2D F9 04 00 87 0F 00 00 2F" +
                    "FA 04 00 89 10 00 00 2B FB 04 00 85 11 00 00 4D" +
                    "FC 04 00 A7 12 00 00 2B FD 04 00 85 13 00 00 2F" +
                    "FE 04 00 89 14 00 00 4D FF 04 00 A7 15 00 00 59" +
                    "00 05 00 B3 16 00 00 7D 01 05 00 D7 17 00 00 AD" +
                    "02 05 00 07 19 00 00 B7 03 05 00 11 1A 00 00 47" +
                    "05 05 00 A1 1B 00 00 7D 06 05 00 D7 1C 00 00 0F" +
                    "08 05 00 69 1E 00 00 5D 09 05 00 B7 1F 00 00 9B" +
                    "0A 05 00 DE 00 00 00 A1 0B 05 00 E4 01 00 00 91" +
                    "0C 05 00 D4 02 00 00 09 0E 05 00 4C 04 00 00 A7" +
                    "0F 05 00 EA 05 00 00 FD 10 05 00 40 07 00 00 93" +
                    "12 05 00 D6 08 00 00 35 14 05 00 78 0A 00 00 63" +
                    "15 05 00 A6 0B 00 00 97 17 05 00 DA 0D 00 00 B1" +
                    "19 05 00 F4 0F 00 00 2B 1C 05 00 6E 12 00 00 69" +
                    "1D 05 00 AC 13 00 00 8D 1E 05 00 D0 14 00 00 D1" +
                    "1F 05 00 14 16 00 00 FF 20 05 00 42 17 00 00 93" +
                    "22 05 00 D6 18 00 00 F1 23 05 00 34 1A 00 00 55" +
                    "25 05 00 98 1B 00 00 83 26 05 00 C6 1C 00 00 BF" +
                    "27 05 00 02 1E 00 00 13 29 05 00 56 1F 00 00 7D" +
                    "2A 05 00 B5 00 00 00 F3 2B 05 00 2B 02 00 00 79" +
                    "2D 05 00 B1 03 00 00 D7 2E 05 00 0F 05 00 00 11" +
                    "30 05 00 49 06 00 00 19 32 05 00 51 08 00 00 2B" +
                    "33 05 00 63 09 00 00 33 34 05 00 6B 0A 00 00 AF" +
                    "35 05 00 E7 0B 00 00 8F 36 05 00 C7 0C 00 00 01" +
                    "38 05 00 39 0E 00 00 49 39 05 00 81 0F 00 00 AD" +
                    "3A 05 00 E5 10 00 00 29 3C 05 00 61 12 00 00 0F" +
                    "3E 05 00 47 14 00 00 99 3F 05 00 D1 15 00 00 79" +
                    "41 05 00 B1 17 00 00 E7 43 05 00 1F 1A 00 00 D1" +
                    "46 05 00 09 1D 00 00 29 49 05 00 61 1F 00 00 CF" +
                    "4B 05 00 EE 01 00 00 F9 4D 05 00 18 04 00 00 E5" +
                    "50 05 00 04 07 00 00 51 53 05 00 70 09 00 00 C1" +
                    "55 05 00 E0 0B 00 00 5F 58 05 00 7E 0E 00 00 9D" +
                    "59 05 00 BC 0F 00 00 8F 5A 05 00 AE 10 00 00 9D" +
                    "5C 05 00 BC 12 00 00 CB 5D 05 00 EA 13 00 00 8F" +
                    "5F 05 00 AE 15 00 00 B1 60 05 00 D0 16 00 00 FB" +
                    "61 05 00 1A 18 00 00 29 63 05 00 48 19 00 00 BD" +
                    "64 05 00 DC 1A 00 00 67 66 05 00 86 1C 00 00 21" +
                    "68 05 00 40 1E 00 00 83 69 05 00 A2 1F 00 00 F9" +
                    "6A 05 00 0B 01 00 00 EB 6C 05 00 FD 02 00 00 B5" +
                    "6E 05 00 C7 04 00 00 19 70 05 00 2B 06 00 00 11" +
                    "72 05 00 23 08 00 00 53 73 05 00 65 09 00 00 99" +
                    "74 05 00 AB 0A 00 00 23 76 05 00 35 0C 00 00 13" +
                    "78 05 00 25 0E 00 00 91 79 05 00 A3 0F 00 00 E3" +
                    "7A 05 00 F5 10 00 00 B7 7B 05 00 C9 11 00 00 B7" +
                    "7C 05 00 C9 12 00 00 B7 7D 05 00 C9 13 00 00 81" +
                    "7F 05 00 93 15 00 00 E7 80 05 00 F9 16 00 00 67" +
                    "82 05 00 79 18 00 00 6B 83 05 00 7D 19 00 00 8B" +
                    "84 05 00 9D 1A 00 00 71 86 05 00 83 1C 00 00 F5" +
                    "87 05 00 07 1E 00 00                           " +

                    //Offset=0x0008DB37(580407) recno=160 sid=0x003C size=0x1048(4168)
                    "                     3C 00 48 10 47 8A 05 00 4E" +
                    "00 00 00 45 8B 05 00 4C 01 00 00 59 8C 05 00 60" +
                    "02 00 00 91 8D 05 00 98 03 00 00 77 8F 05 00 7E" +
                    "05 00 00 B9 90 05 00 C0 06 00 00 D3 91 05 00 DA" +
                    "07 00 00 75 93 05 00 7C 09 00 00 2F 95 05 00 36" +
                    "0B 00 00 F1 96 05 00 F8 0C 00 00 DB 98 05 00 E2" +
                    "0E 00 00 3D 9A 05 00 44 10 00 00 41 9C 05 00 48" +
                    "12 00 00 A1 9D 05 00 A8 13 00 00 F7 9E 05 00 FE" +
                    "14 00 00 71 A0 05 00 78 16 00 00 11 A2 05 00 18" +
                    "18 00 00 0F A3 05 00 16 19 00 00 69 A4 05 00 70" +
                    "1A 00 00 89 A6 05 00 90 1C 00 00 65 A8 05 00 6C" +
                    "1E 00 00 E3 AA 05 00 CF 00 00 00 AF AC 05 00 9B" +
                    "02 00 00 91 AE 05 00 7D 04 00 00 B7 B0 05 00 A3" +
                    "06 00 00 85 B2 05 00 71 08 00 00 D9 B3 05 00 C5" +
                    "09 00 00 4F B5 05 00 3B 0B 00 00 BD B6 05 00 A9" +
                    "0C 00 00 AF B8 05 00 9B 0E 00 00 63 BA 05 00 4F" +
                    "10 00 00 35 BC 05 00 21 12 00 00 5B BD 05 00 47" +
                    "13 00 00 53 BE 05 00 3F 14 00 00 9F BF 05 00 8B" +
                    "15 00 00 17 C2 05 00 03 18 00 00 CB C3 05 00 B7" +
                    "19 00 00 FB C4 05 00 E7 1A 00 00 89 C6 05 00 75" +
                    "1C 00 00 67 C7 05 00 53 1D 00 00 D7 C8 05 00 C3" +
                    "1E 00 00 0B CA 05 00 F7 1F 00 00 0D CB 05 00 EB" +
                    "00 00 00 F7 CB 05 00 D5 01 00 00 03 CD 05 00 E1" +
                    "02 00 00 13 CE 05 00 F1 03 00 00 33 CF 05 00 11" +
                    "05 00 00 51 D0 05 00 2F 06 00 00 49 D1 05 00 27" +
                    "07 00 00 43 D2 05 00 21 08 00 00 63 D4 05 00 41" +
                    "0A 00 00 93 D5 05 00 71 0B 00 00 C9 D6 05 00 A7" +
                    "0C 00 00 FF D7 05 00 DD 0D 00 00 73 D9 05 00 51" +
                    "0F 00 00 27 DB 05 00 05 11 00 00 8B DC 05 00 69" +
                    "12 00 00 C3 DD 05 00 A1 13 00 00 3F DF 05 00 1D" +
                    "15 00 00 E5 E0 05 00 C3 16 00 00 FF E1 05 00 DD" +
                    "17 00 00 29 E3 05 00 07 19 00 00 3F E4 05 00 1D" +
                    "1A 00 00 15 E5 05 00 F3 1A 00 00 F9 E5 05 00 D7" +
                    "1B 00 00 C5 E6 05 00 A3 1C 00 00 C9 E7 05 00 A7" +
                    "1D 00 00 27 E9 05 00 05 1F 00 00 A1 EA 05 00 73" +
                    "00 00 00 6B EC 05 00 3D 02 00 00 A3 ED 05 00 75" +
                    "03 00 00 B5 EE 05 00 87 04 00 00 E9 EF 05 00 BB" +
                    "05 00 00 77 F1 05 00 49 07 00 00 B9 F2 05 00 8B" +
                    "08 00 00 C5 F4 05 00 97 0A 00 00 0D F6 05 00 DF" +
                    "0B 00 00 C3 F7 05 00 95 0D 00 00 E5 F9 05 00 B7" +
                    "0F 00 00 B5 FB 05 00 87 11 00 00 6B FE 05 00 3D" +
                    "14 00 00 DD 00 06 00 AF 16 00 00 0B 03 06 00 DD" +
                    "18 00 00 9F 05 06 00 71 1B 00 00 7B 08 06 00 4D" +
                    "1E 00 00 A7 0A 06 00 62 00 00 00 59 0D 06 00 14" +
                    "03 00 00 3D 0F 06 00 F8 04 00 00 9B 10 06 00 56" +
                    "06 00 00 17 12 06 00 D2 07 00 00 6F 13 06 00 2A" +
                    "09 00 00 59 14 06 00 14 0A 00 00 1F 16 06 00 DA" +
                    "0B 00 00 49 17 06 00 04 0D 00 00 D1 18 06 00 8C" +
                    "0E 00 00 C1 1A 06 00 7C 10 00 00 2B 1C 06 00 E6" +
                    "11 00 00 9F 1D 06 00 5A 13 00 00 77 1F 06 00 32" +
                    "15 00 00 CD 20 06 00 88 16 00 00 B7 21 06 00 72" +
                    "17 00 00 9D 22 06 00 58 18 00 00 61 23 06 00 1C" +
                    "19 00 00 33 24 06 00 EE 19 00 00 67 25 06 00 22" +
                    "1B 00 00 4F 26 06 00 0A 1C 00 00 3F 27 06 00 FA" +
                    "1C 00 00 17 28 06 00 D2 1D 00 00 25 29 06 00 E0" +
                    "1E 00 00 19 2A 06 00 D4 1F 00 00 DF 2A 06 00 96" +
                    "00 00 00 CF 2B 06 00 86 01 00 00 9D 2C 06 00 54" +
                    "02 00 00 CD 2D 06 00 84 03 00 00 9F 2E 06 00 56" +
                    "04 00 00 7F 2F 06 00 36 05 00 00 49 30 06 00 00" +
                    "06 00 00 21 31 06 00 D8 06 00 00 17 32 06 00 CE" +
                    "07 00 00 05 33 06 00 BC 08 00 00 FF 33 06 00 B6" +
                    "09 00 00 DD 34 06 00 94 0A 00 00 B1 35 06 00 68" +
                    "0B 00 00 99 36 06 00 50 0C 00 00 73 37 06 00 2A" +
                    "0D 00 00 95 38 06 00 4C 0E 00 00 8D 39 06 00 44" +
                    "0F 00 00 A9 3A 06 00 60 10 00 00 F3 3B 06 00 AA" +
                    "11 00 00 19 3D 06 00 D0 12 00 00 2D 3E 06 00 E4" +
                    "13 00 00 49 3F 06 00 00 15 00 00 FF 40 06 00 B6" +
                    "16 00 00 9D 42 06 00 54 18 00 00 4D 44 06 00 04" +
                    "1A 00 00 9B 46 06 00 52 1C 00 00 BF 48 06 00 76" +
                    "1E 00 00 19 4B 06 00 D9 00 00 00 DB 4D 06 00 9B" +
                    "03 00 00 53 4F 06 00 13 05 00 00 EB 50 06 00 AB" +
                    "06 00 00 25 52 06 00 E5 07 00 00 9B 53 06 00 5B" +
                    "09 00 00 03 55 06 00 C3 0A 00 00 85 56 06 00 45" +
                    "0C 00 00 E9 58 06 00 A9 0E 00 00 EF 59 06 00 AF" +
                    "0F 00 00 61 5B 06 00 21 11 00 00 2B 5D 06 00 EB" +
                    "12 00 00 57 5F 06 00 17 15 00 00 D1 60 06 00 91" +
                    "16 00 00 15 63 06 00 D5 18 00 00 97 65 06 00 57" +
                    "1B 00 00 D3 67 06 00 93 1D 00 00 31 69 06 00 F1" +
                    "1E 00 00 47 6A 06 00 07 20 00 00 FB 6B 06 00 97" +
                    "01 00 00 43 6D 06 00 DF 02 00 00 7B 6E 06 00 17" +
                    "04 00 00 39 70 06 00 D5 05 00 00 BD 71 06 00 59" +
                    "07 00 00 6B 73 06 00 07 09 00 00 95 74 06 00 31" +
                    "0A 00 00 59 76 06 00 F5 0B 00 00 C7 77 06 00 63" +
                    "0D 00 00 BF 79 06 00 5B 0F 00 00 AD 7B 06 00 49" +
                    "11 00 00 9B 7D 06 00 37 13 00 00 FB 7E 06 00 97" +
                    "14 00 00 2D 81 06 00 C9 16 00 00 5B 82 06 00 F7" +
                    "17 00 00 D5 83 06 00 71 19 00 00 B1 85 06 00 4D" +
                    "1B 00 00 73 87 06 00 0F 1D 00 00 29 89 06 00 C5" +
                    "1E 00 00 AF 8A 06 00 34 00 00 00 C9 8B 06 00 4E" +
                    "01 00 00 39 8D 06 00 BE 02 00 00 B1 8E 06 00 36" +
                    "04 00 00 01 90 06 00 86 05 00 00 6D 91 06 00 F2" +
                    "06 00 00 C7 92 06 00 4C 08 00 00 21 94 06 00 A6" +
                    "09 00 00 51 95 06 00 D6 0A 00 00 61 96 06 00 E6" +
                    "0B 00 00 F1 97 06 00 76 0D 00 00 F5 98 06 00 7A" +
                    "0E 00 00 37 9A 06 00 BC 0F 00 00 79 9B 06 00 FE" +
                    "10 00 00 09 9D 06 00 8E 12 00 00 DD 9D 06 00 62" +
                    "13 00 00 1F 9F 06 00 A4 14 00 00 29 A0 06 00 AE" +
                    "15 00 00 3B A1 06 00 C0 16 00 00 AF A2 06 00 34" +
                    "18 00 00 E3 A3 06 00 68 19 00 00 3D A5 06 00 C2" +
                    "1A 00 00 91 A6 06 00 16 1C 00 00 3F A8 06 00 C4" +
                    "1D 00 00 1B AA 06 00 A0 1F 00 00 1B AB 06 00 8A" +
                    "00 00 00 63 AC 06 00 D2 01 00 00 75 AD 06 00 E4" +
                    "02 00 00 91 AE 06 00 00 04 00 00 E7 AF 06 00 56" +
                    "05 00 00 41 B1 06 00 B0 06 00 00 85 B2 06 00 F4" +
                    "07 00 00 83 B3 06 00 F2 08 00 00 CD B4 06 00 3C" +
                    "0A 00 00 B7 B5 06 00 26 0B 00 00 ED B6 06 00 5C" +
                    "0C 00 00 D9 B7 06 00 48 0D 00 00 B3 B9 06 00 22" +
                    "0F 00 00 63 BB 06 00 D2 10 00 00 E3 BC 06 00 52" +
                    "12 00 00 53 BE 06 00 C2 13 00 00 83 BF 06 00 F2" +
                    "14 00 00 AF C0 06 00 1E 16 00 00 8F C2 06 00 FE" +
                    "17 00 00 0F C4 06 00 7E 19 00 00 DF C5 06 00 4E" +
                    "1B 00 00 77 C7 06 00 E6 1C 00 00 D9 C8 06 00 48" +
                    "1E 00 00 C5 CA 06 00 35 00 00 00 D9 CB 06 00 49" +
                    "01 00 00 29 CD 06 00 99 02 00 00 F3 CE 06 00 63" +
                    "04 00 00 0B D1 06 00 7B 06 00 00 45 D2 06 00 B5" +
                    "07 00 00 59 D4 06 00 C9 09 00 00 87 D6 06 00 F7" +
                    "0B 00 00 DD D7 06 00 4D 0D 00 00 2D DA 06 00 9D" +
                    "0F 00 00 2D DC 06 00 9D 11 00 00 B7 DD 06 00 27" +
                    "13 00 00 09 DF 06 00 79 14 00 00 FB E0 06 00 6B" +
                    "16 00 00 E9 E2 06 00 59 18 00 00 47 E4 06 00 B7" +
                    "19 00 00 21 E6 06 00 91 1B 00 00 03 E8 06 00 73" +
                    "1D 00 00 D1 E9 06 00 41 1F 00 00 45 EC 06 00 95" +
                    "01 00 00 63 EE 06 00 B3 03 00 00 C9 F1 06 00 19" +
                    "07 00 00 BD F4 06 00 0D 0A 00 00 57 F7 06 00 A7" +
                    "0C 00 00 9B FA 06 00 EB 0F 00 00 37 FD 06 00 87" +
                    "12 00 00 55 FF 06 00 A5 14 00 00 57 01 07 00 A7" +
                    "16 00 00 D3 03 07 00 23 19 00 00 BF 05 07 00 0F" +
                    "1B 00 00 1D 07 07 00 6D 1C 00 00 15 08 07 00 65" +
                    "1D 00 00 1F 09 07 00 6F 1E 00 00 39 0B 07 00 8E" +
                    "00 00 00 D5 0C 07 00 2A 02 00 00 21 0E 07 00 76" +
                    "03 00 00 59 10 07 00 AE 05 00 00 43 12 07 00 98" +
                    "07 00 00 A1 13 07 00 F6 08 00 00 9B 16 07 00 F0" +
                    "0B 00 00 99 18 07 00 EE 0D 00 00 43 1A 07 00 98" +
                    "0F 00 00 D3 1C 07 00 28 12 00 00 4B 1E 07 00 A0" +
                    "13 00 00 E9 1F 07 00 3E 15 00 00 13 21 07 00 68" +
                    "16 00 00 65 22 07 00 BA 17 00 00 BF 23 07 00 14" +
                    "19 00 00 77 25 07 00 CC 1A 00 00 11 27 07 00 66" +
                    "1C 00 00 1B 28 07 00 70 1D 00 00 89 2A 07 00 DE" +
                    "1F 00 00 07 2C 07 00 61 01 00 00 4D 2E 07 00 A7" +
                    "03 00 00 41 31 07 00 9B 06 00 00 F1 33 07 00 4B" +
                    "09 00 00 8B 36 07 00 E5 0B 00 00 15 39 07 00 6F" +
                    "0E 00 00 17 3B 07 00 71 10 00 00 2B 3E 07 00 85" +
                    "13 00 00 61 40 07 00 BB 15 00 00 1D 42 07 00 77" +
                    "17 00 00 29 43 07 00 83 18 00 00 0D 45 07 00 67" +
                    "1A 00 00 2D 46 07 00 87 1B 00 00 DF 47 07 00 39" +
                    "1D 00 00 C7 4A 07 00 04 00 00 00 AB 4D 07 00 E8" +
                    "02 00 00 55 50 07 00 92 05 00 00 C9 52 07 00 06" +
                    "08 00 00 91 55 07 00 CE 0A 00 00 9F 57 07 00 DC" +
                    "0C 00 00 8B 59 07 00 C8 0E 00 00 83 5A 07 00 C0" +
                    "0F 00 00 03 5C 07 00 40 11 00 00 3F 5D 07 00 7C" +
                    "12 00 00 7B 5E 07 00 B8 13 00 00 EF 5F 07 00 2C" +
                    "15 00 00 85 61 07 00 C2 16 00 00 2F 63 07 00 6C" +
                    "18 00 00 BB 65 07 00 F8 1A 00 00 D9 66 07 00 16" +
                    "1C 00 00 21 68 07 00 5E 1D 00 00 4D 69 07 00 8A" +
                    "1E 00 00 E3 6A 07 00 04 00 00 00 CF 6B 07 00 F0" +
                    "00 00 00 6B 6D 07 00 8C 02 00 00 F1 6E 07 00 12" +
                    "04 00 00 6D 70 07 00 8E 05 00 00 6F 71 07 00 90" +
                    "06 00 00 9F 72 07 00 C0 07 00 00 55 74 07 00 76" +
                    "09 00 00 49 76 07 00 6A 0B 00 00 D3 78 07 00 F4" +
                    "0D 00 00 5F 7A 07 00 80 0F 00 00 A9 7B 07 00 CA" +
                    "10 00 00 8F 7D 07 00 B0 12 00 00 B5 7F 07 00 D6" +
                    "14 00 00 F9 81 07 00 1A 17 00 00 57 83 07 00 78" +
                    "18 00 00 A5 84 07 00 C6 19 00 00 C3 85 07 00 E4" +
                    "1A 00 00 FB 86 07 00 1C 1C 00 00 01 89 07 00 22" +
                    "1E 00 00 2B 8A 07 00 4C 1F 00 00 77 8B 07 00 85" +
                    "00 00 00 AD 8C 07 00 BB 01 00 00 4F 8E 07 00 5D" +
                    "03 00 00 99 90 07 00 A7 05 00 00 F9 92 07 00 07" +
                    "08 00 00 AB 94 07 00 B9 09 00 00 33 96 07 00 41" +
                    "0B 00 00 CF 97 07 00 DD 0C 00 00 19 9A 07 00 27" +
                    "0F 00 00 31 9B 07 00 3F 10 00 00 BF 9C 07 00 CD" +
                    "11 00 00 A1 9E 07 00 AF 13 00 00 BF A0 07 00 CD" +
                    "15 00 00 A3 A2 07 00 B1 17 00 00 95 A4 07 00 A3" +
                    "19 00 00 95 A6 07 00 A3 1B 00 00 B7 A8 07 00 C5" +
                    "1D 00 00 9F AA 07 00 AD 1F 00 00 F5 AC 07 00 F2" +
                    "01 00 00 8F AE 07 00 8C 03 00 00 DB B0 07 00 D8" +
                    "05 00 00 8D B2 07 00 8A 07 00 00 11 B4 07 00 0E" +
                    "09 00 00 95 B5 07 00 92 0A 00 00 D3 B7 07 00 D0" +
                    "0C 00 00 5D B9 07 00 5A 0E 00 00 63 BB 07 00 60" +
                    "10 00 00 21 BE 07 00 1E 13 00 00 EB BF 07 00 E8" +
                    "14 00 00 A3 C2 07 00 A0 17 00 00 25 C4 07 00 22" +
                    "19 00 00 F7 C4 07 00 F4 19 00 00 29 C6 07 00 26" +
                    "1B 00 00 BD C7 07 00 BA 1C 00 00 61 C9 07 00 5E" +
                    "1E 00 00 3B CB 07 00 6D 00 00 00 C7 CC 07 00 F9" +
                    "01 00 00 4D CE 07 00 7F 03 00 00 93 D0 07 00 C5" +
                    "05 00 00 53 D2 07 00 85 07 00 00 BF D3 07 00 F1" +
                    "08 00 00 EF D4 07 00 21 0A 00 00 F7 D6 07 00 29" +
                    "0C 00 00 8F D8 07 00 C1 0D 00 00 DB D9 07 00 0D" +
                    "0F 00 00 21 DB 07 00 53 10 00 00 59 DC 07 00 8B" +
                    "11 00 00 25 DE 07 00 57 13 00 00 AB DF 07 00 DD" +
                    "14 00 00 5B E1 07 00 8D 16 00 00 91 E3 07 00 C3" +
                    "18 00 00 03 E5 07 00 35 1A 00 00 29 E6 07 00 5B" +
                    "1B 00 00 41 E7 07 00 73 1C 00 00 87 E8 07 00 B9" +
                    "1D 00 00 F1 E9 07 00 23 1F 00 00 33 EB 07 00 5E" +
                    "00 00 00 37 ED 07 00 62 02 00 00 13 EF 07 00 3E" +
                    "04 00 00 15 F0 07 00 40 05 00 00 13 F1 07 00 3E" +
                    "06 00 00 39 F2 07 00 64 07 00 00 41 F3 07 00 6C" +
                    "08 00 00 25 F4 07 00 50 09 00 00 75 F5 07 00 A0" +
                    "0A 00 00 83 F7 07 00 AE 0C 00 00 AF F9 07 00 DA" +
                    "0E 00 00 E1 FB 07 00 0C 11 00 00 3B FE 07 00 66" +
                    "13 00 00 8B FF 07 00 B6 14 00 00 21 01 08 00 4C" +
                    "16 00 00 89 03 08 00 B4 18 00 00 7D 04 08 00 A8" +
                    "19 00 00 0F 06 08 00 3A 1B 00 00 9F 07 08 00 CA" +
                    "1C 00 00 F3 08 08 00 1E 1E 00 00 19 0A 08 00 44" +
                    "1F 00 00 3D 0C 08 00 59 01 00 00 61 0D 08 00 7D" +
                    "02 00 00 11 0F 08 00 2D 04 00 00 45 10 08 00 61" +
                    "05 00 00 9F 11 08 00 BB 06 00 00 65 13 08 00 81" +
                    "08 00 00 83 15 08 00 9F 0A 00 00 39 17 08 00 55" +
                    "0C 00 00 0D 1A 08 00 29 0F 00 00 71 1C 08 00 8D" +
                    "11 00 00 95 1D 08 00 B1 12 00 00 D9 1E 08 00 F5" +
                    "13 00 00 35 20 08 00 51 15 00 00 99 21 08 00 B5" +
                    "16 00 00 09 24 08 00 25 19 00 00 3F 25 08 00 5B" +
                    "1A 00 00 57 26 08 00 73 1B 00 00 49 27 08 00 65" +
                    "1C 00 00 69 28 08 00 85 1D 00 00 C3 29 08 00 DF" +
                    "1E 00 00 9D 2B 08 00 D9 00 00 00 09 2D 08 00 45" +
                    "02 00 00 29 2E 08 00 65 03 00 00 2F 31 08 00 6B" +
                    "06 00 00 9D 32 08 00 D9 07 00 00 B1 33 08 00 ED" +
                    "08 00 00 0D 36 08 00 49 0B 00 00 BB 37 08 00 F7" +
                    "0C 00 00 19 3A 08 00 55 0F 00 00 79 3C 08 00 B5" +
                    "11 00 00 E1 3D 08 00 1D 13 00 00 6F 3F 08 00 AB" +
                    "14 00 00 5F 40 08 00 9B 15 00 00 8D 41 08 00 C9" +
                    "16 00 00 FF 42 08 00 3B 18 00 00 07 44 08 00 43" +
                    "19 00 00 27 46 08 00 63 1B 00 00 53 47 08 00 8F" +
                    "1C 00 00 47 48 08 00 83 1D 00 00 53 49 08 00 8F" +
                    "1E 00 00 61 4A 08 00 9D 1F 00 00 D7 4B 08 00 1E" +
                    "01 00 00 57 4D 08 00 9E 02 00 00 CB 4E 08 00 12" +
                    "04 00 00 7D 50 08 00 C4 05 00 00 A9 51 08 00 F0" +
                    "06 00 00 41 53 08 00 88 08 00 00 ED 54 08 00 34" +
                    "0A 00 00 17 57 08 00 5E 0C 00 00 91 58 08 00 D8" +
                    "0D 00 00 B7 59 08 00 FE 0E 00 00 0F 5B 08 00 56" +
                    "10 00 00 33 5C 08 00 7A 11 00 00 A7 5D 08 00 EE" +
                    "12 00 00 25 5F 08 00 6C 14 00 00 D5 61 08 00 1C" +
                    "17 00 00 EB 63 08 00 32 19 00 00 A5 65 08 00 EC" +
                    "1A 00 00 E1 67 08 00 28 1D 00 00 9F 69 08 00 E6" +
                    "1E 00 00 DD 6A 08 00 04 00 00 00 6F 6C 08 00 96" +
                    "01 00 00 B7 6D 08 00 DE 02 00 00 6D 6F 08 00 94" +
                    "04 00 00 ED 70 08 00 14 06 00 00 5B 72 08 00 82" +
                    "07 00 00 09 74 08 00 30 09 00 00 43 75 08 00 6A" +
                    "0A 00 00 77 76 08 00 9E 0B 00 00 7D 77 08 00 A4" +
                    "0C 00 00 E3 78 08 00 0A 0E 00 00 87 7B 08 00 AE" +
                    "10 00 00 DF 7C 08 00 06 12 00 00 DF 7D 08 00 06" +
                    "13 00 00 3F 7F 08 00 66 14 00 00 71 80 08 00 98" +
                    "15 00 00 8F 81 08 00 B6 16 00 00 EF 82 08 00 16" +
                    "18 00 00 AB 84 08 00 D2 19 00 00 8D 86 08 00 B4" +
                    "1B 00 00 B9 87 08 00 E0 1C 00 00 47 89 08 00 6E" +
                    "1E 00 00 F7 8A 08 00 04 00 00 00 FB 8B 08 00 08" +
                    "01 00 00 9B 8D 08 00 A8 02 00 00 DF 8E 08 00 EC" +
                    "03 00 00 DB 8F 08 00 E8 04 00 00 BB 90 08 00 C8" +
                    "05 00 00 F3 91 08 00 00 07 00 00 43 93 08 00 50" +
                    "08 00 00 71 94 08 00 7E 09 00 00 5B 96 08 00 68" +
                    "0B 00 00 A9 97 08 00 B6 0C 00 00 0F 99 08 00 1C" +
                    "0E 00 00 C5 9A 08 00 D2 0F 00 00 11 9C 08 00 1E" +
                    "11 00 00 D1 9C 08 00 DE 11 00 00 DD 9D 08 00 EA" +
                    "12 00 00 73 9F 08 00 80 14 00 00 05 A1 08 00 12" +
                    "16 00 00 1D A2 08 00 2A 17 00 00 EB A3 08 00 F8" +
                    "18 00 00 33 A5 08 00 40 1A 00 00 F5 A6 08 00 02" +
                    "1C 00 00 6B A8 08 00 78 1D 00 00 77 A9 08 00 84" +
                    "1E 00 00 A5 AB 08 00 CE 00 00 00 99 AD 08 00 C2" +
                    "02 00 00 ED AF 08 00 16 05 00 00 CB B1 08 00 F4" +
                    "06 00 00 B9 B3 08 00 E2 08 00 00 4F B5 08 00 78" +
                    "0A 00 00 67 B6 08 00 90 0B 00 00 DD B7 08 00 06" +
                    "0D 00 00 67 B9 08 00 90 0E 00 00 C7 BA 08 00 F0" +
                    "0F 00 00                                       ";


    /**
     * deep comparison of two SST records
     */
    public static void assertEquals(SSTRecord expected, SSTRecord actual){
        assertEquals("number of strings", expected.getNumStrings(), actual.getNumStrings());
        assertEquals("number of unique strings", expected.getNumUniqueStrings(), actual.getNumUniqueStrings());
        assertEquals("count of strings", expected.countStrings(), actual.countStrings());
        for ( int k = 0; k < expected.countStrings(); k++ ) {
            UnicodeString us1 = expected.getString(k);
            UnicodeString us2 = actual.getString(k);

            assertTrue("String at idx=" + k, us1.equals(us2));
        }
    }

    /**
     *  ExtSSTRecord can be continued. Ensure we properly read the continue remainder.
     */
    public void test50967() {
        byte[] bytes = HexRead.readFromString(data_50967);

        RecordInputStream in = TestcaseRecordInputStream.create(bytes);
        assertEquals(ExtSSTRecord.sid, in.getSid());
        ExtSSTRecord src = new ExtSSTRecord(in);
        assertEquals(12386, src.getDataSize());
        ExtSSTRecord.InfoSubRecord[] sub1 = src.getInfoSubRecords();

        byte[] serialized = src.serialize();

        in = TestcaseRecordInputStream.create(serialized);
        assertEquals(ExtSSTRecord.sid, in.getSid());
        ExtSSTRecord dst = new ExtSSTRecord(in);
        assertEquals(12386, dst.getDataSize());
        ExtSSTRecord.InfoSubRecord[] sub2 = src.getInfoSubRecords();
        assertEquals(sub1.length, sub2.length);

        for(int i = 0; i < sub1.length; i++){
            ExtSSTRecord.InfoSubRecord s1 = sub1[i];
            ExtSSTRecord.InfoSubRecord s2 = sub2[i];

            assertEquals(s1.getBucketSSTOffset(), s2.getBucketSSTOffset());
            assertEquals(s1.getStreamPos(), s2.getStreamPos());

        }
    }


}
