///////////////////////////////////////////////////////////////////////////////
//                                                                             
// JTOpen (IBM Toolbox for Java - OSS version)                                 
//                                                                             
// Filename: SignonEvent.java
//                                                                             
// The source code contained herein is licensed under the IBM Public License   
// Version 1.0, which has been approved by the Open Source Initiative.         
// Copyright (C) 2004-2005 International Business Machines Corporation and     
// others. All rights reserved.                                                
//                                                                             
///////////////////////////////////////////////////////////////////////////////

package com.ibm.as400.access;

import java.util.EventObject;

/**
 An Event object generated by an {@link AS400 AS400} instance during an attempt to sign-on to the system.  SignonEvent objects are consumed by instances of {@link SignonHandler SignonHandler}.
 For all instances of SignonEvent, the {@link EventObject#getSource getSource()} method always returns the <tt>AS400</tt> instance that originated the event.
 @see AS400#setSignonHandler
 **/
public final class SignonEvent extends EventObject
{
  private AS400SecurityException exception_;
  private boolean reconnecting_;


  /**
   Constructs a SignonEvent object.
   @param system The system.
   @param reconnecting Indicates whether the system object is already connected.
   **/
  SignonEvent(AS400 system, boolean reconnecting)
  {
    // Assume args have been validated.
    super(system);
    reconnecting_ = reconnecting;
  }


  /**
   Constructs a SignonEvent object.
   @param system The system.
   @param reconnecting Indicates whether the system object is already connected.
   @param exc The exception.  null if no exception is associated with this event.
   **/
  SignonEvent(AS400 system, boolean reconnecting, AS400SecurityException exc)
  {
    // Assume args have been validated as non-null.
    super(system);
    reconnecting_ = reconnecting;
    exception_ = exc;  // tolerate null value
  }


  /**
   Returns the exception that is associated with this event, if any.
   @return The exception that is associated with this event, or <tt>null</tt> if no exception is associated with the event.
   @see #hasException
   **/
  public AS400SecurityException getException()
  {
    return exception_;
  }


  /**
   Indicates whether this SignonEvent object has an associated exception.  That is, indicates whether getException() will return non-null.
   @return true if there is an associated exception; false otherwise.
   @see #getException
   **/
  public boolean hasException()
  {
    return (exception_ != null);
  }


  /**
   Indicates whether the system object was previously connected.  That is, indicates whether this is a reconnection (for example, to an additional service) rather than an initial connection.
   @return true if the system object is already connected.
   **/
  public boolean isReconnecting()
  {
    return reconnecting_;
  }

}

