# Copyright, 2018, by Samuel G. D. Williams. <http://www.codeotaku.com>
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

require_relative 'connection_context'

RSpec.describe Protocol::HTTP2::Connection do
	include_context Protocol::HTTP2::Connection
	
	it "can negotiate connection" do
		first_server_frame = nil
		
		first_client_frame = client.send_connection_preface do
			first_server_frame = server.read_connection_preface([])
		end
		
		expect(first_client_frame).to be_kind_of Protocol::HTTP2::SettingsFrame
		expect(first_client_frame).to_not be_acknowledgement
		
		expect(first_server_frame).to be_kind_of Protocol::HTTP2::SettingsFrame
		expect(first_server_frame).to_not be_acknowledgement
		
		frame = client.read_frame
		expect(frame).to be_kind_of Protocol::HTTP2::SettingsFrame
		expect(frame).to be_acknowledgement
		
		frame = server.read_frame
		expect(frame).to be_kind_of Protocol::HTTP2::SettingsFrame
		expect(frame).to be_acknowledgement
		
		expect(client.state).to eq :open
		expect(server.state).to eq :open
	end
	
	context Protocol::HTTP2::PingFrame do
		before do
			client.open!
			server.open!
		end
		
		it "can send ping and receive pong" do
			expect(server).to receive(:receive_ping).once.and_call_original
			
			client.send_ping("12345678")
			
			server.read_frame
			
			expect(client).to receive(:receive_ping).once.and_call_original
			
			client.read_frame
		end
	end
	
	context Protocol::HTTP2::Stream do
		before do
			client.open!
			server.open!
		end
		
		let(:request_data) {"Hello World!"}
		let(:stream) {client.create_stream}
		
		let(:request_headers) {[[':method', 'GET'], [':path', '/'], [':authority', 'localhost']]}
		let(:response_headers) {[[':status', '200']]}
		
		it "can create new stream and send response" do
			stream.send_headers(nil, request_headers)
			expect(stream.id).to eq 1
			
			expect(server).to receive(:receive_headers).once.and_wrap_original do |method, frame|
				headers = method.call(frame)
				
				expect(headers).to be == request_headers
			end
			
			server.read_frame
			expect(server.streams).to_not be_empty
			expect(server.streams[1].state).to be :open
			
			stream.send_data(request_data, Protocol::HTTP2::END_STREAM)
			expect(stream.state).to eq :half_closed_local
			
			expect(server).to receive(:receive_data).and_call_original
			
			data_frame = server.read_frame
			expect(data_frame.unpack).to be == request_data
			expect(server.streams[1].state).to be :half_closed_remote
			
			server.streams[1].send_headers(nil, response_headers, Protocol::HTTP2::END_STREAM)
			
			expect(stream).to receive(:process_headers).once.and_wrap_original do |method, frame|
				headers = method.call(frame)
				
				expect(headers).to be == response_headers
			end
			
			client.read_frame
			expect(stream.state).to eq :closed
			
			expect(stream.remote_window.used).to eq data_frame.length
		end
		
		it "client can handle graceful shutdown" do
			stream.send_headers(nil, request_headers, Protocol::HTTP2::END_STREAM)
			
			# Establish request stream on server:
			server.read_frame
			
			# Graceful shutdown
			server.send_goaway(0)
			
			expect(client.read_frame).to be_a Protocol::HTTP2::GoawayFrame
			expect(client.remote_stream_id).to be == 1
			expect(client).to be_closed
			
			expect(server.streams[1].state).to eq :half_closed_remote
			
			server.streams[1].send_headers(nil, response_headers, Protocol::HTTP2::END_STREAM)
			
			client.read_frame
			
			expect(stream.state).to eq :closed
		end
		
		it "client can handle non-graceful shutdown" do
			stream.send_headers(nil, request_headers, Protocol::HTTP2::END_STREAM)
			
			# Establish request stream on server:
			server.read_frame
			
			# Send connection error to client:
			server.send_goaway(1, "Bugger off!")
			
			expect(stream).to receive(:close).and_call_original
			
			expect do
				client.read_frame
			end.to raise_error(Protocol::HTTP2::GoawayError)
			
			client.close
			
			expect(client.remote_stream_id).to be == 1
			expect(client).to be_closed
		end
		
		it "can stream data" do
			stream.send_headers(nil, request_headers)
			stream.send_data("A")
			stream.send_data("B")
			stream.send_data("C")
			stream.send_data("", Protocol::HTTP2::END_STREAM)
			
			frame = server.read_frame
			expect(frame).to be_a(Protocol::HTTP2::HeadersFrame)
			
			expect(stream.available_frame_size).to be >= 3
			
			frame = server.read_frame
			expect(frame).to be_a(Protocol::HTTP2::DataFrame)
			expect(frame.unpack).to be == "A"
			
			frame = server.read_frame
			expect(frame).to be_a(Protocol::HTTP2::DataFrame)
			expect(frame.unpack).to be == "B"
			
			frame = server.read_frame
			expect(frame).to be_a(Protocol::HTTP2::DataFrame)
			expect(frame.unpack).to be == "C"
			
			frame = server.read_frame
			expect(frame).to be_a(Protocol::HTTP2::DataFrame)
			expect(frame.unpack).to be == ""
			expect(frame).to be_end_stream
		end
	end
end
