// SPDX-FileCopyrightText: 2024 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "backendloader_p.h"
#include "inferenceplugin.h"
#include "systemenv.h"
#include "modelinfo.h"

#include <QDirIterator>
#include <QSettings>
#include <QStandardPaths>

#include <iostream>

GLOBAL_USE_NAMESPACE

BackendLoaderPrivate::BackendLoaderPrivate(BackendLoader *parent) : q(parent)
{

}

QList<BackendMetaObjectPointer> BackendLoaderPrivate::sorted() const
{
    QList<BackendMetaObjectPointer> ret = backends;
    std::stable_sort(ret.begin(), ret.end(), [](const BackendMetaObjectPointer &t1, const BackendMetaObjectPointer &t2) {
        return t1->extra(kInferenceBackendScore).toFloat() > t2->extra(kInferenceBackendScore).toFloat();
    });
    return ret;
}

void BackendLoaderPrivate::preload(BackendMetaObjectPointer mo)
{
    if (mo->name() == kInferenceBackendLlamaCpp) {
        QString so = llamacppBackend();
        if (!so.isEmpty()) {
            QFileInfo soFile(QString(PLUGIN_BACKEND_DIR) + "/" + so);
            bool ok = mo->extra(soFile.fileName()).toBool();
            if (ok) {
                std::cerr << "load user setted libllama " << so.toStdString() << std::endl;
                so = soFile.absoluteFilePath();
            }
            else {
                std::cerr << "the libllama user setted is unavailable" << so.toStdString() << std::endl;
                so.clear();
            }
        }

        if (so.isEmpty()) {
            static QStringList defaultso = {"libllama-cuda.so", "libllama-avx2.so", "libllama.so"};
            for (const QString &fso : defaultso) {
                bool ok = mo->extra(fso, false).toBool();
                if (ok) {
                    so = QString(PLUGIN_BACKEND_DIR) + "/llama.cpp/" + fso;
                    break;
                }
            }
        }

        if (so.isEmpty()) {
            std::cerr << "no libllama for backend " << mo->name().toStdString() << std::endl;
            return;
        }

        std::cerr << "load libllama " << so.toStdString() << std::endl;
        QLibrary lib(so);
        lib.load();
    }
}

void BackendLoaderPrivate::checkRuntime(BackendMetaObjectPointer mo)
{
    if (mo.isNull())
        return;

    if (mo->name() == kInferenceBackendLlamaCpp) {
        QDirIterator dirItera(QString(PLUGIN_BACKEND_DIR) + "/llama.cpp", { "*.so" },
                              QDir::Filter::Files,
                              QDirIterator::IteratorFlag::NoIteratorFlags);
        QStringList libs;
        float socre = 0;
        while (dirItera.hasNext()) {
            dirItera.next();
            const QString &name = dirItera.fileName();
            libs.append(name);

            const QString &fileName { dirItera.path() + "/" + dirItera.fileName()};
            if (name == "libllama-avx2.so") {
                auto flags = SystemEnv::cpuInstructions();
                if (flags.isEmpty())
                    std::cerr << "fail to get cpu flags by read /proc/cpuinfo" << std::endl;

                bool ok = flags.contains("avx2", Qt::CaseInsensitive);
                if (!ok)
                    std::cerr << "cpu does not support avx2" << std::endl;
                mo->setExtra(name, ok);

                if (socre < 3) {
                    socre = 3;
                    mo->setExtra(kInferenceBackendScore, socre);
                }

            } else if (name == "libllama-cuda.so") {
                bool enable = SystemEnv::vga().contains("nvidia", Qt::CaseInsensitive);
                enable = enable ? SystemEnv::checkLibrary(fileName) : false;
                mo->setExtra(name, enable);

                if (socre < 5) {
                    socre = 5;
                    mo->setExtra(kInferenceBackendScore, socre);
                }
            } else {
                mo->setExtra(name, true);
                if (socre < 1) {
                    socre = 1;
                    mo->setExtra(kInferenceBackendScore, socre);
                }
            }
        }
        mo->setExtra(kInferenceBackendLibs, libs);
    } else if (mo->name() == kInferenceBackendOpenvino) {
        bool cpu = SystemEnv::cpuModelName().contains("intel", Qt::CaseInsensitive);
        bool gpu = SystemEnv::vga().contains("intel", Qt::CaseInsensitive);
        bool enable = cpu || gpu  ? SystemEnv::checkLibrary(mo->fileName()) : false;
        mo->setExtra(kInferenceBackendRTReady, enable);

        float socre = 0;
        if (cpu)
            socre += 2;
        if (gpu)
            socre += 5;

        mo->setExtra(kInferenceBackendScore, socre);
    } else {
        mo->setExtra(kInferenceBackendScore, 1);
    }
}

QString BackendLoaderPrivate::fixedBackend() const
{
    QSettings set(configPath(), QSettings::IniFormat);
    set.beginGroup("backend");
    return set.value("plugin").toString();
}

QString BackendLoaderPrivate::llamacppBackend() const
{
    QSettings set(configPath(), QSettings::IniFormat);
    set.beginGroup("backend");
    return set.value("llama.cpp").toString();
}

QString BackendLoaderPrivate::configPath() const
{
    return QStandardPaths::standardLocations(QStandardPaths::ConfigLocation).first()
            + QString("/deepin/%1/config.conf").arg(EXE_NAME);
}

BackendLoader::BackendLoader(QObject *parent)
    : QObject(parent)
    , d(new BackendLoaderPrivate(this))
{

    QStringList paths{PLUGIN_BACKEND_DIR};
    d->loadPaths = paths;
}

void BackendLoader::setPaths(const QStringList &paths)
{
    d->loadPaths = paths;
}

void BackendLoader::readBackends()
{
    d->backends.clear();

    for (const QString &path : d->loadPaths) {
        QDirIterator dirItera(path, { "*.so" },
                              QDir::Filter::Files,
                              QDirIterator::IteratorFlag::NoIteratorFlags);
        while (dirItera.hasNext()) {
            dirItera.next();
            BackendMetaObjectPointer mataObj(new BackendMetaObject);
            const QString &fileName { dirItera.path() + "/" + dirItera.fileName() };
            QSharedPointer<QPluginLoader> loader(new QPluginLoader);
            mataObj->d->loader = loader;

            loader->setFileName(fileName);
            if (mataObj->iid() == InferencePlugin_Meta_IID) {
                d->checkRuntime(mataObj);
                d->backends.append(mataObj);
            }
        }
    }
}

QList<BackendMetaObjectPointer> BackendLoader::backends() const
{
    return d->backends;
}

QSharedPointer<InferencePlugin> BackendLoader::load(BackendMetaObjectPointer mo) const
{
    if (mo.isNull())
        return nullptr;

    if (!mo->d->loader)
        return nullptr;

    d->preload(mo);

    if (!mo->d->loader->load()) {
        std::cerr << "Failed load plugin: " << mo->d->loader->errorString().toStdString() << std::endl;
        return nullptr;
    }

    QSharedPointer<InferencePlugin> ins(qobject_cast<InferencePlugin *>(mo->d->loader->instance()));
    return ins;
}

BackendMetaObjectPointer BackendLoader::perfect(const QSharedPointer<ModelInfo> &model, QString *matchedFormat, QString *matchedArch) const
{
    BackendMetaObjectPointer ret;
    if (model.isNull())
        return ret;

    QString fixed = d->fixedBackend();
    auto sortedBackends = d->sorted();

    // check fixed
    if (!fixed.isEmpty()) {
        for (auto bk : sortedBackends) {
            if (QFileInfo(bk->fileName()).fileName() == fixed) {
                auto formats = bk->suportedFormats();
                for (const QString &fmt : model->formats()) {
                    if (formats.contains(fmt, Qt::CaseInsensitive)) {
                        auto archs = bk->suportedArchitectures();
                        // check architectures
                        for (const QString &arch : model->architectures(fmt)) {
                            if (archs.contains(arch, Qt::CaseInsensitive) && isRuntimeSupported(bk)) {
                                if (matchedFormat)
                                    *matchedFormat = fmt;
                                if (matchedArch)
                                    *matchedArch = arch;
                                std::cerr << QString("using user configed backend %0").arg(fixed).toStdString() << std::endl;
                                return bk;
                            }
                        }
                    }
                }
                std::cerr << QString("user configed backend %0 does not support model %1").arg(fixed).arg(model->name()).toStdString() << std::endl;
                break;
            }
        }
    }

    for (auto bk : sortedBackends) {
        // check format
        auto formats = bk->suportedFormats();
        for (const QString &fmt : model->formats()) {
            if (formats.contains(fmt, Qt::CaseInsensitive)) {
                auto archs = bk->suportedArchitectures();
                // check architectures
                for (const QString &arch : model->architectures(fmt)) {
                    if (archs.contains(arch, Qt::CaseInsensitive) && isRuntimeSupported(bk)) {
                        if (matchedFormat)
                            *matchedFormat = fmt;
                        if (matchedArch)
                            *matchedArch = arch;
                        return bk;
                    }
                }
            }
        }
    }

    return ret;
}

bool BackendLoader::isRuntimeSupported(BackendMetaObjectPointer mo) const
{
    if (mo->name() == kInferenceBackendLlamaCpp) {
        auto libs = mo->extra(kInferenceBackendLibs).toStringList();
        for (auto lib : libs) {
            if (mo->extra(lib).toBool())
                return true;
        }
        return false;
    } else if (mo->name() == kInferenceBackendOpenvino) {
        return mo->extra(kInferenceBackendRTReady).toBool();
    }

    return true;
}




