// SPDX-FileCopyrightText: 2024 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "httpserver_p.h"

#include <QDebug>
#include <QDateTime>
#include <QJsonDocument>
#include <QJsonObject>

GLOBAL_USE_NAMESPACE

HttpServerPrivate::HttpServerPrivate(HttpServer *parent) : q(parent)
{

}

bool HttpServerPrivate::isPortInUse(int port)
{
    char cmd[64];
    snprintf(cmd, sizeof(cmd), "netstat -tuln | grep :%d", port);
    FILE* pipe = popen(cmd, "r");
    if (!pipe) {
        std::cerr << "ERROR: popen(netstat) failed!" << std::endl;
        return false;
    }

    char buffer[256];
    while (fgets(buffer, sizeof(buffer), pipe) != nullptr) {
        if (strstr(buffer, "LISTEN")) {
            pclose(pipe);
            return true;
        }
    }

    pclose(pipe);
    return false;
}

bool HttpServerPrivate::inerApi(const httplib::Request &req, httplib::Response &res)
{
    if (req.path == "/health") {
        res.status == 200;
        QVariantHash hash;
        int cur = QDateTime::currentSecsSinceEpoch();
        hash.insert("status", "ok");
        hash.insert("time", cur);
        hash.insert("idle", lastWorkTime > 0 ? cur - lastWorkTime : -1);
        res.set_content(QJsonDocument(QJsonObject::fromVariantHash(hash))
                        .toJson(QJsonDocument::Compact).toStdString(), "application/json");
        return true;
    }

    return false;
}

HttpServer::HttpServer(QObject *parent)
    : QThread(parent)
    , d(new HttpServerPrivate(this))
{

}

bool HttpServer::initialize(QString host, int port)
{
    if (d->hserve)
        return false;

    if (host.isEmpty())
        host = "127.0.0.1";

    // create http sever
   d->hserve = new httplib::Server();
   d->hserve->set_read_timeout(600);
   d->hserve->set_write_timeout(600);

   if (!d->hserve->bind_to_port(host.toStdString(), port)) {
       std::cerr << QString("couldn't bind to server socket: hostname=%0 port=%1").arg(host).arg(port).toStdString()
                 << std::endl;
       return false;
   } else
        qDebug() << "http server listen" << host << port;

   d->hserve->set_error_handler([](const httplib::Request &, httplib::Response & res) {
       if (res.status == 404) {
           res.set_content("{\"error\",\"File Not Found\"}", "application/json; charset=utf-8");
       }
   });

   // api 验证
   d->hserve->set_pre_routing_handler([this](const httplib::Request & req, httplib::Response & res) {
       if (d->inerApi(req, res))
           return httplib::Server::HandlerResponse::Handled;
       return httplib::Server::HandlerResponse::Unhandled;
   });


   // iner api
   registerAPI(Get, "/health");

   return true;
}

bool HttpServer::registerAPI(HttpServer::ReqType type, const QString &api)
{
    if (api.isEmpty())
        return false;

    auto handler = [this](const httplib::Request &req, httplib::Response & res){
        HttpContext ctx{&req, &res};
        // block，there is in work thread on http
        emit this->newRequset(&ctx);
    };
    if (type == Get)
       d->hserve->Get(api.toStdString(), handler);
    else
       d->hserve->Post(api.toStdString(), handler);

   return true;
}

int HttpServer::randomPort()
{
    // 尝试10次
    int count = 10;
    while (count--) {
        int port = qrand() % 9999 + 30000;
        if (!HttpServerPrivate::isPortInUse(port))
            return port;
    }

    return -1;
}

void HttpServer::setLastWorkTime(int seconds)
{
    d->lastWorkTime = seconds;
}

QString HttpServer::getPath(HttpContext *ctx)
{
    return QString::fromStdString(ctx->req->path);
}

QString HttpServer::getBody(HttpContext *ctx)
{
    return QString::fromStdString(ctx->req->body);
}

void HttpServer::setContent(HttpContext *ctx, const QString &content, const QString &type)
{
    ctx->res->set_content(content.toStdString(), type.toStdString());
}

void HttpServer::setStatus(deepin_modelhub::HttpContext *ctx, int st)
{
    ctx->res->status = st;
}

void HttpServer::setChunckProvider(HttpContext *ctx, chunckProvider cp, chunckComplete cc, const QString &type)
{
    const auto chunked_content_provider = [cp](size_t, httplib::DataSink & sink) {
        while (true) {
            QString out;
            bool stop = false;
            if (cp(out, stop)) {
                auto str = out.toStdString();
                if (!sink.write(str.c_str(), str.size()))
                    return false;

                if (stop)
                    break;
            } else {
                std::string error = R"(error:{""})";
                if (!sink.write(error.c_str(), error.size()))
                    return false;
                break;
            }
        }
        sink.done();
        return true;
    };

    ctx->res->set_chunked_content_provider(type.toStdString(), chunked_content_provider, cc);
}

void HttpServer::run()
{
    qDebug() << "http server is running.";
    bool ret = d->hserve->listen_after_bind();
    qDebug() << "http server exit." << ret;
}

HttpServer::~HttpServer()
{
    if (d->hserve) {
        d->hserve->stop();
    }

    if (!isFinished())
        wait(2000);

    delete d->hserve;
    d->hserve = nullptr;
}

