// SPDX-FileCopyrightText: 2011 - 2022 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: LGPL-3.0-or-later

#include "multiscreenwidget.h"
#include "brightnesswidget.h"
#include "scalingwidget.h"
#include "resolutionwidget.h"
#include "refreshratewidget.h"
#include "rotatewidget.h"
#include "secondaryscreendialog.h"
#include "../../mainwindow.h"
#include "widgets/settingsitem.h"
#include "modules/display/displaymodel.h"
#include "modules/display/monitorcontrolwidget.h"
#include "modules/display/monitorindicator.h"
#include "modules/display/recognizewidget.h"
#include "window/gsettingwatcher.h"
#include "window/dconfigwatcher.h"

#include <QVBoxLayout>
#include <QApplication>
#include <QComboBox>
#include <QDesktopWidget>
#include <QScreen>
#include <QKeyEvent>

using namespace dcc::widgets;
using namespace dcc::display;
using namespace DCC_NAMESPACE::display;
DWIDGET_USE_NAMESPACE

const int ComboxWidth = 300;

MultiScreenWidget::MultiScreenWidget(QWidget *parent)
    : QWidget(parent)
    , m_contentLayout(new QVBoxLayout(this))
    , m_monitorControlWidget(new MonitorControlWidget(240, this)) //根据产品设计更改出事高度为240
    , m_fullIndication(new MonitorIndicator(this))
    , m_modeSettingsItem(new SettingsItem(this))
    , m_modeCombox(new QComboBox(m_modeSettingsItem))
    , m_primarySettingsItem(new SettingsItem(this))
    , m_primaryCombox(new QComboBox(m_primarySettingsItem))
    , m_brightnessSpacerItem(new QSpacerItem(0, 20))
    , m_brightnessWidget(new BrightnessWidget(this))
    , m_scalingWidget(new ScalingWidget(this))
    , m_resolutionWidget(new ResolutionWidget(300, this))
    , m_refreshRateWidget(new RefreshRateWidget(300, this))
    , m_rotateWidget(new RotateWidget(300, this))
    , m_model(nullptr)
{
    //初始化列表无法进行静态翻译
    //~ contents_path /display/Multiple Displays
    m_multiSettingLabel = new TitleLabel(tr("Multiple Displays"), this);
    //~ contents_path /display/Mode
    m_modeLabel = new QLabel(tr("Mode"), this);
    //~ contents_path /display/Main Scree
    m_primaryLabel = new QLabel(tr("Main Screen"), this);

    m_monitorControlWidget->setAccessibleName("MultiScreenWidget_monitorControl");
    m_fullIndication->setAccessibleName("fullIndication");

    m_contentLayout->setSpacing(0);
    m_contentLayout->setContentsMargins(56, 20, 56, 0);
    m_contentLayout->addWidget(m_monitorControlWidget);
    m_contentLayout->addSpacing(20);
    m_contentLayout->addWidget(m_multiSettingLabel);
    QHBoxLayout *modeLayout = new QHBoxLayout(m_modeSettingsItem);
    modeLayout->setContentsMargins(10, 0, 10, 0);
    modeLayout->addWidget(m_modeLabel);
    modeLayout->addWidget(m_modeCombox);
    m_modeCombox->setFocusPolicy(Qt::NoFocus);
    m_modeCombox->setMinimumWidth(ComboxWidth);
    m_modeCombox->setMinimumHeight(36);
    m_modeSettingsItem->addBackground();
    m_modeSettingsItem->setMinimumHeight(48);
    m_modeSettingsItem->setLayout(modeLayout);
    m_contentLayout->addSpacing(10);
    m_contentLayout->addWidget(m_modeSettingsItem);

    QHBoxLayout *primaryLayout = new QHBoxLayout(m_primarySettingsItem);
    primaryLayout->setContentsMargins(10, 0, 10, 0);
    primaryLayout->addWidget(m_primaryLabel);
    primaryLayout->addWidget(m_primaryCombox);
    m_primaryCombox->setFocusPolicy(Qt::NoFocus);
    m_primaryCombox->setMinimumWidth(ComboxWidth);
    m_primaryCombox->setMinimumHeight(36);
    m_primarySettingsItem->addBackground();
    m_primarySettingsItem->setMinimumHeight(48);
    m_primarySettingsItem->setLayout(primaryLayout);
    m_contentLayout->addSpacing(10);
    m_contentLayout->addWidget(m_primarySettingsItem);

    m_contentLayout->addSpacerItem(m_brightnessSpacerItem);
    m_contentLayout->addWidget(m_brightnessWidget);
    m_contentLayout->addSpacing(20);
    m_contentLayout->addWidget(m_scalingWidget);
    m_contentLayout->addSpacing(30);
    m_contentLayout->addWidget(m_resolutionWidget);
    m_contentLayout->addSpacing(20);
    m_contentLayout->addWidget(m_refreshRateWidget);
    m_contentLayout->addSpacing(20);
    m_contentLayout->addWidget(m_rotateWidget);
    m_contentLayout->addStretch();

    setLayout(m_contentLayout);

    // 使用GSettings来控制显示状态
    GSettingWatcher::instance()->bind("displayResolution", m_resolutionWidget);
    GSettingWatcher::instance()->bind("displayRefreshRate", m_refreshRateWidget);
    GSettingWatcher::instance()->bind("displayRotate", m_rotateWidget);
    GSettingWatcher::instance()->bind("displayMultipleDisplays", m_multiSettingLabel);
    GSettingWatcher::instance()->bind("displayMultipleDisplays", m_modeSettingsItem);
    GSettingWatcher::instance()->bind("displayMultipleDisplays", m_primarySettingsItem);
    connect(GSettingWatcher::instance(), SIGNAL(notifyGSettingsChanged(const QString &, const QString &)), this, SLOT(onGSettingsChanged(const QString &, const QString &)));
// 先屏蔽dconfig配置,待后续进行配置迁移时统一替换为dconfig配置
//    DConfigWatcher::instance()->bind(DConfigWatcher::display,"mainScreen", m_primarySettingsItem);
//    DConfigWatcher::instance()->bind(DConfigWatcher::display,"multiscreenMode", m_modeSettingsItem);

    QDesktopWidget *desktopwidget = QApplication::desktop();
    connect(desktopwidget,SIGNAL(workAreaResized(int)),this,SLOT(onResetSecondaryScreenDlg()));

    // 仅HDMI屏切换到以HDMI为主屏的扩展模式时，只触发显示模式变化信号，不会触发主屏变化信号，此时只会执行一次initSecondaryScreenDialog，
    // 此时获取QApplication::screens数据不准确，只获取到一个屏幕，另外一个屏幕无法获取，导致副界面位置不对
    // 在SecondaryScreenDialog::resetDialog中，m_monitor->getQScreen()为空
    connect(qApp, &QApplication::screenAdded, this, [ = ]{
        if (qEnvironmentVariable("XDG_SESSION_TYPE").contains("wayland")) {
            initSecondaryScreenDialog();
        }
    });

}

MultiScreenWidget::~MultiScreenWidget()
{
    for (auto dlg : m_secondaryScreenDlgList) {
        dlg->deleteLater();
    }
    m_secondaryScreenDlgList.clear();

    for(auto widget : m_recognizeWidget) {
        widget->deleteLater();
    }
    m_recognizeWidget.clear();
    delete m_fullIndication;

    GSettingWatcher::instance()->erase("displayResolution", m_resolutionWidget);
    GSettingWatcher::instance()->erase("displayRefreshRate", m_refreshRateWidget);
    GSettingWatcher::instance()->erase("displayRotate", m_rotateWidget);
    GSettingWatcher::instance()->erase("displayMultipleDisplays", m_multiSettingLabel);
    GSettingWatcher::instance()->erase("displayMultipleDisplays", m_modeSettingsItem);
    GSettingWatcher::instance()->erase("displayMultipleDisplays", m_primarySettingsItem);
// 先屏蔽dconfig配置,待后续进行配置迁移时统一替换为dconfig配置
//    DConfigWatcher::instance()->erase(DConfigWatcher::display,"mainScreen", m_primarySettingsItem);
//    DConfigWatcher::instance()->erase(DConfigWatcher::display,"multiscreenMode", m_modeSettingsItem);

}

void MultiScreenWidget::setModel(dcc::display::DisplayModel *model)
{
    m_model = model;

    initModeList();
    initPrimaryList();

    connect(m_model, &DisplayModel::displayModeChanged, m_monitorControlWidget, &MonitorControlWidget::setScreensMerged);
    connect(m_model, &DisplayModel::displayModeChanged, this, [=](const int mode) {
        m_monitorControlWidget->setMergeMode(mode == MERGE_MODE);
        if (mode == MERGE_MODE) {
            m_modeCombox->blockSignals(true);
            m_modeCombox->setCurrentIndex(0);
            m_modeCombox->blockSignals(false);
            m_primarySettingsItem->setVisible(false);
            m_brightnessWidget->showBrightness();
            m_brightnessWidget->setVisible(m_model->brightnessEnable());
            m_monitorControlWidget->setModel(m_model);
            for (auto dlg : m_secondaryScreenDlgList) {
                dlg->deleteLater();
            }
            m_secondaryScreenDlgList.clear();
        } else if (mode == EXTEND_MODE) {
            m_modeCombox->blockSignals(true);
            m_modeCombox->setCurrentIndex(1);
            m_modeCombox->blockSignals(false);
            m_primarySettingsItem->setVisible(true);
            m_brightnessWidget->showBrightness(m_model->primaryMonitor());
            m_monitorControlWidget->setModel(m_model);
            initSecondaryScreenDialog();
        } else if (m_model->displayMode() == SINGLE_MODE) {
            auto monitorList = m_model->monitorList();
            for (int idx = 0; idx < monitorList.size(); ++idx) {
                auto monitor = monitorList[idx];
                if (monitor->enable()) {
                    setModeCurrentIndex(monitor->name());
                    m_monitorControlWidget->setModel(m_model, monitor);
                    break;
                }
            }

            m_primarySettingsItem->setVisible(false);
            m_brightnessWidget->showBrightness(m_model->primaryMonitor());

            for (auto dlg : m_secondaryScreenDlgList) {
                dlg->deleteLater();
            }
            m_secondaryScreenDlgList.clear();
        }

        m_displayModeIndex = m_modeCombox->currentIndex();
        qInfo() << " User end set mode : " << mode << " , m_displayModeIndex :" << m_displayModeIndex;
    });
    connect(m_model, &DisplayModel::primaryScreenChanged, this, [=](const QString &name) {
        for (int idx = 0; idx < m_primaryCombox->count(); ++idx) {
            if (name == m_primaryCombox->itemText(idx)) {
                m_primaryCombox->blockSignals(true);
                m_primaryCombox->setCurrentIndex(idx);
                m_primaryCombox->blockSignals(false);
                break;
            }
        }

        if (m_model->displayMode() == MERGE_MODE) {
            m_brightnessWidget->showBrightness();
        } else if (m_model->displayMode() == EXTEND_MODE) {
            m_brightnessWidget->showBrightness(m_model->primaryMonitor());
        } else if (m_model->displayMode() == SINGLE_MODE) {
            m_monitorControlWidget->setModel(m_model, m_model->primaryMonitor());
            m_brightnessWidget->showBrightness(m_model->primaryMonitor());
            auto monitorList = m_model->monitorList();
            for (int idx = 0; idx < monitorList.size(); ++idx) {
                auto monitor = monitorList[idx];
                if (monitor->enable()) {
                    setModeCurrentIndex(monitor->name());
                    break;
                }
            }
        }

        m_resolutionWidget->setMonitor(m_model->primaryMonitor());
        m_refreshRateWidget->setMonitor(m_model->primaryMonitor());
        m_rotateWidget->setMonitor(m_model->primaryMonitor());

        for (const auto &monitor : m_model->monitorList()) {
            if (name == monitor->name()) {
                for (auto screen : QGuiApplication::screens()) {
                    disconnect(screen, &QScreen::geometryChanged, this, &MultiScreenWidget::onResetFullIndication);
                }
                QScreen *screen = m_model->primaryMonitor()->getQScreen();
                if(!screen)
                    continue;

                connect(screen, &QScreen::geometryChanged, this, &MultiScreenWidget::onResetFullIndication);

                m_fullIndication->setGeometry(screen->geometry());
                m_fullIndication->move(screen->geometry().topLeft());
                m_fullIndication->setVisible(true);
                QTimer::singleShot(1000, this, [=] { m_fullIndication->setVisible(false); });
                m_brightnessSpacerItem->changeSize(0, monitor->canBrightness() ? 20 : 0);
                m_brightnessWidget->setVisible(monitor->canBrightness());
                break;
            }
        }

        Q_EMIT requestGatherEnabled(true);
        initSecondaryScreenDialog();
    });
    connect(m_model, &DisplayModel::resolutionRefreshEnableChanged, this, [=](const bool enable) {
        Q_UNUSED(enable)
        onGSettingsChanged("displayMultipleDisplays", GSettingWatcher::instance()->getStatus("displayMultipleDisplays"));
    });
    connect(m_model, &DisplayModel::brightnessEnableChanged, this, [=](const bool enable) {
        const bool visible = enable && m_model->primaryMonitor() && m_model->primaryMonitor()->canBrightness();
        m_brightnessSpacerItem->changeSize(0, visible ? 20 : 0);
        m_brightnessWidget->setVisible(visible);
    });

    connect(m_monitorControlWidget, &MonitorControlWidget::requestMonitorPress, this, &MultiScreenWidget::onMonitorPress);
    connect(m_monitorControlWidget, &MonitorControlWidget::requestMonitorRelease, this, &MultiScreenWidget::onMonitorRelease);
    connect(m_monitorControlWidget, &MonitorControlWidget::requestRecognize, this, &MultiScreenWidget::requestRecognize);
    connect(m_monitorControlWidget, &MonitorControlWidget::requestSetMonitorPosition, this, &MultiScreenWidget::onRequestSetMonitorPosition);
    connect(m_monitorControlWidget, &MonitorControlWidget::requestGatherWindows, this, &MultiScreenWidget::onGatherWindows);
    connect(this, &MultiScreenWidget::requestGatherEnabled, m_monitorControlWidget, &MonitorControlWidget::onGatherEnabled);

    connect(m_modeCombox, static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged), this, [=](int idx) {
        if (idx <= 1 && m_model->displayMode() != idx + 1) {
            Q_EMIT requestSwitchMode(idx + 1);
        } else if (idx > 1 && (m_model->displayMode() != SINGLE_MODE || (m_model->monitorList()[idx - 2]->name() != m_model->primary() && !m_model->primary().isEmpty()))) {
            Q_EMIT requestSwitchMode(SINGLE_MODE, m_model->monitorList()[idx - 2]->name());
        }

        //用户设置暂不生效，待后端数据返回后再进行设置
        m_modeCombox->blockSignals(true);
        m_modeCombox->setCurrentIndex(m_displayModeIndex);
        m_modeCombox->blockSignals(false);
    });
    connect(m_primaryCombox, static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged), this, [=](int idx) {
        Q_EMIT requestSetPrimary(m_primaryCombox->itemText(idx));
    });

    connect(m_brightnessWidget, &BrightnessWidget::requestSetColorTemperature, this, &MultiScreenWidget::requestSetColorTemperature);
    connect(m_brightnessWidget, &BrightnessWidget::requestSetMonitorBrightness, this, &MultiScreenWidget::requestSetMonitorBrightness);
    connect(m_brightnessWidget, &BrightnessWidget::requestAmbientLightAdjustBrightness, this, &MultiScreenWidget::requestAmbientLightAdjustBrightness);
    connect(m_brightnessWidget, &BrightnessWidget::requestSetMethodAdjustCCT, this, &MultiScreenWidget::requestSetMethodAdjustCCT);
    connect(m_scalingWidget, &ScalingWidget::requestUiScaleChange, this, &MultiScreenWidget::requestUiScaleChange);
    connect(m_resolutionWidget, &ResolutionWidget::requestSetResolution, this, &MultiScreenWidget::requestSetResolution);
    connect(m_resolutionWidget, &ResolutionWidget::requestSetFillMode, this, &MultiScreenWidget::requestSetFillMode);
    connect(m_resolutionWidget, &ResolutionWidget::requestCurrFillModeChanged, this, &MultiScreenWidget::requestCurrFillModeChanged);
    connect(m_refreshRateWidget, &RefreshRateWidget::requestSetResolution, this, &MultiScreenWidget::requestSetResolution);
    connect(m_rotateWidget, &RotateWidget::requestSetRotate, this, &MultiScreenWidget::requestSetRotate);

    connect(this, &MultiScreenWidget::requestRecognize, this, &MultiScreenWidget::onRequestRecognize);

    m_monitorControlWidget->setScreensMerged(m_model->displayMode());
    m_monitorControlWidget->setModel(m_model, m_model->displayMode() == SINGLE_MODE ? m_model->primaryMonitor() : nullptr);

    onGSettingsChanged("displayMultipleDisplays", GSettingWatcher::instance()->getStatus("displayMultipleDisplays"));

    m_brightnessWidget->setMode(m_model);
    m_brightnessWidget->showBrightness(m_model->displayMode() == MERGE_MODE ? nullptr : m_model->primaryMonitor());
    const bool brightnessIsEnabled = m_model->brightnessEnable() && m_model->primaryMonitor() && m_model->primaryMonitor()->canBrightness();
    m_brightnessWidget->setVisible(brightnessIsEnabled);
    m_brightnessSpacerItem->changeSize(0, brightnessIsEnabled ? 20 : 0);
    m_scalingWidget->setModel(m_model);
    m_resolutionWidget->setModel(m_model, m_model->primaryMonitor());
    m_refreshRateWidget->setModel(m_model, m_model->primaryMonitor());
    m_rotateWidget->setModel(m_model, m_model->primaryMonitor());
    m_displayModeIndex = m_modeCombox->currentIndex();

    initSecondaryScreenDialog();
}

void MultiScreenWidget::onGSettingsChanged(const QString &gsettingsName, const QString &setting)
{
    if (m_model && gsettingsName == "displayMultipleDisplays" && !setting.isEmpty()) {
        bool multipleDisplays = m_model->resolutionRefreshEnable() && (setting != "Hidden");
        m_multiSettingLabel->setVisible(multipleDisplays);
        m_modeSettingsItem->setVisible(multipleDisplays);
        m_primarySettingsItem->setVisible(multipleDisplays && m_model->displayMode() == EXTEND_MODE);
        bool multipleEnable = m_model->resolutionRefreshEnable() && (setting != "Disabled");
        m_multiSettingLabel->setEnabled(multipleEnable);
        m_modeSettingsItem->setEnabled(multipleEnable);
        m_primarySettingsItem->setEnabled(multipleEnable);
    }
}

void MultiScreenWidget::initModeList()
{
    m_modeCombox->addItem(tr("Duplicate"), "Duplicate");
    m_modeCombox->addItem(tr("Extend"), "Extend");

    auto monitorList = m_model->monitorList();
    for (int idx = 0; idx < monitorList.size(); ++idx) {
        auto monitor = monitorList[idx];
        connect(monitor, &Monitor::enableChanged, this, [ this ](bool enable){
            Q_UNUSED(enable)
            if(m_model->displayMode() == SINGLE_MODE){
                for (auto moni : m_model->monitorList()) {
                    if(moni->enable()){
                        setModeCurrentIndex(moni->name());
                    }
                    break;
                }
            }
        });

        if(monitorList.size() <= 2)
            m_modeCombox->addItem(tr("Only on %1").arg(monitor->name()), monitor->name());

        int displayMode = m_model->displayMode();
        if (displayMode == MERGE_MODE) {
            m_modeCombox->blockSignals(true);
            m_modeCombox->setCurrentIndex(0);
            m_modeCombox->blockSignals(false);
        } else if (displayMode == EXTEND_MODE) {
            m_modeCombox->blockSignals(true);
            m_modeCombox->setCurrentIndex(1);
            m_modeCombox->blockSignals(false);
        } else if (displayMode == SINGLE_MODE && monitor->enable()) {
            setModeCurrentIndex(monitor->name());
        }
    }
}

void MultiScreenWidget::initPrimaryList()
{
    for (const auto &monitor : m_model->monitorList()) {
        m_primaryCombox->addItem(monitor->name());
        if (monitor->name() == m_model->primary()) {
            m_primaryCombox->setCurrentText(m_model->primary());
        }
    }
}

void MultiScreenWidget::initSecondaryScreenDialog()
{
    // 在每次显示模式切换时，先清空m_secondaryScreenDlgList
    for (auto dlg : m_secondaryScreenDlgList) {
        dlg->deleteLater();
    }

    m_secondaryScreenDlgList.clear();

    if (m_model->displayMode() == EXTEND_MODE) {
        for (const auto &monitor : m_model->monitorList()) {
            if (monitor == m_model->primaryMonitor()) {
                requestSetMainwindowRect(m_model->primaryMonitor(), true);
                continue;
            }

            SecondaryScreenDialog *dlg = new SecondaryScreenDialog(this);
            dlg->setAttribute(Qt::WA_WState_WindowOpacitySet);
            dlg->setModel(m_model, monitor);
            connect(dlg, &SecondaryScreenDialog::requestRecognize, this, &MultiScreenWidget::requestRecognize);
            connect(dlg, &SecondaryScreenDialog::requestSetMonitorBrightness, this, &MultiScreenWidget::requestSetMonitorBrightness);
            connect(dlg, &SecondaryScreenDialog::requestAmbientLightAdjustBrightness, this, &MultiScreenWidget::requestAmbientLightAdjustBrightness);
            connect(dlg, &SecondaryScreenDialog::requestSetResolution, this, &MultiScreenWidget::requestSetResolution);
            connect(dlg, &SecondaryScreenDialog::requestSetFillMode, this, &MultiScreenWidget::requestSetFillMode);
            connect(dlg, &SecondaryScreenDialog::requestCurrFillModeChanged, this, &MultiScreenWidget::requestCurrFillModeChanged);
            connect(dlg, &SecondaryScreenDialog::requestSetRotate, this, &MultiScreenWidget::requestSetRotate);
            connect(dlg, &SecondaryScreenDialog::requestGatherWindows, this, &MultiScreenWidget::onGatherWindows);
            connect(dlg, &SecondaryScreenDialog::requestCloseRecognize, this, &MultiScreenWidget::onRequestCloseRecognize);
            connect(this, &MultiScreenWidget::requestGatherEnabled, dlg, &SecondaryScreenDialog::requestGatherEnabled);
            m_secondaryScreenDlgList.append(dlg);

            // 要先移动后显示
            dlg->resetDialog();
        }
        activateWindow();

        if (!qgetenv("WAYLAND_DISPLAY").isEmpty()) {
            QTimer::singleShot(10, this, [=] {
                for (auto dlg : m_secondaryScreenDlgList) {
                    dlg->setWindowFlags(Qt::CoverWindow);
                    dlg->show();
                    QTimer::singleShot(50, dlg, [dlg] { // 此处延时刷新子窗口位置，是因为子窗口在主屏闪现的问题(bug153993)，待窗管处理之后回退
                        dlg->resetDialog();
                    });
                }
            });
        }
    }
}

void MultiScreenWidget::onGatherWindows(const QPoint cursor)
{
    Q_EMIT requestGatherEnabled(false);

    for (const auto &monitor : m_model->monitorList()) {
        auto screen = monitor->getQScreen();
        auto mrt = screen->geometry();

        if (mrt.contains(cursor.x(), cursor.y())) {
            for (QWidget *w : qApp->topLevelWidgets()) {
                if (DMainWindow *mainWin = qobject_cast<DMainWindow *>(w)) {
                    auto rt = mainWin->rect();
                    if (rt.width() > screen->geometry().width())
                        rt.setWidth(screen->geometry().width());

                    if (rt.height() > screen->geometry().height())
                        rt.setHeight(screen->geometry().height());

                    auto tsize = (mrt.size() - rt.size()) / 2;
                    rt.moveTo(screen->geometry().topLeft().x() + tsize.width(), screen->geometry().topLeft().y() + tsize.height());
                    mainWin->setGeometry(rt);
                }
            }

            for (auto dlg : m_secondaryScreenDlgList) {
                auto rt = dlg->rect();
                if (rt.width() > screen->geometry().width())
                    rt.setWidth(screen->geometry().width());

                if (rt.height() > screen->geometry().height())
                    rt.setHeight(screen->geometry().height());

                auto tsize = (mrt.size() - rt.size()) / 2;
                rt.moveTo(screen->geometry().topLeft().x() + tsize.width(), screen->geometry().topLeft().y() + tsize.height());
                dlg->QDialog::setGeometry(rt);
                // 将副窗口置顶
                dlg->activateWindow();
            }
            break;
        }
    }
}

void MultiScreenWidget::onMonitorPress(Monitor *monitor)
{
    QScreen *screen = monitor->getQScreen();
    if(!screen)
        return;

    m_fullIndication->setGeometry(screen->geometry());
    m_fullIndication->move(screen->geometry().topLeft());
    m_fullIndication->setVisible(true);

    QTimer::singleShot(1000, this, [=] { m_fullIndication->setVisible(false); });
}

void MultiScreenWidget::onMonitorRelease(Monitor *monitor)
{
    Q_UNUSED(monitor)
    m_fullIndication->setVisible(false);
    QTimer::singleShot(1000, this, [=] { requestSetMainwindowRect(m_model->primaryMonitor(), false); });
}

void MultiScreenWidget::onRequestSetMonitorPosition(QHash<dcc::display::Monitor *, QPair<int, int>> monitorPosition)
{
    Q_EMIT requestSetMonitorPosition(monitorPosition);
}

void MultiScreenWidget::onResetFullIndication(const QRect &geometry)
{
    m_fullIndication->setGeometry(geometry);
    m_fullIndication->move(geometry.topLeft());
}

void MultiScreenWidget::onResetSecondaryScreenDlg()
{
    for (int i = 0; i < m_secondaryScreenDlgList.count(); ++i) {
        SecondaryScreenDialog *screenDialog = m_secondaryScreenDlgList.at(i);
        Q_ASSERT(screenDialog);
        screenDialog->setWindowOpacity(1);
    }
}

void MultiScreenWidget::onRequestRecognize()
{
    for(auto widget : m_recognizeWidget) {
        widget->deleteLater();
    }
    m_recognizeWidget.clear();

    // 复制模式
    if (m_model->displayMode() == MERGE_MODE) {
        QString text = m_model->monitorList().first()->name();
        for (int idx = 1; idx < m_model->monitorList().size(); idx++) {
            text += QString(" = %1").arg(m_model->monitorList()[idx]->name());
        }

        // 所在显示器不存在显示框
        if (m_recognizeWidget.value(text) == nullptr) {
            dcc::display::RecognizeWidget *widget = new dcc::display::RecognizeWidget(m_model->monitorList()[0], text);
            m_recognizeWidget[text] = widget;
        }
    } else { // 扩展模式
        for (auto monitor : m_model->monitorList()) {
            // 所在显示器不存在显示框
            if (m_recognizeWidget.value(monitor->name()) == nullptr) {
                dcc::display::RecognizeWidget *widget = new dcc::display::RecognizeWidget(monitor, monitor->name());
                m_recognizeWidget[monitor->name()] = widget;
            }
        }
    }

    this->setFocus(); //获取焦点响应键盘事件
}

void MultiScreenWidget::onRequestCloseRecognize()
{
    disconnect(this, &MultiScreenWidget::requestRecognize, this, &MultiScreenWidget::onRequestRecognize);

    for(auto widget : m_recognizeWidget) {
        widget->deleteLater();
    }
    m_recognizeWidget.clear();

    connect(this, &MultiScreenWidget::requestRecognize, this, &MultiScreenWidget::onRequestRecognize);
}

void MultiScreenWidget::keyPressEvent(QKeyEvent *e)
{
    if(e->key() == Qt::Key_Escape) {
        onRequestCloseRecognize();
    }

    QWidget::keyPressEvent(e);
}

void MultiScreenWidget::setModeCurrentIndex(const QString &monitorName)
{
    if(monitorName.isEmpty())
        return;

    qDebug() << Q_FUNC_INFO << monitorName;
    int index = m_modeCombox->findData(monitorName);
    if (index != -1) {
        m_modeCombox->blockSignals(true);
        m_modeCombox->setCurrentIndex(index);
        m_modeCombox->blockSignals(false);
    }
}

void MultiScreenWidget::onMainwindowStateChanged(int type)
{
    qDebug() << Q_FUNC_INFO << type;
    if(QEvent::Show == type || QEvent::Hide == type ){
        for (auto item : m_secondaryScreenDlgList) {
            if(item != nullptr)
                item->setVisible(QEvent::Show == type);
        }
    }
}
