#ifndef __ASM_SYSTEM_H
#define __ASM_SYSTEM_H

#include <xen/lib.h>
#include <xen/bitops.h>
#include <asm/processor.h>

#define read_sreg(name)                                         \
({  unsigned int __sel;                                         \
    asm volatile ( "mov %%" STR(name) ",%0" : "=r" (__sel) );   \
    __sel;                                                      \
})

#define wbinvd() \
    asm volatile ( "wbinvd" : : : "memory" )

#define clflush(a) \
    asm volatile ( "clflush (%0)" : : "r"(a) )

#define nop() \
    asm volatile ( "nop" )

#define xchg(ptr,v) \
    ((__typeof__(*(ptr)))__xchg((unsigned long)(v),(ptr),sizeof(*(ptr))))

struct __xchg_dummy { unsigned long a[100]; };
#define __xg(x) ((volatile struct __xchg_dummy *)(x))

#include <asm/x86_64/system.h>

/*
 * Note: no "lock" prefix even on SMP: xchg always implies lock anyway
 * Note 2: xchg has side effect, so that attribute volatile is necessary,
 *   but generally the primitive is invalid, *ptr is output argument. --ANK
 */
static always_inline unsigned long __xchg(
    unsigned long x, volatile void *ptr, int size)
{
    switch ( size )
    {
    case 1:
        asm volatile ( "xchgb %b0,%1"
                       : "=q" (x)
                       : "m" (*__xg(ptr)), "0" (x)
                       : "memory" );
        break;
    case 2:
        asm volatile ( "xchgw %w0,%1"
                       : "=r" (x)
                       : "m" (*__xg(ptr)), "0" (x)
                       : "memory" );
        break;
    case 4:
        asm volatile ( "xchgl %k0,%1"
                       : "=r" (x)
                       : "m" (*__xg(ptr)), "0" (x)
                       : "memory" );
        break;
    case 8:
        asm volatile ( "xchgq %0,%1"
                       : "=r" (x)
                       : "m" (*__xg(ptr)), "0" (x)
                       : "memory" );
        break;
    }
    return x;
}

/*
 * Atomic compare and exchange.  Compare OLD with MEM, if identical,
 * store NEW in MEM.  Return the initial value in MEM.  Success is
 * indicated by comparing RETURN with OLD.
 */

static always_inline unsigned long __cmpxchg(
    volatile void *ptr, unsigned long old, unsigned long new, int size)
{
    unsigned long prev;
    switch ( size )
    {
    case 1:
        asm volatile ( "lock; cmpxchgb %b1,%2"
                       : "=a" (prev)
                       : "q" (new), "m" (*__xg(ptr)),
                       "0" (old)
                       : "memory" );
        return prev;
    case 2:
        asm volatile ( "lock; cmpxchgw %w1,%2"
                       : "=a" (prev)
                       : "r" (new), "m" (*__xg(ptr)),
                       "0" (old)
                       : "memory" );
        return prev;
    case 4:
        asm volatile ( "lock; cmpxchgl %k1,%2"
                       : "=a" (prev)
                       : "r" (new), "m" (*__xg(ptr)),
                       "0" (old)
                       : "memory" );
        return prev;
    case 8:
        asm volatile ( "lock; cmpxchgq %1,%2"
                       : "=a" (prev)
                       : "r" (new), "m" (*__xg(ptr)),
                       "0" (old)
                       : "memory" );
        return prev;
    }
    return old;
}

static always_inline unsigned long cmpxchg_local_(
    void *ptr, unsigned long old, unsigned long new, unsigned int size)
{
    unsigned long prev = ~old;

    switch ( size )
    {
    case 1:
        asm volatile ( "cmpxchgb %b2, %1"
                       : "=a" (prev), "+m" (*(uint8_t *)ptr)
                       : "q" (new), "0" (old) );
        break;
    case 2:
        asm volatile ( "cmpxchgw %w2, %1"
                       : "=a" (prev), "+m" (*(uint16_t *)ptr)
                       : "r" (new), "0" (old) );
        break;
    case 4:
        asm volatile ( "cmpxchgl %k2, %1"
                       : "=a" (prev), "+m" (*(uint32_t *)ptr)
                       : "r" (new), "0" (old) );
        break;
    case 8:
        asm volatile ( "cmpxchgq %2, %1"
                       : "=a" (prev), "+m" (*(uint64_t *)ptr)
                       : "r" (new), "0" (old) );
        break;
    }

    return prev;
}

#define cmpxchgptr(ptr,o,n) ({                                          \
    const __typeof__(**(ptr)) *__o = (o);                               \
    __typeof__(**(ptr)) *__n = (n);                                     \
    ((__typeof__(*(ptr)))__cmpxchg((ptr),(unsigned long)__o,            \
                                   (unsigned long)__n,sizeof(*(ptr)))); \
})

/*
 * Undefined symbol to cause link failure if a wrong size is used with
 * arch_fetch_and_add().
 */
extern unsigned long __bad_fetch_and_add_size(void);

static always_inline unsigned long __xadd(
    volatile void *ptr, unsigned long v, int size)
{
    switch ( size )
    {
    case 1:
        asm volatile ( "lock; xaddb %b0,%1"
                       : "+r" (v), "+m" (*__xg(ptr))
                       :: "memory");
        return v;
    case 2:
        asm volatile ( "lock; xaddw %w0,%1"
                       : "+r" (v), "+m" (*__xg(ptr))
                       :: "memory");
        return v;
    case 4:
        asm volatile ( "lock; xaddl %k0,%1"
                       : "+r" (v), "+m" (*__xg(ptr))
                       :: "memory");
        return v;
    case 8:
        asm volatile ( "lock; xaddq %q0,%1"
                       : "+r" (v), "+m" (*__xg(ptr))
                       :: "memory");

        return v;
    default:
        return __bad_fetch_and_add_size();
    }
}

/*
 * Atomically add @v to the 1, 2, 4, or 8 byte value at @ptr.  Returns
 * the previous value.
 *
 * This is a full memory barrier.
 */
#define arch_fetch_and_add(ptr, v) \
    ((typeof(*(ptr)))__xadd(ptr, (typeof(*(ptr)))(v), sizeof(*(ptr))))

/*
 * Mandatory barriers, for enforced ordering of reads and writes, e.g. for use
 * with MMIO devices mapped with reduced cacheability.
 */
#define mb()            asm volatile ( "mfence" ::: "memory" )
#define rmb()           asm volatile ( "lfence" ::: "memory" )
#define wmb()           asm volatile ( "sfence" ::: "memory" )

/*
 * SMP barriers, for ordering of reads and writes between CPUs, most commonly
 * used with shared memory.
 *
 * Both Intel and AMD agree that, from a programmer's viewpoint:
 *  Loads cannot be reordered relative to other loads.
 *  Stores cannot be reordered relative to other stores.
 *  Loads may be reordered ahead of a unaliasing stores.
 *
 * Refer to the vendor system programming manuals for further details.
 */
#define smp_mb()        mb()
#define smp_rmb()       barrier()
#define smp_wmb()       barrier()

#define set_mb(var, value) do { xchg(&var, value); } while (0)
#define set_wmb(var, value) do { var = value; smp_wmb(); } while (0)

/**
 * array_index_mask_nospec() - generate a mask that is ~0UL when the
 *      bounds check succeeds and 0 otherwise
 * @index: array element index
 * @size: number of elements in array
 *
 * Returns:
 *     0 - (index < size)
 */
static inline unsigned long array_index_mask_nospec(unsigned long index,
                                                    unsigned long size)
{
    unsigned long mask;

    asm volatile ( "cmp %[size], %[index]; sbb %[mask], %[mask];"
                   : [mask] "=r" (mask)
                   : [size] "g" (size), [index] "r" (index) );

    return mask;
}

/* Override default implementation in nospec.h. */
#define array_index_mask_nospec array_index_mask_nospec

#define local_irq_disable()     asm volatile ( "cli" : : : "memory" )
#define local_irq_enable()      asm volatile ( "sti" : : : "memory" )

/* used in the idle loop; sti takes one instruction cycle to complete */
#define safe_halt()     asm volatile ( "sti; hlt" : : : "memory" )
/* used when interrupts are already enabled or to shutdown the processor */
#define halt()          asm volatile ( "hlt" : : : "memory" )

#define local_save_flags(x)                                      \
({                                                               \
    BUILD_BUG_ON(sizeof(x) != sizeof(long));                     \
    asm volatile ( "pushf" __OS " ; pop" __OS " %0" : "=g" (x)); \
})
#define local_irq_save(x)                                        \
({                                                               \
    local_save_flags(x);                                         \
    local_irq_disable();                                         \
})
#define local_irq_restore(x)                                     \
({                                                               \
    BUILD_BUG_ON(sizeof(x) != sizeof(long));                     \
    asm volatile ( "pushfq\n\t"                                  \
                   "andq %0, (%%rsp)\n\t"                        \
                   "orq  %1, (%%rsp)\n\t"                        \
                   "popfq"                                       \
                   : : "i?r" ( ~X86_EFLAGS_IF ),                 \
                       "ri" ( (x) & X86_EFLAGS_IF ) );           \
})

static inline int local_irq_is_enabled(void)
{
    unsigned long flags;
    local_save_flags(flags);
    return !!(flags & X86_EFLAGS_IF);
}

#define BROKEN_ACPI_Sx          0x0001
#define BROKEN_INIT_AFTER_S1    0x0002

void trap_init(void);
void init_idt_traps(void);
void load_system_tables(void);
void percpu_traps_init(void);
void subarch_percpu_traps_init(void);

#endif
