/*
 * Decompiled with CFR 0.152.
 */
package org.glassfish.cluster.ssh.sftp;

import com.jcraft.jsch.ChannelSftp;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.SftpATTRS;
import com.jcraft.jsch.SftpException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.glassfish.cluster.ssh.launcher.SSHException;
import org.glassfish.cluster.ssh.sftp.SFTPPath;

public class SFTPClient
implements AutoCloseable {
    private static final System.Logger LOG = System.getLogger(SFTPClient.class.getName());
    private static final Predicate<ChannelSftp.LsEntry> PREDICATE_NO_DOTS = p -> !".".equals(p.getFilename()) && !"..".equals(p.getFilename());
    private final ChannelSftp sftpChannel;

    public SFTPClient(ChannelSftp channel) throws SSHException {
        this.sftpChannel = channel;
        try {
            this.sftpChannel.connect();
        }
        catch (JSchException e) {
            throw new SSHException("Failed to connect to the SFTP server. Is it correctly configured on the server?", e);
        }
    }

    @Override
    public void close() {
        if (this.sftpChannel != null) {
            this.sftpChannel.disconnect();
        }
    }

    public SFTPPath getHome() throws SSHException {
        try {
            return SFTPPath.of(this.sftpChannel.getHome());
        }
        catch (SftpException e) {
            throw new SSHException("Could not resolve SFTP Home path.", e);
        }
    }

    public void mkdirs(SFTPPath path) throws SSHException {
        if (this.existsDirectory(path)) {
            return;
        }
        SFTPPath current = SFTPPath.ofAbsolutePath(new String[0]);
        for (Path part : path) {
            if (this.existsDirectory(current = current.resolve(part))) continue;
            try {
                this.sftpChannel.mkdir(current.toString());
            }
            catch (SftpException e) {
                throw new SSHException("Failed to create the directory " + String.valueOf(path) + ".", e);
            }
        }
    }

    public boolean existsDirectory(SFTPPath path) throws SSHException {
        SftpATTRS attrs = this.stat(path);
        return attrs != null && attrs.isDir();
    }

    public boolean isEmptyDirectory(SFTPPath path) throws SSHException {
        SftpATTRS attrs = this.stat(path);
        return attrs != null && attrs.isDir() && this.ls(path, e -> true).isEmpty();
    }

    public void rmDir(SFTPPath path, boolean onlyContent, SFTPPath ... exclude) throws SSHException {
        if (!this.exists(path)) {
            return;
        }
        this.cd(path.getParent());
        List<ChannelSftp.LsEntry> content = this.lsDetails(path, p -> true);
        for (ChannelSftp.LsEntry entry : content) {
            String filename = entry.getFilename();
            SFTPPath entryPath = path.resolve(filename);
            if (SFTPClient.isExcludedFromDeletion(filename, exclude)) {
                LOG.log(System.Logger.Level.TRACE, "Skipping excluded {0}", entryPath);
                continue;
            }
            if (entry.getAttrs().isDir()) {
                this.rmDir(entryPath, false, SFTPClient.getSubDirectoryExclusions(filename, exclude));
                continue;
            }
            LOG.log(System.Logger.Level.TRACE, "Deleting file {0}", entryPath);
            this.rm(entryPath);
        }
        if (!onlyContent) {
            try {
                this.sftpChannel.cd(path.getParent().toString());
                LOG.log(System.Logger.Level.TRACE, "Deleting directory {0}", path);
                this.sftpChannel.rmdir(path.toString());
            }
            catch (SftpException e) {
                throw new SSHException("Failed to delete directory: " + String.valueOf(path) + ".", e);
            }
        }
    }

    private static boolean isExcludedFromDeletion(String firstName, SFTPPath ... exclusions) {
        if (exclusions == null) {
            return false;
        }
        return Arrays.stream(exclusions).filter(p -> p.getNameCount() == 1).anyMatch(p -> p.getFileName().toString().equals(firstName));
    }

    private static SFTPPath[] getSubDirectoryExclusions(String firstName, SFTPPath ... exclusions) {
        if (exclusions == null) {
            return new SFTPPath[0];
        }
        return (SFTPPath[])Arrays.stream(exclusions).filter(p -> p.getNameCount() > 1).filter(p -> p.startsWith(firstName)).map(p -> p.subpath(1, p.getNameCount())).toArray(SFTPPath[]::new);
    }

    public void put(File localFile, SFTPPath remoteFile) throws SSHException {
        try {
            this.sftpChannel.cd(remoteFile.getParent().toString());
            this.sftpChannel.put(localFile.getAbsolutePath(), remoteFile.toString());
        }
        catch (SftpException e) {
            throw new SSHException("Failed to upload the local file " + String.valueOf(localFile) + " to remote file " + String.valueOf(remoteFile) + ".", e);
        }
    }

    public void download(SFTPPath remoteFile, Path localFile) throws SSHException {
        try (InputStream inputStream = this.sftpChannel.get(remoteFile.toString());){
            Files.copy(inputStream, localFile, new CopyOption[0]);
        }
        catch (SftpException | IOException e) {
            throw new SSHException("Failed to download the remote file " + String.valueOf(remoteFile) + " to local file " + String.valueOf(localFile) + ".", e);
        }
    }

    public void rm(SFTPPath path) throws SSHException {
        try {
            this.sftpChannel.cd(path.getParent().toString());
            this.sftpChannel.rm(path.toString());
        }
        catch (SftpException e) {
            throw new SSHException("Failed to remove path " + String.valueOf(path) + ".", e);
        }
    }

    public void mv(SFTPPath remoteSource, SFTPPath remoteTarget) throws SSHException {
        try {
            this.sftpChannel.cd(remoteTarget.getParent().toString());
            this.sftpChannel.rename(remoteSource.toString(), remoteTarget.toString());
        }
        catch (SftpException e) {
            throw new SSHException("Failed to move the directory " + String.valueOf(remoteSource) + " to " + String.valueOf(remoteTarget) + ".", e);
        }
    }

    public boolean exists(SFTPPath path) throws SSHException {
        return this.stat(path) != null;
    }

    public SftpATTRS stat(SFTPPath path) throws SSHException {
        try {
            return this.sftpChannel.stat(path.toString());
        }
        catch (SftpException e) {
            if (e.id == 2) {
                return null;
            }
            throw new SSHException("Failed to call SFTP stat for " + String.valueOf(path) + ".", e);
        }
    }

    public SftpATTRS lstat(SFTPPath path) throws SSHException {
        try {
            return this.sftpChannel.lstat(path.toString());
        }
        catch (SftpException e) {
            if (e.id == 2) {
                return null;
            }
            throw new SSHException("Failed to call SFTP lstat for " + String.valueOf(path) + ".", e);
        }
    }

    public void setTimeModified(SFTPPath path, long millisSinceUnixEpoch) throws SSHException {
        try {
            this.sftpChannel.setMtime(path.toString(), (int)(millisSinceUnixEpoch / 1000L));
        }
        catch (SftpException e) {
            throw new SSHException("Failed to set time modification for path " + String.valueOf(path) + ".", e);
        }
    }

    public void chmod(SFTPPath path, int permissions) throws SSHException {
        try {
            this.sftpChannel.chmod(permissions, path.toString());
        }
        catch (SftpException e) {
            throw new SSHException("Failed to call chmod for remote path " + String.valueOf(path) + " and permissions " + permissions + ".", e);
        }
    }

    public void cd(SFTPPath path) throws SSHException {
        try {
            this.sftpChannel.cd(path.toString());
        }
        catch (SftpException e) {
            throw new SSHException("Failed to change the remote directory to " + String.valueOf(path) + ".", e);
        }
    }

    public List<String> ls(SFTPPath path, Predicate<ChannelSftp.LsEntry> filter) throws SSHException {
        try {
            return this.sftpChannel.ls(path.toString()).stream().filter(filter.and(PREDICATE_NO_DOTS)).map(ChannelSftp.LsEntry::getFilename).collect(Collectors.toList());
        }
        catch (SftpException e) {
            throw new SSHException("Failed to list remote directory " + String.valueOf(path) + ".", e);
        }
    }

    public List<ChannelSftp.LsEntry> lsDetails(SFTPPath path, Predicate<ChannelSftp.LsEntry> filter) throws SSHException {
        try {
            return this.sftpChannel.ls(path.toString()).stream().filter(filter.and(PREDICATE_NO_DOTS)).collect(Collectors.toList());
        }
        catch (SftpException e) {
            throw new SSHException("Failed to list remote directory " + String.valueOf(path) + ".", e);
        }
    }
}

