/*
 * Copyright (c) 2012, 2014, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 * $Id: JAXRSClient.java 73513 2014-02-21 12:29:14Z supol $
 */
package com.sun.ts.tests.jaxrs.api.rs.internalservererrorexception;

import java.io.IOException;

import javax.ws.rs.InternalServerErrorException;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import com.sun.ts.tests.jaxrs.common.JAXRSCommonClient;

/*
 * @class.setup_props: webServerHost;
 *                     webServerPort;
 *                     ts_home;
 */
public class JAXRSClient extends JAXRSCommonClient {

	private static final long serialVersionUID = -588383752025277814L;
	private static final Status STATUS = Status.INTERNAL_SERVER_ERROR;
	protected static final String MESSAGE = "TCK InternalServerErrorException description";
	protected static final String HOST = "www.jcp.org";

	/**
	 * Entry point for different-VM execution. It should delegate to method
	 * run(String[], PrintWriter, PrintWriter), and this method should not
	 * contain any test configuration.
	 */
	public static void main(String[] args) {
		new JAXRSClient().run(args);
	}

	/* Run test */

	/*
	 * @testName: constructorTest
	 * @assertion_ids: JAXRS:JAVADOC:319;
	 * 					JAXRS:JAVADOC:12;
	 * @test_Strategy: Construct a new internal server error exception.
	 * 
	 * 		getResponse
	 */
	public void constructorTest() throws Fault {
		InternalServerErrorException e = new InternalServerErrorException();
		assertResponse(e);
	}

	/*
	 * @testName: constructorResponseTest
	 * @assertion_ids: JAXRS:JAVADOC:320;
	 * 					JAXRS:JAVADOC:12;
	 * @test_Strategy: Construct a new internal server error exception.
	 * 					java.lang.IllegalArgumentException - in case the status
	 * 					code set in the response is not HTTP 500.
	 * 
	 * 		getResponse
	 */
	public void constructorResponseTest() throws Fault {
		Response response = buildResponse();
		InternalServerErrorException e = new InternalServerErrorException(
				response);
		assertResponse(e, HOST);
	}

	/*
	 * @testName: constructorResponseThrowsExceptionTest
	 * @assertion_ids: JAXRS:JAVADOC:320;
	 * @test_Strategy: Construct a new internal server error exception.
	 * 					java.lang.IllegalArgumentException - in case the status
	 * 					code set in the response is not HTTP 500.
	 */
	public void constructorResponseThrowsExceptionTest() throws Fault {
		for (Status status : Status.values())
			if (status != STATUS)
				try {
					Response response = Response.status(status).build();
					InternalServerErrorException e = new InternalServerErrorException(
							response);
					fault("No exception has been thrown for status", status,
							"; exception", e);
				} catch (IllegalArgumentException e) {
					logMsg("IllegalArgumentException has been thrown as expected for status",
							status);
				}
	}

	/*
	 * @testName: constructorThrowableTest
	 * @assertion_ids: JAXRS:JAVADOC:321;
	 * 					JAXRS:JAVADOC:12;
	 * @test_Strategy: Construct a new internal server error exception.
	 * 					cause - the underlying cause of the exception.
	 * 
	 * 		getResponse
	 */
	public void constructorThrowableTest() throws Fault {
		Throwable[] throwables = new Throwable[] { new RuntimeException(),
				new IOException(), new Error(), new Throwable() };
		for (Throwable t : throwables) {
			InternalServerErrorException e = new InternalServerErrorException(t);
			assertResponse(e);
			assertCause(e, t);
		}
	}

	/*
	 * @testName: constructorResponseThrowableTest
	 * @assertion_ids: JAXRS:JAVADOC:322;
	 * 					JAXRS:JAVADOC:12;
	 * @test_Strategy: Construct a new internal server error exception.
	 * 					java.lang.IllegalArgumentException - in case the status
	 * 					code set in the response is not HTTP 500.
	 * 
	 * 		getResponse
	 */
	public void constructorResponseThrowableTest() throws Fault {
		Response response = buildResponse();
		Throwable[] throwables = new Throwable[] { new RuntimeException(),
				new IOException(), new Error(), new Throwable() };
		for (Throwable t : throwables) {
			InternalServerErrorException e = new InternalServerErrorException(
					response, t);
			assertResponse(e, HOST);
			assertCause(e, t);
		}
	}

	/*
	 * @testName: constructorResponseThrowableThrowsExceptionTest
	 * @assertion_ids: JAXRS:JAVADOC:322;
	 * @test_Strategy: Construct a new internal server error exception.
	 * 					java.lang.IllegalArgumentException - in case the status
	 * 					code set in the response is not HTTP 500.
	 */
	public void constructorResponseThrowableThrowsExceptionTest() throws Fault {
		for (Status status : Status.values())
			if (status != STATUS)
				try {
					Response response = Response.status(status).build();
					InternalServerErrorException e = new InternalServerErrorException(
							response, new Throwable());
					fault("No exception has been thrown for status", status,
							"exception", e);
				} catch (IllegalArgumentException e) {
					logMsg("IllegalArgumentException has been thrown as expected for status",
							status);
				}
	}

	/*
	 * @testName: constructorStringTest
	 * @assertion_ids: JAXRS:JAVADOC:1070;
	 * @test_Strategy: message - the detail message (which is saved for later 
	 * 					retrieval by the Throwable.getMessage() method).
	 */
	public void constructorStringTest() throws Fault {
		InternalServerErrorException e = new InternalServerErrorException(
				MESSAGE);
		assertMessage(e);
		assertResponse(e);
	}

	/*
	 * @testName: constructorStringResponseTest
	 * @assertion_ids: JAXRS:JAVADOC:1071;
	 * 					JAXRS:JAVADOC:12;
	 * @test_Strategy: Construct a new internal server error exception.
	 * 
	 * 		getResponse
	 */
	public void constructorStringResponseTest() throws Fault {
		Response response = buildResponse();
		InternalServerErrorException e = new InternalServerErrorException(
				MESSAGE, response);
		assertResponse(e, HOST);
		assertMessage(e);
	}

	/*
	 * @testName: constructorStringResponseThrowsIAETest
	 * @assertion_ids: JAXRS:JAVADOC:1071;
	 * @test_Strategy: Construct a new internal server error exception.
	 * 					java.lang.IllegalArgumentException - in case the status
	 * 					code set in the response is not HTTP 500.
	 */
	public void constructorStringResponseThrowsIAETest() throws Fault {
		for (Status status : Status.values())
			if (status != STATUS)
				try {
					Response response = Response.status(status).build();
					InternalServerErrorException e = new InternalServerErrorException(
							MESSAGE, response);
					fault("No exception has been thrown for status", status,
							"; exception", e);
				} catch (IllegalArgumentException e) {
					logMsg("IllegalArgumentException has been thrown as expected for status",
							status);
				}
	}

	/*
	 * @testName: constructorStringThrowableTest
	 * @assertion_ids: JAXRS:JAVADOC:1072;
	 * 					JAXRS:JAVADOC:12;
	 * @test_Strategy: Construct a new internal server error exception.
	 * 
	 * 		getResponse
	 */
	public void constructorStringThrowableTest() throws Fault {
		Throwable[] throwables = new Throwable[] { new RuntimeException(),
				new IOException(), new Error(), new Throwable() };
		for (Throwable t : throwables) {
			InternalServerErrorException e = new InternalServerErrorException(
					MESSAGE, t);
			assertResponse(e);
			assertCause(e, t);
			assertMessage(e);
		}
	}

	/*
	 * @testName: constructorStringResponseThrowableTest
	 * @assertion_ids: JAXRS:JAVADOC:1073;
	 * 					JAXRS:JAVADOC:12;
	 * @test_Strategy: Construct a new internal server error exception.
	 * 
	 * 		getResponse
	 */
	public void constructorStringResponseThrowableTest() throws Fault {
		Response response = buildResponse();
		Throwable[] throwables = new Throwable[] { new RuntimeException(),
				new IOException(), new Error(), new Throwable() };
		for (Throwable t : throwables) {
			InternalServerErrorException e = new InternalServerErrorException(
					MESSAGE, response, t);
			assertResponse(e, HOST);
			assertCause(e, t);
			assertMessage(e);
		}
	}

	/*
	 * @testName: constructorStringResponseThrowableThrowsIEATest
	 * @assertion_ids: JAXRS:JAVADOC:1073;
	 * @test_Strategy: Construct a new internal server error exception.
	 * 					java.lang.IllegalArgumentException - in case the status
	 * 					code set in the response is not HTTP 500.
	 */
	public void constructorStringResponseThrowableThrowsIEATest() throws Fault {
		for (Status status : Status.values())
			if (status != STATUS)
				try {
					Response response = Response.status(status).build();
					InternalServerErrorException e = new InternalServerErrorException(
							MESSAGE, response, new Throwable());
					fault("No exception has been thrown for status", status,
							"exception", e);
				} catch (IllegalArgumentException e) {
					logMsg("IllegalArgumentException has been thrown as expected for status",
							status);
				}
	}

	// /////////////////////////////////////////////////////////////
	protected Response buildResponse() {
		Response r = Response.status(STATUS).header(HttpHeaders.HOST, HOST)
				.build();
		return r;
	}

	protected void assertResponse(InternalServerErrorException e) throws Fault {
		assertNotNull(e.getResponse(), "#getResponse is null");
		Response response = e.getResponse();
		assertEqualsInt(response.getStatus(), STATUS.getStatusCode(),
				"response cobtains unexpected status", response.getStatus());
		logMsg("response contains expected", STATUS, "status");
	}

	/**
	 * Check the given exception contains a prebuilt response containing the
	 * http header HOST
	 */
	protected void assertResponse(InternalServerErrorException e, String host)
			throws Fault {
		assertResponse(e);
		String header = e.getResponse().getHeaderString(HttpHeaders.HOST);
		assertNotNull(header, "http header", HttpHeaders.HOST,
				" of response is null");
		assertEquals(host, header, "Found unexpected http", HttpHeaders.HOST,
				"header", header);
		logMsg("Found expected http", HttpHeaders.HOST, "header");
	}

	protected void assertCause(WebApplicationException e, Throwable expected)
			throws Fault {
		assertEquals(e.getCause(), expected,
				"#getCause does not contain expected", expected, "but",
				e.getCause());
		logMsg("getCause contains expected", expected);
	}

	protected void assertMessage(InternalServerErrorException e) throws Fault {
		assertNotNull(e.getMessage(), "getMessage() is null");
		assertContains(e.getMessage(), MESSAGE, "Unexpected getMessage()",
				e.getMessage());
		logMsg("found expected getMessage()=", e.getMessage());
	}
}
