/**
 * Copyright (c) 2017 TypeFox GmbH (http://www.typefox.io) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.elk.graph.text.ide.contentassist;

import com.google.common.base.Objects;
import com.google.common.base.Predicate;
import com.google.inject.Inject;
import com.google.inject.Provider;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import org.eclipse.elk.core.data.ILayoutMetaData;
import org.eclipse.elk.core.data.LayoutAlgorithmData;
import org.eclipse.elk.core.data.LayoutMetaDataService;
import org.eclipse.elk.core.data.LayoutOptionData;
import org.eclipse.elk.core.options.CoreOptions;
import org.eclipse.elk.graph.ElkEdge;
import org.eclipse.elk.graph.ElkEdgeSection;
import org.eclipse.elk.graph.ElkGraphElement;
import org.eclipse.elk.graph.ElkLabel;
import org.eclipse.elk.graph.ElkNode;
import org.eclipse.elk.graph.ElkPort;
import org.eclipse.elk.graph.impl.ElkPropertyToValueMapEntryImpl;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.text.ElkGraphTextUtil;
import org.eclipse.elk.graph.text.services.ElkGraphGrammarAccess;
import org.eclipse.elk.graph.util.ElkGraphUtil;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtend.lib.annotations.FinalFieldsConstructor;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.Assignment;
import org.eclipse.xtext.CrossReference;
import org.eclipse.xtext.Keyword;
import org.eclipse.xtext.conversion.impl.IDValueConverter;
import org.eclipse.xtext.ide.editor.contentassist.ContentAssistContext;
import org.eclipse.xtext.ide.editor.contentassist.ContentAssistEntry;
import org.eclipse.xtext.ide.editor.contentassist.IIdeContentProposalAcceptor;
import org.eclipse.xtext.ide.editor.contentassist.IProposalConflictHelper;
import org.eclipse.xtext.ide.editor.contentassist.IdeContentProposalProvider;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.util.Strings;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Special content assist proposals for the ELK Graph language.
 */
@SuppressWarnings("all")
public class ElkGraphProposalProvider extends IdeContentProposalProvider {
  @FinalFieldsConstructor
  public static class SectionShapeFilter implements Predicate<IEObjectDescription> {
    private static final int INCOMING = 0;
    
    private static final int OUTGOING = 1;
    
    private final ElkEdgeSection section;
    
    private final int type;
    
    @Override
    public boolean apply(final IEObjectDescription input) {
      boolean _switchResult = false;
      final int type = this.type;
      switch (type) {
        case ElkGraphProposalProvider.SectionShapeFilter.INCOMING:
          _switchResult = this.isInList(input, this.section.getParent().getSources());
          break;
        case ElkGraphProposalProvider.SectionShapeFilter.OUTGOING:
          _switchResult = this.isInList(input, this.section.getParent().getTargets());
          break;
        default:
          _switchResult = true;
          break;
      }
      return _switchResult;
    }
    
    private boolean isInList(final IEObjectDescription input, final List<? extends EObject> list) {
      boolean _xblockexpression = false;
      {
        final EObject object = input.getEObjectOrProxy();
        boolean _xifexpression = false;
        boolean _eIsProxy = object.eIsProxy();
        if (_eIsProxy) {
          final Function1<EObject, Boolean> _function = new Function1<EObject, Boolean>() {
            @Override
            public Boolean apply(final EObject it) {
              URI _uRI = EcoreUtil.getURI(it);
              URI _eObjectURI = input.getEObjectURI();
              return Boolean.valueOf(Objects.equal(_uRI, _eObjectURI));
            }
          };
          _xifexpression = IterableExtensions.exists(list, _function);
        } else {
          _xifexpression = list.contains(object);
        }
        _xblockexpression = _xifexpression;
      }
      return _xblockexpression;
    }
    
    public SectionShapeFilter(final ElkEdgeSection section, final int type) {
      super();
      this.section = section;
      this.type = type;
    }
  }
  
  private static final Set<String> DISABLED_KEYWORDS = Collections.<String>unmodifiableSet(CollectionLiterals.<String>newHashSet("}", "]"));
  
  private ElkGraphGrammarAccess grammar;
  
  private IDValueConverter idValueConverter;
  
  @Inject
  private IProposalConflictHelper conflictHelper;
  
  @Inject
  public void initialize(final Provider<IDValueConverter> idValueConverterProvider, final ElkGraphGrammarAccess grammarAccess) {
    IDValueConverter _get = idValueConverterProvider.get();
    final Procedure1<IDValueConverter> _function = new Procedure1<IDValueConverter>() {
      @Override
      public void apply(final IDValueConverter it) {
        it.setRule(grammarAccess.getIDRule());
      }
    };
    IDValueConverter _doubleArrow = ObjectExtensions.<IDValueConverter>operator_doubleArrow(_get, _function);
    this.idValueConverter = _doubleArrow;
    this.grammar = grammarAccess;
  }
  
  @Override
  protected boolean filterKeyword(final Keyword keyword, final ContentAssistContext context) {
    return ((!ElkGraphProposalProvider.DISABLED_KEYWORDS.contains(keyword.getValue())) && (!Objects.equal(keyword.getValue(), context.getPrefix())));
  }
  
  @Override
  protected void _createProposals(final Keyword keyword, final ContentAssistContext context, final IIdeContentProposalAcceptor acceptor) {
    boolean _filterKeyword = this.filterKeyword(keyword, context);
    if (_filterKeyword) {
      final ContentAssistEntry entry = this.getProposalCreator().createProposal(keyword.getValue(), context);
      if ((entry != null)) {
        entry.setKind(ContentAssistEntry.KIND_KEYWORD);
        entry.setSource(keyword);
        acceptor.accept(entry, this.getProposalPriorities().getKeywordPriority(keyword.getValue(), entry));
      }
    }
  }
  
  @Override
  protected void _createProposals(final Assignment assignment, final ContentAssistContext context, final IIdeContentProposalAcceptor acceptor) {
    boolean _matched = false;
    Assignment _keyAssignment_0 = this.grammar.getPropertyAccess().getKeyAssignment_0();
    if (Objects.equal(assignment, _keyAssignment_0)) {
      _matched=true;
      this.completePropertyKey(context, acceptor);
    }
    if (!_matched) {
      Assignment _valueAssignment_2_0 = this.grammar.getPropertyAccess().getValueAssignment_2_0();
      if (Objects.equal(assignment, _valueAssignment_2_0)) {
        _matched=true;
      }
      if (!_matched) {
        Assignment _valueAssignment_2_1 = this.grammar.getPropertyAccess().getValueAssignment_2_1();
        if (Objects.equal(assignment, _valueAssignment_2_1)) {
          _matched=true;
        }
      }
      if (!_matched) {
        Assignment _valueAssignment_2_2 = this.grammar.getPropertyAccess().getValueAssignment_2_2();
        if (Objects.equal(assignment, _valueAssignment_2_2)) {
          _matched=true;
        }
      }
      if (!_matched) {
        Assignment _valueAssignment_2_3 = this.grammar.getPropertyAccess().getValueAssignment_2_3();
        if (Objects.equal(assignment, _valueAssignment_2_3)) {
          _matched=true;
        }
      }
      if (_matched) {
        this.completePropertyValue(context, acceptor);
      }
    }
    if (!_matched) {
      super._createProposals(assignment, context, acceptor);
    }
  }
  
  protected void completePropertyKey(final ContentAssistContext context, final IIdeContentProposalAcceptor acceptor) {
    EObject _currentModel = context.getCurrentModel();
    final EObject model = _currentModel;
    boolean _matched = false;
    if (model instanceof ElkNode) {
      _matched=true;
      if (((((ElkNode)model).getParent() == null) || (!((ElkNode)model).getChildren().isEmpty()))) {
        this.proposeProperties(((ElkGraphElement)model), ElkGraphTextUtil.getAlgorithm(((ElkGraphElement)model)), LayoutOptionData.Target.PARENTS, context, acceptor);
      }
      ElkNode _parent = ((ElkNode)model).getParent();
      boolean _tripleNotEquals = (_parent != null);
      if (_tripleNotEquals) {
        this.proposeProperties(((ElkGraphElement)model), ElkGraphTextUtil.getAlgorithm(((ElkNode)model).getParent()), LayoutOptionData.Target.NODES, context, acceptor);
      }
    }
    if (!_matched) {
      if (model instanceof ElkEdge) {
        _matched=true;
        this.proposeProperties(((ElkGraphElement)model), ElkGraphTextUtil.getAlgorithm(((ElkGraphElement)model)), LayoutOptionData.Target.EDGES, context, acceptor);
      }
    }
    if (!_matched) {
      if (model instanceof ElkPort) {
        _matched=true;
        this.proposeProperties(((ElkGraphElement)model), ElkGraphTextUtil.getAlgorithm(((ElkGraphElement)model)), LayoutOptionData.Target.PORTS, context, acceptor);
      }
    }
    if (!_matched) {
      if (model instanceof ElkLabel) {
        _matched=true;
        this.proposeProperties(((ElkGraphElement)model), ElkGraphTextUtil.getAlgorithm(((ElkGraphElement)model)), LayoutOptionData.Target.LABELS, context, acceptor);
      }
    }
  }
  
  protected void proposeProperties(final ElkGraphElement element, final LayoutAlgorithmData algorithmData, final LayoutOptionData.Target targetType, final ContentAssistContext context, final IIdeContentProposalAcceptor acceptor) {
    boolean _existsConflict = this.conflictHelper.existsConflict("a", context);
    if (_existsConflict) {
      return;
    }
    final LayoutMetaDataService metaDataService = LayoutMetaDataService.getInstance();
    final Function1<LayoutOptionData, Boolean> _function = new Function1<LayoutOptionData, Boolean>() {
      @Override
      public Boolean apply(final LayoutOptionData o) {
        return Boolean.valueOf(((((targetType == null) || o.getTargets().contains(targetType)) && (((algorithmData == null) || algorithmData.knowsOption(o)) || Objects.equal(CoreOptions.ALGORITHM, o))) && ((element == null) || (!element.getProperties().map().containsKey(o)))));
      }
    };
    final Iterable<LayoutOptionData> filteredOptions = IterableExtensions.<LayoutOptionData>filter(metaDataService.getOptionData(), _function);
    for (final LayoutOptionData option : filteredOptions) {
      {
        final boolean matchesName = ((!context.getPrefix().isEmpty()) && option.getName().toLowerCase().contains(context.getPrefix().toLowerCase()));
        final List<String> idSplit = Strings.split(option.getId(), ".");
        List<String> _xifexpression = null;
        if ((!matchesName)) {
          _xifexpression = Strings.split(context.getPrefix(), ".");
        }
        final List<String> prefixSplit = _xifexpression;
        boolean foundMatch = false;
        int _size = idSplit.size();
        int i = (_size - 1);
        if (((i >= 1) && Objects.equal(option.getGroup(), idSplit.get((i - 1))))) {
          i--;
        }
        while (((i >= 0) && (!foundMatch))) {
          {
            final Iterable<String> suffix = IterableExtensions.<String>drop(idSplit, i);
            if (((metaDataService.getOptionDataBySuffix(IterableExtensions.join(suffix, ".")) != null) && (matchesName || this.startsWith(suffix, prefixSplit)))) {
              foundMatch = true;
            } else {
              i--;
            }
          }
        }
        if (foundMatch) {
          final Iterable<String> suffix = IterableExtensions.<String>drop(idSplit, i);
          ContentAssistEntry _contentAssistEntry = new ContentAssistEntry();
          final Procedure1<ContentAssistEntry> _function_1 = new Procedure1<ContentAssistEntry>() {
            @Override
            public void apply(final ContentAssistEntry it) {
              it.setProposal(ElkGraphProposalProvider.this.convert(suffix));
              it.setPrefix(context.getPrefix());
              it.setKind(ContentAssistEntry.KIND_PROPERTY);
              it.setLabel(IterableExtensions.join(suffix, "."));
              it.setDescription(ElkGraphProposalProvider.this.getDescription(option));
              it.setDocumentation(option.getDescription());
              it.setSource(option);
            }
          };
          final ContentAssistEntry entry = ObjectExtensions.<ContentAssistEntry>operator_doubleArrow(_contentAssistEntry, _function_1);
          acceptor.accept(entry, this.getProposalPriorities().getDefaultPriority(entry));
        }
      }
    }
  }
  
  /**
   * There's another option that allows to select a layout algorithm other than {@link CoreOptions#ALGORITHM}.
   *  To avoid a dependency to that plugin, the option is hard-coded here.
   */
  public static final String DISCO_LAYOUT_ALG_ID = "org.eclipse.elk.disco.componentCompaction.componentLayoutAlgorithm";
  
  protected void completePropertyValue(final ContentAssistContext context, final IIdeContentProposalAcceptor acceptor) {
    final EObject model = context.getCurrentModel();
    if ((model instanceof ElkPropertyToValueMapEntryImpl)) {
      final IProperty<?> property = ((ElkPropertyToValueMapEntryImpl)model).getKey();
      if ((property instanceof LayoutOptionData)) {
        if ((Objects.equal(CoreOptions.ALGORITHM, property) || Objects.equal(((LayoutOptionData)property).getId(), ElkGraphProposalProvider.DISCO_LAYOUT_ALG_ID))) {
          this.proposeAlgorithms(context, acceptor);
        } else {
          this.typeAwarePropertyValueProposal(((LayoutOptionData)property), context, acceptor);
        }
      }
    }
  }
  
  private Object typeAwarePropertyValueProposal(final LayoutOptionData property, final ContentAssistContext context, final IIdeContentProposalAcceptor acceptor) {
    Object _switchResult = null;
    LayoutOptionData.Type _type = property.getType();
    if (_type != null) {
      switch (_type) {
        case BOOLEAN:
        case ENUM:
        case ENUMSET:
          final String[] choices = property.getChoices();
          for (int i = 0; (i < choices.length); i++) {
            {
              final String proposal = choices[i];
              final Enum<?> enumVal = property.getEnumValue(i);
              final StringBuilder displayString = new StringBuilder(proposal);
              int priority = 3;
              boolean _isExperimentalPropertyValue = ElkGraphUtil.isExperimentalPropertyValue(enumVal);
              if (_isExperimentalPropertyValue) {
                displayString.append(" - Experimental");
                priority = 1;
              } else {
                boolean _isAdvancedPropertyValue = ElkGraphUtil.isAdvancedPropertyValue(enumVal);
                if (_isAdvancedPropertyValue) {
                  displayString.append(" - Advanced");
                  priority = 2;
                }
              }
              final Procedure1<ContentAssistEntry> _function = new Procedure1<ContentAssistEntry>() {
                @Override
                public void apply(final ContentAssistEntry it) {
                  it.setLabel(displayString.toString());
                  it.setSource(property);
                }
              };
              final ContentAssistEntry entry = this.getProposalCreator().createProposal(proposal, context, ContentAssistEntry.KIND_VALUE, _function);
              acceptor.accept(entry, this.getProposalPriorities().getDefaultPriority(entry));
            }
          }
          break;
        case DOUBLE:
          final Procedure1<ContentAssistEntry> _function = new Procedure1<ContentAssistEntry>() {
            @Override
            public void apply(final ContentAssistEntry it) {
              it.setLabel(property.getType().toString());
              it.setSource(property);
            }
          };
          final ContentAssistEntry entry = this.getProposalCreator().createProposal("0.0", context, ContentAssistEntry.KIND_VALUE, _function);
          acceptor.accept(entry, this.getProposalPriorities().getDefaultPriority(entry));
          break;
        case INT:
          final Procedure1<ContentAssistEntry> _function_1 = new Procedure1<ContentAssistEntry>() {
            @Override
            public void apply(final ContentAssistEntry it) {
              it.setLabel(property.getType().toString());
              it.setSource(property);
            }
          };
          final ContentAssistEntry entry_1 = this.getProposalCreator().createProposal("0", context, ContentAssistEntry.KIND_VALUE, _function_1);
          acceptor.accept(entry_1, this.getProposalPriorities().getDefaultPriority(entry_1));
          break;
        case OBJECT:
          String _xtrycatchfinallyexpression = null;
          try {
            String _string = property.getOptionClass().newInstance().toString();
            String _plus = ("\"" + _string);
            _xtrycatchfinallyexpression = (_plus + "\"");
          } catch (final Throwable _t) {
            if (_t instanceof InstantiationException) {
              _xtrycatchfinallyexpression = "";
            } else if (_t instanceof IllegalAccessException) {
              _xtrycatchfinallyexpression = "";
            } else {
              throw Exceptions.sneakyThrow(_t);
            }
          }
          final String proposal = _xtrycatchfinallyexpression;
          final Procedure1<ContentAssistEntry> _function_2 = new Procedure1<ContentAssistEntry>() {
            @Override
            public void apply(final ContentAssistEntry it) {
              it.setLabel(property.getType().toString());
              it.setSource(property);
            }
          };
          final ContentAssistEntry entry_2 = this.getProposalCreator().createProposal(proposal, context, ContentAssistEntry.KIND_VALUE, _function_2);
          acceptor.accept(entry_2, this.getProposalPriorities().getDefaultPriority(entry_2));
          break;
        default:
          _switchResult = null;
          break;
      }
    } else {
      _switchResult = null;
    }
    return _switchResult;
  }
  
  protected void proposeAlgorithms(final ContentAssistContext context, final IIdeContentProposalAcceptor acceptor) {
    boolean _existsConflict = this.conflictHelper.existsConflict("a", context);
    if (_existsConflict) {
      return;
    }
    final LayoutMetaDataService metaDataService = LayoutMetaDataService.getInstance();
    Collection<LayoutAlgorithmData> _algorithmData = metaDataService.getAlgorithmData();
    for (final LayoutAlgorithmData algorithm : _algorithmData) {
      {
        final boolean matchesName = ((!context.getPrefix().isEmpty()) && algorithm.getName().toLowerCase().contains(context.getPrefix().toLowerCase()));
        final List<String> idSplit = Strings.split(algorithm.getId(), ".");
        List<String> _xifexpression = null;
        if ((!matchesName)) {
          _xifexpression = Strings.split(context.getPrefix(), ".");
        }
        final List<String> prefixSplit = _xifexpression;
        boolean foundMatch = false;
        int _size = idSplit.size();
        int i = (_size - 1);
        while (((i >= 0) && (!foundMatch))) {
          {
            final Iterable<String> suffix = IterableExtensions.<String>drop(idSplit, i);
            if (((metaDataService.getAlgorithmDataBySuffix(IterableExtensions.join(suffix, ".")) != null) && (matchesName || this.startsWith(suffix, prefixSplit)))) {
              foundMatch = true;
            } else {
              i--;
            }
          }
        }
        if (foundMatch) {
          final Iterable<String> suffix = IterableExtensions.<String>drop(idSplit, i);
          ContentAssistEntry _contentAssistEntry = new ContentAssistEntry();
          final Procedure1<ContentAssistEntry> _function = new Procedure1<ContentAssistEntry>() {
            @Override
            public void apply(final ContentAssistEntry it) {
              it.setProposal(ElkGraphProposalProvider.this.convert(suffix));
              it.setPrefix(context.getPrefix());
              it.setKind(ContentAssistEntry.KIND_VALUE);
              it.setLabel(IterableExtensions.join(suffix, "."));
              it.setDescription(ElkGraphProposalProvider.this.getDescription(algorithm));
              it.setDocumentation(algorithm.getDescription());
              it.setSource(algorithm);
            }
          };
          final ContentAssistEntry entry = ObjectExtensions.<ContentAssistEntry>operator_doubleArrow(_contentAssistEntry, _function);
          acceptor.accept(entry, this.getProposalPriorities().getDefaultPriority(entry));
        }
      }
    }
  }
  
  private boolean startsWith(final Iterable<String> strings, final List<String> prefix) {
    boolean _isEmpty = prefix.isEmpty();
    if (_isEmpty) {
      return true;
    }
    final List<String> stringList = IterableExtensions.<String>toList(strings);
    for (int i = 0; (i < ((stringList.size() - prefix.size()) + 1)); i++) {
      {
        int j = 0;
        boolean matches = true;
        while (((j < prefix.size()) && matches)) {
          {
            matches = stringList.get((i + j)).startsWith(prefix.get(j));
            j++;
          }
        }
        if (matches) {
          return true;
        }
      }
    }
    return false;
  }
  
  private String convert(final Iterable<String> suffix) {
    final Function1<String, String> _function = new Function1<String, String>() {
      @Override
      public String apply(final String it) {
        return ElkGraphProposalProvider.this.idValueConverter.toString(it);
      }
    };
    return IterableExtensions.join(IterableExtensions.<String, String>map(suffix, _function), ".");
  }
  
  private String getDescription(final ILayoutMetaData data) {
    StringConcatenation _builder = new StringConcatenation();
    String _name = data.getName();
    _builder.append(_name);
    _builder.append(" (");
    String _id = data.getId();
    _builder.append(_id);
    _builder.append(")");
    return _builder.toString();
  }
  
  @Override
  protected Predicate<IEObjectDescription> getCrossrefFilter(final CrossReference reference, final ContentAssistContext context) {
    final EObject model = context.getCurrentModel();
    if ((model instanceof ElkEdgeSection)) {
      boolean _matched = false;
      CrossReference _incomingShapeElkConnectableShapeCrossReference_4_0_0_2_0 = this.grammar.getElkEdgeSectionAccess().getIncomingShapeElkConnectableShapeCrossReference_4_0_0_2_0();
      if (Objects.equal(reference, _incomingShapeElkConnectableShapeCrossReference_4_0_0_2_0)) {
        _matched=true;
      }
      if (!_matched) {
        CrossReference _incomingShapeElkConnectableShapeCrossReference_1_0_0_2_0 = this.grammar.getElkSingleEdgeSectionAccess().getIncomingShapeElkConnectableShapeCrossReference_1_0_0_2_0();
        if (Objects.equal(reference, _incomingShapeElkConnectableShapeCrossReference_1_0_0_2_0)) {
          _matched=true;
        }
      }
      if (_matched) {
        return new ElkGraphProposalProvider.SectionShapeFilter(((ElkEdgeSection)model), ElkGraphProposalProvider.SectionShapeFilter.INCOMING);
      }
      if (!_matched) {
        CrossReference _outgoingShapeElkConnectableShapeCrossReference_4_0_1_2_0 = this.grammar.getElkEdgeSectionAccess().getOutgoingShapeElkConnectableShapeCrossReference_4_0_1_2_0();
        if (Objects.equal(reference, _outgoingShapeElkConnectableShapeCrossReference_4_0_1_2_0)) {
          _matched=true;
        }
        if (!_matched) {
          CrossReference _outgoingShapeElkConnectableShapeCrossReference_1_0_1_2_0 = this.grammar.getElkSingleEdgeSectionAccess().getOutgoingShapeElkConnectableShapeCrossReference_1_0_1_2_0();
          if (Objects.equal(reference, _outgoingShapeElkConnectableShapeCrossReference_1_0_1_2_0)) {
            _matched=true;
          }
        }
        if (_matched) {
          return new ElkGraphProposalProvider.SectionShapeFilter(((ElkEdgeSection)model), ElkGraphProposalProvider.SectionShapeFilter.OUTGOING);
        }
      }
    }
    final URI resourceURI = model.eResource().getURI();
    final Predicate<IEObjectDescription> _function = new Predicate<IEObjectDescription>() {
      @Override
      public boolean apply(final IEObjectDescription candidate) {
        URI _trimFragment = candidate.getEObjectURI().trimFragment();
        return Objects.equal(_trimFragment, resourceURI);
      }
    };
    return _function;
  }
}
