/*******************************************************************************
 * Copyright (c) Philipps University of Marburg. All rights reserved. 
 * This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Philipps University of Marburg - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.refactor.refactoring.runtime.ltk.change;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.emf.compare.diff.metamodel.DiffModel;
import org.eclipse.emf.compare.diff.service.DiffService;
import org.eclipse.emf.compare.match.MatchOptions;
import org.eclipse.emf.compare.match.metamodel.MatchModel;
import org.eclipse.emf.compare.match.service.MatchService;
import org.eclipse.emf.compare.ui.IModelCompareInputProvider;
import org.eclipse.emf.compare.ui.ModelCompareInput;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.change.ChangeDescription;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.refactor.refactoring.interfaces.IDataManagement;
import org.eclipse.emf.refactor.refactoring.runtime.history.EmfRefactorRefactoringDescriptor;
import org.eclipse.emf.refactor.refactoring.runtime.ltk.command.PreviewCommand;
import org.eclipse.emf.refactor.refactoring.runtime.ltk.command.RefactoringCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.ChangeDescriptor;
import org.eclipse.ltk.core.refactoring.RefactoringChangeDescriptor;
import org.eclipse.ltk.core.refactoring.RefactoringDescriptor;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;

/**
 * Class used for providing a LTK Change concerning the execution
 * of a specific EMF model refactoring.
 * @generated NOT
 * @author Thorsten Arendt
 */
public class RefactoringChange extends Change 
				implements IModelCompareInputProvider{

	private IDataManagement idataManagement;
	
	private String refactoringId; 

	/**
	 * Name of the RefactoringChange.
	 */
	private final String name;
	
	/**
	 * Root object of the EMF model.
	 */
	private final EObject root;
	
	/**
	 * EditingDomain object of the EMF model.
	 */
	private final EditingDomain editingDomain;
	
	/**
	 * Command that executes the EMF model refactoring.
	 */
	private final RefactoringCommand refactoringCommand;
	
	/**
	 * Default constructor.
	 * @param name Name of the RefactoringChange.
	 * @param root Root object of the EMF model.
	 * @param editingDomain EditingDomain object of the EMF model.
	 * @param changeDescription ChangeDescription object for model 
	 * refactoring execution.
	 * @param refactoringToApply Runnable object for model refactoring 
	 * execution.
	 * @param enableChangeRecorder Flag whether the model refactoring
	 * execution shall be recorded.
	 */
	public RefactoringChange(String name, EObject root, 
			EditingDomain editingDomain, ChangeDescription changeDescription,
			Runnable refactoringToApply, boolean enableChangeRecorder) {
		super();
		this.name = name;
		this.root = root;
		this.editingDomain=editingDomain;
		if(null != changeDescription){
			refactoringCommand = new RefactoringCommand(name,changeDescription);
		}else{
			refactoringCommand = new RefactoringCommand
					(name, refactoringToApply, this.root, enableChangeRecorder);			
		}
		this.idataManagement = null;
		this.refactoringId = null;
	}
	
	public RefactoringChange(String name, EObject root, 
			EditingDomain editingDomain, ChangeDescription changeDescription,
			Runnable refactoringToApply, boolean enableChangeRecorder, IDataManagement dataManagement, String id) {
		this(name, root, editingDomain, changeDescription, refactoringToApply, enableChangeRecorder);
		this.idataManagement = dataManagement;
		this.refactoringId = id;
	}

	/**
	 * @see org.eclipse.ltk.core.refactoring.Change#getName()
	 */
	@Override
	public String getName() {
		return this.name;
	}

	/**
	 * @see org.eclipse.ltk.core.refactoring.Change#
	 * initializeValidationData(org.eclipse.core.runtime.IProgressMonitor)
	 */
	@Override
	public void initializeValidationData(IProgressMonitor pm) {
		// do nothing
	}

	/**
	 * @see org.eclipse.ltk.core.refactoring.Change#
	 * isValid(org.eclipse.core.runtime.IProgressMonitor)
	 */
	@Override
	public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException,
			OperationCanceledException {
		return new RefactoringStatus();
	}
	
	@Override
	public ChangeDescriptor getDescriptor() {
		// Anpassung AW: Alles nicht optimal. - Der Descriptor msste eigentlich mit allen Informationen weiter hinten/unten in der Vererbungskette initialisiert werden.
		return new RefactoringChangeDescriptor(new EmfRefactorRefactoringDescriptor(refactoringId, "project", "description", "comment", RefactoringDescriptor.NONE, idataManagement));
	}

	/**
	 * @see org.eclipse.ltk.core.refactoring.Change#
	 * perform(org.eclipse.core.runtime.IProgressMonitor)
	 */
	@Override
	public Change perform(IProgressMonitor pm) throws CoreException {
		this.editingDomain.getCommandStack().execute(refactoringCommand);
//		// Anpassung AW
		if (editingDomain instanceof TransactionalEditingDomain) {
			try {
				root.eResource().save(Collections.EMPTY_MAP);
				System.out.println("AW: manuelles speichern!");
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
//		// END AW
		return 
			new UndoRedoChange(refactoringCommand, editingDomain, name, root);	
	}
	
	/**
	 * @see org.eclipse.ltk.core.refactoring.Change#getModifiedElement()
	 */
	@Override
	public Object getModifiedElement() {
		return this.root;
	}

	/**
	 * @see org.eclipse.emf.compare.ui.IModelCompareInputProvider#
	 * getModelCompareInput()
	 */
	@Override
	public ModelCompareInput getModelCompareInput() {
		try {
			//Perform Refactoring
			PreviewCommand previewCommand = 
							new PreviewCommand(refactoringCommand,this.root);
			this.editingDomain.getCommandStack().execute(previewCommand);
			//Generate DiffModel:
			Map<String, Object> options = new HashMap<String, Object>();
			options.put(MatchOptions.OPTION_IGNORE_XMI_ID, new Boolean(true));
			MatchModel matchModel = null;
			try {
				matchModel = MatchService.doMatch(this.root,
										previewCommand.getRootCopy(), options);
				DiffModel diffModel = DiffService.doDiff(matchModel);
				return new ModelCompareInput(matchModel, diffModel);	
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		throw new RuntimeException("Could not generate DiffModel");
	}

}
