package pam.refactorings;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;

import PAM.ClientNode;
import PAM.Cooling;
import PAM.NetworkNode;
import PAM.NetworkObjectLink;
import PAM.Nodes;
import PAM.Room;
import PAM.ServerNode;
import PAM.UninterruptiblePowerSupply;

public abstract class Optimizer {

	public static List<ClientNode> getBetterClientNodes(ClientNode client) {
		List<ClientNode> clientNodes = new ArrayList<ClientNode>();
		for (TreeIterator<EObject> iter = client.eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof ClientNode) {
				ClientNode clientNode = (ClientNode) eObject;
				if (clientNode != client
						&& isEnergeticImprovement(clientNode, client)
						&& hasAtLeastSameFunctionality(clientNode, client)) {
					clientNodes.add(clientNode);
				}
			}
		}	
		return clientNodes;
	}
	
	public static List<ServerNode> getBetterServerNodes(ServerNode server) {
		List<ServerNode> serverNodes = new ArrayList<ServerNode>();
		for (TreeIterator<EObject> iter = server.eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof ServerNode) {
				ServerNode serverNode = (ServerNode) eObject;
				if (serverNode != server
						&& isEnergeticImprovement(serverNode, server)
						&& hasAtLeastSameFunctionality(serverNode, server)) {
					serverNodes.add(serverNode);
				}
			}
		}
		return serverNodes;
	}
	
	public static List<NetworkNode> getBetterNetworkNodes(NetworkNode nwNode) {
		List<NetworkNode> networkNodes = new ArrayList<NetworkNode>();
		for (TreeIterator<EObject> iter = nwNode.eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof NetworkNode) {
				NetworkNode networkNode = (NetworkNode) eObject;
				if (networkNode != nwNode
						&& isEnergeticImprovement(networkNode, nwNode)
						&& hasAtLeastSameFunctionality(networkNode, nwNode)) {
					networkNodes.add(networkNode);
				}
			}
		}
		return networkNodes;
	}
	
	public static ArrayList<NetworkNodePair> getBetterNetworkNodePairs(NetworkNode nwNode) {
		ArrayList<NetworkNodePair> networkNodePairs = new ArrayList<NetworkNodePair>();
		for (TreeIterator<EObject> iter1 = nwNode.eResource().getAllContents(); iter1.hasNext(); ) {
			EObject eObject1 = (EObject) iter1.next();
			if (eObject1 instanceof NetworkNode) {
				NetworkNode networkNode1 = (NetworkNode) eObject1;
				for (TreeIterator<EObject> iter2 = nwNode.eResource().getAllContents(); iter2.hasNext(); ) {
					EObject eObject2 = (EObject) iter2.next();
					if (eObject2 instanceof NetworkNode) {
						NetworkNode networkNode2 = (NetworkNode) eObject2;		
						if (networkNode1 != nwNode && networkNode2 != nwNode) {
							if (isEnergeticImprovement(networkNode1, networkNode2, nwNode)
								&& hasAtLeastSameFunctionality(networkNode1, networkNode2, nwNode)) {
								NetworkNodePair networkNodePair = new NetworkNodePair(networkNode1, networkNode2);
								if (! networkNodePairs.contains(networkNodePair)) networkNodePairs.add(networkNodePair);
							}
						}
					}
				}
			}
		}
		return networkNodePairs;
	}
	
	public static ArrayList<CoolingPair> getBetterCoolingPairs(Cooling cooling) {
		ArrayList<CoolingPair> coolingPairs = new ArrayList<CoolingPair>();
		for (TreeIterator<EObject> iter1 = cooling.eResource().getAllContents(); iter1.hasNext(); ) {
			EObject eObject1 = (EObject) iter1.next();
			if (eObject1 instanceof Cooling) {
				Cooling cooling1 = (Cooling) eObject1;
				for (TreeIterator<EObject> iter2 = cooling.eResource().getAllContents(); iter2.hasNext(); ) {
					EObject eObject2 = (EObject) iter2.next();
					if (eObject2 instanceof Cooling) {
						Cooling cooling2 = (Cooling) eObject2;		
						if (cooling1 != cooling && cooling2 != cooling) {
							if (isEnergeticImprovement(cooling1, cooling2, cooling)
								&& hasAtLeastSameFunctionality(cooling1, cooling2, cooling)) {
								CoolingPair coolingPair = new CoolingPair(cooling1, cooling2);
								if (! coolingPairs.contains(coolingPair)) coolingPairs.add(coolingPair);
							}
						}
					}
				}
			}
		}
		return coolingPairs;
	}

	public static List<CoolingPair> getMergeableCoolingPairs(Cooling cooling) {
		ArrayList<CoolingPair> coolingPairs = new ArrayList<CoolingPair>();
		List<CoolingPair> coolingPairCandidates = getCoolingPairsWithinSameRoom(cooling);
		for (CoolingPair pair : coolingPairCandidates) {
			if (! Optimizer.getBetterCooling(pair).isEmpty()) {
				if (! coolingPairs.contains(pair)) coolingPairs.add(pair);
			}
		}
		return coolingPairs;
	}

	public static List<NetworkNodePair> getMergeableNetworkNodePairs(NetworkNode networkNode) {
		ArrayList<NetworkNodePair> nwnPairs = new ArrayList<NetworkNodePair>();
		List<NetworkNodePair> nwnPairCandidates = getNetworkNodePairs(networkNode);
		for (NetworkNodePair pair : nwnPairCandidates) {
			if (! Optimizer.getBetterNetworkNode(pair).isEmpty()) {
				if (! nwnPairs.contains(pair)) nwnPairs.add(pair);
			}
		}
		return nwnPairs;
	}

	private static List<NetworkNodePair> getNetworkNodePairs(NetworkNode networkNode) {
		ArrayList<NetworkNodePair> nwnPairs = new ArrayList<NetworkNodePair>();
		List<EObject> allLinks = getAllLinks(networkNode.eResource());
		for (EObject eObject : allLinks) {
			NetworkObjectLink link = (NetworkObjectLink) eObject;
			if (link.getConnect0() instanceof NetworkNode) {
				if (link.getConnect0() == networkNode) {
					NetworkNodePair pair = new NetworkNodePair(networkNode, link.getConnect1());
					if (! nwnPairs.contains(pair)) nwnPairs.add(pair);
				}
				if (link.getConnect1() == networkNode) {
					NetworkNodePair pair = new NetworkNodePair((NetworkNode) link.getConnect0(), networkNode);
					if (! nwnPairs.contains(pair)) nwnPairs.add(pair);
				}
			}
		}
		return nwnPairs;
	}

	private static List<EObject> getAllLinks(Resource resource) {
		List<EObject> allLinks = new ArrayList<EObject>();
		TreeIterator<EObject> iter = resource.getAllContents();
		while (iter.hasNext()) {
			EObject eObject = iter.next();
			if (eObject instanceof NetworkObjectLink) {
				allLinks.add((NetworkObjectLink) eObject);
			}
		}
		return allLinks;
	}

	public static List<Cooling> getOptimalCooling(CoolingPair pair) {
		ArrayList<Cooling> betterCoolings = getBetterCooling(pair);
		ArrayList<Cooling> optimalCoolings = new ArrayList<Cooling>();
		for (Cooling c : betterCoolings) {
			boolean isOptimal = true;
			for (Cooling candidate : betterCoolings) {
				if (c != candidate && isEnergeticImprovement(candidate, c)) {
					isOptimal = false;
					break;
				}
			}
			if (isOptimal) {
				optimalCoolings.add(c);
			}
		}
		return optimalCoolings;
	}

	public static List<NetworkNode> getOptimalNetworkNode(NetworkNodePair pair) {
		ArrayList<NetworkNode> betterNetworkNodes = getBetterNetworkNode(pair);
		ArrayList<NetworkNode> optimalNetworkNodes = new ArrayList<NetworkNode>();
		for (NetworkNode node : betterNetworkNodes) {
			boolean isOptimal = true;
			for (NetworkNode candidate : betterNetworkNodes) {
				if (node != candidate && isEnergeticImprovement(candidate, node)) {
					isOptimal = false;
					break;
				}
			}
			if (isOptimal) {
				optimalNetworkNodes.add(node);
			}
		}
		return optimalNetworkNodes;
	}

	public static List<UninterruptiblePowerSupply> getOptimalUPS(UninterruptiblePowerSupplyPair pair) {
		ArrayList<UninterruptiblePowerSupply> betterUninterruptiblePowerSupplies = getBetterUPS(pair);
		ArrayList<UninterruptiblePowerSupply> optimalUninterruptiblePowerSupplies = new ArrayList<UninterruptiblePowerSupply>();
		for (UninterruptiblePowerSupply ups : betterUninterruptiblePowerSupplies) {
			boolean isOptimal = true;
			for (UninterruptiblePowerSupply candidate : betterUninterruptiblePowerSupplies) {
				if (ups != candidate && isEnergeticImprovement(candidate, ups)) {
					isOptimal = false;
					break;
				}
			}
			if (isOptimal) {
				optimalUninterruptiblePowerSupplies.add(ups);
			}
		}
		return optimalUninterruptiblePowerSupplies;
	}

	public static ArrayList<Cooling> getBetterCooling(CoolingPair pair) {
		ArrayList<Cooling> coolings = new ArrayList<Cooling>();
		for (TreeIterator<EObject> iter = pair.getCooling1().eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof Cooling) {
				Cooling cooling = (Cooling) eObject;
				if (cooling != pair.getCooling1() && cooling != pair.getCooling2()) {
					if (isEnergeticImprovement(cooling, pair) && hasAtLeastSameFunctionality(cooling, pair)) {
							if (! coolings.contains(cooling)) coolings.add(cooling);
					}
				}
			}
		}
		return coolings;
	}

	public static ArrayList<NetworkNode> getBetterNetworkNode(NetworkNodePair pair) {
		ArrayList<NetworkNode> nwNodes = new ArrayList<NetworkNode>();
		for (TreeIterator<EObject> iter = pair.getNode1().eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof NetworkNode) {
				NetworkNode nwNode = (NetworkNode) eObject;
				if (nwNode != pair.getNode1() && nwNode != pair.getNode2()) {
					if (isEnergeticImprovement(nwNode, pair) && hasAtLeastSameFunctionality(nwNode, pair)) {
							if (! nwNodes.contains(nwNode)) nwNodes.add(nwNode);
					}
				}
			}
		}
		return nwNodes;
	}

	private static boolean hasAtLeastSameFunctionality(NetworkNode nwNode, NetworkNodePair pair) {
		return (nwNode.getMax_Throughput() >= (pair.getNode1().getMax_Throughput() + pair.getNode2().getMax_Throughput())
				&& (getIncomingLinks(nwNode).size() >= (getIncomingLinks(pair.getNode1()).size() + getIncomingLinks(pair.getNode2()).size() - 2))
				);
	}

	private static boolean isEnergeticImprovement(NetworkNode nwNode, NetworkNodePair pair) {
		return (nwNode.getMax_Watt() < (pair.getNode1().getMax_Watt() + pair.getNode2().getMax_Watt()));
	}

	private static boolean hasAtLeastSameFunctionality(Cooling cooling, CoolingPair pair) {
		return (cooling.getCooling_Capacity() >= (pair.getCooling1().getCooling_Capacity() + pair.getCooling2().getCooling_Capacity()));
	}

	private static boolean isEnergeticImprovement(Cooling cooling, CoolingPair pair) {
		return (cooling.getMax_Watt() < (pair.getCooling1().getMax_Watt() + pair.getCooling2().getMax_Watt()));
	}

	public static ArrayList<UninterruptiblePowerSupplyPair> getBetterUPSPairs(UninterruptiblePowerSupply ups) {
		ArrayList<UninterruptiblePowerSupplyPair> upsPairs = new ArrayList<UninterruptiblePowerSupplyPair>();
		for (TreeIterator<EObject> iter1 = ups.eResource().getAllContents(); iter1.hasNext(); ) {
			EObject eObject1 = (EObject) iter1.next();
			if (eObject1 instanceof UninterruptiblePowerSupply) {
				UninterruptiblePowerSupply ups1 = (UninterruptiblePowerSupply) eObject1;
				for (TreeIterator<EObject> iter2 = ups.eResource().getAllContents(); iter2.hasNext(); ) {
					EObject eObject2 = (EObject) iter2.next();
					if (eObject2 instanceof UninterruptiblePowerSupply) {
						UninterruptiblePowerSupply ups2 = (UninterruptiblePowerSupply) eObject2;		
						if (ups1 != ups && ups2 != ups) {
							if (isEnergeticImprovement(ups1, ups2, ups)
								&& hasAtLeastSameFunctionality(ups1, ups2, ups)) {
								UninterruptiblePowerSupplyPair upsPair = new UninterruptiblePowerSupplyPair(ups1, ups2);
								if (! upsPairs.contains(upsPair)) upsPairs.add(upsPair);
							}
						}
					}
				}
			}
		}
		return upsPairs;
	}

	public static ArrayList<UninterruptiblePowerSupply> getBetterUPS(UninterruptiblePowerSupplyPair pair) {
		ArrayList<UninterruptiblePowerSupply> upss = new ArrayList<UninterruptiblePowerSupply>();
		for (TreeIterator<EObject> iter = pair.getUps1().eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof UninterruptiblePowerSupply) {
				UninterruptiblePowerSupply ups = (UninterruptiblePowerSupply) eObject;
				if (ups != pair.getUps1() && ups != pair.getUps2()) {
					if (isEnergeticImprovement(ups, pair) && hasAtLeastSameFunctionality(ups, pair)) {
							if (! upss.contains(ups)) upss.add(ups);
					}
				}
			}
		}
		return upss;
	}

	public static List<UninterruptiblePowerSupplyPair> getMergeableUPSPairs(UninterruptiblePowerSupply ups) {
		ArrayList<UninterruptiblePowerSupplyPair> upsPairs = new ArrayList<UninterruptiblePowerSupplyPair>();
		List<UninterruptiblePowerSupplyPair> upsPairCandidates = getUPSPairsWithinSameRoom(ups);
		for (UninterruptiblePowerSupplyPair pair : upsPairCandidates) {
			if (! Optimizer.getBetterUPS(pair).isEmpty()) {
				if (! upsPairs.contains(pair)) upsPairs.add(pair);
			}
		}
		return upsPairs;
	}

	private static List<UninterruptiblePowerSupplyPair> getUPSPairsWithinSameRoom(UninterruptiblePowerSupply ups) {
		List<UninterruptiblePowerSupplyPair> upsPairs = new ArrayList<UninterruptiblePowerSupplyPair>();
		List<EObject> allUninterruptiblePowerSupplies = getAllUninterruptiblePowerSupplies(ups);
		Room room1 = getOwningRoom(ups);
		for (EObject eObject : allUninterruptiblePowerSupplies) {
			UninterruptiblePowerSupply candidate = (UninterruptiblePowerSupply) eObject;
			if (candidate != ups) {
				Room room2 = getOwningRoom(candidate);
				if (room1.equals(room2)) {
					UninterruptiblePowerSupplyPair pair = new UninterruptiblePowerSupplyPair(ups, candidate);
					if (! upsPairs.contains(pair)) upsPairs.add(pair);
				}
			}
		}
		return upsPairs;
	}

	private static List<EObject> getAllUninterruptiblePowerSupplies(EObject eObject) {
		List<EObject> allCoolings = new ArrayList<EObject>();
		TreeIterator<EObject> iter = eObject.eResource().getAllContents();
		while (iter.hasNext()) {
			EObject eObjectTmp = iter.next();
			if (eObjectTmp instanceof UninterruptiblePowerSupply) {
				allCoolings.add((UninterruptiblePowerSupply) eObjectTmp);
			}
		}
		return allCoolings;
	}

	private static boolean hasAtLeastSameFunctionality(UninterruptiblePowerSupply ups, UninterruptiblePowerSupplyPair pair) {
		return (ups.getOut_Watt() >= (pair.getUps1().getOut_Watt() + pair.getUps2().getOut_Watt()));
	}

	private static boolean isEnergeticImprovement(UninterruptiblePowerSupply ups, UninterruptiblePowerSupplyPair pair) {
		return (getEfficiency(ups) < (getEfficiency(pair.getUps1()) + getEfficiency(pair.getUps2())));
	}

	public static List<UninterruptiblePowerSupply> getBetterUPSwithoutContext(UninterruptiblePowerSupply ups) {
		List<UninterruptiblePowerSupply> upss = new ArrayList<UninterruptiblePowerSupply>();
		for (TreeIterator<EObject> iter = ups.eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof UninterruptiblePowerSupply) {
				UninterruptiblePowerSupply uninterruptiblePowerSupply = (UninterruptiblePowerSupply) eObject;
				if (uninterruptiblePowerSupply != ups
						&& isEnergeticImprovement(uninterruptiblePowerSupply, ups)
						&& hasAtLeastSameFunctionality(uninterruptiblePowerSupply, ups)) {
					upss.add(uninterruptiblePowerSupply);
				}
			}
		}
		return upss;
	}

	public static List<UninterruptiblePowerSupply> getBetterUPSwithContext(UninterruptiblePowerSupply ups) {
		List<UninterruptiblePowerSupply> upss = new ArrayList<UninterruptiblePowerSupply>();
		for (TreeIterator<EObject> iter = ups.eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof UninterruptiblePowerSupply) {
				UninterruptiblePowerSupply uninterruptiblePowerSupply = (UninterruptiblePowerSupply) eObject;
				if (uninterruptiblePowerSupply != ups
						&& isEnergeticImprovement(uninterruptiblePowerSupply, ups)
						&& hasRequiredFunctionality(uninterruptiblePowerSupply, ups)) {
					upss.add(uninterruptiblePowerSupply);
				}
			}
		}
		return upss;
	}

	public static List<Cooling> getBetterCoolingWithoutContext(Cooling cooling) {
		List<Cooling> coolings = new ArrayList<Cooling>();
		for (TreeIterator<EObject> iter = cooling.eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof Cooling) {
				Cooling c = (Cooling) eObject; 
				if (c != cooling
						&& isEnergeticImprovement(c, cooling)
						&& hasAtLeastSameFunctionality(c, cooling)) {
					coolings.add(c);
				}
			}
		}
		return coolings;
	}

	public static List<Cooling> getBetterCoolingWithContext(Cooling cooling) {
		List<Cooling> coolings = new ArrayList<Cooling>();
		for (TreeIterator<EObject> iter = cooling.eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof Cooling) {
				Cooling c = (Cooling) eObject; 
				if (c != cooling
						&& isEnergeticImprovement(c, cooling)
						&& hasRequiredFunctionality(c, cooling)) {
					coolings.add(c);
				}
			}
		}
		return coolings;
	}

	public static List<ClientNode> getOptimalClientNodes(ClientNode client) {
		List<ClientNode> optimalClientNodes = new ArrayList<ClientNode>();
		List<ClientNode> clientNodes = getBetterClientNodes(client);
		for (ClientNode cn : clientNodes) {
			boolean isOptimal = true;
			for (ClientNode candidate : clientNodes) {
				if (cn != candidate && isEnergeticImprovement(candidate, cn)) {
					isOptimal = false;
					break;
				}
			}
			if (isOptimal) {
				optimalClientNodes.add(cn);
			}
		}
		return optimalClientNodes;
	}
	
	public static List<ServerNode> getOptimalServerNodes(ServerNode server) {
		List<ServerNode> optimalServerNodes = new ArrayList<ServerNode>();
		List<ServerNode> serverNodes = getBetterServerNodes(server);
		for (ServerNode sn : serverNodes) {
			boolean isOptimal = true;
			for (ServerNode candidate : serverNodes) {
				if (sn != candidate && isEnergeticImprovement(candidate, sn)) {
					isOptimal = false;
					break;
				}
			}
			if (isOptimal) {
				optimalServerNodes.add(sn);
			}
		}
		return optimalServerNodes;
	}
	
	public static List<Cooling> getOptimalCoolingWithoutContext(Cooling cooling) {
		List<Cooling> optimalCoolings = new ArrayList<Cooling>();
		List<Cooling> coolings = getBetterCoolingWithoutContext(cooling);
		for (Cooling c : coolings) {
			boolean isOptimal = true;
			for (Cooling candidate : coolings) {
				if (c != candidate && isEnergeticImprovement(candidate, c)) {
					isOptimal = false;
					break;
				}
			}
			if (isOptimal) {
				optimalCoolings.add(c);
			}
		}
		return optimalCoolings;
	}
	
	public static List<Cooling> getOptimalCoolingWithContext(Cooling cooling) {
		List<Cooling> optimalCoolings = new ArrayList<Cooling>();
		List<Cooling> coolings = getBetterCoolingWithContext(cooling);
		for (Cooling c : coolings) {
			boolean isOptimal = true;
			for (Cooling candidate : coolings) {
				if (c != candidate && isEnergeticImprovement(candidate, c)) {
					isOptimal = false;
					break;
				}
			}
			if (isOptimal) {
				optimalCoolings.add(c);
			}
		}
		return optimalCoolings;
	}
	
	public static List<CoolingPair> getOptimalCoolingPairs(Cooling cooling) {
		ArrayList<CoolingPair> optimalCoolingPairs = new ArrayList<CoolingPair>();
		ArrayList<CoolingPair> coolingPairs = getBetterCoolingPairs(cooling);
		for (CoolingPair coolingPair : coolingPairs) {
			boolean isOptimal = true;
			for (CoolingPair candidate : coolingPairs) {
				if (coolingPair != candidate && isEnergeticImprovement(candidate, coolingPair)) {
					isOptimal = false;
					break;
				}
			}
			if (isOptimal) {
				optimalCoolingPairs.add(coolingPair);
			}
		}
		return optimalCoolingPairs;
	}

	private static boolean isEnergeticImprovement(CoolingPair p1, CoolingPair p2) {
		return (p1.getCooling1().getMax_Watt() + p1.getCooling2().getMax_Watt()) < (p2.getCooling1().getMax_Watt() + p2.getCooling2().getMax_Watt());
	}

	public static List<UninterruptiblePowerSupply> getOptimalUPSwithoutContext(UninterruptiblePowerSupply ups) {
		List<UninterruptiblePowerSupply> optimalUninterruptiblePowerSupplys = new ArrayList<UninterruptiblePowerSupply>();
		List<UninterruptiblePowerSupply> upss = getBetterUPSwithoutContext(ups);
		for (UninterruptiblePowerSupply unintPS : upss) {
			boolean isOptimal = true;
			for (UninterruptiblePowerSupply candidate : upss) {
				if (unintPS != candidate && isEnergeticImprovement(candidate, unintPS)) {
					isOptimal = false;
					break;
				}
			}
			if (isOptimal) {
				optimalUninterruptiblePowerSupplys.add(unintPS);
			}
		}
		return optimalUninterruptiblePowerSupplys;
	}

	public static List<UninterruptiblePowerSupply> getOptimalUPSwithContext(UninterruptiblePowerSupply ups) {
		List<UninterruptiblePowerSupply> optimalUninterruptiblePowerSupplys = new ArrayList<UninterruptiblePowerSupply>();
		List<UninterruptiblePowerSupply> upss = getBetterUPSwithContext(ups);
		for (UninterruptiblePowerSupply unintPS : upss) {
			boolean isOptimal = true;
			for (UninterruptiblePowerSupply candidate : upss) {
				if (unintPS != candidate && isEnergeticImprovement(candidate, unintPS)) {
					isOptimal = false;
					break;
				}
			}
			if (isOptimal) {
				optimalUninterruptiblePowerSupplys.add(unintPS);
			}
		}
		return optimalUninterruptiblePowerSupplys;
	}
	
	public static ArrayList<UninterruptiblePowerSupplyPair> getOptimalUPSPairs(UninterruptiblePowerSupply ups) {
		ArrayList<UninterruptiblePowerSupplyPair> optimalUPSPairs = new ArrayList<UninterruptiblePowerSupplyPair>();
		ArrayList<UninterruptiblePowerSupplyPair> upsPairs = getBetterUPSPairs(ups);
		for (UninterruptiblePowerSupplyPair upsPair : upsPairs) {
			boolean isOptimal = true;
			for (UninterruptiblePowerSupplyPair candidate : upsPairs) {
				if (upsPair != candidate && isEnergeticImprovement(candidate, upsPair)) {
					isOptimal = false;
					break;
				}
			}
			if (isOptimal) {
				optimalUPSPairs.add(upsPair);
			}
		}
		return optimalUPSPairs;
	}

	private static boolean isEnergeticImprovement(
			UninterruptiblePowerSupplyPair upsp1,
			UninterruptiblePowerSupplyPair upsp2) {
		return (getEfficiency(upsp1.getUps1()) + getEfficiency(upsp1.getUps2()) < getEfficiency(upsp2.getUps1()) + getEfficiency(upsp2.getUps2()));
	}

	public static List<NetworkNode> getOptimalNetworkNodes(NetworkNode network) {
		List<NetworkNode> optimalNetworkNodes = new ArrayList<NetworkNode>();
		List<NetworkNode> networkNodes = getBetterNetworkNodes(network);
		for (NetworkNode node : networkNodes) {
			boolean isOptimal = true;
			for (NetworkNode candidate : networkNodes) {
				if (node != candidate && isEnergeticImprovement(candidate, node)) {
					isOptimal = false;
					break;
				}
			}
			if (isOptimal) {
				optimalNetworkNodes.add(node);
			}
		}
		return optimalNetworkNodes;
	}
	
	public static ArrayList<NetworkNodePair> getOptimalNetworkNodePairs(NetworkNode nwNode) {
		ArrayList<NetworkNodePair> optimalNetworkNodePairs = new ArrayList<NetworkNodePair>();
		ArrayList<NetworkNodePair> networkNodePairs = getBetterNetworkNodePairs(nwNode);
		for (NetworkNodePair nnPair : networkNodePairs) {
			boolean isOptimal = true;
			for (NetworkNodePair candidate : networkNodePairs) {
				if (nnPair != candidate && isEnergeticImprovement(candidate, nnPair)) {
					isOptimal = false;
					break;
				}
			}
			if (isOptimal) {
				optimalNetworkNodePairs.add(nnPair);
			}
		}
		return optimalNetworkNodePairs;
	}

	private static boolean isEnergeticImprovement(NetworkNodePair p1, NetworkNodePair p2) {
		return (p1.getNode1().getMax_Watt() + p1.getNode2().getMax_Watt()) < (p2.getNode1().getMax_Watt() + p2.getNode2().getMax_Watt());
	}

	private static boolean hasAtLeastSameFunctionality(ClientNode candidate, ClientNode client) {
		return candidate.getMFLOPs() >= client.getMFLOPs();
	}
	
	private static boolean hasAtLeastSameFunctionality(ServerNode candidate, ServerNode server) {
		return candidate.getMFLOPs() >= server.getMFLOPs() && candidate.getMax_Capacity() >= server.getMax_Capacity();
	}
	
	private static boolean hasAtLeastSameFunctionality(NetworkNode candidate, NetworkNode nwNode) {
		return numberOfIncomingLinks(candidate) >= numberOfIncomingLinks(nwNode) && candidate.getMax_Throughput() >= nwNode.getMax_Throughput();
	}
	
	private static boolean hasAtLeastSameFunctionality(NetworkNode candidate1, NetworkNode candidate2, NetworkNode nwNode) {
		return numberOfIncomingLinks(candidate1) < numberOfIncomingLinks(nwNode)
				&& numberOfIncomingLinks(candidate2) < numberOfIncomingLinks(nwNode)
				&& (numberOfIncomingLinks(candidate1) + numberOfIncomingLinks(candidate2)) >= (numberOfIncomingLinks(nwNode) + 2)
				&& (candidate1.getMax_Throughput() + candidate2.getMax_Throughput()) >= nwNode.getMax_Throughput();
	}
	
	private static boolean hasAtLeastSameFunctionality(UninterruptiblePowerSupply candidate, UninterruptiblePowerSupply ups) {
		return candidate.getOut_Watt() >= ups.getOut_Watt();
	}
	
	private static boolean hasAtLeastSameFunctionality(UninterruptiblePowerSupply candidate1, 
			UninterruptiblePowerSupply candidate2, UninterruptiblePowerSupply ups) {
		return 
//				candidate1.getOut_Watt() < ups.getOut_Watt()
//				&& candidate2.getOut_Watt() < ups.getOut_Watt()
//				&& 
				(candidate1.getOut_Watt() + candidate2.getOut_Watt()) >= ups.getOut_Watt();
	}

	private static boolean hasAtLeastSameFunctionality(Cooling candidate, Cooling cooling) {
		return candidate.getCooling_Capacity() >= cooling.getCooling_Capacity();
	}

	private static boolean hasAtLeastSameFunctionality(Cooling candidate1, Cooling candidate2, Cooling cooling) {
		return candidate1.getCooling_Capacity() < cooling.getCooling_Capacity()
				&& candidate2.getCooling_Capacity() < cooling.getCooling_Capacity()
				&& (candidate1.getCooling_Capacity() + candidate2.getCooling_Capacity()) >= cooling.getCooling_Capacity();
	}

	private static boolean hasRequiredFunctionality(UninterruptiblePowerSupply candidate, UninterruptiblePowerSupply ups) {
		Room room = getOwningRoom(ups);		
		return getRequiredMaxWatt(room) <= candidate.getOut_Watt();
	}

	private static boolean hasRequiredFunctionality(Cooling candidate, Cooling cooling) {
		Room room = getOwningRoom(cooling);		
		return getRequiredMaxWatt(room) <= candidate.getCooling_Capacity();
	}
	
	@SuppressWarnings("unchecked")
	public static int getRequiredMaxWatt(Room room) {
		int requiredMaxWatt = 0;
		List<Room> allRooms = getAllRooms(room);
		for (Room r : allRooms) {
			EList<Nodes> nodes = r.getContains();
			for (Nodes n : nodes) {
				requiredMaxWatt += n.getMax_Watt();
			}
		}	
		return requiredMaxWatt;
	}
	
	@SuppressWarnings("unchecked")
	public static int getProducedWatt(Room room) {
		int producedWatt = 0;
		EList<UninterruptiblePowerSupply> upss = room.getApplies();
		for (UninterruptiblePowerSupply ups : upss) {
			producedWatt += ups.getOut_Watt();
		}
		return producedWatt;
	}

	@SuppressWarnings("unchecked")
	private static int getCoolingCapacity(Room room) {
		int cc = 0;
		EList<Cooling> coolings = room.getIncludes();
		for (Cooling cooling : coolings) {
			cc += cooling.getCooling_Capacity();
		}
		return cc;
	}
	
	@SuppressWarnings("unchecked")
	private static List<Room> getAllRooms(Room room) {
		List<Room> allRooms = new ArrayList<Room>();
		allRooms.add(room); 
		EList<Room> subrooms = room.getSubrooms();
		for (Room room1 : subrooms) {
			allRooms.addAll(getAllRooms(room1));
		}		
		return allRooms;
	}

	public static Room getOwningRoom(UninterruptiblePowerSupply ups) {
		for (TreeIterator<EObject> iter = ups.eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof Room) {
				Room room = (Room) eObject;
				if (room.getApplies().contains(ups)) return room;
			}
		}
		return null;
	}
	
	public static Room getOwningRoom(NetworkNode nwNode) {
		for (TreeIterator<EObject> iter = nwNode.eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof Room) {
				Room room = (Room) eObject;
				if (room.getContains().contains(nwNode)) return room;
			}
		}
		return null;
	}

	public static Room getOwningRoom(Cooling cooling) {
		for (TreeIterator<EObject> iter = cooling.eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof Room) {
				Room room = (Room) eObject;
				if (room.getIncludes().contains(cooling)) return room;
			}
		}
		return null;
	}

	public static int numberOfIncomingLinks(NetworkNode nwNode) {
		int numberOfIncomingLinks = 0;
		for (TreeIterator<EObject> iter = nwNode.eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof NetworkObjectLink) {
				NetworkObjectLink networkObjectLink = (NetworkObjectLink) eObject;
				if (networkObjectLink.getConnect0() == nwNode) numberOfIncomingLinks++;
				if (networkObjectLink.getConnect1() == nwNode) numberOfIncomingLinks++;
			}
		}
		return numberOfIncomingLinks;
	}
	
	public static ArrayList<NetworkObjectLink> getIncomingLinks(NetworkNode nwNode) {
		ArrayList<NetworkObjectLink> incomingLinks = new ArrayList<NetworkObjectLink>();
		for (TreeIterator<EObject> iter = nwNode.eResource().getAllContents(); iter.hasNext(); ) {
			EObject eObject = (EObject) iter.next();
			if (eObject instanceof NetworkObjectLink) {
				NetworkObjectLink networkObjectLink = (NetworkObjectLink) eObject;
				if (networkObjectLink.getConnect0() == nwNode) incomingLinks.add(networkObjectLink);
				if (networkObjectLink.getConnect1() == nwNode) incomingLinks.add(networkObjectLink);
			}
		}
		return incomingLinks;
	}

	private static boolean isEnergeticImprovement(ClientNode candidate, ClientNode node) {
		return candidate.getMax_Watt() < node.getMax_Watt();
	}
	
	private static boolean isEnergeticImprovement(ServerNode candidate, ServerNode node) {
		boolean efficiency1 = (candidate.getMax_Watt() < node.getMax_Watt())
				&& (candidate.getAct_Watt() <= node.getAct_Watt())
				&& (candidate.getIdle_Watt() <= node.getIdle_Watt());
		boolean efficiency2 = (candidate.getMax_Watt() <= node.getMax_Watt())
				&& (candidate.getAct_Watt() < node.getAct_Watt())
				&& (candidate.getIdle_Watt() <= node.getIdle_Watt());
		boolean efficiency3 = (candidate.getMax_Watt() <= node.getMax_Watt())
				&& (candidate.getAct_Watt() <= node.getAct_Watt())
				&& (candidate.getIdle_Watt() < node.getIdle_Watt());
		return efficiency1 || efficiency2 || efficiency3;
	}
	
	private static boolean isEnergeticImprovement(NetworkNode candidate, NetworkNode node) {
		return candidate.getMax_Watt() < node.getMax_Watt();
	}
	
	private static boolean isEnergeticImprovement(NetworkNode candidate1, NetworkNode candidate2, NetworkNode node) {
		return (candidate1.getMax_Watt() + candidate2.getMax_Watt()) < node.getMax_Watt();
	}
	
	private static boolean isEnergeticImprovement(UninterruptiblePowerSupply candidate, UninterruptiblePowerSupply ups) {
		return getEfficiency(candidate) < getEfficiency(ups);
	}

	private static boolean isEnergeticImprovement(UninterruptiblePowerSupply candidate1, 
			UninterruptiblePowerSupply candidate2, UninterruptiblePowerSupply ups) {
		return (getEfficiency(candidate1) + getEfficiency(candidate2)) < getEfficiency(ups);
	}

	private static boolean isEnergeticImprovement(Cooling candidate, Cooling cooling) {
		return candidate.getMax_Watt() < cooling.getMax_Watt();
	}

	private static boolean isEnergeticImprovement(Cooling candidate1,
			Cooling candidate2, Cooling cooling) {
		return (candidate1.getMax_Watt() + candidate2.getMax_Watt()) < cooling.getMax_Watt();
	}

	private static double getEfficiency(UninterruptiblePowerSupply ups) {
		return (((ups.getOut_Watt()/ups.getEfficiency())*100) - ups.getOut_Watt());
	}

	public static boolean isDeletable(UninterruptiblePowerSupply ups) {
		Room owningRoom = getOwningRoom(ups);
		int requiredWatt = getRequiredMaxWatt(owningRoom);
		int producedWatt = getProducedWatt(owningRoom);
		return (requiredWatt <=  (producedWatt - ups.getOut_Watt()));
	}

	public static boolean isDeletable(Cooling cooling) {
		Room owningRoom = getOwningRoom(cooling);
		int requiredWatt = getRequiredMaxWatt(owningRoom);
		int coolingCapacity = getCoolingCapacity(owningRoom);
		return (requiredWatt <=  (coolingCapacity - cooling.getCooling_Capacity()));
	}

	@SuppressWarnings("unchecked")
	public static boolean isOptimalDeletable(UninterruptiblePowerSupply ups) {
		if (isDeletable(ups)) {
			Room owningRoom = getOwningRoom(ups);
			EList<UninterruptiblePowerSupply> upss = owningRoom.getApplies();
			for (UninterruptiblePowerSupply candidate : upss) {
				if (candidate != ups && isDeletable(candidate) && isEnergeticImprovement(ups, candidate)) {
					return false;
				}
			}
			return true;
		}
		return false;
	}

	@SuppressWarnings("unchecked")
	public static boolean isOptimalDeletable(Cooling cooling) {
		if (isDeletable(cooling)) {
			Room owningRoom = getOwningRoom(cooling);
			EList<Cooling> coolings = owningRoom.getIncludes();
			for (Cooling candidate : coolings) {
				if (candidate != cooling && isDeletable(candidate) && isEnergeticImprovement(cooling, candidate)) {
					return false;
				}
			}
			return true;
		}
		return false;
	}
	
	private static List<CoolingPair> getCoolingPairsWithinSameRoom(Cooling cooling) {
		List<CoolingPair> coolingPairs = new ArrayList<CoolingPair>();
		List<EObject> allCoolings = getAllCoolings(cooling);
		Room room1 = getOwningRoom(cooling);
		for (EObject eObject : allCoolings) {
			Cooling candidate = (Cooling) eObject;
			if (candidate != cooling) {
				Room room2 = getOwningRoom(candidate);
				if (room1.equals(room2)) {
					CoolingPair pair = new CoolingPair(cooling, candidate);
					if (! coolingPairs.contains(pair)) coolingPairs.add(pair);
				}
			}
		}
		return coolingPairs;
	}
	
	private static List<EObject> getAllCoolings(EObject eObject) {
		List<EObject> allCoolings = new ArrayList<EObject>();
		TreeIterator<EObject> iter = eObject.eResource().getAllContents();
		while (iter.hasNext()) {
			EObject eObjectTmp = iter.next();
			if (eObjectTmp instanceof Cooling) {
				allCoolings.add((Cooling) eObjectTmp);
			}
		}
		return allCoolings;
	}

}
