/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.k3.al.annotationprocessor;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import fr.inria.diverse.k3.al.annotationprocessor.Aspect;
import fr.inria.diverse.k3.al.annotationprocessor.AspectProcessor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.xtend.lib.macro.CodeGenerationContext;
import org.eclipse.xtend.lib.macro.TransformationContext;
import org.eclipse.xtend.lib.macro.declaration.AnnotationReference;
import org.eclipse.xtend.lib.macro.declaration.ClassDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MethodDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableClassDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableMethodDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableTypeDeclaration;
import org.eclipse.xtend.lib.macro.declaration.ParameterDeclaration;
import org.eclipse.xtend.lib.macro.declaration.Type;
import org.eclipse.xtend.lib.macro.declaration.TypeDeclaration;
import org.eclipse.xtend.lib.macro.declaration.TypeReference;
import org.eclipse.xtend.lib.macro.file.Path;
import org.eclipse.xtend.lib.macro.services.GlobalTypeLookup;
import org.eclipse.xtend.lib.macro.services.TypeLookup;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * A tool class containing helper operations for k3.
 */
@SuppressWarnings("all")
public abstract class Helper {
  /**
   * The name of the parameter 'with' of the annotation aspect.
   */
  public final static String annotationWith = "with";
  
  /**
   * The name of the parameter 'className' of the annotation aspect.
   */
  public final static String annotationName = "className";
  
  /**
   * The name of the parameter 'transactionSupport' of the annotation aspect.
   */
  public final static String annotationTransactionSupportName = "transactionSupport";
  
  /**
   * Sorts the given classes following the inheritance order. Top classes are sorted at the end and
   * down classes are sorted the beginning.
   * 
   * @param classes classes
   * @param ctx transformation context
   */
  public static void sortClasses(final List<MutableClassDeclaration> classes, final TypeLookup ctx) {
    if (((classes == null) || (classes.size() < 2))) {
      return;
    }
    final int size = classes.size();
    int firstPosModif = (-1);
    boolean stable = false;
    final MutableClassDeclaration[] list = new MutableClassDeclaration[size];
    final HashMap<MutableClassDeclaration, HashSet<MutableClassDeclaration>> map = new HashMap<MutableClassDeclaration, HashSet<MutableClassDeclaration>>();
    Set<MutableClassDeclaration> listTmp = null;
    MutableClassDeclaration tmp = null;
    classes.<MutableClassDeclaration>toArray(list);
    final Consumer<MutableClassDeclaration> _function = (MutableClassDeclaration cl) -> {
      final HashSet<MutableClassDeclaration> st = new HashSet<MutableClassDeclaration>();
      Helper.getSuperClasses(cl, st, ctx);
      map.put(cl, st);
    };
    classes.forEach(_function);
    while ((!stable)) {
      {
        stable = true;
        final int start = Math.max(0, firstPosModif);
        firstPosModif = (-1);
        ExclusiveRange _doubleDotLessThan = new ExclusiveRange(start, (size - 1), true);
        for (final Integer i : _doubleDotLessThan) {
          {
            listTmp = map.get(list[((i).intValue() + 1)]);
            boolean _contains = listTmp.contains(list[(i).intValue()]);
            if (_contains) {
              stable = false;
              tmp = list[(i).intValue()];
              list[(i).intValue()] = list[((i).intValue() + 1)];
              list[((i).intValue() + 1)] = tmp;
              if (((firstPosModif > ((i).intValue() - 1)) || (firstPosModif == (-1)))) {
                firstPosModif = Math.max(0, ((i).intValue() - 1));
              }
            } else {
              listTmp = map.get(list[(i).intValue()]);
              boolean _contains_1 = listTmp.contains(list[((i).intValue() + 1)]);
              boolean _not = (!_contains_1);
              if (_not) {
                boolean sortedOnce = false;
                int j = ((i).intValue() - 1);
                listTmp = map.get(list[((i).intValue() + 1)]);
                while (((j >= 0) && (!sortedOnce))) {
                  boolean _contains_2 = listTmp.contains(list[j]);
                  if (_contains_2) {
                    tmp = list[j];
                    list[j] = list[((i).intValue() + 1)];
                    list[((i).intValue() + 1)] = tmp;
                    stable = false;
                    sortedOnce = true;
                    if (((start > (j - 1)) || (firstPosModif == (-1)))) {
                      firstPosModif = Math.max(0, (j - 1));
                    }
                  } else {
                    j = (j - 1);
                  }
                }
              }
            }
          }
        }
      }
    }
    map.clear();
    classes.clear();
    classes.addAll(IterableExtensions.<MutableClassDeclaration>toList(((Iterable<MutableClassDeclaration>)Conversions.doWrapArray(list))));
  }
  
  /**
   * On an aspect class, find its direct "super" aspect classes following either the traditional "extends"
   * or following the "with" attribute of the aspect annotation
   * 
   * It works by looking for types in the compilation unit classpath
   * 
   * primary extends, comes first, then the secondary "with" in order of appearance in the declaration
   * 
   * @param clazz class declaration
   * @param ctx transformation context (used for type lookup)
   * @return a set of class declarations that are either
   */
  public static List<ClassDeclaration> getDirectPrimaryAndSecondarySuperClasses(final ClassDeclaration clazz, final GlobalTypeLookup ctx) {
    final List<ClassDeclaration> res = CollectionLiterals.<ClassDeclaration>newArrayList();
    TypeReference _extendedClass = clazz.getExtendedClass();
    boolean _tripleNotEquals = (_extendedClass != null);
    if (_tripleNotEquals) {
      Type _findTypeGlobally = ctx.findTypeGlobally(clazz.getExtendedClass().getName());
      final ClassDeclaration l = ((ClassDeclaration) _findTypeGlobally);
      if ((l != null)) {
        res.add(l);
      }
    }
    final Function1<String, ClassDeclaration> _function = (String n) -> {
      Type _findTypeGlobally_1 = ctx.findTypeGlobally(n);
      return ((ClassDeclaration) _findTypeGlobally_1);
    };
    Iterables.<ClassDeclaration>addAll(res, IterableExtensions.<ClassDeclaration>filterNull(ListExtensions.<String, ClassDeclaration>map(Helper.getWithClassNames(clazz), _function)));
    return res;
  }
  
  /**
   * Fill superClasses with all super classes of classDecl
   * Applied on a normal class, will return full hierarchy of java extends only
   * Applied on an aspect class, it returns the hierarchy following both the java extends, and the <i>with</i>
   * attribute of the @aspect annotation
   */
  public static void getAllPrimaryAndSecondarySuperClasses(final List<ClassDeclaration> superClasses, final ClassDeclaration classDecl, @Extension final GlobalTypeLookup context) {
    final Function1<ClassDeclaration, Boolean> _function = (ClassDeclaration directsuperclass) -> {
      boolean _contains = superClasses.contains(directsuperclass);
      return Boolean.valueOf((!_contains));
    };
    final List<ClassDeclaration> directSuperClasses = IterableExtensions.<ClassDeclaration>toList(IterableExtensions.<ClassDeclaration>filter(Helper.getDirectPrimaryAndSecondarySuperClasses(classDecl, context), _function));
    superClasses.addAll(directSuperClasses);
    final Consumer<ClassDeclaration> _function_1 = (ClassDeclaration directsuperclass) -> {
      Helper.getAllPrimaryAndSecondarySuperClasses(superClasses, directsuperclass, context);
    };
    directSuperClasses.forEach(_function_1);
  }
  
  /**
   * Completes the list 'res' with all the super types of the given class 'clazz'.
   * 
   * Will look only in types in the current Compilation Unit
   * 
   * @param clazz class declaration
   * @param res returned list of super types of clazz
   * @param ctx transformation context
   */
  public static void getSuperClasses(final MutableClassDeclaration clazz, final Set<MutableClassDeclaration> res, @Extension final TypeLookup ctx) {
    boolean _contains = res.contains(clazz);
    if (_contains) {
      return;
    }
    res.add(clazz);
    TypeReference _extendedClass = clazz.getExtendedClass();
    boolean _tripleNotEquals = (_extendedClass != null);
    if (_tripleNotEquals) {
      final MutableClassDeclaration l = ctx.findClass(clazz.getExtendedClass().getName());
      if ((l != null)) {
        Helper.getSuperClasses(l, res, ctx);
      }
      final Function1<String, MutableClassDeclaration> _function = (String n) -> {
        return ctx.findClass(n);
      };
      final Consumer<MutableClassDeclaration> _function_1 = (MutableClassDeclaration cl) -> {
        if ((cl != null)) {
          Helper.getSuperClasses(cl, res, ctx);
        }
      };
      ListExtensions.<String, MutableClassDeclaration>map(Helper.getWithClassNames(clazz), _function).forEach(_function_1);
    }
  }
  
  public static List<MutableClassDeclaration> sortByClassInheritance(final MutableClassDeclaration clazz, final List<? extends MutableClassDeclaration> classes, @Extension final TypeLookup context) {
    final Set<MutableClassDeclaration> listTmp = new HashSet<MutableClassDeclaration>();
    final Set<MutableClassDeclaration> listRes = new HashSet<MutableClassDeclaration>();
    Helper.getSuperClasses(clazz, listRes, context);
    final Consumer<MutableClassDeclaration> _function = (MutableClassDeclaration c) -> {
      boolean _contains = listRes.contains(c);
      boolean _not = (!_contains);
      if (_not) {
        listTmp.clear();
        Helper.getSuperClasses(c, listTmp, context);
        boolean _contains_1 = listTmp.contains(clazz);
        if (_contains_1) {
          listRes.add(c);
        }
      }
    };
    classes.forEach(_function);
    final List<MutableClassDeclaration> list = IterableExtensions.<MutableClassDeclaration>toList(listRes);
    Helper.sortClasses(list, context);
    return list;
  }
  
  /**
   * Computes the names of the classes provided by the parameter 'with' of the annotation 'aspect'.
   * @param clazz aspect class
   * @return a list of names
   */
  public static List<String> getWithClassNames(final TypeDeclaration clazz) {
    final Function1<TypeReference, String> _function = (TypeReference it) -> {
      return it.getName();
    };
    return ListExtensions.<TypeReference, String>map(Helper.getAnnotationWithType(clazz), _function);
  }
  
  /**
   * Getting the classes identified by the parameter 'with'.
   * The returned list is never null but can be empty.
   * @param cl type declaration
   * @return list of type reference
   */
  public static List<TypeReference> getAnnotationWithType(final TypeDeclaration cl) {
    if (((cl == null) || (cl.getAnnotations() == null))) {
      return Collections.<TypeReference>emptyList();
    }
    try {
      final Function1<AnnotationReference, Boolean> _function = (AnnotationReference it) -> {
        TypeReference[] _classArrayValue = it.getClassArrayValue(Helper.annotationWith);
        return Boolean.valueOf((_classArrayValue != null));
      };
      final AnnotationReference annot = IterableExtensions.findFirst(cl.getAnnotations(), _function);
      if ((annot == null)) {
        return Collections.<TypeReference>emptyList();
      }
      TypeReference[] _classArrayValue = annot.getClassArrayValue(Helper.annotationWith);
      return new ArrayList<TypeReference>((Collection<? extends TypeReference>)Conversions.doWrapArray(_classArrayValue));
    } catch (final Throwable _t) {
      if (_t instanceof NullPointerException) {
        final NullPointerException ex = (NullPointerException)_t;
        return Collections.<TypeReference>emptyList();
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  public static TypeReference getAnnotationAspectType(final TypeDeclaration cl) {
    if (((cl == null) || (cl.getAnnotations() == null))) {
      return null;
    }
    try {
      final Function1<AnnotationReference, Boolean> _function = (AnnotationReference a) -> {
        return Boolean.valueOf((Objects.equal(a.getAnnotationTypeDeclaration().getSimpleName(), Aspect.class.getSimpleName()) && (a.getClassValue(Helper.annotationName) != null)));
      };
      final AnnotationReference annot = IterableExtensions.findFirst(cl.getAnnotations(), _function);
      if ((annot == null)) {
        return null;
      }
      return annot.getClassValue(Helper.annotationName);
    } catch (final Throwable _t) {
      if (_t instanceof NullPointerException) {
        final NullPointerException ex = (NullPointerException)_t;
        return null;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  /**
   * Computes the name of the class to aspectize identified by the annotation 'aspect'.
   * 
   * @param clazz type declaration
   * @return the name of the aspectized class
   */
  public static String getAspectedClassName(final TypeDeclaration clazz) {
    final TypeReference type = Helper.getAnnotationAspectType(clazz);
    if ((type == null)) {
      return "";
    }
    return type.getName();
  }
  
  public static List<MethodDeclaration> sortByMethodInheritance(final Set<MethodDeclaration> methods, final List<String> inheritOrder) {
    final Function1<String, Iterable<MethodDeclaration>> _function = (String classe) -> {
      final Function1<MethodDeclaration, Boolean> _function_1 = (MethodDeclaration it) -> {
        String _simpleName = it.getDeclaringType().getSimpleName();
        return Boolean.valueOf(Objects.equal(_simpleName, classe));
      };
      return IterableExtensions.<MethodDeclaration>filter(methods, _function_1);
    };
    return IterableExtensions.<MethodDeclaration>toList(Iterables.<MethodDeclaration>concat(ListExtensions.<String, Iterable<MethodDeclaration>>map(inheritOrder, _function)));
  }
  
  /**
   * Fill s with super classes of c, ordered by hierarchy
   * (the first element is the direct super type of c)
   * 
   * @param s list of class declarations
   * 
   * @param c class declaration
   * @param context transformation context
   */
  public static void getSuperClass(final List<ClassDeclaration> s, final ClassDeclaration c, @Extension final TypeLookup context) {
    TypeReference _extendedClass = c.getExtendedClass();
    boolean _tripleNotEquals = (_extendedClass != null);
    if (_tripleNotEquals) {
      Type _findTypeGlobally = context.findTypeGlobally(c.getExtendedClass().getName());
      final ClassDeclaration l = ((ClassDeclaration) _findTypeGlobally);
      boolean _equals = Objects.equal(l, c);
      if (_equals) {
        return;
      }
      if ((l != null)) {
        s.add(l);
        Helper.getSuperClass(s, l, context);
      }
    }
  }
  
  public static String mkstring(final List<TypeReference> list, final String delimiter, final String before, final String after) {
    int _size = list.size();
    boolean _equals = (_size == 0);
    if (_equals) {
      return "";
    }
    final StringBuffer s = new StringBuffer();
    if ((before != null)) {
      s.append(before);
    }
    final Consumer<TypeReference> _function = (TypeReference e) -> {
      s.append((delimiter + "?"));
    };
    list.forEach(_function);
    if ((after != null)) {
      s.append(after);
    }
    return s.toString().replace((before + delimiter), before);
  }
  
  /**
   * Find first method with the same name in super classes hierarchy
   * @param clazz This class and super classes are the search area
   * @param methodName Method to find
   * @param context transformation context
   * @return first method declaration with the same name asmethodName in super classes hierarchy, or null if none found
   */
  public static MethodDeclaration findMethod(final ClassDeclaration clazz, final MutableMethodDeclaration methodName, @Extension final TransformationContext context) {
    final Function1<MethodDeclaration, Boolean> _function = (MethodDeclaration it) -> {
      String _simpleName = it.getSimpleName();
      String _simpleName_1 = methodName.getSimpleName();
      return Boolean.valueOf(Objects.equal(_simpleName, _simpleName_1));
    };
    MethodDeclaration m = IterableExtensions.findFirst(clazz.getDeclaredMethods(), _function);
    if ((m == null)) {
      TypeReference _extendedClass = clazz.getExtendedClass();
      boolean _tripleEquals = (_extendedClass == null);
      if (_tripleEquals) {
        return null;
      } else {
        Type _findTypeGlobally = context.findTypeGlobally(clazz.getExtendedClass().getName());
        boolean _tripleEquals_1 = (_findTypeGlobally == null);
        if (_tripleEquals_1) {
          return null;
        } else {
          Type _findTypeGlobally_1 = context.findTypeGlobally(clazz.getExtendedClass().getName());
          return Helper.findMethod(((ClassDeclaration) _findTypeGlobally_1), methodName, context);
        }
      }
    } else {
      return m;
    }
  }
  
  /**
   * Compare the prototype of the two given methods.
   * @param m1 first method declaration
   * @param m2 second method declaration
   * @param alsoCheckFirstArgSelf If true, it checks that m2 equals m1 (respec. m1 equals m2) after being transformed for k3. More precisely, it checks
   * whether the first argument of m2 (respec. m1) is _self to then compare the two prototypes.
   * @return true if the 2 methods have similar signature
   */
  public static boolean isSamePrototype(final MutableMethodDeclaration m1, final MutableMethodDeclaration m2, final boolean alsoCheckFirstArgSelf) {
    final boolean nameOk = (((m1 != null) && (m2 != null)) && Objects.equal(m1.getSimpleName(), m2.getSimpleName()));
    final boolean ok = ((nameOk && (IterableExtensions.size(m1.getParameters()) == IterableExtensions.size(m2.getParameters()))) && Helper.isSameType(m1.getReturnType(), m2.getReturnType()));
    if (((nameOk && (!ok)) && alsoCheckFirstArgSelf)) {
      return (((IterableExtensions.size(m1.getParameters()) == (IterableExtensions.size(m2.getParameters()) + 1)) && Objects.equal(IterableExtensions.head(m1.getParameters()).getSimpleName(), AspectProcessor.SELF_VAR_NAME)) || (((IterableExtensions.size(m1.getParameters()) + 1) == IterableExtensions.size(m2.getParameters())) && Objects.equal(IterableExtensions.head(m2.getParameters()).getSimpleName(), AspectProcessor.SELF_VAR_NAME)));
    }
    return ok;
  }
  
  public static boolean isSameType(final TypeReference tr1, final TypeReference tr2) {
    return (Objects.equal(tr1, tr2) || (((tr1 != null) && (tr2 != null)) && Objects.equal(tr1.getSimpleName(), tr2.getSimpleName())));
  }
  
  /**
   * @param cls type declaration
   * @param ctx transformation context
   * 
   * @return the adapter standing between a metaclass in a sub-metamodel
   * and its corresponding metaclass in a super-metamodel
   */
  public static String getAdapterClassName(final MutableTypeDeclaration cls, @Extension final TransformationContext ctx) {
    final Function1<AnnotationReference, Boolean> _function = (AnnotationReference it) -> {
      Object _value = it.getValue("adapter");
      return Boolean.valueOf((_value != null));
    };
    final AnnotationReference ann = IterableExtensions.findFirst(cls.getAnnotations(), _function);
    String _xifexpression = null;
    if ((ann != null)) {
      Object _value = ann.getValue("adapter");
      _xifexpression = ((String) _value);
    } else {
      _xifexpression = "";
    }
    return _xifexpression;
  }
  
  /**
   * simple method signature
   */
  public static String methodSignature(final MethodDeclaration m) {
    StringConcatenation _builder = new StringConcatenation();
    String _simpleName = m.getReturnType().getSimpleName();
    _builder.append(_simpleName);
    _builder.append(" ");
    String _simpleName_1 = m.getSimpleName();
    _builder.append(_simpleName_1);
    _builder.append("(");
    final Function1<ParameterDeclaration, String> _function = (ParameterDeclaration p) -> {
      return p.getType().getSimpleName();
    };
    String _join = IterableExtensions.join(IterableExtensions.map(m.getParameters(), _function), ",");
    _builder.append(_join);
    _builder.append(")");
    return _builder.toString();
  }
  
  /**
   * similar to methodSignature but makes sure to have the signature from the original xtend
   * Ie. optionnaly removes the _self parameter introduced by {@link AspectProcessor#methodProcessingAddSelfStatic(MutableTypeDeclaration) }
   * This can be useful when mixing method declaration from both java and currently processed compilation unit
   */
  public static String initialMethodSignature(final MethodDeclaration m) {
    if ((IterableExtensions.isEmpty(m.getParameters()) || (!Objects.equal(IterableExtensions.head(m.getParameters()).getSimpleName(), AspectProcessor.SELF_VAR_NAME)))) {
      StringConcatenation _builder = new StringConcatenation();
      String _simpleName = m.getReturnType().getSimpleName();
      _builder.append(_simpleName);
      _builder.append(" ");
      String _simpleName_1 = m.getSimpleName();
      _builder.append(_simpleName_1);
      _builder.append("(");
      final Function1<ParameterDeclaration, String> _function = (ParameterDeclaration p) -> {
        return p.getType().getSimpleName();
      };
      String _join = IterableExtensions.join(IterableExtensions.map(m.getParameters(), _function), ",");
      _builder.append(_join);
      _builder.append(")");
      return _builder.toString();
    } else {
      final Iterable<? extends ParameterDeclaration> parameters = IterableExtensions.drop(m.getParameters(), 1);
      StringConcatenation _builder_1 = new StringConcatenation();
      String _simpleName_2 = m.getReturnType().getSimpleName();
      _builder_1.append(_simpleName_2);
      _builder_1.append(" ");
      String _simpleName_3 = m.getSimpleName();
      _builder_1.append(_simpleName_3);
      _builder_1.append("(");
      final Function1<ParameterDeclaration, String> _function_1 = (ParameterDeclaration p) -> {
        return p.getType().getSimpleName();
      };
      String _join_1 = IterableExtensions.join(IterableExtensions.map(parameters, _function_1), ",");
      _builder_1.append(_join_1);
      _builder_1.append(")");
      return _builder_1.toString();
    }
  }
  
  public static void writeContentsIfNew(final Path targetFilePath, final String newContent, @Extension final CodeGenerationContext context) {
    boolean write = false;
    boolean _exists = context.exists(targetFilePath);
    boolean _not = (!_exists);
    if (_not) {
      write = true;
    } else {
      CharSequence _contents = context.getContents(targetFilePath);
      boolean _notEquals = (!Objects.equal(_contents, newContent));
      if (_notEquals) {
        write = true;
      }
    }
    if (write) {
      context.setContents(targetFilePath, newContent);
    }
  }
}
