/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.trace.gemoc.generator.util;

import com.google.common.base.Objects;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.jar.Manifest;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.codegen.ecore.generator.Generator;
import org.eclipse.emf.codegen.ecore.genmodel.GenJDKLevel;
import org.eclipse.emf.codegen.ecore.genmodel.GenModel;
import org.eclipse.emf.codegen.ecore.genmodel.GenModelFactory;
import org.eclipse.emf.codegen.ecore.genmodel.GenPackage;
import org.eclipse.emf.codegen.ecore.genmodel.GenParameter;
import org.eclipse.emf.codegen.ecore.genmodel.generator.GenBaseGeneratorAdapter;
import org.eclipse.emf.codegen.ecore.genmodel.util.GenModelUtil;
import org.eclipse.emf.codegen.util.CodeGenUtil;
import org.eclipse.emf.common.util.BasicMonitor;
import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EParameter;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.plugin.EcorePlugin;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gemoc.trace.gemoc.generator.util.PluginProjectHelper;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.ui.PlatformUI;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.Pure;

@SuppressWarnings("all")
public class StandaloneEMFProjectGenerator {
  protected final EPackage ecoreModel;
  
  protected final String projectName;
  
  @Accessors({ AccessorType.PUBLIC_GETTER, AccessorType.PROTECTED_SETTER })
  protected IProject project;
  
  @Accessors({ AccessorType.PUBLIC_GETTER, AccessorType.PROTECTED_SETTER })
  protected final Set<GenPackage> referencedGenPackages = new HashSet<GenPackage>();
  
  @Accessors({ AccessorType.PUBLIC_GETTER, AccessorType.PROTECTED_SETTER })
  protected final Set<EPackage> rootPackages = new HashSet<EPackage>();
  
  protected GenModel genModel;
  
  /**
   * Helper method to generate code without a job.
   */
  public void generateModelCode() {
    try {
      final IRunnableWithProgress _function = (IProgressMonitor m) -> {
        try {
          this.generateModelCode(m);
        } catch (Throwable _e) {
          throw Exceptions.sneakyThrow(_e);
        }
      };
      PlatformUI.getWorkbench().getActiveWorkbenchWindow().run(false, true, _function);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public final static String MODEL_GEN_FOLDER = "model";
  
  protected IProgressMonitor progressMonitor;
  
  protected ResourceSet resourceSet;
  
  protected URI modelGenFolderURI;
  
  protected String srcFolderPathString;
  
  protected Resource ecoreModelResource;
  
  public StandaloneEMFProjectGenerator(final String projectName, final EPackage p) {
    this.ecoreModel = p;
    this.projectName = projectName;
  }
  
  /**
   * Creates a new EMF project with the ecore file and the genmodel in the "model" folder
   * also mages project, referencedGenPackages and rootPackages available.
   */
  public void generateBaseEMFProject(final IProgressMonitor m) {
    try {
      this.progressMonitor = m;
      ResourceSetImpl _resourceSetImpl = new ResourceSetImpl();
      this.resourceSet = _resourceSetImpl;
      this.project = PluginProjectHelper.createPluginProject(this.projectName, 
        Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("src")), 
        Collections.<IProject>unmodifiableList(CollectionLiterals.<IProject>newArrayList()), 
        Collections.<String>unmodifiableSet(CollectionLiterals.<String>newHashSet()), 
        Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList()), m);
      this.modelGenFolderURI = this.setupModelGenFolder();
      this.srcFolderPathString = this.setupSrcFolder();
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(this.projectName);
      _builder.append("/");
      _builder.append(StandaloneEMFProjectGenerator.MODEL_GEN_FOLDER);
      _builder.append("/");
      String _name = this.ecoreModel.getName();
      _builder.append(_name);
      _builder.append(".ecore");
      this.ecoreModelResource = this.resourceSet.createResource(
        URI.createPlatformResourceURI(_builder.toString(), true));
      this.ecoreModelResource.getContents().add(this.ecoreModel);
      this.save(this.ecoreModelResource);
      this.ecoreModelResource.unload();
      this.ecoreModelResource.load(null);
      this.ecoreModelResource.unload();
      this.ecoreModelResource.load(null);
      this.checkReferencedPackages(this.ecoreModelResource);
      EObject _get = this.ecoreModelResource.getContents().get(0);
      this.genModel = this.generateGenModel(((EPackage) _get), this.modelGenFolderURI);
      this.rootPackages.addAll(IterableExtensions.<EPackage>toSet(Iterables.<EPackage>filter(this.ecoreModelResource.getContents(), EPackage.class)));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Generates the code using the genmodel (within a Job).
   */
  public void generateModelCode(final IProgressMonitor m) throws Exception {
    this.generateCode(this.progressMonitor);
  }
  
  private URI setupModelGenFolder() {
    URI modelGenFolderURI = null;
    final IFolder modelGenFolder = this.project.getFolder(StandaloneEMFProjectGenerator.MODEL_GEN_FOLDER);
    boolean _exists = modelGenFolder.exists();
    boolean _not = (!_exists);
    if (_not) {
      try {
        modelGenFolder.create(true, true, null);
      } catch (final Throwable _t) {
        if (_t instanceof CoreException) {
          final CoreException e = (CoreException)_t;
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("The folder \'");
          _builder.append(StandaloneEMFProjectGenerator.MODEL_GEN_FOLDER);
          _builder.append("\' could not be created.");
          throw new RuntimeException(_builder.toString(), e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    modelGenFolderURI = URI.createPlatformResourceURI(modelGenFolder.getFullPath().toString(), true);
    return modelGenFolderURI;
  }
  
  private String setupSrcFolder() {
    String srcFolderPathString = null;
    final IFolder srcFolder = this.project.getFolder("src");
    boolean _exists = srcFolder.exists();
    boolean _not = (!_exists);
    if (_not) {
      try {
        srcFolder.create(true, true, null);
      } catch (final Throwable _t) {
        if (_t instanceof CoreException) {
          final CoreException e = (CoreException)_t;
          throw new RuntimeException("The source folder \'src\' could not be created.", e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    srcFolderPathString = srcFolder.getFullPath().toString();
    return srcFolderPathString;
  }
  
  private void checkReferencedPackages(final Resource xmofModelResource) {
    final Set<URI> missingPackages = new HashSet<URI>();
    final Map<EObject, Collection<EStructuralFeature.Setting>> externalCrossReferences = EcoreUtil.ExternalCrossReferencer.find(xmofModelResource);
    Set<EObject> _keySet = externalCrossReferences.keySet();
    for (final EObject eObject : _keySet) {
      boolean _eIsProxy = eObject.eIsProxy();
      if (_eIsProxy) {
        missingPackages.add(EcoreUtil.getURI(eObject).trimFragment());
      }
    }
    int _size = missingPackages.size();
    boolean _greaterThan = (_size > 0);
    if (_greaterThan) {
      String _xifexpression = null;
      int _size_1 = missingPackages.size();
      boolean _equals = (_size_1 == 1);
      if (_equals) {
        _xifexpression = "";
      } else {
        _xifexpression = "s";
      }
      String _plus = ("Unable to load the following referenced resource" + _xifexpression);
      String _plus_1 = (_plus + ": ");
      String _string = missingPackages.toString();
      final String message = (_plus_1 + _string);
      throw new RuntimeException(message);
    }
  }
  
  protected GenModel generateGenModel(final EPackage rootEPackage, final URI modelGenFolderURI) {
    final Resource genModelResource = this.createGenModel(rootEPackage);
    EObject _get = genModelResource.getContents().get(0);
    final GenModel genModel = ((GenModel) _get);
    this.setInitializeByLoad(genModel);
    this.save(genModelResource);
    return genModel;
  }
  
  protected Resource createGenModel(final EPackage rootEPackage) {
    final Resource ecoreModelResource = rootEPackage.eResource();
    final String genModelFileName = ecoreModelResource.getURI().trimFileExtension().appendFileExtension("genmodel").lastSegment().toString();
    final URI genModelURI = this.modelGenFolderURI.appendSegment(genModelFileName);
    final Resource genModelResource = this.resourceSet.createResource(genModelURI);
    final GenModel genModel = GenModelFactory.eINSTANCE.createGenModel();
    genModelResource.getContents().add(genModel);
    final IFolder srcFolder = this.project.getFolder("src");
    genModel.setModelDirectory(srcFolder.getFullPath().toString());
    genModel.getForeignModel().add(ecoreModelResource.getURI().toString());
    genModel.setModelName(this.getModelName(genModelURI));
    genModel.setModelPluginID(this.getPluginID(genModelURI));
    genModel.setRootExtendsClass("org.eclipse.emf.ecore.impl.MinimalEObjectImpl$Container");
    genModel.setComplianceLevel(this.getComplicanceLevel());
    genModel.setImportOrganizing(true);
    genModel.initialize(Collections.<EPackage>singleton(rootEPackage));
    this.setMissingParameterTypes(genModel);
    this.fixUsedGenPackages(genModel);
    return genModelResource;
  }
  
  private Set<GenModel> fixedGenModels = new HashSet<GenModel>();
  
  /**
   * Tries to fix the "usedGenPackages" collection of a genmodel (and recursively of all genmodels it references)
   * 1) remove all usedGenPackages that have a null genModel (for a mysterious reason...)
   * 2) use the magical method 'computeMissingGenPackages' to find missing packages, and add them to usedGenPackages
   * 3) as a bonus, store all referenced gen packages in 'referencedGenPackages' for later use
   */
  private void fixUsedGenPackages(final GenModel genModel) {
    boolean _contains = this.fixedGenModels.contains(genModel);
    boolean _not = (!_contains);
    if (_not) {
      this.fixedGenModels.add(genModel);
      final Function1<GenPackage, Boolean> _function = (GenPackage p) -> {
        GenModel _genModel = p.getGenModel();
        return Boolean.valueOf(Objects.equal(_genModel, null));
      };
      CollectionExtensions.<GenPackage>removeAll(genModel.getUsedGenPackages(), IterableExtensions.<GenPackage>filter(ImmutableList.<GenPackage>copyOf(genModel.getUsedGenPackages()), _function));
      final List<GenPackage> missingGenPackages = this.computeMissingGenPackages(genModel);
      for (final GenPackage genPackage : missingGenPackages) {
        this.fixUsedGenPackages(genPackage.getGenModel());
      }
      this.referencedGenPackages.addAll(missingGenPackages);
      this.referencedGenPackages.addAll(genModel.getGenPackages());
      genModel.getUsedGenPackages().addAll(missingGenPackages);
    }
  }
  
  protected String getModelName(final URI genModelURI) {
    final String genModelFileName = genModelURI.trimFileExtension().lastSegment();
    String _upperCase = genModelFileName.substring(0, 1).toUpperCase();
    String _substring = genModelFileName.substring(1);
    final String modelName = (_upperCase + _substring);
    return modelName;
  }
  
  protected String getPluginID(final URI uri) {
    String pluginID = "";
    final IFile manifestFile = this.project.getFolder("META-INF").getFile("MANIFEST.MF");
    try {
      InputStream _contents = manifestFile.getContents();
      final Manifest manifest = new Manifest(_contents);
      String symbolicName = manifest.getMainAttributes().getValue("Bundle-SymbolicName");
      boolean _notEquals = (!Objects.equal(symbolicName, null));
      if (_notEquals) {
        final int index = symbolicName.indexOf(";");
        if ((index > 0)) {
          symbolicName = symbolicName.substring(0, index);
        }
        pluginID = symbolicName.trim();
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        String _string = manifestFile.getFullPath().toString();
        String _plus = ("Could not find manifest file \'" + _string);
        String _plus_1 = (_plus + "\'.");
        throw new RuntimeException(_plus_1, e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return pluginID;
  }
  
  private GenJDKLevel getComplicanceLevel() {
    final String complianceLevel = CodeGenUtil.EclipseUtil.getJavaComplianceLevel(this.project);
    boolean _equals = "1.4".equals(complianceLevel);
    if (_equals) {
      return GenJDKLevel.JDK14_LITERAL;
    } else {
      boolean _equals_1 = "1.5".equals(complianceLevel);
      if (_equals_1) {
        return GenJDKLevel.JDK50_LITERAL;
      } else {
        boolean _equals_2 = "1.6".equals(complianceLevel);
        if (_equals_2) {
          return GenJDKLevel.JDK60_LITERAL;
        } else {
          boolean _equals_3 = "1.7".equals(complianceLevel);
          if (_equals_3) {
            return GenJDKLevel.JDK70_LITERAL;
          } else {
            return GenJDKLevel.JDK80_LITERAL;
          }
        }
      }
    }
  }
  
  /**
   * In case of missing parameter types, the types are temporarily set to
   * EObject
   * 
   * @param genModel
   */
  private void setMissingParameterTypes(final GenModel genModel) {
    Set<GenParameter> _set = IteratorExtensions.<GenParameter>toSet(Iterators.<GenParameter>filter(genModel.eAllContents(), GenParameter.class));
    for (final GenParameter genModelElement : _set) {
      {
        final GenParameter genParameter = ((GenParameter) genModelElement);
        final EParameter ecoreParameter = genParameter.getEcoreParameter();
        EClassifier _eType = ecoreParameter.getEType();
        boolean _equals = Objects.equal(_eType, null);
        if (_equals) {
          ecoreParameter.setEType(EcorePackage.eINSTANCE.getEObject());
        }
      }
    }
  }
  
  protected List<GenPackage> computeMissingGenPackages(final GenModel genModel) {
    final List<GenPackage> missingGenPackages = new UniqueEList<GenPackage>();
    final Map<String, URI> genModelLocationMapTargetEnvironment = EcorePlugin.getEPackageNsURIToGenModelLocationMap(true);
    final Map<String, URI> genModelLocationMapEnvironment = EcorePlugin.getEPackageNsURIToGenModelLocationMap(false);
    List<EPackage> _missingPackages = genModel.getMissingPackages();
    for (final EPackage ePackage : _missingPackages) {
      boolean _notEquals = (!Objects.equal(ePackage, null));
      if (_notEquals) {
        URI missingGenModelURI = genModelLocationMapEnvironment.get(ePackage.getNsURI());
        boolean _equals = Objects.equal(missingGenModelURI, null);
        if (_equals) {
          missingGenModelURI = genModelLocationMapTargetEnvironment.get(ePackage.getNsURI());
        }
        boolean _equals_1 = Objects.equal(missingGenModelURI, null);
        if (_equals_1) {
          String _nsURI = ePackage.getNsURI();
          String _plus = ("Unable to load generator model of required package \'" + _nsURI);
          String _plus_1 = (_plus + "\'.");
          throw new RuntimeException(_plus_1);
        }
        Resource missingGenModelResource = null;
        try {
          missingGenModelResource = this.resourceSet.getResource(missingGenModelURI, true);
        } catch (final Throwable _t) {
          if (_t instanceof RuntimeException) {
            String _nsURI_1 = ePackage.getNsURI();
            String _plus_2 = ("Unable to load generator model of required package \'" + _nsURI_1);
            String _plus_3 = (_plus_2 + "\'.");
            throw new RuntimeException(_plus_3);
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
        EObject _get = missingGenModelResource.getContents().get(0);
        final GenModel missingGenModel = ((GenModel) _get);
        missingGenPackages.addAll(missingGenModel.getGenPackages());
      }
    }
    return missingGenPackages;
  }
  
  protected void setInitializeByLoad(final GenModel genModel) {
    EList<GenPackage> _genPackages = genModel.getGenPackages();
    for (final GenPackage genPackage : _genPackages) {
      this.setInitializeByLoad(genPackage);
    }
  }
  
  private void setInitializeByLoad(final GenPackage genPackage) {
    genPackage.setLoadInitialization(false);
    List<GenPackage> _subGenPackages = genPackage.getSubGenPackages();
    for (final GenPackage subGenPackage : _subGenPackages) {
      this.setInitializeByLoad(subGenPackage);
    }
  }
  
  private List<Diagnostic> expandDiagnostics(final Diagnostic d) {
    final ArrayList<Diagnostic> result = new ArrayList<Diagnostic>();
    result.add(d);
    result.addAll(d.getChildren());
    List<Diagnostic> _children = d.getChildren();
    for (final Diagnostic c : _children) {
      result.addAll(this.expandDiagnostics(c));
    }
    return result;
  }
  
  private String exceptionToStackString(final Throwable e) {
    final StringWriter sw = new StringWriter();
    PrintWriter _printWriter = new PrintWriter(sw);
    e.printStackTrace(_printWriter);
    final String exceptionAsString = sw.toString();
    return exceptionAsString;
  }
  
  private String diagnosticErrorsToString(final Diagnostic diagnostic) {
    final Function1<Diagnostic, Boolean> _function = (Diagnostic d) -> {
      int _severity = d.getSeverity();
      return Boolean.valueOf((_severity == Diagnostic.ERROR));
    };
    final Set<Diagnostic> errors = IterableExtensions.<Diagnostic>toSet(IterableExtensions.<Diagnostic>filter(this.expandDiagnostics(diagnostic), _function));
    final Function1<Diagnostic, Boolean> _function_1 = (Diagnostic e) -> {
      Throwable _exception = e.getException();
      return Boolean.valueOf((!Objects.equal(_exception, null)));
    };
    final Function1<Diagnostic, String> _function_2 = (Diagnostic e) -> {
      return this.exceptionToStackString(e.getException());
    };
    final Set<String> exceptions = IterableExtensions.<String>toSet(IterableExtensions.<Diagnostic, String>map(IterableExtensions.<Diagnostic>filter(errors, _function_1), _function_2));
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _hasElements = false;
      for(final String e : exceptions) {
        if (!_hasElements) {
          _hasElements = true;
          _builder.append("Encountered exceptions:\n");
        } else {
          _builder.appendImmediate("\n", "");
        }
        _builder.append("- ");
        _builder.append(e);
        _builder.newLineIfNotEmpty();
      }
      if (_hasElements) {
        _builder.append("\n");
      }
    }
    {
      boolean _hasElements_1 = false;
      for(final Diagnostic e_1 : errors) {
        if (!_hasElements_1) {
          _hasElements_1 = true;
          _builder.append("Encountered diagnostic errors:\n");
        } else {
          _builder.appendImmediate("\n", "");
        }
        _builder.append("- ");
        String _message = e_1.getMessage();
        _builder.append(_message);
        _builder.append(" ");
        _builder.append("\t\t\t");
      }
      if (_hasElements_1) {
        _builder.append("\n");
      }
    }
    _builder.newLineIfNotEmpty();
    return _builder.toString();
  }
  
  protected void generateCode(final IProgressMonitor progressMonitor) throws Exception {
    boolean success = false;
    this.prepareGenModelForCodeGeneration(this.genModel);
    final Generator generator = GenModelUtil.createGenerator(this.genModel);
    final boolean canGenerate = generator.canGenerate(this.genModel, GenBaseGeneratorAdapter.MODEL_PROJECT_TYPE);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Cannot generate code of EPackage ");
    String _name = this.ecoreModel.getName();
    _builder.append(_name);
    String message = _builder.toString();
    if (canGenerate) {
      Diagnostic diagnostic = null;
      String otherMessage = "";
      try {
        diagnostic = generator.generate(this.genModel, GenBaseGeneratorAdapter.MODEL_PROJECT_TYPE, 
          BasicMonitor.toMonitor(progressMonitor));
      } catch (final Throwable _t) {
        if (_t instanceof Throwable) {
          final Throwable t = (Throwable)_t;
          otherMessage = this.exceptionToStackString(t);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      if (((!Objects.equal(diagnostic, null)) && (diagnostic.getSeverity() == Diagnostic.OK))) {
        success = true;
      } else {
        boolean _notEquals = (!Objects.equal(diagnostic, null));
        if (_notEquals) {
          String _message = message;
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append(": ");
          String _diagnosticErrorsToString = this.diagnosticErrorsToString(diagnostic);
          _builder_1.append(_diagnosticErrorsToString);
          _builder_1.append(".");
          message = (_message + _builder_1);
        } else {
          String _message_1 = message;
          message = (_message_1 + otherMessage);
        }
      }
    } else {
      String _message_2 = message;
      message = (_message_2 + "generator.canGenerate returns false.");
    }
    if ((!success)) {
      throw new Exception(message);
    }
  }
  
  protected void prepareGenModelForCodeGeneration(final GenModel genModel) {
    genModel.reconcile();
    genModel.setCanGenerate(true);
  }
  
  private void save(final Resource resource) {
    try {
      resource.save(Collections.EMPTY_MAP);
    } catch (final Throwable _t) {
      if (_t instanceof IOException) {
        final IOException e = (IOException)_t;
        URI _uRI = resource.getURI();
        String _plus = ("Could not save resource \'" + _uRI);
        String _plus_1 = (_plus + "\'.");
        throw new RuntimeException(_plus_1, e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  @Pure
  public IProject getProject() {
    return this.project;
  }
  
  protected void setProject(final IProject project) {
    this.project = project;
  }
  
  @Pure
  public Set<GenPackage> getReferencedGenPackages() {
    return this.referencedGenPackages;
  }
  
  @Pure
  public Set<EPackage> getRootPackages() {
    return this.rootPackages;
  }
}
