/**
 * Copyright (c) 2017 Inria.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.xdsmlframework.test.lib;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import fr.inria.diverse.melange.metamodel.melange.ModelTypingSpace;
import java.util.List;
import java.util.function.Consumer;
import java.util.zip.ZipFile;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.compare.Comparison;
import org.eclipse.emf.compare.EMFCompare;
import org.eclipse.emf.compare.Match;
import org.eclipse.emf.compare.diff.DefaultDiffEngine;
import org.eclipse.emf.compare.diff.FeatureFilter;
import org.eclipse.emf.compare.scope.DefaultComparisonScope;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.gemoc.trace.gemoc.generator.TraceAddonGeneratorIntegration;
import org.eclipse.gemoc.xdsmlframework.ide.ui.XDSMLFrameworkUI;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.pde.internal.core.natures.PDE;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.IOverwriteQuery;
import org.eclipse.ui.internal.wizards.datatransfer.ZipLeveledStructureProvider;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;
import org.eclipse.ui.wizards.datatransfer.ImportOperation;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.junit4.ui.util.IResourcesSetupUtil;
import org.eclipse.xtext.junit4.ui.util.JavaProjectSetupUtil;
import org.eclipse.xtext.resource.DerivedStateAwareResource;
import org.eclipse.xtext.ui.XtextProjectHelper;
import org.eclipse.xtext.ui.editor.XtextEditor;
import org.eclipse.xtext.ui.editor.outline.IOutlineNode;
import org.eclipse.xtext.ui.editor.outline.impl.OutlinePage;
import org.eclipse.xtext.ui.editor.utils.EditorUtils;
import org.eclipse.xtext.ui.resource.XtextResourceSetProvider;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.junit.Assert;

/**
 * Class containing helper methods for testing a workspace in a GEMOC Language workbench
 */
@SuppressWarnings("all")
public class MelangeWorkspaceTestHelper extends WorkspaceTestHelper {
  private static final String MELANGE_CMD_GENERATE_ALL = "fr.inria.diverse.melange.GenerateAll";
  
  private static final String MELANGE_CMD_GENERATE_ADAPTERS = "fr.inria.diverse.melange.GenerateAdapters";
  
  private static final String MELANGE_CMD_GENERATE_LANGUAGES = "fr.inria.diverse.melange.GenerateLanguages";
  
  private static final String MELANGE_CMD_GENERATE_INTERFACES = "fr.inria.diverse.melange.GenerateInterfaces";
  
  private static final String MELANGE_CMD_GENERATE_TRACE = "org.eclipse.gemoc.execution.sequential.javaxdsml.ide.ui.commands.GenerateTraceAddonPlugin";
  
  private static final String MELANGE_CMD_CLEAN_ALL = "fr.inria.diverse.melange.CleanAll";
  
  private static final String MELANGE_EDITOR_ID = "fr.inria.diverse.melange.Melange";
  
  @Inject
  private XtextResourceSetProvider rsProvider;
  
  @Override
  public void init() {
    Display _default = Display.getDefault();
    _default.syncExec(new Runnable() {
      @Override
      public void run() {
        try {
          PlatformUI.getWorkbench().showPerspective(XDSMLFrameworkUI.ID_PERSPECTIVE, PlatformUI.getWorkbench().getActiveWorkbenchWindow());
          MelangeWorkspaceTestHelper.this.closeWelcomePage();
        } catch (Throwable _e) {
          throw Exceptions.sneakyThrow(_e);
        }
      }
    });
  }
  
  public IProject deployMelangeProject(final String projectName, final String zipLocation) {
    try {
      final IJavaProject newProject = JavaProjectSetupUtil.createJavaProject(projectName);
      JavaProjectSetupUtil.addSourceFolder(newProject, "src");
      JavaProjectSetupUtil.addSourceFolder(newProject, "src-gen");
      IResourcesSetupUtil.addNature(newProject.getProject(), XtextProjectHelper.NATURE_ID);
      IResourcesSetupUtil.addNature(newProject.getProject(), PDE.PLUGIN_NATURE);
      IResourcesSetupUtil.addBuilder(newProject.getProject(), XtextProjectHelper.BUILDER_ID);
      IResourcesSetupUtil.addBuilder(newProject.getProject(), PDE.MANIFEST_BUILDER_ID);
      IResourcesSetupUtil.addBuilder(newProject.getProject(), PDE.SCHEMA_BUILDER_ID);
      Path _path = new Path("org.eclipse.xtend.XTEND_CONTAINER");
      JavaProjectSetupUtil.addToClasspath(newProject, 
        JavaCore.newContainerEntry(_path));
      Path _path_1 = new Path("org.eclipse.pde.core.requiredPlugins");
      JavaProjectSetupUtil.addToClasspath(newProject, 
        JavaCore.newContainerEntry(_path_1));
      final ZipFile zip = new ZipFile(zipLocation);
      final ZipLeveledStructureProvider structureProvider = new ZipLeveledStructureProvider(zip);
      final IOverwriteQuery queryOverwrite = new IOverwriteQuery() {
        @Override
        public String queryOverwrite(final String file) {
          return IOverwriteQuery.ALL;
        }
      };
      IPath _fullPath = newProject.getProject().getFullPath();
      Object _root = structureProvider.getRoot();
      ImportOperation _importOperation = new ImportOperation(_fullPath, _root, structureProvider, queryOverwrite);
      NullProgressMonitor _nullProgressMonitor = new NullProgressMonitor();
      _importOperation.run(_nullProgressMonitor);
      zip.close();
      return newProject.getProject();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public void generateAll(final String melangeFile) {
    WorkspaceTestHelper.invokeCommandOnSelectedFile(MelangeWorkspaceTestHelper.MELANGE_CMD_GENERATE_ALL, melangeFile);
  }
  
  public void generateAdapters(final String melangeFile) {
    WorkspaceTestHelper.invokeCommandOnSelectedFile(MelangeWorkspaceTestHelper.MELANGE_CMD_GENERATE_ADAPTERS, melangeFile);
  }
  
  public void generateLanguages(final String melangeFile) {
    WorkspaceTestHelper.invokeCommandOnSelectedFile(MelangeWorkspaceTestHelper.MELANGE_CMD_GENERATE_LANGUAGES, melangeFile);
  }
  
  public void generateInterfaces(final String melangeFile) {
    WorkspaceTestHelper.invokeCommandOnSelectedFile(MelangeWorkspaceTestHelper.MELANGE_CMD_GENERATE_INTERFACES, melangeFile);
  }
  
  public void generateTrace(final String melangeFile, final String languageName, final String targetProjectName) {
    abstract class __MelangeWorkspaceTestHelper_3 extends Job {
      __MelangeWorkspaceTestHelper_3(final String arg0) {
        super(arg0);
      }
      
      Exception reportedJobException;
    }
    
    try {
      IWorkspaceRoot _root = ResourcesPlugin.getWorkspace().getRoot();
      Path _path = new Path(melangeFile);
      final IFile mlgFile = _root.getFile(_path);
      String _string = melangeFile.toString();
      String _plus = ("Generating trace addon plugin for " + _string);
      final __MelangeWorkspaceTestHelper_3 j = new __MelangeWorkspaceTestHelper_3(_plus) {
        @Override
        protected IStatus run(final IProgressMonitor monitor) {
          try {
            TraceAddonGeneratorIntegration.generateAddon(mlgFile, languageName, targetProjectName, true, monitor);
          } catch (final Throwable _t) {
            if (_t instanceof Exception) {
              final Exception e = (Exception)_t;
              this.reportedJobException = e;
            } else {
              throw Exceptions.sneakyThrow(_t);
            }
          }
          return new Status(Status.OK, "org.eclipse.gemoc.trace.gemoc.ui", "Multidimensional Trace addon plugin generated.");
        }
      };
      j.schedule();
      j.join();
      boolean _notEquals = (!Objects.equal(j.reportedJobException, null));
      if (_notEquals) {
        throw j.reportedJobException;
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public void cleanAll(final String melangeFile) {
    Display _default = Display.getDefault();
    _default.syncExec(new Runnable() {
      @Override
      public void run() {
        WorkspaceTestHelper.invokeCommandOnSelectedFile(MelangeWorkspaceTestHelper.MELANGE_CMD_CLEAN_ALL, melangeFile);
      }
    });
  }
  
  public XtextEditor openEditor(final String melangeFile) {
    try {
      final IWorkspace ws = ResourcesPlugin.getWorkspace();
      IWorkspaceRoot _root = ws.getRoot();
      Path _path = new Path(melangeFile);
      final IFile mlgFile = _root.getFile(_path);
      final IWorkbench wb = PlatformUI.getWorkbench();
      IWorkbenchPage _activePage = wb.getActiveWorkbenchWindow().getActivePage();
      FileEditorInput _fileEditorInput = new FileEditorInput(mlgFile);
      final IEditorPart openEditor = _activePage.openEditor(_fileEditorInput, MelangeWorkspaceTestHelper.MELANGE_EDITOR_ID);
      final XtextEditor xtextEditor = EditorUtils.getXtextEditor(openEditor);
      if ((xtextEditor != null)) {
        xtextEditor.selectAndReveal(0, 0);
        xtextEditor.getInternalSourceViewer().setSelectedRange(0, 0);
        xtextEditor.getInternalSourceViewer().getTextWidget().setFocus();
        return xtextEditor;
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        e.printStackTrace();
        Assert.fail(e.getMessage());
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return null;
  }
  
  public TreeViewer getOutline(final String melangeFile) {
    final XtextEditor editor = this.openEditor(melangeFile);
    IContentOutlinePage _adapter = editor.<IContentOutlinePage>getAdapter(IContentOutlinePage.class);
    final OutlinePage outlinePage = ((OutlinePage) _adapter);
    final TreeViewer treeViewer = outlinePage.getTreeViewer();
    return treeViewer;
  }
  
  /**
   * Check for each aspect from {@link aspects} that K3-generated files are inside {@link project}
   * 
   * @param aspects Pairs of [AspectName->AspectedClass]
   */
  public void assertK3AspectsExists(final List<Pair<String, String>> aspects, final String project) {
    final String ASPECTS_NS = (project + ".aspects");
    final String ASPECTS_FOLDER = ASPECTS_NS.replaceAll("\\.", "/");
    final String SRC_GEN = "src-gen";
    final Consumer<Pair<String, String>> _function = (Pair<String, String> asp) -> {
      final String aspectName = asp.getKey();
      final String targetClass = asp.getValue();
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(project);
      _builder.append("/");
      _builder.append(SRC_GEN);
      _builder.append("/");
      _builder.append(ASPECTS_FOLDER);
      _builder.append("/");
      _builder.append(aspectName);
      _builder.append(".java");
      final String aspect = _builder.toString();
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append(project);
      _builder_1.append("/");
      _builder_1.append(SRC_GEN);
      _builder_1.append("/");
      _builder_1.append(ASPECTS_FOLDER);
      _builder_1.append("/");
      _builder_1.append(aspectName);
      _builder_1.append(targetClass);
      _builder_1.append("AspectContext.java");
      final String context = _builder_1.toString();
      StringConcatenation _builder_2 = new StringConcatenation();
      _builder_2.append(project);
      _builder_2.append("/");
      _builder_2.append(SRC_GEN);
      _builder_2.append("/");
      _builder_2.append(ASPECTS_FOLDER);
      _builder_2.append("/");
      _builder_2.append(aspectName);
      _builder_2.append(targetClass);
      _builder_2.append("AspectProperties.java");
      final String properties = _builder_2.toString();
      this.assertFileExists(aspect);
      this.assertFileExists(context);
      this.assertFileExists(properties);
    };
    aspects.forEach(_function);
  }
  
  /**
   * Check for each aspect from {@link aspects} that K3-generated files are NOT inside {@link project}
   * 
   * @param aspects Pairs of [AspectName->AspectedClass]
   */
  public void assertK3AspectsDontExists(final List<Pair<String, String>> aspects, final String project) {
    final String ASPECTS_NS = (project + ".aspects");
    final String ASPECTS_FOLDER = ASPECTS_NS.replaceAll("\\.", "/");
    final String SRC_GEN = "src-gen";
    final Consumer<Pair<String, String>> _function = (Pair<String, String> asp) -> {
      final String aspectName = asp.getKey();
      final String targetClass = asp.getValue();
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(project);
      _builder.append("/");
      _builder.append(SRC_GEN);
      _builder.append("/");
      _builder.append(ASPECTS_FOLDER);
      _builder.append("/");
      _builder.append(aspectName);
      _builder.append(".java");
      final String aspect = _builder.toString();
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append(project);
      _builder_1.append("/");
      _builder_1.append(SRC_GEN);
      _builder_1.append("/");
      _builder_1.append(ASPECTS_FOLDER);
      _builder_1.append("/");
      _builder_1.append(aspectName);
      _builder_1.append(targetClass);
      _builder_1.append("AspectContext.java");
      final String context = _builder_1.toString();
      StringConcatenation _builder_2 = new StringConcatenation();
      _builder_2.append(project);
      _builder_2.append("/");
      _builder_2.append(SRC_GEN);
      _builder_2.append("/");
      _builder_2.append(ASPECTS_FOLDER);
      _builder_2.append("/");
      _builder_2.append(aspectName);
      _builder_2.append(targetClass);
      _builder_2.append("AspectProperties.java");
      final String properties = _builder_2.toString();
      this.assertFileDontExists(aspect);
      this.assertFileDontExists(context);
      this.assertFileDontExists(properties);
    };
    aspects.forEach(_function);
  }
  
  /**
   * Returns the EPackage for generated model type {@link mtName}
   * in {@link project}
   */
  public EPackage getMT(final IProject project, final String mtName) {
    final ResourceSetImpl rs = new ResourceSetImpl();
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("platform:/resource/");
    String _name = project.getName();
    _builder.append(_name);
    _builder.append("/model-gen/");
    _builder.append(mtName);
    _builder.append(".ecore");
    final Resource res = rs.getResource(URI.createURI(_builder.toString()), true);
    EObject _head = IterableExtensions.<EObject>head(res.getContents());
    return ((EPackage) _head);
  }
  
  public void assertMatch(final EPackage pkg, final String refEcore) {
    final ResourceSetImpl rs = new ResourceSetImpl();
    final URI uri = URI.createURI(refEcore);
    final Resource res = rs.getResource(uri, true);
    EObject _head = IterableExtensions.<EObject>head(res.getContents());
    final EPackage ref = ((EPackage) _head);
    final DefaultComparisonScope scope = new DefaultComparisonScope(pkg, ref, null);
    EMFCompare.Builder _builder = EMFCompare.builder();
    final Comparison comparison = _builder.setDiffEngine(
      new DefaultDiffEngine() {
        @Override
        public FeatureFilter createFeatureFilter() {
          return new FeatureFilter() {
            @Override
            public boolean isIgnoredReference(final Match match, final EReference ref) {
              return (Objects.equal(ref, EcorePackage.Literals.EMODEL_ELEMENT__EANNOTATIONS) || super.isIgnoredReference(match, ref));
            }
            
            @Override
            public boolean checkForOrderingChanges(final EStructuralFeature f) {
              return false;
            }
          };
        }
      }).build().compare(scope);
    boolean _isEmpty = comparison.getDifferences().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      Assert.fail(IterableExtensions.join(comparison.getDifferences(), ", "));
    }
    Assert.assertTrue(comparison.getDifferences().isEmpty());
  }
  
  /**
   * Assert node contains at least pkg
   */
  public void assertMatch(final EPackage pkg, final IOutlineNode node) {
    Assert.assertEquals(pkg.getName(), node.getText().toString());
    final EList<EPackage> subPack = pkg.getESubpackages();
    final EList<EClassifier> classes = pkg.getEClassifiers();
    final List<IOutlineNode> subNodes = node.getChildren();
    final Consumer<EPackage> _function = (EPackage p) -> {
      final Function1<IOutlineNode, Boolean> _function_1 = (IOutlineNode it) -> {
        String _string = it.getText().toString();
        String _name = p.getName();
        return Boolean.valueOf(Objects.equal(_string, _name));
      };
      final IOutlineNode n = IterableExtensions.<IOutlineNode>findFirst(subNodes, _function_1);
      Assert.assertNotNull(n);
      this.assertMatch(p, n);
    };
    subPack.forEach(_function);
    final Consumer<EClassifier> _function_1 = (EClassifier c) -> {
      final Function1<IOutlineNode, Boolean> _function_2 = (IOutlineNode it) -> {
        String _string = it.getText().toString();
        String _name = c.getName();
        return Boolean.valueOf(Objects.equal(_string, _name));
      };
      final IOutlineNode n = IterableExtensions.<IOutlineNode>findFirst(subNodes, _function_2);
      Assert.assertNotNull(n);
      if ((c instanceof EClass)) {
        this.assertMatch(((EClass)c), n);
      }
    };
    classes.forEach(_function_1);
  }
  
  public void assertMatch(final EClass cls, final IOutlineNode node) {
    Assert.assertEquals(cls.getName(), node.getText().toString());
    final EList<EReference> ref = cls.getEAllReferences();
    final EList<EAttribute> att = cls.getEAllAttributes();
    final EList<EOperation> op = cls.getEAllOperations();
    final List<IOutlineNode> subNodes = node.getChildren();
    final Consumer<EReference> _function = (EReference r) -> {
      final Function1<IOutlineNode, Boolean> _function_1 = (IOutlineNode it) -> {
        String _string = it.getText().toString();
        String _name = r.getName();
        String _plus = (_name + " : ");
        String _name_1 = r.getEType().getName();
        String _plus_1 = (_plus + _name_1);
        return Boolean.valueOf(Objects.equal(_string, _plus_1));
      };
      final IOutlineNode n = IterableExtensions.<IOutlineNode>findFirst(subNodes, _function_1);
      Assert.assertNotNull(n);
    };
    ref.forEach(_function);
    final Consumer<EAttribute> _function_1 = (EAttribute a) -> {
      final Function1<IOutlineNode, Boolean> _function_2 = (IOutlineNode it) -> {
        String _string = it.getText().toString();
        String _name = a.getName();
        String _plus = (_name + " : ");
        String _name_1 = a.getEType().getName();
        String _plus_1 = (_plus + _name_1);
        return Boolean.valueOf(Objects.equal(_string, _plus_1));
      };
      final IOutlineNode n = IterableExtensions.<IOutlineNode>findFirst(subNodes, _function_2);
      Assert.assertNotNull(n);
    };
    att.forEach(_function_1);
    final Consumer<EOperation> _function_2 = (EOperation o) -> {
      final Function1<IOutlineNode, Boolean> _function_3 = (IOutlineNode it) -> {
        String _string = it.getText().toString();
        String _name = o.getName();
        String _plus = (_name + " : ");
        String _name_1 = o.getEType().getName();
        String _plus_1 = (_plus + _name_1);
        return Boolean.valueOf(Objects.equal(_string, _plus_1));
      };
      final IOutlineNode n = IterableExtensions.<IOutlineNode>findFirst(subNodes, _function_3);
      Assert.assertNotNull(n);
    };
    op.forEach(_function_2);
  }
  
  public void assertMatch(final String refEcore, final IOutlineNode node) {
    final ResourceSetImpl rs = new ResourceSetImpl();
    final URI uri = URI.createURI(refEcore);
    final Resource res = rs.getResource(uri, true);
    EObject _head = IterableExtensions.<EObject>head(res.getContents());
    final EPackage ref = ((EPackage) _head);
    this.assertMatch(ref, node);
  }
  
  public ModelTypingSpace getResource(final String projectName, final String melangeFile) {
    final IProject melangeProject = this.getProject(projectName);
    final ResourceSet rs = this.rsProvider.get(melangeProject);
    final URI uri = URI.createPlatformResourceURI(melangeFile, true);
    Resource _resource = rs.getResource(uri, true);
    final DerivedStateAwareResource res = ((DerivedStateAwareResource) _resource);
    res.installDerivedState(false);
    EObject _head = IterableExtensions.<EObject>head(res.getContents());
    return ((ModelTypingSpace) _head);
  }
}
