/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.builder;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Injector;
import fr.inria.diverse.melange.ast.AspectExtensions;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.builder.AbstractBuilder;
import fr.inria.diverse.melange.builder.BuilderError;
import fr.inria.diverse.melange.builder.ImportBuilder;
import fr.inria.diverse.melange.builder.InheritanceBuilder;
import fr.inria.diverse.melange.builder.MergeBuilder;
import fr.inria.diverse.melange.builder.ModelTypingSpaceBuilder;
import fr.inria.diverse.melange.builder.OperatorBuilder;
import fr.inria.diverse.melange.builder.SliceBuilder;
import fr.inria.diverse.melange.builder.WeaveBuilder;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import fr.inria.diverse.melange.lib.EcoreMerger;
import fr.inria.diverse.melange.metamodel.melange.Import;
import fr.inria.diverse.melange.metamodel.melange.Inheritance;
import fr.inria.diverse.melange.metamodel.melange.Language;
import fr.inria.diverse.melange.metamodel.melange.Merge;
import fr.inria.diverse.melange.metamodel.melange.ModelTypingSpace;
import fr.inria.diverse.melange.metamodel.melange.Operator;
import fr.inria.diverse.melange.metamodel.melange.Slice;
import fr.inria.diverse.melange.metamodel.melange.Weave;
import fr.inria.diverse.melange.utils.ErrorHelper;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * General builder for a {@link Language}.
 */
@SuppressWarnings("all")
public class LanguageBuilder extends AbstractBuilder {
  @Inject
  private ErrorHelper errorHelper;
  
  @Inject
  private Injector injector;
  
  @Inject
  @Extension
  private AspectExtensions _aspectExtensions;
  
  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;
  
  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;
  
  @Inject
  private EcoreMerger ecoreMerger;
  
  /**
   * The {@link ModelTypingSpaceBuilder} that builds the {@link ModelTypingSpace}
   * to which the {@link Language} belongs.
   */
  private ModelTypingSpaceBuilder root;
  
  /**
   * The {@link Language} being built by this builder.
   */
  private Language source;
  
  /**
   * Whether the language is currently being processed or not.
   */
  private boolean isBuilding = false;
  
  /**
   * The list of all {@link OperatorBuilder} being used to build this
   * {@link Language}.
   */
  private List<OperatorBuilder<? extends Operator>> builders;
  
  public LanguageBuilder(final Language l, final ModelTypingSpaceBuilder root) {
    super();
    this.source = l;
    this.root = root;
  }
  
  @Override
  public void preBuild() {
    super.preBuild();
    this.isBuilding = true;
  }
  
  @Override
  public void postBuild() {
    this.isBuilding = false;
    final Consumer<BuilderError> _function = new Consumer<BuilderError>() {
      @Override
      public void accept(final BuilderError e) {
        LanguageBuilder.this.errorHelper.addError(e.getLocation(), e.getMessage());
      }
    };
    this.errors.forEach(_function);
    boolean _isGeneratedByMelange = this._languageExtensions.isGeneratedByMelange(this.source);
    if (_isGeneratedByMelange) {
      final Consumer<EPackage> _function_1 = new Consumer<EPackage>() {
        @Override
        public void accept(final EPackage it) {
          String _externalPackageUri = LanguageBuilder.this._languageExtensions.getExternalPackageUri(LanguageBuilder.this.source);
          String _name = it.getName();
          String _plus = (_externalPackageUri + _name);
          String _plus_1 = (_plus + "/");
          it.setNsURI(_plus_1);
        }
      };
      this.model.forEach(_function_1);
    }
    this.bindOppositeReferences();
  }
  
  @Override
  public void make() {
    boolean _isEmpty = this.source.getOperators().isEmpty();
    if (_isEmpty) {
      return;
    }
    final Function1<Operator, Boolean> _function = new Function1<Operator, Boolean>() {
      @Override
      public Boolean apply(final Operator it) {
        return Boolean.valueOf((!(it instanceof Weave)));
      }
    };
    final List<Operator> otherOperators = IterableExtensions.<Operator>toList(IterableExtensions.<Operator>filter(this.source.getOperators(), _function));
    final Function1<Weave, Boolean> _function_1 = new Function1<Weave, Boolean>() {
      @Override
      public Boolean apply(final Weave it) {
        String _aspectWildcardImport = it.getAspectWildcardImport();
        return Boolean.valueOf((_aspectWildcardImport == null));
      }
    };
    final Comparator<Weave> _function_2 = new Comparator<Weave>() {
      @Override
      public int compare(final Weave wA, final Weave wB) {
        int _xifexpression = (int) 0;
        JvmTypeReference _aspectTypeRef = wA.getAspectTypeRef();
        String _aspectAnnotationValue = null;
        if (_aspectTypeRef!=null) {
          _aspectAnnotationValue=LanguageBuilder.this._aspectExtensions.getAspectAnnotationValue(_aspectTypeRef);
        }
        boolean _tripleNotEquals = (_aspectAnnotationValue != null);
        if (_tripleNotEquals) {
          _xifexpression = 1;
        } else {
          _xifexpression = (-1);
        }
        return _xifexpression;
      }
    };
    final Function1<Weave, Operator> _function_3 = new Function1<Weave, Operator>() {
      @Override
      public Operator apply(final Weave it) {
        return ((Operator) it);
      }
    };
    final List<Operator> aspectOperators = IterableExtensions.<Operator>toList(ListExtensions.<Weave, Operator>map(IterableExtensions.<Weave>sortWith(IterableExtensions.<Weave>filter(Iterables.<Weave>filter(this.source.getOperators(), Weave.class), _function_1), _function_2), _function_3));
    this.builders = CollectionLiterals.<OperatorBuilder<? extends Operator>>newArrayList();
    this.builders.addAll(this.createBuilders(otherOperators));
    final Consumer<OperatorBuilder<? extends Operator>> _function_4 = new Consumer<OperatorBuilder<? extends Operator>>() {
      @Override
      public void accept(final OperatorBuilder<? extends Operator> builder) {
        builder.build();
        LanguageBuilder.this.errors.addAll(builder.errors);
      }
    };
    this.builders.forEach(_function_4);
    this.model = IterableExtensions.<EPackage>toSet(EcoreUtil.<EPackage>copyAll(IterableExtensions.<OperatorBuilder<? extends Operator>>head(this.builders).model));
    final Consumer<OperatorBuilder<? extends Operator>> _function_5 = new Consumer<OperatorBuilder<? extends Operator>>() {
      @Override
      public void accept(final OperatorBuilder<? extends Operator> builder) {
        LanguageBuilder.this.errors.addAll(LanguageBuilder.this.merge(LanguageBuilder.this.model, builder.model, builder.source));
      }
    };
    IterableExtensions.<OperatorBuilder<? extends Operator>>drop(this.builders, 1).forEach(_function_5);
    final List<OperatorBuilder<? extends Operator>> weaveBuilders = this.createBuilders(aspectOperators);
    final Consumer<OperatorBuilder<? extends Operator>> _function_6 = new Consumer<OperatorBuilder<? extends Operator>>() {
      @Override
      public void accept(final OperatorBuilder<? extends Operator> builder) {
        builder.build();
        LanguageBuilder.this.errors.addAll(builder.errors);
        LanguageBuilder.this.errors.addAll(LanguageBuilder.this.merge(LanguageBuilder.this.model, builder.model, builder.source));
      }
    };
    weaveBuilders.forEach(_function_6);
    this.builders.addAll(weaveBuilders);
    if (((this.model == null) || this.model.isEmpty())) {
      String _name = this.source.getName();
      String _plus = ("Can\'t build " + _name);
      BuilderError _builderError = new BuilderError(_plus, this.source);
      this.errors.add(_builderError);
    }
  }
  
  /**
   * Merges the {@link EPackage}s {@code merged} into the {@link EPackage}s
   * {@code base}. Conflicts that may arise during the merge are returned as
   * {@link BuilderError}s.
   * 
   * @see EcoreMerger#merge
   */
  public List<BuilderError> merge(final Set<EPackage> base, final Set<EPackage> merged, final Operator context) {
    if ((merged != null)) {
      this.ecoreMerger.merge(base, merged);
    }
    final Function1<EcoreMerger.Conflict, BuilderError> _function = new Function1<EcoreMerger.Conflict, BuilderError>() {
      @Override
      public BuilderError apply(final EcoreMerger.Conflict it) {
        String _string = it.toString();
        return new BuilderError(_string, context);
      }
    };
    return ListExtensions.<EcoreMerger.Conflict, BuilderError>map(this.ecoreMerger.getConflicts(), _function);
  }
  
  /**
   * Returns the list of {@link OperatorBuilder} used to build the different
   * {@link Operator} of {@code operators}.
   */
  public List<OperatorBuilder<? extends Operator>> createBuilders(final List<Operator> operators) {
    final ArrayList<OperatorBuilder<? extends Operator>> res = CollectionLiterals.<OperatorBuilder<? extends Operator>>newArrayList();
    final Consumer<Operator> _function = new Consumer<Operator>() {
      @Override
      public void accept(final Operator op) {
        OperatorBuilder<? extends Operator> _switchResult = null;
        boolean _matched = false;
        if (op instanceof Inheritance) {
          _matched=true;
          _switchResult = new InheritanceBuilder(((Inheritance)op), LanguageBuilder.this.root);
        }
        if (!_matched) {
          if (op instanceof Merge) {
            _matched=true;
            _switchResult = new MergeBuilder(((Merge)op), LanguageBuilder.this.root);
          }
        }
        if (!_matched) {
          if (op instanceof Slice) {
            _matched=true;
            _switchResult = new SliceBuilder(((Slice)op), LanguageBuilder.this.root);
          }
        }
        if (!_matched) {
          if (op instanceof Import) {
            _matched=true;
            _switchResult = new ImportBuilder(((Import)op));
          }
        }
        if (!_matched) {
          if (op instanceof Weave) {
            _matched=true;
            _switchResult = new WeaveBuilder(((Weave)op), LanguageBuilder.this.model);
          }
        }
        final OperatorBuilder<? extends Operator> builder = _switchResult;
        res.add(builder);
        LanguageBuilder.this.injector.injectMembers(builder);
      }
    };
    operators.forEach(_function);
    return res;
  }
  
  public boolean isBuilding() {
    return this.isBuilding;
  }
  
  public Language getSource() {
    return this.source;
  }
  
  public WeaveBuilder findBuilder(final Weave w) {
    final Function1<WeaveBuilder, Boolean> _function = new Function1<WeaveBuilder, Boolean>() {
      @Override
      public Boolean apply(final WeaveBuilder it) {
        return Boolean.valueOf((it.source == w));
      }
    };
    return IterableExtensions.<WeaveBuilder>findFirst(Iterables.<WeaveBuilder>filter(this.builders, WeaveBuilder.class), _function);
  }
  
  /**
   * Return the list of Operators' builder
   */
  public List<OperatorBuilder<? extends Operator>> getSubBuilders() {
    return this.builders;
  }
  
  private void bindOppositeReferences() {
    final Function1<EPackage, List<EClass>> _function = new Function1<EPackage, List<EClass>>() {
      @Override
      public List<EClass> apply(final EPackage it) {
        return LanguageBuilder.this._ecoreExtensions.getAllClasses(it);
      }
    };
    final Set<EClass> allClasses = IterableExtensions.<EClass>toSet(Iterables.<EClass>concat(IterableExtensions.<EPackage, List<EClass>>map(this.model, _function)));
    final Function1<EClass, EList<EReference>> _function_1 = new Function1<EClass, EList<EReference>>() {
      @Override
      public EList<EReference> apply(final EClass it) {
        return it.getEAllReferences();
      }
    };
    final Function1<EReference, Boolean> _function_2 = new Function1<EReference, Boolean>() {
      @Override
      public Boolean apply(final EReference it) {
        final Function1<EAnnotation, Boolean> _function = new Function1<EAnnotation, Boolean>() {
          @Override
          public Boolean apply(final EAnnotation it) {
            String _source = it.getSource();
            return Boolean.valueOf(Objects.equal(_source, "opposite"));
          }
        };
        return Boolean.valueOf(IterableExtensions.<EAnnotation>exists(it.getEAnnotations(), _function));
      }
    };
    final Iterable<EReference> allOpposites = IterableExtensions.<EReference>filter(Iterables.<EReference>concat(IterableExtensions.<EClass, EList<EReference>>map(allClasses, _function_1)), _function_2);
    final Consumer<EReference> _function_3 = new Consumer<EReference>() {
      @Override
      public void accept(final EReference ref) {
        final Function1<EAnnotation, Boolean> _function = new Function1<EAnnotation, Boolean>() {
          @Override
          public Boolean apply(final EAnnotation it) {
            String _source = it.getSource();
            return Boolean.valueOf(Objects.equal(_source, "opposite"));
          }
        };
        final EAnnotation annot = IterableExtensions.<EAnnotation>findFirst(ref.getEAnnotations(), _function);
        final String opRefName = annot.getDetails().get("value");
        EClassifier _eType = ref.getEType();
        if ((_eType instanceof EClass)) {
          EClassifier _eType_1 = ref.getEType();
          final Function1<EReference, Boolean> _function_1 = new Function1<EReference, Boolean>() {
            @Override
            public Boolean apply(final EReference it) {
              String _name = it.getName();
              return Boolean.valueOf(Objects.equal(_name, opRefName));
            }
          };
          final EReference opRef = IterableExtensions.<EReference>findFirst(((EClass) _eType_1).getEAllReferences(), _function_1);
          if (((opRef != null) && (opRef.getEOpposite() == null))) {
            ref.setEOpposite(opRef);
            opRef.setEOpposite(ref);
          }
        }
        ref.getEAnnotations().remove(annot);
      }
    };
    allOpposites.forEach(_function_3);
  }
}
