/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.k3.al.annotationprocessor;

import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.eclipse.xtend.lib.macro.CodeGenerationContext;
import org.eclipse.xtend.lib.macro.TransformationContext;
import org.eclipse.xtend.lib.macro.ValidationContext;
import org.eclipse.xtend.lib.macro.declaration.MutableClassDeclaration;
import org.eclipse.xtend.lib.macro.declaration.Type;
import org.eclipse.xtend.lib.macro.declaration.TypeReference;
import org.eclipse.xtend.lib.macro.file.Path;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * This class is in charge of building and updating the property file that list all the Aspect classes for a given Class
 * It must be called within the doGenerateCode() of the ClassProcessor
 * This builder will do nothing if called without annotated element
 * 
 * This builder used a map with weak reference in order to be able to  use a single instance for a given project
 * see getAspectMappingBuilder() to get the instance
 * 
 * This allows to collect work done in several doTransform of several parallel jobs and assemble them when doing the doGenerateCode (ie. when calling the writeProperty)
 * once the writeProperty has been called once, the map is reset to null in order to restart from scratch when calling the readCurrentMapping again
 */
@SuppressWarnings("all")
public class AspectMappingBuilder {
  public static String ASPECTMAPPING_FOLDER = "META-INF/xtend-gen";

  private List<? extends MutableClassDeclaration> classes;

  private Path targetFilePath;

  private String projectName;

  /**
   * main key: project name
   */
  private static Map<String, WeakReference<AspectMappingBuilder>> projectsAspectMappingBuilder = new HashMap<String, WeakReference<AspectMappingBuilder>>();

  /**
   * use getAspectMappingBuilder() in order to get an instance associated to the project
   */
  private AspectMappingBuilder(final String projectName) {
    this.projectName = projectName;
  }

  public static AspectMappingBuilder getAspectMappingBuilder(final String projectName) {
    final WeakReference<AspectMappingBuilder> weakRef = AspectMappingBuilder.projectsAspectMappingBuilder.get(projectName);
    if (((weakRef != null) && (weakRef.get() != null))) {
      return weakRef.get();
    } else {
      final AspectMappingBuilder result = new AspectMappingBuilder(projectName);
      WeakReference<AspectMappingBuilder> _weakReference = new WeakReference<AspectMappingBuilder>(result);
      AspectMappingBuilder.projectsAspectMappingBuilder.put(projectName, _weakReference);
      return result;
    }
  }

  /**
   * internal map
   */
  private final Map<String, List<String>> mapping = CollectionLiterals.<String, List<String>>newHashMap();

  /**
   * Rebuild mapping from existing property file
   * @param classes classes
   * @param context transformation context
   */
  public void readCurrentMapping(final List<? extends MutableClassDeclaration> classes, @Extension final TransformationContext context) {
    this.classes = classes;
    int _size = classes.size();
    boolean _greaterThan = (_size > 0);
    if (_greaterThan) {
      final Path filePath = IterableExtensions.head(classes).getCompilationUnit().getFilePath();
      Path _projectFolder = context.getProjectFolder(filePath);
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("/");
      _builder.append(AspectMappingBuilder.ASPECTMAPPING_FOLDER);
      _builder.append("/");
      String _lastSegment = context.getProjectFolder(filePath).getLastSegment();
      _builder.append(_lastSegment);
      _builder.append(".k3_aspect_mapping.properties");
      this.targetFilePath = _projectFolder.append(_builder.toString());
    }
    int _size_1 = classes.size();
    boolean _greaterThan_1 = (_size_1 > 0);
    if (_greaterThan_1) {
      final Properties properties = new Properties();
      boolean _exists = context.exists(this.targetFilePath);
      if (_exists) {
        try {
          properties.load(context.getContentsAsStream(this.targetFilePath));
          final BiConsumer<Object, Object> _function = (Object propKey, Object commaSeparatedPropvalues) -> {
            final Function1<String, String> _function_1 = (String s) -> {
              return s.trim();
            };
            final List<String> propValues = ListExtensions.<String, String>map(((List<String>)Conversions.doWrapArray(((String) commaSeparatedPropvalues).split(","))), _function_1);
            final Consumer<String> _function_2 = (String propValue) -> {
              this.addMapping(((String) propKey), propValue);
            };
            propValues.forEach(_function_2);
          };
          properties.forEach(_function);
        } catch (final Throwable _t) {
          if (_t instanceof IOException) {
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
      }
    }
  }

  /**
   * returns the list of all declared aspects
   * If called during the codeGeneration phase it contains
   * all aspect classesof the current project
   */
  public List<String> getAllDeclaredAspects() {
    final List<String> result = CollectionLiterals.<String>newArrayList();
    synchronized (this.mapping) {
      final Function1<List<String>, Boolean> _function = (List<String> l) -> {
        return Boolean.valueOf(result.addAll(l));
      };
      IterableExtensions.<List<String>>forall(this.mapping.values(), _function);
    }
    return result;
  }

  /**
   * try to clean unused mappings
   * @param context transformation context
   */
  public void cleanUnusedMapping(@Extension final ValidationContext context) {
    synchronized (this.mapping) {
      int _size = this.classes.size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        final List<String> keytoRemove = new ArrayList<String>();
        final HashMap<String, List<String>> mappingCopy = new HashMap<String, List<String>>(this.mapping);
        final BiConsumer<String, List<String>> _function = (String key, List<String> valueList) -> {
          final Function1<String, Boolean> _function_1 = (String value) -> {
            Type _findTypeGlobally = context.findTypeGlobally(value);
            return Boolean.valueOf((_findTypeGlobally != null));
          };
          final List<String> newValueList = IterableExtensions.<String>toList(IterableExtensions.<String>filter(valueList, _function_1));
          this.mapping.put(key, newValueList);
          int _size_1 = newValueList.size();
          boolean _equals = (_size_1 == 0);
          if (_equals) {
            keytoRemove.add(key);
          }
        };
        mappingCopy.forEach(_function);
        for (final String key : keytoRemove) {
          this.mapping.remove(key);
        }
      }
    }
  }

  public void addMappingForAnnotatedSourceElements() {
    for (final MutableClassDeclaration annotatedSourceElement : this.classes) {
      {
        final TypeReference aspectizedClassType = Helper.getAnnotationAspectType(annotatedSourceElement);
        if ((aspectizedClassType != null)) {
          this.addMapping(aspectizedClassType.getName(), annotatedSourceElement.getQualifiedName());
        }
      }
    }
  }

  /**
   * @param context code generation context
   */
  public void writePropertyFile(@Extension final CodeGenerationContext context) {
    if (((this.classes != null) && (this.classes.size() > 0))) {
      synchronized (this.mapping) {
        StringConcatenation _builder = new StringConcatenation();
        String buf = _builder.toString();
        List<Map.Entry<String, List<String>>> _collect = this.mapping.entrySet().stream().sorted(Map.Entry.<String, List<String>>comparingByKey()).collect(Collectors.<Map.Entry<String, List<String>>>toList());
        for (final Map.Entry<String, List<String>> entrySet : _collect) {
          StringConcatenation _builder_1 = new StringConcatenation();
          String _string = buf.toString();
          _builder_1.append(_string);
          _builder_1.newLineIfNotEmpty();
          String _key = entrySet.getKey();
          _builder_1.append(_key);
          _builder_1.append(" = ");
          {
            List<String> _value = entrySet.getValue();
            boolean _hasElements = false;
            for(final String aString : _value) {
              if (!_hasElements) {
                _hasElements = true;
              } else {
                _builder_1.appendImmediate(", ", "");
              }
              _builder_1.append(aString);
            }
          }
          buf = _builder_1.toString();
        }
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append("# List of the Java classes that have been aspectized and name of the aspect classes separated by comma");
        _builder_2.newLine();
        String _string_1 = buf.toString();
        _builder_2.append(_string_1);
        final String contents = _builder_2.toString();
        Helper.writeContentsIfNew(this.targetFilePath, contents, context);
      }
    }
  }

  private void addMapping(final String aspectizedClassName, final String aspectClassName) {
    synchronized (this.mapping) {
      List<String> existingListForAspectizedElement = this.mapping.get(aspectizedClassName);
      if ((existingListForAspectizedElement == null)) {
        existingListForAspectizedElement = CollectionLiterals.<String>newArrayList();
        this.mapping.put(aspectizedClassName, existingListForAspectizedElement);
      }
      boolean _contains = existingListForAspectizedElement.contains(aspectClassName);
      boolean _not = (!_contains);
      if (_not) {
        existingListForAspectizedElement.add(aspectClassName);
      }
    }
  }
}
