/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.k3.al.annotationprocessor;

import com.google.common.base.Objects;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.xtend.lib.macro.AbstractClassProcessor;
import org.eclipse.xtend.lib.macro.CodeGenerationContext;
import org.eclipse.xtend.lib.macro.RegisterGlobalsContext;
import org.eclipse.xtend.lib.macro.TransformationContext;
import org.eclipse.xtend.lib.macro.ValidationContext;
import org.eclipse.xtend.lib.macro.declaration.AnnotationReference;
import org.eclipse.xtend.lib.macro.declaration.ClassDeclaration;
import org.eclipse.xtend.lib.macro.declaration.CompilationStrategy;
import org.eclipse.xtend.lib.macro.declaration.CompilationUnit;
import org.eclipse.xtend.lib.macro.declaration.FieldDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MethodDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableClassDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableConstructorDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableFieldDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableMethodDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableParameterDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableTypeDeclaration;
import org.eclipse.xtend.lib.macro.declaration.ParameterDeclaration;
import org.eclipse.xtend.lib.macro.declaration.Type;
import org.eclipse.xtend.lib.macro.declaration.TypeReference;
import org.eclipse.xtend.lib.macro.declaration.Visibility;
import org.eclipse.xtend.lib.macro.expression.Expression;
import org.eclipse.xtend.lib.macro.file.Path;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * AbstractClassProcessor that processes @Aspect annotation on classes.
 * 
 * <img src="doc-files/AspectProcessor_SD.png" width="500" alt="AspectProcessor Sequence Diagram">
 * 
 * <br>See <a href="https://www.eclipse.org/xtend/documentation/204_activeannotations.html">xtend annotation processor documentation</a>
 */
@SuppressWarnings("all")
public class AspectProcessor extends AbstractClassProcessor {
  private static final Logger LOGGER = Logger.getLogger(AspectProcessor.class.getPackage().getName());

  private Map<MutableClassDeclaration, List<MutableClassDeclaration>> listResMap = CollectionLiterals.<MutableClassDeclaration, List<MutableClassDeclaration>>newHashMap();

  private AspectMappingBuilder aspectMappingBuilder;

  private final ProjectStaticDispatchBuilder projectStaticDispatchBuilder = new ProjectStaticDispatchBuilder();

  public static final String CTX_NAME = "AspectContext";

  public static final String PROP_NAME = "AspectProperties";

  public static final String OVERRIDE_METHOD = OverrideAspectMethod.class.getSimpleName();

  public static final String STEP = Step.class.getSimpleName();

  public static final String PROP_VAR_NAME = "_self_";

  public static final String SELF_VAR_NAME = "_self";

  public static final String PRIV_PREFIX = "_privk3_";

  public static final String PRIV_CONSTRUCTOR_POSTFIX = "_constructor_initializer";

  public static final String RESULT_VAR = "result";

  public static final String DISPATCH_POINTCUT_KEY = "#DispatchPointCut_before#";

  public static Boolean lock = Boolean.valueOf(false);

  /**
   * Phase 1: Register properties and context helpers
   */
  @Override
  public void doRegisterGlobals(final ClassDeclaration annotatedClass, final RegisterGlobalsContext context) {
    String _qualifiedName = annotatedClass.getQualifiedName();
    String _plus = ("Phase 1: doRegisterGlobals: " + _qualifiedName);
    AspectProcessor.LOGGER.log(Level.FINER, _plus);
    final TypeReference type = Helper.getAnnotationAspectType(annotatedClass);
    if ((type != null)) {
      final String className = type.getSimpleName();
      String _qualifiedName_1 = annotatedClass.getQualifiedName();
      String _plus_1 = (_qualifiedName_1 + className);
      String _plus_2 = (_plus_1 + AspectProcessor.PROP_NAME);
      context.registerClass(_plus_2);
      String _qualifiedName_2 = annotatedClass.getQualifiedName();
      String _plus_3 = (_qualifiedName_2 + className);
      String _plus_4 = (_plus_3 + AspectProcessor.CTX_NAME);
      context.registerClass(_plus_4);
    }
  }

  @Override
  public void doRegisterGlobals(final List<? extends ClassDeclaration> annotatedClasses, @Extension final RegisterGlobalsContext context) {
    final Function1<ClassDeclaration, String> _function = (ClassDeclaration c) -> {
      return c.getSimpleName();
    };
    String _join = IterableExtensions.join(ListExtensions.map(annotatedClasses, _function), ", ");
    String _plus = ("Phase 1: " + _join);
    AspectProcessor.LOGGER.log(Level.FINE, _plus);
    super.doRegisterGlobals(annotatedClasses, context);
  }

  private List<? extends MutableClassDeclaration> mclasses = null;

  /**
   * Phase 2: Transform aspected class' fields and methods
   */
  @Override
  public void doTransform(final List<? extends MutableClassDeclaration> classes, @Extension final TransformationContext context) {
    final Function1<MutableClassDeclaration, String> _function = (MutableClassDeclaration c) -> {
      return c.getSimpleName();
    };
    String _join = IterableExtensions.join(ListExtensions.map(classes, _function), ", ");
    String _plus = ("Phase 2: " + _join);
    AspectProcessor.LOGGER.log(Level.FINE, _plus);
    this.mclasses = classes;
    for (final MutableClassDeclaration clazz : classes) {
      {
        final List<MutableClassDeclaration> listRes = Helper.sortByClassInheritance(clazz, classes, context);
        final Function1<MutableClassDeclaration, String> _function_1 = (MutableClassDeclaration it) -> {
          return it.getSimpleName();
        };
        final List<String> inheritList = ListExtensions.<MutableClassDeclaration, String>map(listRes, _function_1);
        this.listResMap.put(clazz, listRes);
        final TypeReference typeRef = Helper.getAnnotationAspectType(clazz);
        if ((typeRef == null)) {
          context.addError(clazz, "The aspectized class cannot be resolved.");
        } else {
          final String className = typeRef.getSimpleName();
          final String identifier = typeRef.getName();
          final Map<MutableMethodDeclaration, String> bodies = CollectionLiterals.<MutableMethodDeclaration, String>newHashMap();
          this.fieldsProcessing(context, clazz, className, identifier, bodies);
          this.methodsProcessing(clazz, context, identifier, bodies, inheritList, className);
          this.constructorsProcessing(clazz, context, identifier, bodies, className);
          this.aspectContextMaker(context, clazz, className, identifier);
        }
      }
    }
    this.aspectMappingBuilder = AspectMappingBuilder.getAspectMappingBuilder(context.getProjectFolder(IterableExtensions.head(classes).getCompilationUnit().getFilePath()).toString());
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("aspectMappingBuilder.allDeclaredAspects.size=");
    int _size = this.aspectMappingBuilder.getAllDeclaredAspects().size();
    _builder.append(_size);
    _builder.append(" ");
    String _join_1 = IterableExtensions.join(this.aspectMappingBuilder.getAllDeclaredAspects(), ", ");
    _builder.append(_join_1);
    AspectProcessor.LOGGER.log(Level.FINER, _builder.toString());
    this.aspectMappingBuilder.readCurrentMapping(classes, context);
    this.aspectMappingBuilder.addMappingForAnnotatedSourceElements();
  }

  /**
   * Phase 3: Validation
   * perform some cleanup
   */
  @Override
  public void doValidate(final List<? extends ClassDeclaration> annotatedClasses, @Extension final ValidationContext context) {
    final Function1<ClassDeclaration, String> _function = (ClassDeclaration c) -> {
      return c.getSimpleName();
    };
    String _join = IterableExtensions.join(ListExtensions.map(annotatedClasses, _function), ", ");
    String _plus = ("Phase 3: " + _join);
    AspectProcessor.LOGGER.log(Level.FINE, _plus);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("before cleanUnusedMapping: aspectMappingBuilder.allDeclaredAspects.size=");
    int _size = this.aspectMappingBuilder.getAllDeclaredAspects().size();
    _builder.append(_size);
    _builder.append(" ");
    String _join_1 = IterableExtensions.join(this.aspectMappingBuilder.getAllDeclaredAspects(), ", ");
    _builder.append(_join_1);
    AspectProcessor.LOGGER.log(Level.FINER, _builder.toString());
    this.aspectMappingBuilder.cleanUnusedMapping(context);
    StringConcatenation _builder_1 = new StringConcatenation();
    _builder_1.append("after cleanUnusedMapping: aspectMappingBuilder.allDeclaredAspects.size=");
    int _size_1 = this.aspectMappingBuilder.getAllDeclaredAspects().size();
    _builder_1.append(_size_1);
    _builder_1.append(" ");
    String _join_2 = IterableExtensions.join(this.aspectMappingBuilder.getAllDeclaredAspects(), ", ");
    _builder_1.append(_join_2);
    AspectProcessor.LOGGER.log(Level.FINER, _builder_1.toString());
    super.doValidate(annotatedClasses, context);
  }

  /**
   * Phase 4: use an additional code generator to produce some additional files:
   * - generates the .k3_aspect_mapping.properties file that can be used later by other tool to inspect aspects
   * - generates the AspectJ file needed to bypass some limitations;
   */
  @Override
  public void doGenerateCode(final List<? extends ClassDeclaration> annotatedSourceElements, @Extension final CodeGenerationContext context) {
    final Function1<ClassDeclaration, String> _function = (ClassDeclaration c) -> {
      return c.getSimpleName();
    };
    String _join = IterableExtensions.join(ListExtensions.map(annotatedSourceElements, _function), ", ");
    String _plus = ("Phase 4: " + _join);
    AspectProcessor.LOGGER.log(Level.INFO, _plus);
    this.aspectMappingBuilder.writePropertyFile(context);
    for (final ClassDeclaration classDecl : annotatedSourceElements) {
      this.generateAspectJCodeForClass(classDecl, context);
    }
    for (final ClassDeclaration classDecl_1 : annotatedSourceElements) {
      this.injectDispatchInParentAspects(classDecl_1, context);
    }
    final Function1<ClassDeclaration, String> _function_1 = (ClassDeclaration c) -> {
      return c.getSimpleName();
    };
    String _join_1 = IterableExtensions.join(ListExtensions.map(annotatedSourceElements, _function_1), ", ");
    String _plus_1 = ("Phase 4: write writeTempStaticDispatchFile (" + _join_1);
    String _plus_2 = (_plus_1 + ")");
    AspectProcessor.LOGGER.log(Level.FINE, _plus_2);
    final Function1<ClassDeclaration, CompilationUnit> _function_2 = (ClassDeclaration classDecl_2) -> {
      return classDecl_2.getCompilationUnit();
    };
    final Consumer<CompilationUnit> _function_3 = (CompilationUnit cu) -> {
      this.projectStaticDispatchBuilder.writeTempStaticDispatchFile(cu, context);
    };
    ListExtensions.map(annotatedSourceElements, _function_2).forEach(_function_3);
  }

  /**
   * Change the signature of the original method to makes it static
   * -	add _self parameter
   * -	makes it static
   * -	deal with the @Abstract annotation
   */
  private void methodProcessingAddSelfStatic(final MutableMethodDeclaration m, final String identifier, @Extension final TransformationContext cxt) {
    if ((IterableExtensions.isEmpty(m.getParameters()) || (!Objects.equal(IterableExtensions.head(m.getParameters()).getSimpleName(), AspectProcessor.SELF_VAR_NAME)))) {
      final ArrayList<Pair<String, TypeReference>> l = new ArrayList<Pair<String, TypeReference>>();
      Iterable<? extends MutableParameterDeclaration> _parameters = m.getParameters();
      for (final MutableParameterDeclaration p1 : _parameters) {
        String _simpleName = p1.getSimpleName();
        TypeReference _type = p1.getType();
        Pair<String, TypeReference> _pair = new Pair<String, TypeReference>(_simpleName, _type);
        l.add(_pair);
      }
      IterableExtensions.toList(m.getParameters()).clear();
      m.addParameter(AspectProcessor.SELF_VAR_NAME, cxt.newTypeReference(identifier));
      for (final Pair<String, TypeReference> param : l) {
        m.addParameter(param.getKey(), param.getValue());
      }
      boolean _isAbstract = m.isAbstract();
      if (_isAbstract) {
        m.addAnnotation(cxt.newAnnotationReference(cxt.findTypeGlobally(Abstract.class)));
      }
    }
    m.setStatic(true);
  }

  private void methodProcessingAddSuper(final MutableMethodDeclaration m, final MutableClassDeclaration clazz, final String aspectizedClassName, @Extension final TransformationContext cxt) {
    final Function1<AnnotationReference, Boolean> _function = (AnnotationReference it) -> {
      String _simpleName = it.getAnnotationTypeDeclaration().getSimpleName();
      return Boolean.valueOf(Objects.equal(_simpleName, AspectProcessor.OVERRIDE_METHOD));
    };
    boolean _exists = IterableExtensions.exists(m.getAnnotations(), _function);
    boolean _not = (!_exists);
    if (_not) {
      return;
    }
    final Function1<TypeReference, ClassDeclaration> _function_1 = (TypeReference cl) -> {
      Type _findTypeGlobally = cxt.findTypeGlobally(cl.getName());
      return ((ClassDeclaration) _findTypeGlobally);
    };
    final Set<ClassDeclaration> superClasses = IterableExtensions.<ClassDeclaration>toSet(IterableExtensions.<ClassDeclaration>filterNull(ListExtensions.<TypeReference, ClassDeclaration>map(Helper.getAnnotationWithType(clazz), _function_1)));
    ClassDeclaration _xifexpression = null;
    TypeReference _extendedClass = clazz.getExtendedClass();
    boolean _tripleEquals = (_extendedClass == null);
    if (_tripleEquals) {
      _xifexpression = null;
    } else {
      Type _findTypeGlobally = cxt.findTypeGlobally(
        clazz.getExtendedClass().getName());
      _xifexpression = ((ClassDeclaration) _findTypeGlobally);
    }
    final ClassDeclaration superCl = _xifexpression;
    if ((superCl != null)) {
      superClasses.add(superCl);
    }
    boolean _isEmpty = superClasses.isEmpty();
    if (_isEmpty) {
      return;
    }
    final Function1<ClassDeclaration, MethodDeclaration> _function_2 = (ClassDeclaration sc) -> {
      return Helper.findMethod(sc, m, cxt);
    };
    final Function1<MethodDeclaration, Boolean> _function_3 = (MethodDeclaration it) -> {
      final Function1<AnnotationReference, Boolean> _function_4 = (AnnotationReference it_1) -> {
        String _simpleName = it_1.getAnnotationTypeDeclaration().getSimpleName();
        return Boolean.valueOf(Objects.equal(_simpleName, "Abstract"));
      };
      AnnotationReference _findFirst = IterableExtensions.findFirst(it.getAnnotations(), _function_4);
      return Boolean.valueOf((_findFirst == null));
    };
    final Iterable<MethodDeclaration> superMeths = IterableExtensions.<MethodDeclaration>filter(IterableExtensions.<MethodDeclaration>filterNull(IterableExtensions.<ClassDeclaration, MethodDeclaration>map(superClasses, _function_2)), _function_3);
    int _size = IterableExtensions.size(superMeths);
    final boolean multiSuper = (_size > 1);
    final Consumer<MethodDeclaration> _function_4 = (MethodDeclaration sm) -> {
      final String superAspectedClassName = IterableExtensions.<String>last(((Iterable<String>)Conversions.doWrapArray(Helper.getAspectedClassName(sm.getDeclaringType()).split("\\."))));
      String _xifexpression_1 = null;
      if (multiSuper) {
        _xifexpression_1 = (("super_" + superAspectedClassName) + "_");
      } else {
        _xifexpression_1 = "super_";
      }
      final String superNamePrefix = _xifexpression_1;
      String _simpleName = m.getSimpleName();
      String _plus = (superNamePrefix + _simpleName);
      final Procedure1<MutableMethodDeclaration> _function_5 = (MutableMethodDeclaration it) -> {
        final StringBuilder paramsList = new StringBuilder();
        it.setVisibility(Visibility.PRIVATE);
        it.setStatic(true);
        it.setReturnType(m.getReturnType());
        Iterable<? extends MutableParameterDeclaration> _parameters = m.getParameters();
        for (final MutableParameterDeclaration p : _parameters) {
          it.addParameter(p.getSimpleName(), p.getType());
        }
        final Function1<MutableParameterDeclaration, String> _function_6 = (MutableParameterDeclaration it_1) -> {
          return it_1.getSimpleName();
        };
        paramsList.append(IterableExtensions.join(IterableExtensions.map(m.getParameters(), _function_6), ","));
        final CompilationStrategy _function_7 = (CompilationStrategy.CompilationContext it_1) -> {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("final ");
          String _name = cxt.newTypeReference(sm.getDeclaringType()).getName();
          String _plus_1 = (_name + superAspectedClassName);
          String _plus_2 = (_plus_1 + AspectProcessor.PROP_NAME);
          _builder.append(_plus_2);
          _builder.append(" ");
          _builder.append(AspectProcessor.PROP_VAR_NAME);
          _builder.append(" = ");
          String _name_1 = cxt.newTypeReference(sm.getDeclaringType()).getName();
          String _plus_3 = (_name_1 + superAspectedClassName);
          String _plus_4 = (_plus_3 + AspectProcessor.CTX_NAME);
          _builder.append(_plus_4);
          _builder.append(".getSelf(");
          _builder.append(AspectProcessor.SELF_VAR_NAME);
          _builder.append(");");
          _builder.newLineIfNotEmpty();
          {
            String _name_2 = sm.getReturnType().getName();
            boolean _notEquals = (!Objects.equal(_name_2, "void"));
            if (_notEquals) {
              _builder.append("return ");
            }
          }
          _builder.append(" ");
          String _name_3 = cxt.newTypeReference(sm.getDeclaringType()).getName();
          _builder.append(_name_3);
          _builder.append(".");
          String _simpleName_1 = m.getSimpleName();
          String _plus_5 = (AspectProcessor.PRIV_PREFIX + _simpleName_1);
          _builder.append(_plus_5);
          _builder.append("(");
          _builder.append(AspectProcessor.PROP_VAR_NAME);
          _builder.append(", ");
          _builder.append(paramsList);
          _builder.append(");");
          _builder.newLineIfNotEmpty();
          return _builder;
        };
        it.setBody(_function_7);
        cxt.setPrimarySourceElement(it, m);
      };
      clazz.addMethod(_plus, _function_5);
    };
    superMeths.forEach(_function_4);
  }

  /**
   * used annotations: ReplaceAspectMethod
   * Add "_hidden_" at the beginning of the replaced method name
   */
  private void methodProcessingAddHidden(final MutableMethodDeclaration m, final String identifier, @Extension final TransformationContext cxt) {
    final Function1<AnnotationReference, Boolean> _function = (AnnotationReference it) -> {
      String _simpleName = it.getAnnotationTypeDeclaration().getSimpleName();
      return Boolean.valueOf(Objects.equal(_simpleName, "ReplaceAspectMethod"));
    };
    boolean _exists = IterableExtensions.exists(m.getAnnotations(), _function);
    if (_exists) {
      final MutableClassDeclaration cl = cxt.findClass(identifier);
      if ((cl != null)) {
        final Function1<MutableMethodDeclaration, Boolean> _function_1 = (MutableMethodDeclaration m2) -> {
          return Boolean.valueOf((Objects.equal(m2.getSimpleName(), m.getSimpleName()) && (IterableExtensions.size(m2.getParameters()) == (IterableExtensions.size(m.getParameters()) - 1))));
        };
        final MutableMethodDeclaration m2 = IterableExtensions.findFirst(cl.getDeclaredMethods(), _function_1);
        String _simpleName = m.getSimpleName();
        String _plus = ("_hidden_" + _simpleName);
        m2.setSimpleName(_plus);
      }
    }
  }

  /**
   * Add the _privk3_ method with the original body computed by xtend
   */
  private void methodProcessingAddPriv(final MutableMethodDeclaration m, final MutableClassDeclaration clazz, final String aspectizedClassName, final Map<MutableMethodDeclaration, String> bodies, @Extension final TransformationContext cxt) {
    String _simpleName = m.getSimpleName();
    String _plus = (AspectProcessor.PRIV_PREFIX + _simpleName);
    final Procedure1<MutableMethodDeclaration> _function = (MutableMethodDeclaration it) -> {
      cxt.setPrimarySourceElement(it, m);
      it.setVisibility(Visibility.PROTECTED);
      it.setStatic(true);
      it.setAbstract(false);
      it.setReturnType(m.getReturnType());
      boolean _isAbstract = m.isAbstract();
      if (_isAbstract) {
        final CompilationStrategy _function_1 = (CompilationStrategy.CompilationContext it_1) -> {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("throw new java.lang.RuntimeException(\"Not implemented\");");
          return _builder;
        };
        it.setBody(_function_1);
      } else {
        Expression _body = m.getBody();
        boolean _tripleEquals = (_body == null);
        if (_tripleEquals) {
          final CompilationStrategy _function_2 = (CompilationStrategy.CompilationContext it_1) -> {
            return bodies.get(m);
          };
          it.setBody(_function_2);
        } else {
          it.setBody(m.getBody());
        }
      }
      String _qualifiedName = clazz.getQualifiedName();
      String _plus_1 = (_qualifiedName + aspectizedClassName);
      String _plus_2 = (_plus_1 + AspectProcessor.PROP_NAME);
      it.addParameter(AspectProcessor.PROP_VAR_NAME, 
        cxt.newTypeReference(cxt.findClass(_plus_2)));
      Iterable<? extends MutableParameterDeclaration> _parameters = m.getParameters();
      for (final MutableParameterDeclaration p : _parameters) {
        it.addParameter(p.getSimpleName(), p.getType());
      }
    };
    clazz.addMethod(_plus, _function);
  }

  /**
   * Change the body of the original method in order to call the appropriate _privk3_ methods.
   * Ie. does a dispatch that search in the aspect hierarchy the method that need to be called
   */
  private void methodProcessingChangeBody(final MutableMethodDeclaration m, final MutableClassDeclaration clazz, @Extension final TransformationContext cxt, final List<String> inheritList, final String className) {
    final MutableTypeDeclaration dt = m.getDeclaringType();
    final Function1<MutableParameterDeclaration, String> _function = (MutableParameterDeclaration it) -> {
      return it.getSimpleName();
    };
    String parametersString = IterableExtensions.join(IterableExtensions.map(m.getParameters(), _function), ",");
    final Function1<AnnotationReference, Boolean> _function_1 = (AnnotationReference it) -> {
      String _simpleName = it.getAnnotationTypeDeclaration().getSimpleName();
      return Boolean.valueOf(Objects.equal(_simpleName, AspectProcessor.STEP));
    };
    final boolean isStep = IterableExtensions.exists(m.getAnnotations(), _function_1);
    final String returnInstruction = this.getReturnInstruction(m, cxt);
    final boolean hasReturn = returnInstruction.contains("return");
    final StringBuilder callSB = new StringBuilder();
    final String resultVar = "result";
    StringConcatenation _builder = new StringConcatenation();
    String _name = cxt.newTypeReference(dt).getName();
    _builder.append(_name);
    _builder.append(".");
    String _simpleName = m.getSimpleName();
    String _plus = (AspectProcessor.PRIV_PREFIX + _simpleName);
    _builder.append(_plus);
    _builder.append("(_self_, ");
    String _aspectedClassName = Helper.getAspectedClassName(dt);
    String _plus_1 = ("(" + _aspectedClassName);
    String _plus_2 = (_plus_1 + ")");
    String _plus_3 = (_plus_2 + AspectProcessor.SELF_VAR_NAME);
    String _replaceFirst = parametersString.replaceFirst(AspectProcessor.SELF_VAR_NAME, _plus_3);
    _builder.append(_replaceFirst);
    _builder.append(")");
    String privcall = _builder.toString();
    if (isStep) {
      String _xifexpression = null;
      boolean _contains = parametersString.contains(",");
      if (_contains) {
        int _indexOf = parametersString.indexOf(",");
        int _plus_4 = (_indexOf + 1);
        _xifexpression = parametersString.substring(_plus_4);
      } else {
        _xifexpression = "";
      }
      final String parametersWithoutSelf = _xifexpression;
      privcall = this.surroundWithStepCommandExecution(className, m.getSimpleName(), privcall, hasReturn, resultVar, parametersWithoutSelf);
    } else {
      if (hasReturn) {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append(resultVar);
        _builder_1.append(" = ");
        _builder_1.append(privcall);
        privcall = _builder_1.toString();
      }
    }
    final Consumer<String> _function_2 = (String dpc) -> {
      StringConcatenation _builder_2 = new StringConcatenation();
      _builder_2.append(dpc);
      _builder_2.newLineIfNotEmpty();
      callSB.append(_builder_2);
    };
    this.projectStaticDispatchBuilder.findExistingDispatchCalls(m, cxt).forEach(_function_2);
    StringConcatenation _builder_2 = new StringConcatenation();
    _builder_2.append("// ");
    _builder_2.append(AspectProcessor.DISPATCH_POINTCUT_KEY);
    _builder_2.append(" ");
    String _initialMethodSignature = Helper.initialMethodSignature(m);
    _builder_2.append(_initialMethodSignature);
    _builder_2.newLineIfNotEmpty();
    _builder_2.append("if (");
    _builder_2.append(AspectProcessor.SELF_VAR_NAME);
    _builder_2.append(" instanceof ");
    String _aspectedClassName_1 = Helper.getAspectedClassName(dt);
    _builder_2.append(_aspectedClassName_1);
    _builder_2.append("){");
    _builder_2.newLineIfNotEmpty();
    _builder_2.append("\t");
    _builder_2.append(privcall, "\t");
    _builder_2.append(";");
    _builder_2.newLineIfNotEmpty();
    _builder_2.append("}");
    callSB.append(_builder_2);
    m.setAbstract(false);
    final CompilationStrategy _function_3 = (CompilationStrategy.CompilationContext it) -> {
      return this.getBody(clazz, className, callSB.toString(), returnInstruction);
    };
    m.setBody(_function_3);
  }

  private boolean hasReturnType(final MutableMethodDeclaration declaration, @Extension final TransformationContext cxt) {
    TypeReference _returnType = declaration.getReturnType();
    TypeReference _newTypeReference = cxt.newTypeReference("void");
    return (!Objects.equal(_returnType, _newTypeReference));
  }

  private String getReturnInstruction(final MutableMethodDeclaration declaration, @Extension final TransformationContext cxt) {
    String ret = "";
    boolean _hasReturnType = this.hasReturnType(declaration, cxt);
    if (_hasReturnType) {
      boolean _isInferred = declaration.getReturnType().isInferred();
      boolean _not = (!_isInferred);
      if (_not) {
        String _name = declaration.getReturnType().getName();
        String _plus = ("return (" + _name);
        String _plus_1 = (_plus + ")result;");
        ret = _plus_1;
      } else {
        cxt.addError(declaration, 
          "Cannot infer return type. Please specify the return type of this method.");
        ret = "return result;";
      }
    } else {
      ret = "";
    }
    return ret;
  }

  private String surroundWithStepCommandExecution(final String className, final String methodName, final String code, final boolean hasReturn, final String resultVar, final String parameters) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("fr.inria.diverse.k3.al.annotationprocessor.stepmanager.StepCommand command = new fr.inria.diverse.k3.al.annotationprocessor.stepmanager.StepCommand() {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("@Override");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("public void execute() {");
    _builder.newLine();
    {
      if (hasReturn) {
        _builder.append("\t\t");
        _builder.append("addToResult(");
        _builder.append(code, "\t\t");
        _builder.append(");");
        _builder.newLineIfNotEmpty();
      } else {
        _builder.append("\t\t");
        _builder.append(code, "\t\t");
        _builder.append(";");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("};");
    _builder.newLine();
    _builder.append("fr.inria.diverse.k3.al.annotationprocessor.stepmanager.IStepManager stepManager = fr.inria.diverse.k3.al.annotationprocessor.stepmanager.StepManagerRegistry.getInstance().findStepManager(_self);");
    _builder.newLine();
    _builder.append("if (stepManager != null) {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("stepManager.executeStep(_self, new Object[] {");
    _builder.append(parameters, "\t");
    _builder.append("}, command, \"");
    _builder.append(className, "\t");
    _builder.append("\", \"");
    _builder.append(methodName, "\t");
    _builder.append("\");");
    _builder.newLineIfNotEmpty();
    _builder.append("} else {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("command.execute();");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    {
      if (hasReturn) {
        _builder.append(resultVar);
        _builder.append(" = command.getResult().get();");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder.toString();
  }

  /**
   * Creates the body for the method that is able to take of the <i>_self_</i> property.
   * Ie call that is able to know about the companion object *AspectProperties.
   * @param clazz type of the aspect (used to compute the name of AspectProperties class)
   * @param className
   * @param call
   */
  private CharSequence getBody(final MutableClassDeclaration clazz, final String className, final String call, final String returnStatement) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("final ");
    String _qualifiedName = clazz.getQualifiedName();
    String _plus = (_qualifiedName + className);
    String _plus_1 = (_plus + AspectProcessor.PROP_NAME);
    _builder.append(_plus_1);
    _builder.append(" ");
    _builder.append(AspectProcessor.PROP_VAR_NAME);
    _builder.append(" = ");
    String _qualifiedName_1 = clazz.getQualifiedName();
    String _plus_2 = (_qualifiedName_1 + className);
    String _plus_3 = (_plus_2 + AspectProcessor.CTX_NAME);
    _builder.append(_plus_3);
    _builder.append(".getSelf(");
    _builder.append(AspectProcessor.SELF_VAR_NAME);
    _builder.append(");");
    _builder.newLineIfNotEmpty();
    {
      boolean _contains = returnStatement.contains("return");
      if (_contains) {
        _builder.append("Object result = null;");
        _builder.newLine();
      }
    }
    String _string = call.toString();
    _builder.append(_string);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.append(returnStatement);
    return _builder;
  }

  /**
   * In the case of multi-inheritance, operations of the aspects that cannot be classically extended must be added
   * to the class.
   */
  private void methodProcessingAddMultiInheritMeth(final MutableClassDeclaration clazz, final String identifier, @Extension final TransformationContext cxt) {
    final Function1<TypeReference, Boolean> _function = (TypeReference cl) -> {
      TypeReference _extendedClass = clazz.getExtendedClass();
      return Boolean.valueOf((!Objects.equal(cl, _extendedClass)));
    };
    final Function1<TypeReference, MutableClassDeclaration> _function_1 = (TypeReference cl) -> {
      return cxt.findClass(cl.getName());
    };
    final Iterable<MutableClassDeclaration> superClasses = IterableExtensions.<MutableClassDeclaration>filterNull(IterableExtensions.<TypeReference, MutableClassDeclaration>map(IterableExtensions.<TypeReference>filter(Helper.getAnnotationWithType(clazz), _function), _function_1));
    final Set<MutableClassDeclaration> scs = CollectionLiterals.<MutableClassDeclaration>newHashSet();
    final Consumer<MutableClassDeclaration> _function_2 = (MutableClassDeclaration sc) -> {
      Helper.getSuperClasses(sc, scs, cxt);
    };
    superClasses.forEach(_function_2);
    final Consumer<MutableClassDeclaration> _function_3 = (MutableClassDeclaration sc) -> {
      final Function1<MutableMethodDeclaration, Boolean> _function_4 = (MutableMethodDeclaration dm) -> {
        return Boolean.valueOf((((!Objects.equal(dm.getVisibility(), Visibility.PRIVATE)) && (!dm.getSimpleName().startsWith(AspectProcessor.PRIV_PREFIX))) && 
          (!IterableExtensions.exists(clazz.getDeclaredMethods(), ((Function1<MutableMethodDeclaration, Boolean>) (MutableMethodDeclaration dm2) -> {
            return Boolean.valueOf(Helper.isSamePrototype(dm, dm2, true));
          })))));
      };
      final Consumer<MutableMethodDeclaration> _function_5 = (MutableMethodDeclaration dm) -> {
        final Procedure1<MutableMethodDeclaration> _function_6 = (MutableMethodDeclaration it) -> {
          cxt.setPrimarySourceElement(it, dm);
          it.setVisibility(dm.getVisibility());
          it.setStatic(true);
          it.setFinal(false);
          it.setAbstract(false);
          it.setReturnType(dm.getReturnType());
          it.addParameter(AspectProcessor.SELF_VAR_NAME, cxt.newTypeReference(identifier));
          final Consumer<MutableParameterDeclaration> _function_7 = (MutableParameterDeclaration par) -> {
            it.addParameter(par.getSimpleName(), par.getType());
          };
          IterableExtensions.drop(dm.getParameters(), 1).forEach(_function_7);
        };
        final MutableMethodDeclaration me = clazz.addMethod(dm.getSimpleName(), _function_6);
        this.methodProcessingAddSelfStatic(me, identifier, cxt);
        final Function1<MutableParameterDeclaration, String> _function_7 = (MutableParameterDeclaration it) -> {
          return it.getSimpleName();
        };
        final String params = IterableExtensions.join(IterableExtensions.map(me.getParameters(), _function_7), ",");
        final CompilationStrategy _function_8 = (CompilationStrategy.CompilationContext it) -> {
          StringConcatenation _builder = new StringConcatenation();
          {
            if (((me.getReturnType() == null) || Objects.equal(me.getReturnType().getSimpleName(), "void"))) {
            } else {
              _builder.append("return ");
            }
          }
          String _simpleName = sc.getSimpleName();
          _builder.append(_simpleName);
          _builder.append(".");
          String _simpleName_1 = dm.getSimpleName();
          _builder.append(_simpleName_1);
          _builder.append("(");
          _builder.append(params);
          _builder.append(");");
          return _builder;
        };
        me.setBody(_function_8);
      };
      IterableExtensions.filter(sc.getDeclaredMethods(), _function_4).forEach(_function_5);
      final Function1<MutableFieldDeclaration, Boolean> _function_6 = (MutableFieldDeclaration it) -> {
        String _simpleName = it.getSimpleName();
        return Boolean.valueOf((!Objects.equal(_simpleName, AspectProcessor.PROP_VAR_NAME)));
      };
      final Consumer<MutableFieldDeclaration> _function_7 = (MutableFieldDeclaration fi) -> {
        final String clName = fi.getDeclaringType().getSimpleName();
        final Procedure1<MutableMethodDeclaration> _function_8 = (MutableMethodDeclaration it) -> {
          cxt.setPrimarySourceElement(it, fi);
          it.setStatic(true);
          it.setReturnType(fi.getType());
          it.addParameter(AspectProcessor.SELF_VAR_NAME, cxt.newTypeReference(identifier));
          final CompilationStrategy _function_9 = (CompilationStrategy.CompilationContext it_1) -> {
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("return ");
            _builder.append(clName);
            _builder.append(".");
            String _simpleName = fi.getSimpleName();
            _builder.append(_simpleName);
            _builder.append("(");
            _builder.append(AspectProcessor.SELF_VAR_NAME);
            _builder.append(");");
            return _builder;
          };
          it.setBody(_function_9);
        };
        clazz.addMethod(fi.getSimpleName(), _function_8);
        boolean _isFinal = fi.isFinal();
        boolean _not = (!_isFinal);
        if (_not) {
          final Procedure1<MutableMethodDeclaration> _function_9 = (MutableMethodDeclaration it) -> {
            cxt.setPrimarySourceElement(it, fi);
            it.setStatic(true);
            it.setReturnType(cxt.newTypeReference("void"));
            it.addParameter(AspectProcessor.SELF_VAR_NAME, cxt.newTypeReference(identifier));
            it.addParameter(fi.getSimpleName(), fi.getType());
            final CompilationStrategy _function_10 = (CompilationStrategy.CompilationContext it_1) -> {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append(clName);
              _builder.append(".");
              String _simpleName = fi.getSimpleName();
              _builder.append(_simpleName);
              _builder.append("(");
              _builder.append(AspectProcessor.SELF_VAR_NAME);
              _builder.append(", ");
              String _simpleName_1 = fi.getSimpleName();
              _builder.append(_simpleName_1);
              _builder.append(");");
              return _builder;
            };
            it.setBody(_function_10);
          };
          clazz.addMethod(fi.getSimpleName(), _function_9);
        }
      };
      IterableExtensions.filter(sc.getDeclaredFields(), _function_6).forEach(_function_7);
    };
    scs.forEach(_function_3);
  }

  private void methodsProcessing(final MutableClassDeclaration clazz, final TransformationContext cxt, final String identifier, final Map<MutableMethodDeclaration, String> bodies, final List<String> inheritList, final String aspectizedClassName) {
    Iterable<? extends MutableMethodDeclaration> _declaredMethods = clazz.getDeclaredMethods();
    for (final MutableMethodDeclaration m : _declaredMethods) {
      boolean _checkAnnotationprocessorCorrect = this.checkAnnotationprocessorCorrect(m, clazz, cxt);
      if (_checkAnnotationprocessorCorrect) {
        final Function1<AnnotationReference, Boolean> _function = (AnnotationReference an) -> {
          String _qualifiedName = an.getAnnotationTypeDeclaration().getQualifiedName();
          return Boolean.valueOf(Objects.equal(_qualifiedName, "java.lang.Override"));
        };
        m.removeAnnotation(IterableExtensions.findFirst(m.getAnnotations(), _function));
        this.methodProcessingAddSelfStatic(m, identifier, cxt);
        this.methodProcessingAddSuper(m, clazz, aspectizedClassName, cxt);
        this.methodProcessingAddHidden(m, identifier, cxt);
        this.methodProcessingAddPriv(m, clazz, aspectizedClassName, bodies, cxt);
        this.methodProcessingChangeBody(m, clazz, cxt, inheritList, aspectizedClassName);
      } else {
        cxt.addError(m, "Cannot find a super method in the aspect hierarchy.");
      }
    }
    this.methodProcessingAddMultiInheritMeth(clazz, identifier, cxt);
  }

  /**
   * Checks that the given method of the given class is correctly tagged with the annotation OverrideAspectMethod, i.e.
   * checks that a super method exists in its hierarchy.
   */
  private boolean checkAnnotationprocessorCorrect(final MutableMethodDeclaration m, final MutableClassDeclaration clazz, final TransformationContext cxt) {
    final Function1<AnnotationReference, Boolean> _function = (AnnotationReference it) -> {
      String _simpleName = it.getAnnotationTypeDeclaration().getSimpleName();
      return Boolean.valueOf(Objects.equal(_simpleName, AspectProcessor.OVERRIDE_METHOD));
    };
    boolean _exists = IterableExtensions.exists(m.getAnnotations(), _function);
    boolean _not = (!_exists);
    if (_not) {
      return true;
    }
    final List<ClassDeclaration> supers = Helper.getDirectPrimaryAndSecondarySuperClasses(clazz, cxt);
    boolean _isEmpty = supers.isEmpty();
    if (_isEmpty) {
      String _simpleName = m.getSimpleName();
      String _plus = ("Missing superclass: declaring the method " + _simpleName);
      String _plus_1 = (_plus + " with @OverrideAspectMethod implies to have at least a parent class");
      cxt.addError(clazz, _plus_1);
      return false;
    }
    final Function1<ClassDeclaration, Boolean> _function_1 = (ClassDeclaration superCl) -> {
      MethodDeclaration _findMethod = Helper.findMethod(superCl, m, cxt);
      return Boolean.valueOf((_findMethod != null));
    };
    return IterableExtensions.<ClassDeclaration>exists(supers, _function_1);
  }

  private void constructorsProcessing(final MutableClassDeclaration clazz, final TransformationContext cxt, final String identifier, final Map<MutableMethodDeclaration, String> bodies, final String className) {
    Iterable<? extends MutableConstructorDeclaration> _declaredConstructors = clazz.getDeclaredConstructors();
    for (final MutableConstructorDeclaration c : _declaredConstructors) {
      Expression _body = c.getBody();
      boolean _tripleNotEquals = (_body != null);
      if (_tripleNotEquals) {
        cxt.addError(c, 
          "Constructors not supported in aspect. Please consider using the @AspectInitializer annotation instead.");
      }
    }
  }

  /**
   * Create the class which links classes with their aspects
   */
  private void aspectContextMaker(@Extension final TransformationContext context, final MutableClassDeclaration clazz, final String className, final String identifier) {
    String _qualifiedName = clazz.getQualifiedName();
    String _plus = (_qualifiedName + className);
    String _plus_1 = (_plus + AspectProcessor.CTX_NAME);
    final MutableClassDeclaration holderClass = context.findClass(_plus_1);
    if ((holderClass == null)) {
      return;
    }
    context.setPrimarySourceElement(holderClass, clazz);
    holderClass.setVisibility(Visibility.PUBLIC);
    final Procedure1<MutableConstructorDeclaration> _function = (MutableConstructorDeclaration it) -> {
      it.setVisibility(Visibility.PRIVATE);
      context.setPrimarySourceElement(it, clazz);
    };
    holderClass.addConstructor(_function);
    final Procedure1<MutableFieldDeclaration> _function_1 = (MutableFieldDeclaration it) -> {
      it.setVisibility(Visibility.PUBLIC);
      it.setStatic(true);
      it.setFinal(true);
      it.setType(context.newTypeReference(holderClass));
      final CompilationStrategy _function_2 = (CompilationStrategy.CompilationContext it_1) -> {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("new ");
        String _simpleName = holderClass.getSimpleName();
        _builder.append(_simpleName);
        _builder.append("()");
        return _builder;
      };
      it.setInitializer(_function_2);
      context.setPrimarySourceElement(it, clazz);
    };
    holderClass.addField("INSTANCE", _function_1);
    final Procedure1<MutableMethodDeclaration> _function_2 = (MutableMethodDeclaration it) -> {
      it.setVisibility(Visibility.PUBLIC);
      it.setStatic(true);
      it.addParameter("_self", context.newTypeReference(identifier));
      String _qualifiedName_1 = clazz.getQualifiedName();
      String _plus_2 = (_qualifiedName_1 + className);
      String _plus_3 = (_plus_2 + AspectProcessor.PROP_NAME);
      it.setReturnType(context.newTypeReference(context.findClass(_plus_3)));
      final CompilationStrategy _function_3 = (CompilationStrategy.CompilationContext it_1) -> {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("\t\t");
        _builder.append("if (!INSTANCE.map.containsKey(_self))");
        _builder.newLine();
        _builder.append("\t\t\t");
        _builder.append("INSTANCE.map.put(_self, new ");
        String _qualifiedName_2 = clazz.getQualifiedName();
        String _plus_4 = (_qualifiedName_2 + className);
        String _plus_5 = (_plus_4 + AspectProcessor.PROP_NAME);
        _builder.append(_plus_5, "\t\t\t");
        _builder.append("());");
        _builder.newLineIfNotEmpty();
        _builder.append("\t\t");
        _builder.append("return INSTANCE.map.get(_self);");
        return _builder;
      };
      it.setBody(_function_3);
      context.setPrimarySourceElement(it, clazz);
    };
    holderClass.addMethod("getSelf", _function_2);
    final Procedure1<MutableFieldDeclaration> _function_3 = (MutableFieldDeclaration it) -> {
      it.setVisibility(Visibility.PRIVATE);
      it.setStatic(false);
      String _qualifiedName_1 = clazz.getQualifiedName();
      String _plus_2 = (_qualifiedName_1 + className);
      String _plus_3 = (_plus_2 + AspectProcessor.PROP_NAME);
      it.setType(context.newTypeReference("java.util.Map", context.newTypeReference(identifier), 
        context.newTypeReference(context.findClass(_plus_3))));
      final CompilationStrategy _function_4 = (CompilationStrategy.CompilationContext it_1) -> {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("new java.util.WeakHashMap<");
        String _mkstring = Helper.mkstring(context.newTypeReference(identifier).getActualTypeArguments(), ",", "<", ">");
        String _plus_4 = (identifier + _mkstring);
        _builder.append(_plus_4);
        _builder.append(", ");
        String _qualifiedName_2 = clazz.getQualifiedName();
        String _plus_5 = (_qualifiedName_2 + className);
        String _plus_6 = (_plus_5 + AspectProcessor.PROP_NAME);
        _builder.append(_plus_6);
        _builder.append(">()");
        return _builder;
      };
      it.setInitializer(_function_4);
      context.setPrimarySourceElement(it, clazz);
    };
    holderClass.addField("map", _function_3);
    final Procedure1<MutableMethodDeclaration> _function_4 = (MutableMethodDeclaration it) -> {
      it.setVisibility(Visibility.PUBLIC);
      it.setStatic(false);
      String _qualifiedName_1 = clazz.getQualifiedName();
      String _plus_2 = (_qualifiedName_1 + className);
      String _plus_3 = (_plus_2 + AspectProcessor.PROP_NAME);
      it.setReturnType(context.newTypeReference("java.util.Map", context.newTypeReference(identifier), 
        context.newTypeReference(context.findClass(_plus_3))));
      final CompilationStrategy _function_5 = (CompilationStrategy.CompilationContext it_1) -> {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("return map;");
        return _builder;
      };
      it.setBody(_function_5);
      context.setPrimarySourceElement(it, clazz);
    };
    holderClass.addMethod("getMap", _function_4);
  }

  /**
   * Move non static fields
   */
  private void fieldProcessingMoveField(final MutableClassDeclaration clazz, final List<MutableFieldDeclaration> toRemove, final List<MutableFieldDeclaration> propertyAspect, final String className, @Extension final TransformationContext context) {
    String _qualifiedName = clazz.getQualifiedName();
    String _plus = (_qualifiedName + className);
    String _plus_1 = (_plus + AspectProcessor.PROP_NAME);
    final MutableClassDeclaration c = context.findClass(_plus_1);
    if ((c == null)) {
      context.addError(clazz, 
        "Cannot resolve the class to aspectise. Check that the classes to aspectise are not in the same project that your aspects.");
    } else {
      Iterable<? extends MutableFieldDeclaration> _declaredFields = clazz.getDeclaredFields();
      for (final MutableFieldDeclaration f : _declaredFields) {
        boolean _isStatic = f.isStatic();
        boolean _not = (!_isStatic);
        if (_not) {
          String _simpleName = f.getSimpleName();
          boolean _notEquals = (!Objects.equal(_simpleName, AspectProcessor.PROP_VAR_NAME));
          if (_notEquals) {
            toRemove.add(f);
            final Function1<AnnotationReference, Boolean> _function = (AnnotationReference it) -> {
              String _simpleName_1 = it.getAnnotationTypeDeclaration().getSimpleName();
              return Boolean.valueOf(Objects.equal(_simpleName_1, "NotAspectProperty"));
            };
            boolean _exists = IterableExtensions.exists(f.getAnnotations(), _function);
            boolean _not_1 = (!_exists);
            if (_not_1) {
              propertyAspect.add(f);
            }
            final Procedure1<MutableFieldDeclaration> _function_1 = (MutableFieldDeclaration it) -> {
              it.setVisibility(Visibility.PUBLIC);
              it.setStatic(f.isStatic());
              it.setFinal(f.isFinal());
              it.setType(f.getType());
              Expression _initializer = f.getInitializer();
              boolean _tripleNotEquals = (_initializer != null);
              if (_tripleNotEquals) {
                it.setInitializer(f.getInitializer());
              }
              context.setPrimarySourceElement(it, f);
            };
            final MutableFieldDeclaration newField = c.addField(f.getSimpleName(), _function_1);
            final Consumer<AnnotationReference> _function_2 = (AnnotationReference an) -> {
              newField.addAnnotation(an);
            };
            f.getAnnotations().forEach(_function_2);
          } else {
            String _qualifiedName_1 = clazz.getQualifiedName();
            String _plus_2 = (_qualifiedName_1 + className);
            String _plus_3 = (_plus_2 + AspectProcessor.PROP_NAME);
            f.setType(context.newTypeReference(context.findClass(_plus_3)));
            f.setStatic(true);
          }
        }
      }
    }
  }

  private void fieldProcessingAddGetterSetter(final MutableClassDeclaration clazz, final List<MutableFieldDeclaration> propertyAspect, final String identifier, final Map<MutableMethodDeclaration, String> bodies, @Extension final TransformationContext context) {
    for (final MutableFieldDeclaration f : propertyAspect) {
      {
        final Procedure1<MutableMethodDeclaration> _function = (MutableMethodDeclaration it) -> {
          it.setReturnType(f.getType());
          it.addParameter(AspectProcessor.SELF_VAR_NAME, context.newTypeReference(identifier));
          context.setPrimarySourceElement(it, f);
          it.setVisibility(f.getVisibility());
          final Consumer<AnnotationReference> _function_1 = (AnnotationReference ann) -> {
            it.addAnnotation(ann);
          };
          f.getAnnotations().forEach(_function_1);
        };
        MutableMethodDeclaration get = clazz.addMethod(f.getSimpleName(), _function);
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("try {");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("for (java.lang.reflect.Method m : _self.getClass().getMethods()) {");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("if (m.getName().equals(\"");
        {
          if ((Objects.equal(f.getType().getSimpleName(), "boolean") || Objects.equal(f.getType().getSimpleName(), "Boolean"))) {
            _builder.append("is");
          } else {
            _builder.append("get");
          }
        }
        String _upperCase = f.getSimpleName().substring(0, 1).toUpperCase();
        String _substring = f.getSimpleName().substring(1);
        String _plus = (_upperCase + _substring);
        _builder.append(_plus, "\t\t");
        _builder.append("\") &&");
        _builder.newLineIfNotEmpty();
        _builder.append("\t\t\t");
        _builder.append("m.getParameterTypes().length == 0) {");
        _builder.newLine();
        _builder.append("\t\t\t\t");
        _builder.append("Object ret = m.invoke(_self);");
        _builder.newLine();
        _builder.append("\t\t\t\t");
        _builder.append("if (ret != null) {");
        _builder.newLine();
        _builder.append("\t\t\t\t\t");
        _builder.append("return (");
        String _qualifiedName = f.getType().getType().getQualifiedName();
        _builder.append(_qualifiedName, "\t\t\t\t\t");
        _builder.append(") ret;");
        _builder.newLineIfNotEmpty();
        _builder.append("\t\t\t\t");
        _builder.append("}");
        {
          boolean _isPrimitive = f.getType().isPrimitive();
          boolean _not = (!_isPrimitive);
          if (_not) {
            _builder.append(" else {");
            _builder.newLineIfNotEmpty();
            _builder.append("\t\t\t\t");
            _builder.append("\t");
            _builder.append("return null;");
            _builder.newLine();
            _builder.append("\t\t\t\t");
            _builder.append("}");
            _builder.newLine();
          }
        }
        _builder.append("\t\t");
        _builder.append("}");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("}");
        _builder.newLine();
        _builder.append("} catch (Exception e) {");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("// Chut !");
        _builder.newLine();
        _builder.append("}");
        _builder.newLine();
        _builder.append("return ");
        _builder.append(AspectProcessor.PROP_VAR_NAME);
        _builder.append(".");
        String _simpleName = f.getSimpleName();
        _builder.append(_simpleName);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        final String gemocHackGetter = _builder.toString();
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append(gemocHackGetter);
        bodies.put(get, _builder_1.toString());
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append("try {");
        _builder_2.newLine();
        _builder_2.append("\t");
        _builder_2.append("for (java.lang.reflect.Method m : _self.getClass().getMethods()) {");
        _builder_2.newLine();
        _builder_2.append("\t\t");
        _builder_2.append("if (m.getName().equals(\"set");
        String _upperCase_1 = f.getSimpleName().substring(0, 1).toUpperCase();
        String _substring_1 = f.getSimpleName().substring(1);
        String _plus_1 = (_upperCase_1 + _substring_1);
        _builder_2.append(_plus_1, "\t\t");
        _builder_2.append("\")");
        _builder_2.newLineIfNotEmpty();
        _builder_2.append("\t\t\t\t");
        _builder_2.append("&& m.getParameterTypes().length == 1) {");
        _builder_2.newLine();
        _builder_2.append("\t\t\t");
        _builder_2.append("m.invoke(_self, ");
        String _simpleName_1 = f.getSimpleName();
        _builder_2.append(_simpleName_1, "\t\t\t");
        _builder_2.append(");");
        _builder_2.newLineIfNotEmpty();
        _builder_2.append("\t\t\t");
        _builder_2.append("setterCalled = true;");
        _builder_2.newLine();
        _builder_2.append("\t\t");
        _builder_2.append("}");
        _builder_2.newLine();
        _builder_2.append("\t");
        _builder_2.append("}");
        _builder_2.newLine();
        _builder_2.append("} catch (Exception e) {");
        _builder_2.newLine();
        _builder_2.append("\t");
        _builder_2.append("// Chut !");
        _builder_2.newLine();
        _builder_2.append("}");
        _builder_2.newLine();
        final String gemocHackSetter = _builder_2.toString();
        boolean _isFinal = f.isFinal();
        boolean _not_1 = (!_isFinal);
        if (_not_1) {
          final Procedure1<MutableMethodDeclaration> _function_1 = (MutableMethodDeclaration it) -> {
            it.setReturnType(context.newTypeReference("void"));
            it.addParameter(AspectProcessor.SELF_VAR_NAME, context.newTypeReference(identifier));
            it.addParameter(f.getSimpleName(), f.getType());
            it.setVisibility(f.getVisibility());
            context.setPrimarySourceElement(it, f);
            final Consumer<AnnotationReference> _function_2 = (AnnotationReference ann) -> {
              it.addAnnotation(ann);
            };
            f.getAnnotations().forEach(_function_2);
          };
          MutableMethodDeclaration set = clazz.addMethod(f.getSimpleName(), _function_1);
          StringConcatenation _builder_3 = new StringConcatenation();
          _builder_3.append("boolean setterCalled = false;");
          _builder_3.newLine();
          _builder_3.append(gemocHackSetter);
          _builder_3.newLineIfNotEmpty();
          _builder_3.append("if (!setterCalled) {");
          _builder_3.newLine();
          _builder_3.append("\t");
          _builder_3.append(AspectProcessor.PROP_VAR_NAME, "\t");
          _builder_3.append(".");
          String _simpleName_2 = f.getSimpleName();
          _builder_3.append(_simpleName_2, "\t");
          _builder_3.append(" = ");
          String _simpleName_3 = f.getSimpleName();
          _builder_3.append(_simpleName_3, "\t");
          _builder_3.append(";");
          _builder_3.newLineIfNotEmpty();
          _builder_3.append("}");
          _builder_3.newLine();
          bodies.put(set, _builder_3.toString());
        }
      }
    }
  }

  /**
   * Move fields of the aspect to the AspectProperties class
   */
  private void fieldsProcessing(@Extension final TransformationContext context, final MutableClassDeclaration clazz, final String className, final String identifier, final Map<MutableMethodDeclaration, String> bodies) {
    final List<MutableFieldDeclaration> toRemove = CollectionLiterals.<MutableFieldDeclaration>newArrayList();
    final List<MutableFieldDeclaration> propertyAspect = CollectionLiterals.<MutableFieldDeclaration>newArrayList();
    this.fieldProcessingMoveField(clazz, toRemove, propertyAspect, className, context);
    this.fieldProcessingAddGetterSetter(clazz, propertyAspect, identifier, bodies, context);
    for (final MutableFieldDeclaration f : toRemove) {
      {
        final MutableFieldDeclaration field = ((MutableFieldDeclaration) f);
        field.remove();
      }
    }
  }

  /**
   * Generate the code and file for a ClassDeclaration
   * Ie. if the ClassDeclaration contains at least one Method
   * with  ReplaceAspectMethod annotation or one field with SynchroField annotation
   */
  private void generateAspectJCodeForClass(final ClassDeclaration classDecl, @Extension final CodeGenerationContext context) {
    final TypeReference typeRef = Helper.getAnnotationAspectType(classDecl);
    final StringBuilder stAspectJ = new StringBuilder();
    boolean doGenerate = false;
    CharSequence _subSequence = typeRef.getName().subSequence(0, typeRef.getName().lastIndexOf("."));
    String _plus = ("package " + _subSequence);
    String _plus_1 = (_plus + ";\n");
    stAspectJ.append(_plus_1);
    String _simpleName = typeRef.getSimpleName();
    String _plus_2 = ("public aspect AspectJ" + _simpleName);
    String _plus_3 = (_plus_2 + "{\n");
    stAspectJ.append(_plus_3);
    Iterable<? extends MethodDeclaration> _declaredMethods = classDecl.getDeclaredMethods();
    for (final MethodDeclaration m : _declaredMethods) {
      final Function1<AnnotationReference, Boolean> _function = (AnnotationReference it) -> {
        String _simpleName_1 = it.getAnnotationTypeDeclaration().getSimpleName();
        return Boolean.valueOf(Objects.equal(_simpleName_1, "ReplaceAspectMethod"));
      };
      boolean _exists = IterableExtensions.exists(m.getAnnotations(), _function);
      if (_exists) {
        doGenerate = true;
        String _simpleName_1 = m.getReturnType().getSimpleName();
        String _plus_4 = (_simpleName_1 + " around (");
        String _name = typeRef.getName();
        String _plus_5 = (_plus_4 + _name);
        String _plus_6 = (_plus_5 + 
          " self)  :target (self) && (call ( ");
        String _name_1 = m.getReturnType().getName();
        String _plus_7 = (_plus_6 + _name_1);
        String _plus_8 = (_plus_7 + " ");
        String _name_2 = typeRef.getName();
        String _plus_9 = (_plus_8 + _name_2);
        String _plus_10 = (_plus_9 + 
          ".");
        String _simpleName_2 = m.getSimpleName();
        String _plus_11 = (_plus_10 + _simpleName_2);
        String _plus_12 = (_plus_11 + "( ");
        stAspectJ.append(_plus_12);
        final Consumer<ParameterDeclaration> _function_1 = (ParameterDeclaration p) -> {
          stAspectJ.append(p.getType().getName());
          boolean _equals = IterableExtensions.last(m.getParameters()).equals(p);
          boolean _not = (!_equals);
          if (_not) {
            stAspectJ.append(",");
          }
        };
        m.getParameters().forEach(_function_1);
        stAspectJ.append(" ) ) ) { ");
        boolean _equals = "void".equals(m.getReturnType().getName());
        boolean _not = (!_equals);
        if (_not) {
          stAspectJ.append("return ");
        }
        String _qualifiedName = classDecl.getQualifiedName();
        String _plus_13 = (_qualifiedName + ".");
        String _simpleName_3 = m.getSimpleName();
        String _plus_14 = (_plus_13 + _simpleName_3);
        String _plus_15 = (_plus_14 + "(self");
        stAspectJ.append(_plus_15);
        for (int i = 0; (i < IterableExtensions.size(m.getParameters())); i++) {
          String _name_3 = (((ParameterDeclaration[])Conversions.unwrapArray(m.getParameters(), ParameterDeclaration.class))[i]).getType().getName();
          String _plus_16 = (("," + 
            "(") + _name_3);
          String _plus_17 = (_plus_16 + ")thisJoinPoint.getArgs()[");
          String _plus_18 = (_plus_17 + Integer.valueOf(i));
          String _plus_19 = (_plus_18 + "]");
          stAspectJ.append(_plus_19);
        }
        stAspectJ.append(" );}\n");
      }
    }
    Iterable<? extends FieldDeclaration> _declaredFields = classDecl.getDeclaredFields();
    for (final FieldDeclaration a : _declaredFields) {
      final Function1<AnnotationReference, Boolean> _function_2 = (AnnotationReference it) -> {
        String _simpleName_4 = it.getAnnotationTypeDeclaration().getSimpleName();
        return Boolean.valueOf(Objects.equal(_simpleName_4, "SynchroField"));
      };
      boolean _exists_1 = IterableExtensions.exists(a.getAnnotations(), _function_2);
      if (_exists_1) {
        doGenerate = true;
        String _name_3 = typeRef.getName();
        String _plus_16 = ("void around (" + _name_3);
        String _plus_17 = (_plus_16 + " self)  :target (self) &&  call ( void ");
        String _simpleName_4 = typeRef.getSimpleName();
        String _plus_18 = (_plus_17 + _simpleName_4);
        String _plus_19 = (_plus_18 + ".");
        stAspectJ.append(_plus_19);
        String _firstUpper = StringExtensions.toFirstUpper(a.getSimpleName());
        String _plus_20 = ("set" + _firstUpper);
        String _plus_21 = (_plus_20 + "(");
        String _name_4 = a.getType().getName();
        String _plus_22 = (_plus_21 + _name_4);
        String _plus_23 = (_plus_22 + ")){");
        stAspectJ.append(_plus_23);
        String _qualifiedName_1 = classDecl.getQualifiedName();
        String _plus_24 = (_qualifiedName_1 + ".");
        String _simpleName_5 = a.getSimpleName();
        String _plus_25 = (_plus_24 + _simpleName_5);
        String _plus_26 = (_plus_25 + "(self, (");
        String _name_5 = a.getType().getName();
        String _plus_27 = (_plus_26 + _name_5);
        String _plus_28 = (_plus_27 + ")thisJoinPoint.getArgs()[0]);");
        stAspectJ.append(_plus_28);
        stAspectJ.append("proceed(self);\n}\n");
        String _name_6 = typeRef.getName();
        String _plus_29 = ("void around (" + _name_6);
        String _plus_30 = (_plus_29 + " self)  :target (self) &&  call ( void ");
        String _qualifiedName_2 = classDecl.getQualifiedName();
        String _plus_31 = (_plus_30 + _qualifiedName_2);
        String _plus_32 = (_plus_31 + ".");
        stAspectJ.append(_plus_32);
        String _simpleName_6 = a.getSimpleName();
        String _plus_33 = (_simpleName_6 + "(");
        String _name_7 = typeRef.getName();
        String _plus_34 = (_plus_33 + _name_7);
        String _plus_35 = (_plus_34 + ",");
        String _name_8 = a.getType().getName();
        String _plus_36 = (_plus_35 + _name_8);
        String _plus_37 = (_plus_36 + ")){");
        stAspectJ.append(_plus_37);
        String _firstUpper_1 = StringExtensions.toFirstUpper(a.getSimpleName());
        String _plus_38 = ("self.set" + _firstUpper_1);
        String _plus_39 = (_plus_38 + "( (");
        String _name_9 = a.getType().getName();
        String _plus_40 = (_plus_39 + _name_9);
        String _plus_41 = (_plus_40 + ")thisJoinPoint.getArgs()[0]);");
        stAspectJ.append(_plus_41);
        stAspectJ.append("proceed(self);\n}\n");
      }
    }
    stAspectJ.append("\n}\n");
    final Path filePath = classDecl.getCompilationUnit().getFilePath();
    Path _targetFolder = context.getTargetFolder(filePath);
    String _replace = typeRef.getName().subSequence(0, typeRef.getName().lastIndexOf(".")).toString().replace(".", "/");
    String _plus_42 = (_replace + "/AspectJ");
    String _simpleName_7 = typeRef.getSimpleName();
    String _plus_43 = (_plus_42 + _simpleName_7);
    String _plus_44 = (_plus_43 + ".aj");
    Path targetFilePath = _targetFolder.append(_plus_44);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("// AspectJ classes that have been aspectized and name");
    _builder.newLine();
    String _string = stAspectJ.toString();
    _builder.append(_string);
    final String contents = _builder.toString();
    if (doGenerate) {
      Helper.writeContentsIfNew(targetFilePath, contents, context);
    }
  }

  /**
   * Change the generated java code of parent aspects of this class (only in the same project) in order
   * to inject the dispatch code to this child classDecl
   */
  private void injectDispatchInParentAspects(final ClassDeclaration classDecl, @Extension final CodeGenerationContext context) {
    try {
      final List<ClassDeclaration> allSuperClasses = CollectionLiterals.<ClassDeclaration>newArrayList();
      Helper.getAllPrimaryAndSecondarySuperClasses(allSuperClasses, classDecl, context);
      StringConcatenation _builder = new StringConcatenation();
      String _simpleName = classDecl.getSimpleName();
      _builder.append(_simpleName);
      _builder.append(" has ");
      int _size = allSuperClasses.size();
      _builder.append(_size);
      _builder.append(" super classes: ");
      final Function1<ClassDeclaration, String> _function = (ClassDeclaration c) -> {
        return c.getSimpleName();
      };
      String _join = IterableExtensions.join(ListExtensions.<ClassDeclaration, String>map(allSuperClasses, _function), ",");
      _builder.append(_join);
      AspectProcessor.LOGGER.log(Level.FINE, _builder.toString());
      Iterable<? extends MethodDeclaration> _declaredMethods = classDecl.getDeclaredMethods();
      for (final MethodDeclaration m : _declaredMethods) {
        {
          final String methodSignature = Helper.initialMethodSignature(m);
          for (final ClassDeclaration superclass : allSuperClasses) {
            if ((IterableExtensions.exists(superclass.getAnnotations(), ((Function1<AnnotationReference, Boolean>) (AnnotationReference it) -> {
              String _simpleName_1 = it.getAnnotationTypeDeclaration().getSimpleName();
              String _simpleName_2 = Aspect.class.getSimpleName();
              return Boolean.valueOf(Objects.equal(_simpleName_1, _simpleName_2));
            })) && 
              IterableExtensions.exists(superclass.getDeclaredMethods(), ((Function1<MethodDeclaration, Boolean>) (MethodDeclaration supermethod) -> {
                String _initialMethodSignature = Helper.initialMethodSignature(supermethod);
                return Boolean.valueOf(Objects.equal(methodSignature, _initialMethodSignature));
              })))) {
              final boolean isSuperClassInProject = this.aspectMappingBuilder.getAllDeclaredAspects().contains(superclass.getQualifiedName());
              StringConcatenation _builder_1 = new StringConcatenation();
              String _simpleName_1 = classDecl.getSimpleName();
              _builder_1.append(_simpleName_1);
              _builder_1.append(" extends ");
              String _simpleName_2 = superclass.getSimpleName();
              _builder_1.append(_simpleName_2);
              _builder_1.append(" isSuperClassInProject = ");
              _builder_1.append(isSuperClassInProject);
              AspectProcessor.LOGGER.log(Level.FINE, _builder_1.toString());
              if (isSuperClassInProject) {
                final Path superclassfilePath = superclass.getCompilationUnit().getFilePath();
                Path _xifexpression = null;
                if (((((Object[])Conversions.unwrapArray(context.getSourceFolder(superclassfilePath).relativize(context.getProjectFolder(superclassfilePath)).getSegments(), Object.class)).length > 1) && 
                  (((Object[])Conversions.unwrapArray(context.getTargetFolder(superclassfilePath).relativize(context.getProjectFolder(superclassfilePath)).getSegments(), Object.class)).length == 1))) {
                  Path _xblockexpression = null;
                  {
                    StringConcatenation _builder_2 = new StringConcatenation();
                    String _simpleName_3 = m.getSimpleName();
                    _builder_2.append(_simpleName_3);
                    _builder_2.append(". Aspect processor is applying targetFolder workaround");
                    AspectProcessor.LOGGER.log(Level.FINE, _builder_2.toString());
                    _xblockexpression = context.getSourceFolder(superclassfilePath).getParent().append(
                      context.getTargetFolder(superclassfilePath).relativize(context.getProjectFolder(superclassfilePath)).toString());
                  }
                  _xifexpression = _xblockexpression;
                } else {
                  Path _xblockexpression_1 = null;
                  {
                    StringConcatenation _builder_2 = new StringConcatenation();
                    String _simpleName_3 = m.getSimpleName();
                    _builder_2.append(_simpleName_3);
                    _builder_2.append(". Aspect processor is NOT applying targetFolder workaround");
                    AspectProcessor.LOGGER.log(Level.FINE, _builder_2.toString());
                    _xblockexpression_1 = context.getTargetFolder(superclassfilePath);
                  }
                  _xifexpression = _xblockexpression_1;
                }
                final Path superclasstargetFolder = _xifexpression;
                String _replace = superclass.getQualifiedName().replace(".", "/");
                String _plus = (_replace + ".java");
                Path superclassjavafile = superclasstargetFolder.append(_plus);
                StringConcatenation _builder_2 = new StringConcatenation();
                String _simpleName_3 = m.getSimpleName();
                _builder_2.append(_simpleName_3);
                _builder_2.append(". Aspect processor is looking for ");
                _builder_2.append(superclassjavafile);
                _builder_2.append(" ");
                AspectProcessor.LOGGER.log(Level.FINE, _builder_2.toString());
                boolean _waitForFileContent = this.waitForFileContent(superclassjavafile, context);
                boolean _not = (!_waitForFileContent);
                if (_not) {
                  StringConcatenation _builder_3 = new StringConcatenation();
                  _builder_3.append("[");
                  _builder_3.append(this);
                  _builder_3.append("] Timeout occured while processing ");
                  String _qualifiedName = classDecl.getQualifiedName();
                  _builder_3.append(_qualifiedName);
                  _builder_3.append(".");
                  String _simpleName_4 = m.getSimpleName();
                  _builder_3.append(_simpleName_4);
                  _builder_3.append(". Aspect processor is waiting for ");
                  _builder_3.append(superclassjavafile);
                  _builder_3.append(" ");
                  AspectProcessor.LOGGER.log(Level.SEVERE, _builder_3.toString());
                } else {
                  synchronized (AspectProcessor.lock) {
                    StringConcatenation _builder_4 = new StringConcatenation();
                    _builder_4.append("injecting ");
                    String _simpleName_5 = classDecl.getSimpleName();
                    _builder_4.append(_simpleName_5);
                    _builder_4.append(" dispatch code in ");
                    _builder_4.append(superclassjavafile);
                    AspectProcessor.LOGGER.log(Level.FINER, _builder_4.toString());
                    String _name = m.getReturnType().getName();
                    final boolean hasReturn = (!Objects.equal(_name, "void"));
                    String _xifexpression_1 = null;
                    boolean _isEmpty = IterableExtensions.isEmpty(m.getParameters());
                    if (_isEmpty) {
                      StringConcatenation _builder_5 = new StringConcatenation();
                      String _aspectedClassName = Helper.getAspectedClassName(classDecl);
                      String _plus_1 = ("(" + _aspectedClassName);
                      String _plus_2 = (_plus_1 + ")");
                      String _plus_3 = (_plus_2 + AspectProcessor.SELF_VAR_NAME);
                      _builder_5.append(_plus_3);
                      _xifexpression_1 = _builder_5.toString();
                    } else {
                      StringConcatenation _builder_6 = new StringConcatenation();
                      _builder_6.append("(");
                      String _aspectedClassName_1 = Helper.getAspectedClassName(classDecl);
                      _builder_6.append(_aspectedClassName_1);
                      _builder_6.append(")");
                      _builder_6.append(AspectProcessor.SELF_VAR_NAME);
                      _builder_6.append(",");
                      final Function1<ParameterDeclaration, String> _function_1 = (ParameterDeclaration it) -> {
                        return it.getSimpleName();
                      };
                      String _join_1 = IterableExtensions.join(IterableExtensions.map(m.getParameters(), _function_1), ",");
                      _builder_6.append(_join_1);
                      _xifexpression_1 = _builder_6.toString();
                    }
                    final String parametersString = _xifexpression_1;
                    StringConcatenation _builder_7 = new StringConcatenation();
                    String _qualifiedName_1 = classDecl.getQualifiedName();
                    _builder_7.append(_qualifiedName_1);
                    _builder_7.append(".");
                    String _simpleName_6 = m.getSimpleName();
                    _builder_7.append(_simpleName_6);
                    _builder_7.append("(");
                    _builder_7.append(parametersString);
                    _builder_7.append(");");
                    String call = _builder_7.toString();
                    if (hasReturn) {
                      StringConcatenation _builder_8 = new StringConcatenation();
                      _builder_8.append(AspectProcessor.RESULT_VAR);
                      _builder_8.append(" = ");
                      _builder_8.append(call);
                      call = _builder_8.toString();
                    }
                    StringConcatenation _builder_9 = new StringConcatenation();
                    _builder_9.append("// BeginInjectInto ");
                    String _qualifiedName_2 = superclass.getQualifiedName();
                    _builder_9.append(_qualifiedName_2);
                    _builder_9.append("#");
                    String _methodSignature = Helper.methodSignature(m);
                    _builder_9.append(_methodSignature);
                    _builder_9.append(" from ");
                    String _qualifiedName_3 = classDecl.getQualifiedName();
                    _builder_9.append(_qualifiedName_3);
                    final String dispatchInjectKey = _builder_9.toString();
                    StringConcatenation _builder_10 = new StringConcatenation();
                    _builder_10.append("\t");
                    _builder_10.append(dispatchInjectKey, "\t");
                    _builder_10.newLineIfNotEmpty();
                    _builder_10.append("\t\t");
                    _builder_10.append("if (");
                    _builder_10.append(AspectProcessor.SELF_VAR_NAME, "\t\t");
                    _builder_10.append(" instanceof ");
                    String _aspectedClassName_2 = Helper.getAspectedClassName(classDecl);
                    _builder_10.append(_aspectedClassName_2, "\t\t");
                    _builder_10.append("){");
                    _builder_10.newLineIfNotEmpty();
                    _builder_10.append("\t\t\t");
                    _builder_10.append(call, "\t\t\t");
                    _builder_10.newLineIfNotEmpty();
                    _builder_10.append("\t\t");
                    _builder_10.append("} else");
                    _builder_10.newLine();
                    _builder_10.append("\t\t");
                    _builder_10.append("// EndInjectInto ");
                    String _qualifiedName_4 = superclass.getQualifiedName();
                    _builder_10.append(_qualifiedName_4, "\t\t");
                    _builder_10.append("#");
                    String _methodSignature_1 = Helper.methodSignature(m);
                    _builder_10.append(_methodSignature_1, "\t\t");
                    _builder_10.append(" from ");
                    String _qualifiedName_5 = classDecl.getQualifiedName();
                    _builder_10.append(_qualifiedName_5, "\t\t");
                    final String dispatchInjectCodeForParent = _builder_10.toString();
                    this.projectStaticDispatchBuilder.add(dispatchInjectCodeForParent);
                    final String aspectJavaFileContent = this.readFileContents(superclassjavafile, context);
                    boolean _contains = aspectJavaFileContent.contains(dispatchInjectKey);
                    boolean _not_1 = (!_contains);
                    if (_not_1) {
                      String _initialMethodSignature = Helper.initialMethodSignature(m);
                      final String pointcutString = ((("// " + AspectProcessor.DISPATCH_POINTCUT_KEY) + " ") + _initialMethodSignature);
                      StringConcatenation _builder_11 = new StringConcatenation();
                      _builder_11.append(dispatchInjectCodeForParent);
                      _builder_11.newLineIfNotEmpty();
                      _builder_11.append("// ");
                      _builder_11.append(AspectProcessor.DISPATCH_POINTCUT_KEY);
                      _builder_11.append(" ");
                      String _methodSignature_2 = Helper.methodSignature(m);
                      _builder_11.append(_methodSignature_2);
                      final String pointcutReplacement = _builder_11.toString();
                      final String newContent = aspectJavaFileContent.replace(pointcutString, pointcutReplacement);
                      context.setContents(superclassjavafile, newContent);
                      int timeout = 40;
                      do {
                        {
                          Thread.sleep(100);
                          timeout--;
                        }
                      } while(((!Objects.equal(context.getContents(superclassjavafile).toString(), newContent)) && (timeout > 0)));
                    }
                  }
                }
              }
            }
          }
        }
      }
      this.projectStaticDispatchBuilder.cleanDeprecatedDispatchFiles(classDecl.getCompilationUnit(), context);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * wait a little for content
   * timeout after 2 seconds
   * does NOT use org.eclipse.xtend.lib.macro.file.Path.getContents() because it seems to trigger events that reschedules some build phase
   * @return true if content has been retreived or false if timeout occured
   */
  private boolean waitForFileContent(final Path file, @Extension final CodeGenerationContext context) {
    try {
      int timeout = 20;
      String _path = context.toURI(file).getPath();
      final File f = new File(_path);
      while (((!f.exists()) && (timeout > 0))) {
        {
          Thread.sleep(100);
          timeout--;
        }
      }
      timeout = 20;
      while (((f.length() == 0) && (timeout > 0))) {
        {
          Thread.sleep(100);
          timeout--;
        }
      }
      return (timeout != 0);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * read the content of the given file
   * does NOT use org.eclipse.xtend.lib.macro.file.Path.getContents() because it seems to trigger events that reschedules some build phase
   * -> do it using plain java so it doesn't trigger eclipse event
   */
  private String readFileContents(final Path file, @Extension final CodeGenerationContext context) {
    try {
      String _path = context.toURI(file).getPath();
      FileReader _fileReader = new FileReader(_path);
      final BufferedReader br = new BufferedReader(_fileReader);
      final StringBuilder sb = new StringBuilder();
      String line = br.readLine();
      while ((line != null)) {
        {
          sb.append(line);
          sb.append(System.lineSeparator());
          line = br.readLine();
        }
      }
      return sb.toString();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
