/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.trace.commons;

import com.google.common.base.Objects;
import com.google.common.collect.Iterators;
import java.util.Set;
import org.eclipse.emf.codegen.ecore.genmodel.GenClassifier;
import org.eclipse.emf.codegen.ecore.genmodel.GenPackage;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public class EcoreCraftingUtil {
  public static EReference addReferenceToClass(final EClass clazz, final String refName, final EClassifier refType) {
    final EReference res = EcoreFactory.eINSTANCE.createEReference();
    res.setName(refName);
    res.setEType(refType);
    clazz.getEStructuralFeatures().add(res);
    return res;
  }

  public static EReference addReferenceToClass(final EClass clazz, final String refName, final EClass refType) {
    if (((((clazz != null) && (refName != null)) && (!Objects.equal(refName, ""))) && (refType != null))) {
      final EReference res = EcoreFactory.eINSTANCE.createEReference();
      res.setName(refName);
      res.setEType(refType);
      clazz.getEStructuralFeatures().add(res);
      return res;
    } else {
      return null;
    }
  }

  public static String getBaseFQN(final EOperation o) {
    final EClass c = o.getEContainingClass();
    String _baseFQN = EcoreCraftingUtil.getBaseFQN(c);
    String _plus = (_baseFQN + ".");
    String _name = o.getName();
    return (_plus + _name);
  }

  public static EAttribute addAttributeToClass(final EClass clazz, final String attName, final EDataType attType) {
    final EAttribute res = EcoreFactory.eINSTANCE.createEAttribute();
    res.setName(attName);
    res.setEType(attType);
    clazz.getEStructuralFeatures().add(res);
    return res;
  }

  public static EStructuralFeature addFeatureToClass(final EClass clazz, final String name, final EClassifier type) {
    EStructuralFeature res = null;
    if ((type instanceof EDataType)) {
      res = EcoreFactory.eINSTANCE.createEAttribute();
    } else {
      if ((type instanceof EClass)) {
        res = EcoreFactory.eINSTANCE.createEReference();
      }
    }
    res.setName(name);
    res.setEType(type);
    clazz.getEStructuralFeatures().add(res);
    return res;
  }

  public static String getFQN(final EClassifier c, final String separator) {
    final EPackage p = c.getEPackage();
    if ((p != null)) {
      String _ePackageFQN = EcoreCraftingUtil.getEPackageFQN(p, separator);
      String _plus = (_ePackageFQN + separator);
      String _name = c.getName();
      return (_plus + _name);
    } else {
      return c.getName();
    }
  }

  public static String getEPackageFQN(final EPackage p, final String separator) {
    final EPackage superP = p.getESuperPackage();
    if ((superP != null)) {
      String _ePackageFQN = EcoreCraftingUtil.getEPackageFQN(superP, separator);
      String _plus = (_ePackageFQN + separator);
      String _name = p.getName();
      return (_plus + _name);
    } else {
      return p.getName();
    }
  }

  public static String getBaseFQN(final EClassifier c) {
    if ((c != null)) {
      final EPackage p = c.getEPackage();
      if ((p != null)) {
        String _baseFQN = EcoreCraftingUtil.getBaseFQN(p);
        String _plus = (_baseFQN + ".");
        String _name = c.getName();
        return (_plus + _name);
      } else {
        return c.getName();
      }
    } else {
      return "";
    }
  }

  public static String getJavaFQN(final EClassifier c, final Set<GenPackage> refGenPackages) {
    return EcoreCraftingUtil.getJavaFQN(c, refGenPackages, false);
  }

  public static String getJavaFQN(final EClassifier c, final Set<GenPackage> refGenPackages, final boolean enforcePrimitiveJavaClasses) {
    Class<?> _instanceClass = c.getInstanceClass();
    boolean _tripleNotEquals = (_instanceClass != null);
    if (_tripleNotEquals) {
      if (enforcePrimitiveJavaClasses) {
        String _canonicalName = c.getInstanceClass().getCanonicalName();
        if (_canonicalName != null) {
          switch (_canonicalName) {
            case "int":
              return "java.lang.Integer";
            case "boolean":
              return "java.lang.Boolean";
          }
        }
      }
      return c.getInstanceClass().getCanonicalName();
    }
    if (((c.getInstanceClassName() != null) && (!Objects.equal(c.getInstanceClassName(), "")))) {
      return c.getInstanceClassName();
    }
    String base = "";
    final GenClassifier gc = EcoreCraftingUtil.getGenClassifier(c, refGenPackages);
    if (((gc != null) && (gc.getGenPackage() != null))) {
      base = EcoreCraftingUtil.getBase(gc.getGenPackage());
    }
    String _baseFQN = EcoreCraftingUtil.getBaseFQN(c);
    return (base + _baseFQN);
  }

  private static String getBase(final GenPackage gp) {
    String base = "";
    if (((gp.getBasePackage() != null) && (gp.getSuperGenPackage() == null))) {
      String _basePackage = gp.getBasePackage();
      String _plus = (_basePackage + ".");
      base = _plus;
    }
    return base;
  }

  public static String getBaseFQN(final EPackage p) {
    final EPackage superP = p.getESuperPackage();
    if ((superP != null)) {
      String _baseFQN = EcoreCraftingUtil.getBaseFQN(superP);
      String _plus = (_baseFQN + ".");
      String _name = p.getName();
      return (_plus + _name);
    } else {
      return p.getName();
    }
  }

  public static String getJavaFQN(final EPackage p, final Set<GenPackage> refGenPackages) {
    String base = "";
    final GenPackage gp = EcoreCraftingUtil.getGenPackage(p, refGenPackages);
    if ((gp != null)) {
      base = EcoreCraftingUtil.getBase(gp);
    }
    String _baseFQN = EcoreCraftingUtil.getBaseFQN(p);
    return (base + _baseFQN);
  }

  public static GenClassifier getGenClassifier(final EClassifier c, final Set<GenPackage> refGenPackages) {
    if ((c != null)) {
      for (final GenPackage gp : refGenPackages) {
        Set<GenClassifier> _set = IteratorExtensions.<GenClassifier>toSet(Iterators.<GenClassifier>filter(gp.eAllContents(), GenClassifier.class));
        for (final GenClassifier gc : _set) {
          {
            final EClassifier ecoreClass = gc.getEcoreClassifier();
            if ((ecoreClass != null)) {
              final String s1 = EcoreCraftingUtil.getBaseFQN(ecoreClass);
              final String s2 = EcoreCraftingUtil.getBaseFQN(c);
              if ((((s1 != null) && (s2 != null)) && s1.equalsIgnoreCase(s2))) {
                return gc;
              }
            }
          }
        }
      }
    }
    return null;
  }

  public static GenPackage getGenPackage(final EPackage p, final Set<GenPackage> refGenPackages) {
    if ((p != null)) {
      for (final GenPackage gp : refGenPackages) {
        {
          final EPackage packageInGenpackage = gp.getEcorePackage();
          if ((packageInGenpackage != null)) {
            final String s1 = EcoreCraftingUtil.getBaseFQN(p);
            final String s2 = EcoreCraftingUtil.getBaseFQN(packageInGenpackage);
            if ((((s1 != null) && (s2 != null)) && s1.equalsIgnoreCase(s2))) {
              return gp;
            }
          }
        }
      }
    }
    return null;
  }

  public static String stringCreate(final EClass c) {
    final EPackage p = c.getEPackage();
    String _baseFQN = EcoreCraftingUtil.getBaseFQN(p);
    String _plus = (_baseFQN + ".");
    String _firstUpper = StringExtensions.toFirstUpper(p.getName());
    String _plus_1 = (_plus + _firstUpper);
    String _plus_2 = (_plus_1 + "Factory.eINSTANCE.create");
    String _name = c.getName();
    String _plus_3 = (_plus_2 + _name);
    return (_plus_3 + "()");
  }

  public static String stringCreateImplicitStep(final EClass c) {
    final EPackage p = c.getEPackage();
    String _baseFQN = EcoreCraftingUtil.getBaseFQN(p);
    String _plus = (_baseFQN + ".");
    String _firstUpper = StringExtensions.toFirstUpper(p.getName());
    String _plus_1 = (_plus + _firstUpper);
    String _plus_2 = (_plus_1 + "Factory.eINSTANCE.create");
    String _name = c.getName();
    String _plus_3 = (_plus_2 + _name);
    return (_plus_3 + 
      "_ImplicitStep()");
  }

  public static String stringGetter(final EStructuralFeature f) {
    if ((f instanceof EAttribute)) {
      boolean _equals = ((EAttribute)f).getEAttributeType().getName().equals("EBoolean");
      if (_equals) {
        String _firstUpper = StringExtensions.toFirstUpper(((EAttribute)f).getName());
        String _plus = ("is" + _firstUpper);
        return (_plus + "()");
      }
    }
    String _firstUpper_1 = StringExtensions.toFirstUpper(f.getName());
    String _plus_1 = ("get" + _firstUpper_1);
    return (_plus_1 + "()");
  }

  public static String stringFeatureID(final EStructuralFeature feature, final EClassifier containingClass, final Set<GenPackage> refGenPackages) {
    final EPackage p = containingClass.getEPackage();
    final GenPackage gp = EcoreCraftingUtil.getGenPackage(p, refGenPackages);
    String _javaFQN = EcoreCraftingUtil.getJavaFQN(p, refGenPackages);
    String _plus = (_javaFQN + ".");
    String _prefix = gp.getPrefix();
    String _plus_1 = (_plus + _prefix);
    String _plus_2 = (_plus_1 + "Package.eINSTANCE.get");
    String _firstUpper = StringExtensions.toFirstUpper(containingClass.getName());
    String _plus_3 = (_plus_2 + _firstUpper);
    String _plus_4 = (_plus_3 + "_");
    String _firstUpper_1 = StringExtensions.toFirstUpper(feature.getName());
    String _plus_5 = (_plus_4 + _firstUpper_1);
    return (_plus_5 + "().getFeatureID()");
  }

  public static String stringClassifierID(final EClassifier c, final Set<GenPackage> refGenPackages) {
    final EPackage p = c.getEPackage();
    final GenPackage gp = EcoreCraftingUtil.getGenPackage(p, refGenPackages);
    String _javaFQN = EcoreCraftingUtil.getJavaFQN(p, refGenPackages);
    String _plus = (_javaFQN + ".");
    String _prefix = gp.getPrefix();
    String _plus_1 = (_plus + _prefix);
    String _plus_2 = (_plus_1 + "Package.eINSTANCE.get");
    String _firstUpper = StringExtensions.toFirstUpper(c.getName());
    String _plus_3 = (_plus_2 + _firstUpper);
    return (_plus_3 + 
      "().getClassifierID()");
  }

  public static String stringGetter(final String s) {
    String _firstUpper = StringExtensions.toFirstUpper(s);
    String _plus = ("get" + _firstUpper);
    return (_plus + "()");
  }

  public static String stringSetter(final EStructuralFeature f, final String value, final Set<GenPackage> genPackages) {
    String _firstUpper = StringExtensions.toFirstUpper(f.getName());
    String _plus = ("set" + _firstUpper);
    String _plus_1 = (_plus + "( ( ");
    String _javaFQN = EcoreCraftingUtil.getJavaFQN(f.getEType(), genPackages);
    String _plus_2 = (_plus_1 + _javaFQN);
    String _plus_3 = (_plus_2 + " )");
    String _plus_4 = (_plus_3 + value);
    return (_plus_4 + ")");
  }

  public static String stringSetter(final EStructuralFeature f, final String value) {
    String _firstUpper = StringExtensions.toFirstUpper(f.getName());
    String _plus = ("set" + _firstUpper);
    String _plus_1 = (_plus + "(");
    String _plus_2 = (_plus_1 + value);
    return (_plus_2 + ")");
  }

  public static String stringSetter(final String f, final String value) {
    String _firstUpper = StringExtensions.toFirstUpper(f);
    String _plus = ("set" + _firstUpper);
    String _plus_1 = (_plus + "(");
    String _plus_2 = (_plus_1 + value);
    return (_plus_2 + ")");
  }
}
