/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.trace.metamodel.generator;

import com.google.common.base.Objects;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import opsemanticsview.OperationalSemanticsView;
import opsemanticsview.Rule;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Second output of the transformation: a class both to access to parts
 * of the trace metamodel, and with links between the original metamodels
 * and the trace metamodel.
 */
@SuppressWarnings("all")
public class TraceMMGenerationTraceability {
  public TraceMMGenerationTraceability(final TraceMMExplorer traceMMExplorer, final OperationalSemanticsView mmext) {
    this.traceMMExplorer = traceMMExplorer;
    this.mmext = mmext;
  }

  @Accessors({ AccessorType.PUBLIC_GETTER, AccessorType.PACKAGE_SETTER })
  private final TraceMMExplorer traceMMExplorer;

  @Accessors({ AccessorType.PUBLIC_GETTER, AccessorType.PACKAGE_SETTER })
  private final OperationalSemanticsView mmext;

  private Set<EClass> runtimeClasses = new HashSet<EClass>();

  void addRuntimeClass(final EClass c) {
    this.runtimeClasses.add(c);
  }

  public Set<EClass> getRuntimeClasses() {
    return ImmutableSet.<EClass>copyOf(this.runtimeClasses);
  }

  private Map<EClass, Set<EStructuralFeature>> mutableProperties = new HashMap<EClass, Set<EStructuralFeature>>();

  void addMutableProperty(final EClass c, final EStructuralFeature r) {
    boolean _containsKey = this.mutableProperties.containsKey(c);
    boolean _not = (!_containsKey);
    if (_not) {
      HashSet<EStructuralFeature> _hashSet = new HashSet<EStructuralFeature>();
      this.mutableProperties.put(c, _hashSet);
    }
    this.mutableProperties.get(c).add(r);
  }

  public Set<EStructuralFeature> getMutablePropertiesOf(final EClass c) {
    boolean _containsKey = this.mutableProperties.containsKey(c);
    if (_containsKey) {
      return ImmutableSet.<EStructuralFeature>copyOf(this.mutableProperties.get(c));
    } else {
      return Collections.<EStructuralFeature>unmodifiableSet(CollectionLiterals.<EStructuralFeature>newHashSet());
    }
  }

  public Set<EStructuralFeature> getAllMutableProperties() {
    return IterableExtensions.<EStructuralFeature>toSet(Iterables.<EStructuralFeature>concat(this.mutableProperties.values()));
  }

  private final HashMap<EClass, EClass> tracedClasses = new HashMap<EClass, EClass>();

  void putTracedClasses(final EClass runtimeClass, final EClass tracedClass) {
    this.tracedClasses.put(runtimeClass, tracedClass);
  }

  public EClass getTracedClass(final EClass mutableClass) {
    return this.tracedClasses.get(mutableClass);
  }

  public Set<EClass> getTracedClassSet() {
    final HashSet<EClass> result = new HashSet<EClass>();
    result.addAll(this.tracedClasses.keySet());
    return result;
  }

  private final HashMap<EStructuralFeature, EClass> dimensionClasses = new HashMap<EStructuralFeature, EClass>();

  public EClass putDimensionClass(final EStructuralFeature property, final EClass dimension) {
    return this.dimensionClasses.put(property, dimension);
  }

  public EClass getDimensionClass(final EStructuralFeature property) {
    return this.dimensionClasses.get(property);
  }

  private final HashMap<EStructuralFeature, EReference> dimensionRefs = new HashMap<EStructuralFeature, EReference>();

  public EReference putDimensionRef(final EStructuralFeature property, final EReference dimensionRef) {
    return this.dimensionRefs.put(property, dimensionRef);
  }

  public EReference getDimensionRef(final EStructuralFeature property) {
    return this.dimensionRefs.get(property);
  }

  public Set<EClass> getNewClasses() {
    return IterableExtensions.<EClass>toSet(this.mmext.getDynamicClasses());
  }

  public boolean hasTracedClass(final EClass mutableClass) {
    return this.tracedClasses.containsKey(mutableClass);
  }

  public Set<EClass> getAllMutableClasses() {
    return this.tracedClasses.keySet();
  }

  public EClass getRealMutableClass(final EClass tracedClass) {
    final Function1<Map.Entry<EClass, EClass>, Boolean> _function = (Map.Entry<EClass, EClass> p) -> {
      EClass _value = p.getValue();
      return Boolean.valueOf(Objects.equal(_value, tracedClass));
    };
    final Map.Entry<EClass, EClass> mutClass = IterableExtensions.<Map.Entry<EClass, EClass>>findFirst(this.tracedClasses.entrySet(), _function);
    boolean _notEquals = (!Objects.equal(mutClass, null));
    if (_notEquals) {
      return mutClass.getKey();
    } else {
      return null;
    }
  }

  private Map<EClass, Set<EReference>> refs_originalObject = new HashMap<EClass, Set<EReference>>();

  void addRefs_originalObject(final EClass c1, final EReference r) {
    boolean _containsKey = this.refs_originalObject.containsKey(c1);
    boolean _not = (!_containsKey);
    if (_not) {
      HashSet<EReference> _hashSet = new HashSet<EReference>();
      this.refs_originalObject.put(c1, _hashSet);
    }
    this.refs_originalObject.get(c1).add(r);
  }

  public Set<EReference> getRefs_originalObject(final EClass class1) {
    final Set<EReference> res = new HashSet<EReference>();
    final Function1<EClass, Set<EReference>> _function = (EClass c) -> {
      return this.getRefs_originalObject(c);
    };
    final Set<EReference> existingRefs = IterableExtensions.<EReference>toSet(Iterables.<EReference>concat(ListExtensions.<EClass, Set<EReference>>map(class1.getEAllSuperTypes(), _function)));
    res.addAll(existingRefs);
    final Set<EReference> refsForThisClass = this.refs_originalObject.get(class1);
    if (((!Objects.equal(refsForThisClass, null)) && (!refsForThisClass.isEmpty()))) {
      res.addAll(refsForThisClass);
    }
    return res;
  }

  private Map<EStructuralFeature, EClass> valueClass = new HashMap<EStructuralFeature, EClass>();

  void putValueClass(final EStructuralFeature r, final EClass c) {
    this.valueClass.put(r, c);
  }

  public EClass getValueClass(final EStructuralFeature s) {
    return this.valueClass.get(s);
  }

  private Map<EStructuralFeature, EReference> stateClassToValueClass = new HashMap<EStructuralFeature, EReference>();

  void putStateClassToValueClass(final EStructuralFeature r1, final EReference r2) {
    this.stateClassToValueClass.put(r1, r2);
  }

  public EReference getStateClassToValueClass(final EStructuralFeature s) {
    boolean _containsValue = this.mutablePropertyToValueProperty.containsValue(s);
    if (_containsValue) {
      final Function1<Map.Entry<EStructuralFeature, EStructuralFeature>, Boolean> _function = (Map.Entry<EStructuralFeature, EStructuralFeature> entry) -> {
        EStructuralFeature _value = entry.getValue();
        return Boolean.valueOf(Objects.equal(_value, s));
      };
      final EStructuralFeature key = IterableExtensions.<Map.Entry<EStructuralFeature, EStructuralFeature>>findFirst(this.mutablePropertyToValueProperty.entrySet(), _function).getKey();
      return this.stateClassToValueClass.get(key);
    } else {
      return this.stateClassToValueClass.get(s);
    }
  }

  private Set<EClass> stepClasses = new HashSet<EClass>();

  void addStepClass(final EClass c) {
    this.stepClasses.add(c);
  }

  public Set<EClass> getStepClasses() {
    return ImmutableSet.<EClass>copyOf(this.stepClasses);
  }

  private final Map<Rule, EClass> stepRuleToStepClass = new HashMap<Rule, EClass>();

  void addStepRuleToStepClass(final Rule stepRule, final EClass stepClass) {
    this.stepRuleToStepClass.put(stepRule, stepClass);
  }

  public EClass getStepClassFromStepRule(final Rule stepRule) {
    return this.stepRuleToStepClass.get(stepRule);
  }

  private Set<EClass> bigStepClasses = new HashSet<EClass>();

  void addBigStepClass(final EClass c) {
    this.bigStepClasses.add(c);
  }

  public Set<EClass> getBigStepClasses() {
    return ImmutableSet.<EClass>copyOf(this.bigStepClasses);
  }

  private Map<EClass, EClass> implicitStepClasses = new HashMap<EClass, EClass>();

  void putImplicitStepClass(final EClass step, final EClass containgClass) {
    this.implicitStepClasses.put(step, containgClass);
  }

  public Set<EClass> getImplicitStepClasses() {
    return ImmutableSet.<EClass>copyOf(this.implicitStepClasses.keySet());
  }

  public EClass getImplicitStepContainingClass(final EClass implicitStepClass) {
    return this.implicitStepClasses.get(implicitStepClass);
  }

  private final Map<EClass, EReference> stepSequences = new HashMap<EClass, EReference>();

  void addStepSequence(final EClass stepClass, final EReference trace) {
    this.stepSequences.put(stepClass, trace);
  }

  public EReference getStepSequence(final EClass stepClass) {
    return this.stepSequences.get(stepClass);
  }

  public boolean hasExeClass(final EClass tracedClass) {
    final Function1<EClass, Boolean> _function = (EClass k) -> {
      EClass _get = this.tracedClasses.get(k);
      return Boolean.valueOf(Objects.equal(_get, tracedClass));
    };
    return IterableExtensions.<EClass>exists(this.tracedClasses.keySet(), _function);
  }

  public EClass getExeClass(final EClass tracedClass) {
    final Function1<EClass, Boolean> _function = (EClass k) -> {
      EClass _get = this.tracedClasses.get(k);
      return Boolean.valueOf(Objects.equal(_get, tracedClass));
    };
    return IterableExtensions.<EClass>findFirst(this.tracedClasses.keySet(), _function);
  }

  private final Map<EStructuralFeature, EStructuralFeature> mutablePropertyToValueProperty = new HashMap<EStructuralFeature, EStructuralFeature>();

  public void putMutablePropertyToValueProperty(final EStructuralFeature mutableProperty, final EStructuralFeature valueProperty) {
    this.mutablePropertyToValueProperty.put(mutableProperty, valueProperty);
  }

  public EStructuralFeature getValuePropertyOfMutableProperty(final EStructuralFeature mutableProperty) {
    return this.mutablePropertyToValueProperty.get(mutableProperty);
  }

  @Pure
  public TraceMMExplorer getTraceMMExplorer() {
    return this.traceMMExplorer;
  }

  @Pure
  public OperationalSemanticsView getMmext() {
    return this.mmext;
  }
}
