package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link LogConfig}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableLogConfig.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableLogConfig implements LogConfig {
  private final String logType;
  private final @Nullable Map<String, String> logOptions;

  private ImmutableLogConfig(
      String logType,
      @Nullable Map<String, String> logOptions) {
    this.logType = logType;
    this.logOptions = logOptions;
  }

  /**
   * @return The value of the {@code logType} attribute
   */
  @JsonProperty("Type")
  @Override
  public String logType() {
    return logType;
  }

  /**
   * @return The value of the {@code logOptions} attribute
   */
  @JsonProperty("Config")
  @Override
  public @Nullable Map<String, String> logOptions() {
    return logOptions;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link LogConfig#logType() logType} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for logType
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableLogConfig withLogType(String value) {
    String newValue = Objects.requireNonNull(value, "logType");
    if (this.logType.equals(newValue)) return this;
    return new ImmutableLogConfig(newValue, this.logOptions);
  }

  /**
   * Copy the current immutable object by replacing the {@link LogConfig#logOptions() logOptions} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to be added to the logOptions map
   * @return A modified copy of {@code this} object
   */
  public final ImmutableLogConfig withLogOptions(@Nullable Map<String, ? extends String> entries) {
    if (this.logOptions == entries) return this;
    @Nullable Map<String, String> newValue = entries == null ? null : createUnmodifiableMap(true, false, entries);
    return new ImmutableLogConfig(this.logType, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableLogConfig} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableLogConfig
        && equalTo(0, (ImmutableLogConfig) another);
  }

  private boolean equalTo(int synthetic, ImmutableLogConfig another) {
    return logType.equals(another.logType)
        && Objects.equals(logOptions, another.logOptions);
  }

  /**
   * Computes a hash code from attributes: {@code logType}, {@code logOptions}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + logType.hashCode();
    h += (h << 5) + Objects.hashCode(logOptions);
    return h;
  }

  /**
   * Prints the immutable value {@code LogConfig} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "LogConfig{"
        + "logType=" + logType
        + ", logOptions=" + logOptions
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link LogConfig} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable LogConfig instance
   */
  public static ImmutableLogConfig copyOf(LogConfig instance) {
    if (instance instanceof ImmutableLogConfig) {
      return (ImmutableLogConfig) instance;
    }
    return ImmutableLogConfig.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableLogConfig ImmutableLogConfig}.
   * <pre>
   * ImmutableLogConfig.builder()
   *    .logType(String) // required {@link LogConfig#logType() logType}
   *    .logOptions(Map&amp;lt;String, String&amp;gt; | null) // nullable {@link LogConfig#logOptions() logOptions}
   *    .build();
   * </pre>
   * @return A new ImmutableLogConfig builder
   */
  public static ImmutableLogConfig.Builder builder() {
    return new ImmutableLogConfig.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableLogConfig ImmutableLogConfig}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements LogConfig.Builder {
    private static final long INIT_BIT_LOG_TYPE = 0x1L;
    private long initBits = 0x1L;

    private String logType;
    private Map<String, String> logOptions = null;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code LogConfig} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(LogConfig instance) {
      Objects.requireNonNull(instance, "instance");
      this.logType(instance.logType());
      @Nullable Map<String, String> logOptionsValue = instance.logOptions();
      if (logOptionsValue != null) {
        putAllLogOptions(logOptionsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link LogConfig#logType() logType} attribute.
     * @param logType The value for logType 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Type")
    public final Builder logType(String logType) {
      this.logType = Objects.requireNonNull(logType, "logType");
      initBits &= ~INIT_BIT_LOG_TYPE;
      return this;
    }

    /**
     * Put one entry to the {@link LogConfig#logOptions() logOptions} map.
     * @param key The key in the logOptions map
     * @param value The associated value in the logOptions map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLogOption(String key, String value) {
      if (this.logOptions == null) {
        this.logOptions = new LinkedHashMap<String, String>();
      }
      this.logOptions.put(
          Objects.requireNonNull(key, "logOptions key"),
          Objects.requireNonNull(value, value == null ? "logOptions value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link LogConfig#logOptions() logOptions} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLogOption(Map.Entry<String, ? extends String> entry) {
      if (this.logOptions == null) {
        this.logOptions = new LinkedHashMap<String, String>();
      }
      String k = entry.getKey();
      String v = entry.getValue();
      this.logOptions.put(
          Objects.requireNonNull(k, "logOptions key"),
          Objects.requireNonNull(v, v == null ? "logOptions value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link LogConfig#logOptions() logOptions} map. Nulls are not permitted as keys or values, but parameter itself can be null
     * @param entries The entries that will be added to the logOptions map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Config")
    public final Builder logOptions(@Nullable Map<String, ? extends String> entries) {
      if (entries == null) {
        this.logOptions = null;
        return this;
      }
      this.logOptions = new LinkedHashMap<String, String>();
      return putAllLogOptions(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link LogConfig#logOptions() logOptions} map. Nulls are not permitted
     * @param entries The entries that will be added to the logOptions map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllLogOptions(Map<String, ? extends String> entries) {
      if (this.logOptions == null) {
        this.logOptions = new LinkedHashMap<String, String>();
      }
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.logOptions.put(
            Objects.requireNonNull(k, "logOptions key"),
            Objects.requireNonNull(v, v == null ? "logOptions value for key: " + k : null));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableLogConfig ImmutableLogConfig}.
     * @return An immutable instance of LogConfig
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableLogConfig build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableLogConfig(logType, logOptions == null ? null : createUnmodifiableMap(false, false, logOptions));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_LOG_TYPE) != 0) attributes.add("logType");
      return "Cannot build LogConfig, some of required attributes are not set " + attributes;
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
