package prefuse.data.column;

import java.util.BitSet;
import java.util.Iterator;
import java.util.Set;

import prefuse.data.DataTypeException;
import prefuse.data.Table;
import prefuse.data.event.ColumnListener;
import prefuse.data.event.EventConstants;
import prefuse.data.event.ExpressionListener;
import prefuse.data.expression.Expression;
import prefuse.data.expression.ExpressionAnalyzer;


/**
 * <p>Column instance that stores values provided by an Expression
 * instance. These expressions can reference other column values within the
 * same table. Values are evaluated when first requested and then cached to
 * increase performance. This column maintains listeners for all referenced
 * columns discovered in the expression and for the expression itself,
 * invalidating all cached entries when an update to either occurs.</p>
 * 
 * <p>
 * WARNING: Infinite recursion, eventually resulting in a StackOverflowError,
 * could occur if an expression refers to its own column, or if two
 * ExpressionColumns have expressions referring to each other. The 
 * responsibility for avoiding such situations is left with client programmers.
 * Note that it is fine for one ExpressionColumn to reference another;
 * however, the graph induced by such references must not contain any cycles.
 * </p>
 * 
 * @author <a href="http://jheer.org">jeffrey heer</a>
 * @see prefuse.data.expression
 */
public class ExpressionColumn extends AbstractColumn {
    
    private Expression m_expr;
    private Table m_table;
    private Set m_columns;
    
    private BitSet m_valid;
    private Column m_cache;
    private Listener m_lstnr;
    
    /**
     * Create a new ExpressionColumn.
     * @param table the table this column is a member of
     * @param expr the expression used to provide the column values
     */
    public ExpressionColumn(Table table, Expression expr) {
        super(expr.getType(table.getSchema()));
        this.m_table = table;
        this.m_expr = expr;
        this.m_lstnr = new Listener();
        
        init();
        
        int nrows = this.m_table.getRowCount();
        this.m_cache = ColumnFactory.getColumn(getColumnType(), nrows);
        this.m_valid = new BitSet(nrows);
        this.m_expr.addExpressionListener(this.m_lstnr);
    }
    
    protected void init() {
        // first remove listeners on any current columns
        if ( this.m_columns != null && this.m_columns.size() > 0 ) {
            Iterator iter = this.m_columns.iterator();
            while ( iter.hasNext() ) {
                String field = (String)iter.next();
                Column col = this.m_table.getColumn(field);
                col.removeColumnListener(this.m_lstnr);
            }
        }
        // now get the current set of columns
        this.m_columns = ExpressionAnalyzer.getReferencedColumns(this.m_expr);
        
        // sanity check table and expression
        Iterator iter = this.m_columns.iterator();
        while ( iter.hasNext() ) {
            String name = (String)iter.next();
            if ( this.m_table.getColumn(name) == null )
                throw new IllegalArgumentException("Table must contain all "
                        + "columns referenced by the expression."
                        + " Bad column name: "+name);
            
        }
        
        // passed check, so now listen to columns
        iter = this.m_columns.iterator();
        while ( iter.hasNext() ) {
            String field = (String)iter.next();
            Column col = this.m_table.getColumn(field);
            col.addColumnListener(this.m_lstnr);
        }
    }
    
    // ------------------------------------------------------------------------
    // Column Metadata
    
    /**
     * @see prefuse.data.column.Column#getRowCount()
     */
    public int getRowCount() {
        return this.m_cache.getRowCount();
    }

    /**
     * @see prefuse.data.column.Column#setMaximumRow(int)
     */
    public void setMaximumRow(int nrows) {
        this.m_cache.setMaximumRow(nrows);
    }

    // ------------------------------------------------------------------------
    // Cache Management
    
    /**
     * Check if this ExpressionColumn has a valid cached value at the given
     * row.
     * @param row the row to check for a valid cache entry
     * @return true if the cache row is valid, false otherwise
     */
    public boolean isCacheValid(int row) {
        return this.m_valid.get(row);
    }
    
    /**
     * Invalidate a range of the cache.
     * @param start the start of the range to invalidate
     * @param end the end of the range to invalidate, inclusive
     */
    public void invalidateCache(int start, int end ) {
        this.m_valid.clear(start, end+1);
    }
    
    // ------------------------------------------------------------------------
    // Data Access Methods    

    /**
     * Has no effect, as all values in this column are derived.
     * @param row the row to revert
     */
    public void revertToDefault(int row) {
        // do nothing, as we don't have default values.
    }
    
    /**
     * @see prefuse.data.column.AbstractColumn#canSet(java.lang.Class)
     */
    public boolean canSet(Class type) {
        return false;
    }
    
    /**
     * @see prefuse.data.column.Column#get(int)
     */
    public Object get(int row) {
        rangeCheck(row);
        if ( isCacheValid(row) ) {
            return this.m_cache.get(row);
        }
        Object val = this.m_expr.get(this.m_table.getTuple(row));
        Class type = val==null ? Object.class : val.getClass();
        if ( this.m_cache.canSet(type) ) {
            this.m_cache.set(val, row);
            this.m_valid.set(row);
        }
        return val;
    }

    /**
     * @see prefuse.data.column.Column#set(java.lang.Object, int)
     */
    public void set(Object val, int row) throws DataTypeException {
        throw new UnsupportedOperationException();
    }
    
    private void rangeCheck(int row) {
        if ( row < 0 || row >= getRowCount() )
            throw new IndexOutOfBoundsException();
    }
    
    // ------------------------------------------------------------------------
    
    /**
     * @see prefuse.data.column.Column#getBoolean(int)
     */
    public boolean getBoolean(int row) throws DataTypeException {
        if ( !canGetBoolean() )
            throw new DataTypeException(boolean.class);
        rangeCheck(row);
        
        if ( isCacheValid(row) ) {
            return this.m_cache.getBoolean(row);
        } else {
            boolean value = this.m_expr.getBoolean(this.m_table.getTuple(row));
            this.m_cache.setBoolean(value, row);
            this.m_valid.set(row);
            return value;
        }
    }

    private void computeNumber(int row) {
        if ( this.m_columnType == int.class || this.m_columnType == byte.class ) {
            this.m_cache.setInt(this.m_expr.getInt(this.m_table.getTuple(row)), row);
        } else if ( this.m_columnType == long.class ) {
            this.m_cache.setLong(this.m_expr.getLong(this.m_table.getTuple(row)), row);
        } else if ( this.m_columnType == float.class ) {
            this.m_cache.setFloat(this.m_expr.getFloat(this.m_table.getTuple(row)), row);
        } else {
            this.m_cache.setDouble(this.m_expr.getDouble(this.m_table.getTuple(row)), row);
        }
        this.m_valid.set(row);
    }
    
    /**
     * @see prefuse.data.column.Column#getInt(int)
     */
    public int getInt(int row) throws DataTypeException {
        if ( !canGetInt() )
            throw new DataTypeException(int.class);
        rangeCheck(row);
        
        if ( !isCacheValid(row) )
            computeNumber(row);
        return this.m_cache.getInt(row);
    }

    /**
     * @see prefuse.data.column.Column#getDouble(int)
     */
    public double getDouble(int row) throws DataTypeException {
        if ( !canGetDouble() )
            throw new DataTypeException(double.class);
        rangeCheck(row);
        
        if ( !isCacheValid(row) )
            computeNumber(row);
        return this.m_cache.getDouble(row);
    }

    /**
     * @see prefuse.data.column.Column#getFloat(int)
     */
    public float getFloat(int row) throws DataTypeException {
        if ( !canGetFloat() )
            throw new DataTypeException(float.class);
        rangeCheck(row);
        
        if ( !isCacheValid(row) )
            computeNumber(row);
        return this.m_cache.getFloat(row);
    }

    /**
     * @see prefuse.data.column.Column#getLong(int)
     */
    public long getLong(int row) throws DataTypeException {
        if ( !canGetLong() )
            throw new DataTypeException(long.class);
        rangeCheck(row);
        
        if ( !isCacheValid(row) )
            computeNumber(row);
        return this.m_cache.getLong(row);
    }
    
    // ------------------------------------------------------------------------
    // Listener Methods

    private class Listener implements ColumnListener, ExpressionListener {
    
        public void columnChanged(int start, int end) {
            // for a single index change with a valid cache value,
            // propagate a change event with the previous value
            if ( start == end && isCacheValid(start) ) {
                if ( !ExpressionColumn.this.m_table.isValidRow(start) ) return;
                
                // invalidate the cache index
                invalidateCache(start, end);
                // fire change event including previous value
                Class type = getColumnType();
                if ( int.class == type ) {
                    fireColumnEvent(start, ExpressionColumn.this.m_cache.getInt(start));
                } else if ( long.class == type ) {
                    fireColumnEvent(start, ExpressionColumn.this.m_cache.getLong(start));
                } else if ( float.class == type ) {
                    fireColumnEvent(start, ExpressionColumn.this.m_cache.getFloat(start));
                } else if ( double.class == type ) {
                    fireColumnEvent(start, ExpressionColumn.this.m_cache.getDouble(start));
                } else if ( boolean.class == type ) {
                    fireColumnEvent(start, ExpressionColumn.this.m_cache.getBoolean(start));
                } else {
                    fireColumnEvent(start, ExpressionColumn.this.m_cache.get(start));
                }
                
            // otherwise send a generic update
            } else {
                // invalidate cache indices
                invalidateCache(start, end);
                // fire change event
                fireColumnEvent(EventConstants.UPDATE, start, end);
            }
        }
        
        public void columnChanged(Column src, int idx, boolean prev) {
            columnChanged(idx, idx);
        }
    
        public void columnChanged(Column src, int idx, double prev) {
            columnChanged(idx, idx);
        }
    
        public void columnChanged(Column src, int idx, float prev) {
            columnChanged(idx, idx);
        }
    
        public void columnChanged(Column src, int type, int start, int end) {
            columnChanged(start, end);
        }
    
        public void columnChanged(Column src, int idx, int prev) {
            columnChanged(idx, idx);
        }
    
        public void columnChanged(Column src, int idx, long prev) {
            columnChanged(idx, idx);
        }
    
        public void columnChanged(Column src, int idx, Object prev) {
            columnChanged(idx, idx);
        }
    
        public void expressionChanged(Expression expr) {
            // mark everything as changed
            columnChanged(0, ExpressionColumn.this.m_cache.getRowCount()-1);
            // re-initialize our setup
            init();
        }
    }
    
} // end of class DerivedColumn
