/**
 * Copyright (c) 2016, 2022 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Shuai Li (CEA LIST) <shuai.li@cea.fr> - Initial API and implementation
 *   Van Cam Pham (CEA LIST) <vancam.pham@cea.fr> - Reverse implementation
 *   Ansgar Radermacher (CEA LIST) - Larger refactoring
 */
package org.eclipse.papyrus.designer.languages.cpp.reverse.reverse;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.eclipse.cdt.core.dom.ast.ExpansionOverlapsBoundaryException;
import org.eclipse.cdt.core.dom.ast.IASTDeclSpecifier;
import org.eclipse.cdt.core.dom.ast.IASTDeclaration;
import org.eclipse.cdt.core.dom.ast.IASTMacroExpansionLocation;
import org.eclipse.cdt.core.dom.ast.IASTNode;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTAliasDeclaration;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTemplateDeclaration;
import org.eclipse.cdt.core.model.ICElement;
import org.eclipse.cdt.core.model.IEnumeration;
import org.eclipse.cdt.core.model.IEnumerator;
import org.eclipse.cdt.core.model.IInclude;
import org.eclipse.cdt.core.model.IMethodDeclaration;
import org.eclipse.cdt.core.model.IParent;
import org.eclipse.cdt.core.model.ISourceReference;
import org.eclipse.cdt.core.model.IStructure;
import org.eclipse.cdt.core.model.IStructureDeclaration;
import org.eclipse.cdt.core.model.IStructureTemplate;
import org.eclipse.cdt.core.model.IStructureTemplateDeclaration;
import org.eclipse.cdt.core.model.ITranslationUnit;
import org.eclipse.cdt.core.model.ITypeDef;
import org.eclipse.cdt.core.model.IUsing;
import org.eclipse.cdt.core.parser.IToken;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Template;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Typedef;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Using;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.ClassifierTemplateParameter;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.ParameterableElement;
import org.eclipse.uml2.uml.PrimitiveType;
import org.eclipse.uml2.uml.RedefinableTemplateSignature;
import org.eclipse.uml2.uml.TemplateParameter;
import org.eclipse.uml2.uml.TemplateSignature;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLFactory;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Pass1: build all classifiers (and packages) on UML level, but do not fill them with contents (attributes and methods, etc.)
 * The objective of having two passes is that all type references (set in pass2) and forward declared elements are already
 * available in the UML model
 */
@SuppressWarnings("all")
public class GetOrCreateP1 {
  /**
   * Create or get the classifiers within a parent (called initially with a translation unit, then recursively)
   * @param model the model to get/create classifiers in
   * @param parent the CDT model parent
   * @return a list of created UML types
   */
  public static void getOrCreateClassifiers(final IParent parent) throws Exception {
    ICElement[] children = ReverseUtils.safeGetChilds(parent);
    for (int i = 0; (i < children.length); i++) {
      {
        ICElement child = children[i];
        boolean _isActive = ReverseUtils.isActive(child);
        if (_isActive) {
          boolean _matched = false;
          if ((child instanceof IStructure)) {
            _matched=true;
            GetOrCreateP1.getOrCreateClassifier(child);
            GetOrCreateP1.getOrCreateClassifiers(((IParent) child));
          }
          if (!_matched) {
            if ((child instanceof IEnumeration)) {
              _matched=true;
              GetOrCreateP1.getOrCreateClassifier(child);
            }
          }
          if (!_matched) {
            if ((child instanceof IParent)) {
              _matched=true;
              GetOrCreateP1.getOrCreateClassifiers(((IParent) child));
            }
          }
          if (!_matched) {
            if ((child instanceof ITypeDef)) {
              _matched=true;
              GetOrCreateP1.getOrCreateClassifier(child);
            }
          }
          if (!_matched) {
            if ((child instanceof IUsing)) {
              _matched=true;
            }
          }
          if (!_matched) {
            if ((child instanceof IInclude)) {
              _matched=true;
              final ITranslationUnit headerUnit = ReverseUtils.getTranslationUnitFromInclude(((IInclude) child), ReverseData.current.project);
              if (((headerUnit != null) && (ReverseUtils.getSourceFolder(headerUnit) != null))) {
                ReverseCpp2Uml.currentCpp2Uml().reverseHeader(headerUnit);
              }
            }
          }
        }
      }
    }
    if ((parent instanceof ISourceReference)) {
      GetOrCreateP1.getOrCreateAlias(((ISourceReference)parent));
    }
  }

  /**
   * Create classifier for a forward declaration, either a (template) class, struct/union
   *  @param container the owner to create a classifier in, must be a class or package
   * @return the associated uml type
   */
  public static Classifier getOrCreateClassifier(final ICElement ideclaration) {
    try {
      final Map<ICElement, EObject> map = ReverseData.current.map;
      if (((((((((ideclaration.getElementType() != ICElement.C_CLASS) && (ideclaration.getElementType() != ICElement.C_TEMPLATE_CLASS)) && 
        (ideclaration.getElementType() != ICElement.C_ENUMERATION)) && 
        (ideclaration.getElementType() != ICElement.C_STRUCT)) && 
        (ideclaration.getElementType() != ICElement.C_UNION)) && (ideclaration.getElementType() != ICElement.C_TEMPLATE_STRUCT)) && 
        (!(ideclaration instanceof ITypeDef))) && (!(ideclaration instanceof IUsing)))) {
        return null;
      }
      Classifier existing = null;
      Namespace container = ReverseUtils.getContainer(ideclaration);
      if ((container instanceof DataType)) {
        container = ((DataType)container).getNearestPackage();
      }
      EObject _get = map.get(ideclaration);
      if ((_get instanceof Classifier)) {
        EObject _get_1 = map.get(ideclaration);
        existing = ((Classifier) _get_1);
      }
      if ((existing != null)) {
        return existing;
      }
      boolean structIsClassLike = GetOrCreateP1.isClassLike(ideclaration);
      final String name = ReverseUtils.stripTemplate(ideclaration.getElementName());
      if ((ideclaration instanceof ITypeDef)) {
        final PrimitiveType typedef = GetOrCreateP1.createTypedef(container, ((ITypeDef)ideclaration));
        map.put(ideclaration, typedef);
        return typedef;
      } else {
        if ((((ideclaration.getElementType() == ICElement.C_CLASS) || 
          (ideclaration.getElementType() == ICElement.C_TEMPLATE_CLASS)) || structIsClassLike)) {
          final IStructureDeclaration iStructure = ((IStructure) ideclaration);
          org.eclipse.uml2.uml.Class classifier = null;
          if ((existing != null)) {
            classifier = ((org.eclipse.uml2.uml.Class) existing);
          } else {
            if ((container instanceof org.eclipse.uml2.uml.Package)) {
              classifier = ((org.eclipse.uml2.uml.Package) container).createOwnedClass(name, false);
            } else {
              Classifier _createNestedClassifier = ((org.eclipse.uml2.uml.Class) container).createNestedClassifier(name, UMLPackage.Literals.CLASS);
              classifier = ((org.eclipse.uml2.uml.Class) _createNestedClassifier);
            }
            ReverseUtils.setXmlID(classifier);
          }
          map.put(ideclaration, classifier);
          int _elementType = iStructure.getElementType();
          boolean _equals = (_elementType == ICElement.C_TEMPLATE_CLASS);
          if (_equals) {
            IStructureTemplate istructureTemplate = ((IStructureTemplate) iStructure);
            final Template template = StereotypeUtil.<Template>applyApp(classifier, Template.class);
            template.setDeclaration(istructureTemplate.getTemplateSignature());
          }
          return classifier;
        } else {
          if (((!structIsClassLike) && 
            (((ideclaration.getElementType() == ICElement.C_STRUCT) || (ideclaration.getElementType() == ICElement.C_UNION)) || 
              (ideclaration.getElementType() == ICElement.C_TEMPLATE_STRUCT)))) {
            DataType dataType = null;
            if ((existing != null)) {
              dataType = ((DataType) existing);
            } else {
              if ((container instanceof org.eclipse.uml2.uml.Package)) {
                Type _createOwnedType = ((org.eclipse.uml2.uml.Package) container).createOwnedType(name, UMLPackage.Literals.DATA_TYPE);
                dataType = ((DataType) _createOwnedType);
              } else {
                Classifier _createNestedClassifier_1 = ((org.eclipse.uml2.uml.Class) container).createNestedClassifier(name, 
                  UMLPackage.Literals.DATA_TYPE);
                dataType = ((DataType) _createNestedClassifier_1);
              }
              ReverseUtils.setXmlID(dataType);
            }
            map.put(ideclaration, dataType);
            return dataType;
          } else {
            int _elementType_1 = ideclaration.getElementType();
            boolean _equals_1 = (_elementType_1 == ICElement.C_ENUMERATION);
            if (_equals_1) {
              Enumeration enumeration = null;
              final IEnumeration iEnum = ((IEnumeration) ideclaration);
              if ((existing != null)) {
                enumeration = ((Enumeration) existing);
              } else {
                if ((container instanceof org.eclipse.uml2.uml.Package)) {
                  enumeration = ((org.eclipse.uml2.uml.Package) container).createOwnedEnumeration(name);
                } else {
                  Classifier _createNestedClassifier_2 = ((org.eclipse.uml2.uml.Class) container).createNestedClassifier(name, 
                    UMLPackage.Literals.ENUMERATION);
                  enumeration = ((Enumeration) _createNestedClassifier_2);
                }
                ReverseUtils.setXmlID(enumeration);
              }
              map.put(ideclaration, enumeration);
              GetOrCreateP1.getOrCreateClassifiers(iEnum);
              ICElement[] _children = iEnum.getChildren();
              for (final ICElement child : _children) {
                if ((child instanceof IEnumerator)) {
                  final EnumerationLiteral literal = enumeration.createOwnedLiteral(((IEnumerator)child).getElementName());
                  ReverseUtils.setXmlID(literal);
                  if (((((IEnumerator)child).getConstantExpression() != null) && (!((IEnumerator)child).getConstantExpression().equals("")))) {
                    ValueSpecification _createSpecification = literal.createSpecification("defaultVal-ue", literal.getEnumeration(), 
                      UMLPackage.Literals.OPAQUE_EXPRESSION);
                    final OpaqueExpression valueSpecification = ((OpaqueExpression) _createSpecification);
                    valueSpecification.getLanguages().add(ReverseCpp2Uml.Cpp_LangID);
                    valueSpecification.getBodies().add(((IEnumerator)child).getConstantExpression());
                  }
                }
              }
              return enumeration;
            }
          }
        }
      }
      return null;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * A C++ struct can be "class-like", e.g. have nested definitions and operations. However, a
   * datatype in UML is more constrained (no nested classifier). Therefore, we need to know
   * whether to map a C++ struct to a UML class instead of a datatype.
   */
  public static boolean isClassLike(final ICElement ideclaration) {
    try {
      if ((((ideclaration.getElementType() == ICElement.C_STRUCT) || (ideclaration.getElementType() == ICElement.C_UNION)) || 
        (ideclaration.getElementType() == ICElement.C_TEMPLATE_STRUCT))) {
        final IStructure iStructure = ((IStructure) ideclaration);
        ICElement[] _children = iStructure.getChildren();
        for (final ICElement child : _children) {
          if ((((child instanceof IMethodDeclaration) || (child instanceof IStructure)) || 
            (child instanceof IStructureTemplateDeclaration))) {
            return true;
          }
        }
        IASTNode node = ASTUtils.findEnclosingNode(((ISourceReference) ideclaration));
        if ((ideclaration instanceof IStructureTemplate)) {
          node = ASTUtils.getDeclSpecifier(node);
        }
        IASTNode[] _children_1 = node.getChildren();
        for (final IASTNode child_1 : _children_1) {
          if ((child_1 instanceof ICPPASTAliasDeclaration)) {
            return true;
          }
        }
      }
      return false;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Create a type definition
   * @param container the owner to create the typedef in, should be either
   *  a class, package or datatype. In case of datatype, nearest package is used
   */
  public static PrimitiveType createTypedef(final Namespace container, final ITypeDef typedef) {
    try {
      PrimitiveType _createPrimitiveType = UMLFactory.eINSTANCE.createPrimitiveType();
      final Procedure1<PrimitiveType> _function = new Procedure1<PrimitiveType>() {
        @Override
        public void apply(final PrimitiveType it) {
          it.setName(typedef.getElementName());
        }
      };
      PrimitiveType primitiveType = ObjectExtensions.<PrimitiveType>operator_doubleArrow(_createPrimitiveType, _function);
      if ((container instanceof org.eclipse.uml2.uml.Package)) {
        EList<Type> _ownedTypes = ((org.eclipse.uml2.uml.Package)container).getOwnedTypes();
        _ownedTypes.add(primitiveType);
      } else {
        if ((container instanceof org.eclipse.uml2.uml.Class)) {
          EList<Classifier> _nestedClassifiers = ((org.eclipse.uml2.uml.Class)container).getNestedClassifiers();
          _nestedClassifiers.add(primitiveType);
        } else {
          EList<Type> _ownedTypes_1 = container.getNearestPackage().getOwnedTypes();
          _ownedTypes_1.add(primitiveType);
        }
      }
      final Typedef typedefSt = StereotypeUtil.<Typedef>applyApp(primitiveType, Typedef.class);
      if (((Objects.equal(typedef.getTypeName(), "struct") || Objects.equal(typedef.getTypeName(), "class")) || Objects.equal(typedef.getTypeName(), "enum"))) {
        ICElement _parent = typedef.getParent();
        int pos = ((List<ICElement>)Conversions.doWrapArray(((IParent) _parent).getChildren())).indexOf(typedef);
        if ((pos > 0)) {
          ICElement _parent_1 = typedef.getParent();
          ICElement childStruct = ((IParent) _parent_1).getChildren()[(pos - 1)];
          if ((((childStruct instanceof IStructure) || (childStruct instanceof IEnumeration)) && 
            Objects.equal(childStruct.getElementName(), ""))) {
            String source = ((ISourceReference) childStruct).getSource();
            Typedef _stereotypeApplication = UMLUtil.<Typedef>getStereotypeApplication(primitiveType, Typedef.class);
            String _replace = source.replace("typedef", 
              "");
            String _plus = (_replace + " typeName");
            _stereotypeApplication.setDefinition(_plus);
          }
        }
      } else {
        IASTNode typedefNode = ASTUtils.findEnclosingNode(typedef);
        String rawSignature = typedefNode.getRawSignature().replaceAll("\\n", "").replaceAll("\\r", "").replaceAll(";", "").replaceAll("\\s+", " ").trim();
        Pattern pattern = Pattern.compile("(\\()(\\s*)(\\*)(.*)(\\))(\\s*)(\\()(.*)(\\))");
        Matcher matcher = pattern.matcher(rawSignature);
        boolean _find = matcher.find();
        if (_find) {
          final String typeName = rawSignature.replaceFirst(Pattern.quote(typedef.getElementName()), "typeName");
          typedefSt.setDefinition(typeName);
        } else {
          typedefSt.setDefinition(typedef.getTypeName());
        }
      }
      ReverseData.current.map.put(typedef, primitiveType);
      return primitiveType;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Search for alias declarations in a structure and create them, if they
   * do not exist yet. This function works on the AST level, as alias declaration (via Using)
   * do not exist on the ICElement level
   */
  public static void getOrCreateAlias(final ISourceReference sourceRef) {
    final Namespace container = ReverseUtils.getContainer(((ICElement) sourceRef), true);
    IASTNode node = ASTUtils.findEnclosingNode(sourceRef);
    if ((node == null)) {
      return;
    }
    if ((sourceRef instanceof IStructureTemplate)) {
      final IASTDeclSpecifier declSpecifier = ASTUtils.getDeclSpecifier(node);
      if ((declSpecifier != null)) {
        node = declSpecifier;
      } else {
        if ((!(node instanceof IASTMacroExpansionLocation))) {
          System.err.println("not in macro?? should not happen");
        }
      }
    }
    IASTNode[] _children = node.getChildren();
    for (final IASTNode child : _children) {
      {
        if ((child instanceof ICPPASTAliasDeclaration)) {
          GetOrCreateP1.getOrCreateAlias(((ICPPASTAliasDeclaration)child), container);
        }
        if ((child instanceof ICPPASTTemplateDeclaration)) {
          final IASTDeclaration declaration = ((ICPPASTTemplateDeclaration)child).getDeclaration();
          if ((declaration instanceof ICPPASTAliasDeclaration)) {
            GetOrCreateP1.getOrCreateAlias(((ICPPASTAliasDeclaration)declaration), container);
          }
        }
      }
    }
  }

  /**
   * Create or get an alias declaration from an ASTAliasDeclaration
   */
  public static void getOrCreateAlias(final ICPPASTAliasDeclaration aliasDecl, final Namespace container) {
    String definition = "";
    try {
      IToken token = aliasDecl.getMappingTypeId().getSyntax();
      while ((token != null)) {
        {
          while ((token.getOffset() > definition.length())) {
            String _definition = definition;
            definition = (_definition + " ");
          }
          String _definition = definition;
          String _string = token.toString();
          definition = (_definition + _string);
          token = token.getNext();
        }
      }
    } catch (final Throwable _t) {
      if (_t instanceof ExpansionOverlapsBoundaryException) {
        definition = aliasDecl.getMappingTypeId().getDeclSpecifier().toString();
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    final String aliasName = aliasDecl.getAlias().toString();
    NamedElement aliasPT = container.getMember(aliasName);
    if ((aliasPT == null)) {
      if ((container instanceof org.eclipse.uml2.uml.Package)) {
        aliasPT = ((org.eclipse.uml2.uml.Package)container).createOwnedPrimitiveType(aliasName);
      } else {
        if ((container instanceof org.eclipse.uml2.uml.Class)) {
          aliasPT = ((org.eclipse.uml2.uml.Class)container).createNestedClassifier(aliasName, UMLPackage.eINSTANCE.getPrimitiveType());
        } else {
          System.err.println("should not happen");
        }
      }
      ReverseUtils.setXmlID(aliasPT);
      final Using usingSt = StereotypeUtil.<Using>applyApp(aliasPT, Using.class);
      usingSt.setDefinition(definition);
    }
  }

  /**
   * Create a template parameter within a template signature. If the
   * signature does not exist, it will be created.
   */
  public static Classifier getOrCreateTemplateParameter(final NamedElement element, final String parameterTypeName, final String keyWord) {
    Classifier ret = null;
    TemplateSignature templateSignature = null;
    if ((element instanceof Classifier)) {
      Classifier classifier = ((Classifier) element);
      if (((classifier.getOwnedTemplateSignature() == null) || 
        (!(classifier.getOwnedTemplateSignature() instanceof RedefinableTemplateSignature)))) {
        TemplateSignature _createOwnedTemplateSignature = classifier.createOwnedTemplateSignature(
          UMLPackage.Literals.REDEFINABLE_TEMPLATE_SIGNATURE);
        templateSignature = ((RedefinableTemplateSignature) _createOwnedTemplateSignature);
        ((RedefinableTemplateSignature) templateSignature).setName(ReverseUtils.TEMPLATE_PARAMETER_SIGNATURE_NAME);
      }
      TemplateSignature _ownedTemplateSignature = classifier.getOwnedTemplateSignature();
      templateSignature = ((RedefinableTemplateSignature) _ownedTemplateSignature);
    } else {
      if ((element instanceof Operation)) {
        Operation operation = ((Operation) element);
        if (((operation.getOwnedTemplateSignature() == null) || 
          (!(operation.getOwnedTemplateSignature() instanceof TemplateSignature)))) {
          TemplateSignature _createOwnedTemplateSignature_1 = operation.createOwnedTemplateSignature(
            UMLPackage.Literals.TEMPLATE_SIGNATURE);
          templateSignature = ((TemplateSignature) _createOwnedTemplateSignature_1);
        }
        TemplateSignature _ownedTemplateSignature_1 = operation.getOwnedTemplateSignature();
        templateSignature = ((TemplateSignature) _ownedTemplateSignature_1);
      } else {
        return null;
      }
    }
    Iterable<ClassifierTemplateParameter> classifierTemplates = Iterables.<ClassifierTemplateParameter>filter(templateSignature.getOwnedParameters(), ClassifierTemplateParameter.class);
    final Function1<ClassifierTemplateParameter, Boolean> _function = new Function1<ClassifierTemplateParameter, Boolean>() {
      @Override
      public Boolean apply(final ClassifierTemplateParameter it) {
        ParameterableElement _ownedParameteredElement = it.getOwnedParameteredElement();
        return Boolean.valueOf((_ownedParameteredElement instanceof Classifier));
      }
    };
    Iterable<ClassifierTemplateParameter> classifierTemplatesContainClassifier = IterableExtensions.<ClassifierTemplateParameter>filter(classifierTemplates, _function);
    final Function1<ClassifierTemplateParameter, ParameterableElement> _function_1 = new Function1<ClassifierTemplateParameter, ParameterableElement>() {
      @Override
      public ParameterableElement apply(final ClassifierTemplateParameter it) {
        return it.getOwnedParameteredElement();
      }
    };
    Iterable<ParameterableElement> containedClassifiers = IterableExtensions.<ClassifierTemplateParameter, ParameterableElement>map(classifierTemplatesContainClassifier, _function_1);
    final Function1<Classifier, Boolean> _function_2 = new Function1<Classifier, Boolean>() {
      @Override
      public Boolean apply(final Classifier it) {
        return Boolean.valueOf(it.getName().equals(parameterTypeName));
      }
    };
    ret = IterableExtensions.<Classifier>head(IterableExtensions.<Classifier>filter(Iterables.<Classifier>filter(containedClassifiers, Classifier.class), _function_2));
    if ((ret == null)) {
      TemplateParameter _createOwnedParameter = templateSignature.createOwnedParameter(
        UMLPackage.Literals.CLASSIFIER_TEMPLATE_PARAMETER);
      ClassifierTemplateParameter classifierTemplate = ((ClassifierTemplateParameter) _createOwnedParameter);
      ParameterableElement _createOwnedParameteredElement = classifierTemplate.createOwnedParameteredElement(UMLPackage.Literals.CLASS);
      ret = ((Classifier) _createOwnedParameteredElement);
      ret.setName(parameterTypeName);
      classifierTemplate.addKeyword(keyWord);
    }
    return ret;
  }
}
