/**
 * Copyright (c) 2022, 2023 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Mohamed Harkat - Initial API and implementation
 *   Ansgar Radermacher - Integration and bug fixes
 */
package org.eclipse.papyrus.designer.languages.python.codegen.gen;

import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.languages.python.codegen.transformation.PythonCodeGenUtils;
import org.eclipse.papyrus.designer.languages.python.profile.python.Main;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@SuppressWarnings("all")
public class PyClasses {
  /**
   * The method that writes the whole class, it kind of acts as a main method here
   * 
   * @param clazz the class to generate code for
   * @param useRelativeImports if true, write imports in a relative way
   */
  public static CharSequence genClassWithNS(final org.eclipse.uml2.uml.Class clazz, final boolean useRelativeImports) {
    StringConcatenation _builder = new StringConcatenation();
    final CharSequence imports = PythonCodeGenUtils.writeImports(clazz, useRelativeImports);
    _builder.newLineIfNotEmpty();
    _builder.append(imports);
    _builder.newLineIfNotEmpty();
    {
      boolean _isBlank = imports.toString().isBlank();
      boolean _not = (!_isBlank);
      if (_not) {
        _builder.newLine();
      }
    }
    CharSequence _genClass = PyClasses.genClass(clazz);
    _builder.append(_genClass);
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  /**
   * Generate the header and the body of the class
   */
  public static CharSequence genClass(final org.eclipse.uml2.uml.Class clazz) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("class ");
    String _name = clazz.getName();
    _builder.append(_name);
    {
      int _size = clazz.getGenerals().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        _builder.append("(");
        {
          boolean _isAbstract = clazz.isAbstract();
          if (_isAbstract) {
            _builder.append("ABC");
          }
        }
        {
          EList<Classifier> _generals = clazz.getGenerals();
          boolean _hasElements = false;
          for(final Classifier superClass : _generals) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(", ", "");
            }
            String _name_1 = superClass.getName();
            _builder.append(_name_1);
          }
        }
        _builder.append(")");
      } else {
        {
          boolean _isAbstract_1 = clazz.isAbstract();
          if (_isAbstract_1) {
            _builder.append("(ABC)");
          }
        }
      }
    }
    _builder.append(":");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      boolean _isEmpty = PyClasses.writeStaticAttributes(clazz).toString().isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        _builder.append("\t");
        CharSequence _writeStaticAttributes = PyClasses.writeStaticAttributes(clazz);
        _builder.append(_writeStaticAttributes, "\t");
        _builder.newLineIfNotEmpty();
        _builder.newLine();
      }
    }
    {
      if (((!PyClasses.hasConstructor(clazz)) && PyClasses.checkIfNotStaticAttributes(clazz))) {
        _builder.append("\t");
        _builder.append("def __init__(self):");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        CharSequence _writeNotStaticAttributes = PyClasses.writeNotStaticAttributes(clazz);
        _builder.append(_writeNotStaticAttributes, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.newLine();
      }
    }
    {
      EList<Operation> _ownedOperations = clazz.getOwnedOperations();
      for(final Operation op : _ownedOperations) {
        _builder.append("\t");
        {
          if ((clazz.isAbstract() && op.isAbstract())) {
            _builder.append("@abstractmethod");
          }
        }
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        String _trim = PyMethods.method(op).toString().trim();
        _builder.append(_trim, "\t");
        _builder.newLineIfNotEmpty();
        {
          boolean _isConstructor = PyMethods.isConstructor(op);
          if (_isConstructor) {
            _builder.append("\t");
            _builder.append("\t");
            CharSequence _writeNotStaticAttributes_1 = PyClasses.writeNotStaticAttributes(clazz);
            _builder.append(_writeNotStaticAttributes_1, "\t\t");
            _builder.newLineIfNotEmpty();
          }
        }
        {
          if ((op.isAbstract() || (PyMethods.mBody(op) == null))) {
            _builder.append("\t");
            _builder.append("\t");
            _builder.append("pass");
            _builder.newLine();
            _builder.newLine();
          } else {
            _builder.append("\t");
            _builder.append("\t");
            String _mBody = PyMethods.mBody(op);
            _builder.append(_mBody, "\t\t");
            _builder.newLineIfNotEmpty();
            _builder.newLine();
          }
        }
      }
    }
    {
      if ((clazz.getOwnedOperations().isEmpty() && clazz.getOwnedAttributes().isEmpty())) {
        _builder.append("\t");
        _builder.append("def __init__(self):");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("pass");
        _builder.newLine();
        _builder.newLine();
      }
    }
    {
      boolean _isApplied = StereotypeUtil.isApplied(clazz, Main.class);
      if (_isApplied) {
        final Main main = UMLUtil.<Main>getStereotypeApplication(clazz, Main.class);
        _builder.newLineIfNotEmpty();
        _builder.newLine();
        _builder.append("def main():");
        _builder.newLine();
        {
          if (((main.getBody() == null) || main.getBody().isEmpty())) {
            _builder.append("\t");
            _builder.append("pass");
            _builder.newLine();
          } else {
            _builder.append("\t");
            String _body = main.getBody();
            _builder.append(_body, "\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.newLine();
        _builder.newLine();
        _builder.append("if __name__ == \'__main__\':");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("main()");
        _builder.newLine();
      }
    }
    return _builder;
  }

  /**
   * A method to write static attributes of the class. the attributes should be declared outside the constructor.
   */
  public static CharSequence writeStaticAttributes(final org.eclipse.uml2.uml.Class clazz) {
    StringConcatenation _builder = new StringConcatenation();
    {
      final Function1<Property, Boolean> _function = new Function1<Property, Boolean>() {
        @Override
        public Boolean apply(final Property it) {
          return Boolean.valueOf(((it.isStatic() && (!it.isComposite())) && (!it.isReadOnly())));
        }
      };
      Iterable<Property> _filter = IterableExtensions.<Property>filter(clazz.getOwnedAttributes(), _function);
      for(final Property attribute : _filter) {
        String _writeAttributeVisibility = PyAttributes.writeAttributeVisibility(attribute);
        _builder.append(_writeAttributeVisibility);
        String _name = attribute.getName();
        _builder.append(_name);
        String _assignType = PyAttributes.assignType(attribute);
        _builder.append(_assignType);
        String _assignAttributeValue = PyAttributes.assignAttributeValue(attribute);
        _builder.append(_assignAttributeValue);
        _builder.newLineIfNotEmpty();
      }
    }
    {
      final Function1<Property, Boolean> _function_1 = new Function1<Property, Boolean>() {
        @Override
        public Boolean apply(final Property it) {
          return Boolean.valueOf(((it.isStatic() && it.isComposite()) && (!it.isReadOnly())));
        }
      };
      Iterable<Property> _filter_1 = IterableExtensions.<Property>filter(clazz.getOwnedAttributes(), _function_1);
      for(final Property attribute_1 : _filter_1) {
        String _writeAttributeVisibility_1 = PyAttributes.writeAttributeVisibility(attribute_1);
        _builder.append(_writeAttributeVisibility_1);
        String _name_1 = attribute_1.getName();
        _builder.append(_name_1);
        String _assignCompositeAttributes = PyAttributes.assignCompositeAttributes(attribute_1);
        _builder.append(_assignCompositeAttributes);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  /**
   * A method to write non static attributes of the class. the attributes should be declared inside a constructor.
   */
  public static CharSequence writeNotStaticAttributes(final org.eclipse.uml2.uml.Class clazz) {
    StringConcatenation _builder = new StringConcatenation();
    {
      final Function1<Property, Boolean> _function = new Function1<Property, Boolean>() {
        @Override
        public Boolean apply(final Property it) {
          return Boolean.valueOf((((!it.isStatic()) && (!it.isComposite())) && (!it.isReadOnly())));
        }
      };
      Iterable<Property> _filter = IterableExtensions.<Property>filter(clazz.getOwnedAttributes(), _function);
      for(final Property attribute : _filter) {
        _builder.append("self.");
        String _writeAttributeVisibility = PyAttributes.writeAttributeVisibility(attribute);
        _builder.append(_writeAttributeVisibility);
        String _name = attribute.getName();
        _builder.append(_name);
        String _assignType = PyAttributes.assignType(attribute);
        _builder.append(_assignType);
        String _assignAttributeValue = PyAttributes.assignAttributeValue(attribute);
        _builder.append(_assignAttributeValue);
        _builder.newLineIfNotEmpty();
      }
    }
    {
      final Function1<Property, Boolean> _function_1 = new Function1<Property, Boolean>() {
        @Override
        public Boolean apply(final Property it) {
          return Boolean.valueOf((((!it.isStatic()) && it.isComposite()) && (!it.isReadOnly())));
        }
      };
      Iterable<Property> _filter_1 = IterableExtensions.<Property>filter(clazz.getOwnedAttributes(), _function_1);
      for(final Property attribute_1 : _filter_1) {
        _builder.append("self.");
        String _writeAttributeVisibility_1 = PyAttributes.writeAttributeVisibility(attribute_1);
        _builder.append(_writeAttributeVisibility_1);
        String _name_1 = attribute_1.getName();
        _builder.append(_name_1);
        String _assignCompositeAttributes = PyAttributes.assignCompositeAttributes(attribute_1);
        _builder.append(_assignCompositeAttributes);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  /**
   * A method to write read only attributes.
   */
  public static CharSequence writeReadOnlyAttributes(final org.eclipse.uml2.uml.Class clazz) {
    StringConcatenation _builder = new StringConcatenation();
    {
      final Function1<Property, Boolean> _function = new Function1<Property, Boolean>() {
        @Override
        public Boolean apply(final Property it) {
          return Boolean.valueOf(it.isReadOnly());
        }
      };
      Iterable<Property> _filter = IterableExtensions.<Property>filter(clazz.getOwnedAttributes(), _function);
      for(final Property attribute : _filter) {
        _builder.append("def ");
        String _name = attribute.getName();
        _builder.append(_name);
        _builder.append("(");
        {
          boolean _isStatic = attribute.isStatic();
          boolean _not = (!_isStatic);
          if (_not) {
            _builder.append("self");
          }
        }
        _builder.append("):");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        {
          boolean _isStatic_1 = attribute.isStatic();
          boolean _not_1 = (!_isStatic_1);
          if (_not_1) {
            _builder.append("self.");
          }
        }
        String _name_1 = attribute.getName();
        _builder.append(_name_1, "\t");
        String _assignType = PyAttributes.assignType(attribute);
        _builder.append(_assignType, "\t");
        _builder.append(" = ");
        String _assignAttributeValue = PyAttributes.assignAttributeValue(attribute);
        _builder.append(_assignAttributeValue, "\t");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  /**
   * a checker that returns true if the class has no static attributes
   * else it returns false
   */
  public static boolean checkIfNotStaticAttributes(final org.eclipse.uml2.uml.Class clazz) {
    EList<Property> _ownedAttributes = clazz.getOwnedAttributes();
    for (final Property attribute : _ownedAttributes) {
      boolean _isStatic = attribute.isStatic();
      boolean _not = (!_isStatic);
      if (_not) {
        return true;
      }
    }
    return false;
  }

  /**
   * Check if the class has a constructor. The constructor would be necessary to declare non static variables
   * returns false if there is a constructor, else true.
   */
  public static boolean hasConstructor(final org.eclipse.uml2.uml.Class clazz) {
    EList<Operation> _ownedOperations = clazz.getOwnedOperations();
    for (final Operation op : _ownedOperations) {
      boolean _isConstructor = PyMethods.isConstructor(op);
      if (_isConstructor) {
        return true;
      }
    }
    return false;
  }

  /**
   * Check if the class has a static main operation .
   */
  public static boolean hasMainOp(final org.eclipse.uml2.uml.Class clazz) {
    EList<Operation> _ownedOperations = clazz.getOwnedOperations();
    for (final Operation op : _ownedOperations) {
      if ((op.getName().equals("main") && op.isStatic())) {
        return true;
      }
    }
    return false;
  }
}
