/**
 * Copyright (c) 2014, 2015 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   Benoit Maggi       - #474408 : order by identifier the generated file
 */
package org.eclipse.papyrus.uml.profile.types.generator;

import com.google.common.base.Objects;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.eclipse.papyrus.infra.types.AbstractAdviceBindingConfiguration;
import org.eclipse.papyrus.infra.types.ElementTypeConfiguration;
import org.eclipse.papyrus.infra.types.ElementTypeSetConfiguration;
import org.eclipse.papyrus.infra.types.ElementTypesConfigurationsFactory;
import org.eclipse.papyrus.infra.types.SpecializationTypeConfiguration;
import org.eclipse.papyrus.uml.profile.types.generator.ApplyStereotypeAdviceRule;
import org.eclipse.papyrus.uml.profile.types.generator.ElementTypeRule;
import org.eclipse.papyrus.uml.profile.types.generator.Identifiers;
import org.eclipse.papyrus.uml.profile.types.generator.ImpliedExtension;
import org.eclipse.papyrus.uml.profile.types.generator.UML;
import org.eclipse.papyrus.uml.profile.types.generator.UMLElementTypes;
import org.eclipse.papyrus.uml.types.core.advices.applystereotype.ApplyStereotypeAdviceConfiguration;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Transformation rule for generating an {@link ElementTypeSetConfiguration} from a UML {@link Profile}.
 */
@Singleton
@SuppressWarnings("all")
public class ConfigurationSetRule {
  @Extension
  private static ElementTypesConfigurationsFactory elementtypesconfigurationsFactory = ElementTypesConfigurationsFactory.eINSTANCE;
  
  @Inject
  @Extension
  private Identifiers _identifiers;
  
  @Inject
  @Extension
  private UML _uML;
  
  @Inject
  @Extension
  private UMLElementTypes _uMLElementTypes;
  
  @Inject
  @Extension
  private ElementTypeRule _elementTypeRule;
  
  @Inject
  @Extension
  private ApplyStereotypeAdviceRule _applyStereotypeAdviceRule;
  
  private static List<ElementTypeConfiguration> elementTypeConfigurationList;
  
  private List<AbstractAdviceBindingConfiguration> adviceBindingConfigurationList;
  
  public static ElementTypeConfiguration addElementType(final ElementTypeConfiguration elementtype) {
    final Function1<ElementTypeConfiguration, Boolean> _function = new Function1<ElementTypeConfiguration, Boolean>() {
      @Override
      public Boolean apply(final ElementTypeConfiguration el) {
        return Boolean.valueOf(el.getIdentifier().equals(elementtype.getIdentifier()));
      }
    };
    ElementTypeConfiguration found = IterableExtensions.<ElementTypeConfiguration>findFirst(ConfigurationSetRule.elementTypeConfigurationList, _function);
    boolean _equals = Objects.equal(found, null);
    if (_equals) {
      ConfigurationSetRule.elementTypeConfigurationList.add(elementtype);
      return elementtype;
    } else {
      return found;
    }
  }
  
  public ElementTypeSetConfiguration toConfigurationSet(final Profile umlProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlProfile);
    final ElementTypeSetConfiguration _result;
    synchronized (_createCache_toConfigurationSet) {
      if (_createCache_toConfigurationSet.containsKey(_cacheKey)) {
        return _createCache_toConfigurationSet.get(_cacheKey);
      }
      ElementTypeSetConfiguration _createElementTypeSetConfiguration = ConfigurationSetRule.elementtypesconfigurationsFactory.createElementTypeSetConfiguration();
      _result = _createElementTypeSetConfiguration;
      _createCache_toConfigurationSet.put(_cacheKey, _result);
    }
    _init_toConfigurationSet(_result, umlProfile);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, ElementTypeSetConfiguration> _createCache_toConfigurationSet = CollectionLiterals.newHashMap();
  
  private void _init_toConfigurationSet(final ElementTypeSetConfiguration it, final Profile umlProfile) {
    ConfigurationSetRule.elementTypeConfigurationList = CollectionLiterals.<ElementTypeConfiguration>newArrayList();
    this.adviceBindingConfigurationList = CollectionLiterals.<AbstractAdviceBindingConfiguration>newArrayList();
    this._identifiers.setIdentifierBase(umlProfile);
    it.setIdentifier(this._identifiers.getQualified("elementTypes"));
    String _elvis = null;
    ElementTypeSetConfiguration _baseUMLElementTypeSet = this._uMLElementTypes.getBaseUMLElementTypeSet();
    String _metamodelNsURI = null;
    if (_baseUMLElementTypeSet!=null) {
      _metamodelNsURI=_baseUMLElementTypeSet.getMetamodelNsURI();
    }
    if (_metamodelNsURI != null) {
      _elvis = _metamodelNsURI;
    } else {
      _elvis = UMLPackage.eNS_URI;
    }
    it.setMetamodelNsURI(_elvis);
    Iterable<ImpliedExtension> _allExtensions = this._uML.getAllExtensions(umlProfile);
    for (final ImpliedExtension ext : _allExtensions) {
      {
        Iterable<? extends ElementTypeConfiguration> _diagramSpecificElementTypes = this._uMLElementTypes.getDiagramSpecificElementTypes(ext.getMetaclass());
        for (final ElementTypeConfiguration element : _diagramSpecificElementTypes) {
          {
            final SpecializationTypeConfiguration elementtype = this._elementTypeRule.toElementType(ext, element);
            ConfigurationSetRule.elementTypeConfigurationList.add(elementtype);
          }
        }
        final Function1<ElementTypeConfiguration, Boolean> _function = new Function1<ElementTypeConfiguration, Boolean>() {
          @Override
          public Boolean apply(final ElementTypeConfiguration it) {
            boolean _hasSemanticSupertype = ConfigurationSetRule.this._uMLElementTypes.hasSemanticSupertype(it);
            return Boolean.valueOf((!_hasSemanticSupertype));
          }
        };
        final Iterable<? extends ElementTypeConfiguration> typesNeedingAdvice = IterableExtensions.filter(this._uMLElementTypes.getDiagramSpecificElementTypes(ext.getMetaclass()), _function);
        for (final ElementTypeConfiguration element_1 : typesNeedingAdvice) {
          {
            final ApplyStereotypeAdviceConfiguration advice = this._applyStereotypeAdviceRule.toAdviceConfiguration(ext.getStereotype(), ext, element_1);
            this.adviceBindingConfigurationList.add(advice);
          }
        }
      }
    }
    final Function1<AbstractAdviceBindingConfiguration, String> _function = new Function1<AbstractAdviceBindingConfiguration, String>() {
      @Override
      public String apply(final AbstractAdviceBindingConfiguration it) {
        return it.getIdentifier();
      }
    };
    it.getAdviceBindingsConfigurations().addAll(IterableExtensions.<AbstractAdviceBindingConfiguration, String>sortBy(this.adviceBindingConfigurationList, _function));
    final Function1<ElementTypeConfiguration, String> _function_1 = new Function1<ElementTypeConfiguration, String>() {
      @Override
      public String apply(final ElementTypeConfiguration it) {
        return it.getIdentifier();
      }
    };
    it.getElementTypeConfigurations().addAll(IterableExtensions.<ElementTypeConfiguration, String>sortBy(ConfigurationSetRule.elementTypeConfigurationList, _function_1));
  }
}
