/**
 * Copyright (c) 2014, 2015, 2018 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Christian W. Damus - Initial API and implementation
 * Ansgar Radermacher - Bug 526155, enable re-generation from profile: copy existing advices
 * Ansgar Radermacher - Bug 526156, reference semantic base element type
 */
package org.eclipse.papyrus.uml.profile.types.generator;

import com.google.common.base.Objects;
import java.util.ArrayList;
import java.util.HashMap;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmf.runtime.emf.type.core.ElementTypeRegistry;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.papyrus.infra.types.AbstractEditHelperAdviceConfiguration;
import org.eclipse.papyrus.infra.types.ElementTypeConfiguration;
import org.eclipse.papyrus.infra.types.ElementTypesConfigurationsFactory;
import org.eclipse.papyrus.infra.types.IconEntry;
import org.eclipse.papyrus.infra.types.SpecializationTypeConfiguration;
import org.eclipse.papyrus.infra.types.core.impl.ConfiguredHintedSpecializationElementType;
import org.eclipse.papyrus.uml.profile.types.generator.ConfigurationSetRule;
import org.eclipse.papyrus.uml.profile.types.generator.Identifiers;
import org.eclipse.papyrus.uml.profile.types.generator.ImpliedExtension;
import org.eclipse.papyrus.uml.profile.types.generator.UMLElementTypes;
import org.eclipse.papyrus.uml.types.core.matchers.stereotype.StereotypeApplicationMatcherConfiguration;
import org.eclipse.papyrus.uml.types.core.matchers.stereotype.StereotypeApplicationMatcherFactory;
import org.eclipse.uml2.uml.Image;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Transformation rule for generating a {@link SpecializationTypeConfiguration} from a UML metaclass {@link Extension}.
 */
@Singleton
@SuppressWarnings("all")
public class ElementTypeRule {
  @Extension
  private static ElementTypesConfigurationsFactory elementtypesconfigurationsFactory = ElementTypesConfigurationsFactory.eINSTANCE;
  
  @Extension
  private static StereotypeApplicationMatcherFactory stereotypeApplicationMatcherConfigurationFactory = StereotypeApplicationMatcherFactory.eINSTANCE;
  
  @Inject
  @Extension
  private UMLElementTypes _uMLElementTypes;
  
  @Inject
  @Extension
  private Identifiers _identifiers;
  
  public SpecializationTypeConfiguration toElementType(final ImpliedExtension umlExtension, final ElementTypeConfiguration supertype) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlExtension, supertype);
    final SpecializationTypeConfiguration _result;
    synchronized (_createCache_toElementType) {
      if (_createCache_toElementType.containsKey(_cacheKey)) {
        return _createCache_toElementType.get(_cacheKey);
      }
      SpecializationTypeConfiguration _createSpecializationTypeConfiguration = ElementTypeRule.elementtypesconfigurationsFactory.createSpecializationTypeConfiguration();
      _result = _createSpecializationTypeConfiguration;
      _createCache_toElementType.put(_cacheKey, _result);
    }
    _init_toElementType(_result, umlExtension, supertype);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, SpecializationTypeConfiguration> _createCache_toElementType = CollectionLiterals.newHashMap();
  
  private void _init_toElementType(final SpecializationTypeConfiguration it, final ImpliedExtension umlExtension, final ElementTypeConfiguration supertype) {
    it.setIdentifier(this._identifiers.toElementTypeID(umlExtension, supertype));
    boolean _hasSemanticSupertype = this._uMLElementTypes.hasSemanticSupertype(supertype);
    if (_hasSemanticSupertype) {
      final String baseTypeId = this._identifiers.toSemanticElementTypeID(umlExtension, this._uMLElementTypes.getElementTypeConfiguration(umlExtension.getMetaclass()));
      final IElementType baseTypeFromRegistry = ElementTypeRegistry.getInstance().getType(baseTypeId);
      if ((baseTypeFromRegistry instanceof ConfiguredHintedSpecializationElementType)) {
        final ElementTypeConfiguration baseType = ((ConfiguredHintedSpecializationElementType) baseTypeFromRegistry).getConfiguration();
        it.getSpecializedTypes().add(baseType);
      } else {
        final SpecializationTypeConfiguration baseType_1 = ElementTypeRule.elementtypesconfigurationsFactory.createSpecializationTypeConfiguration();
        baseType_1.setIdentifier(this._identifiers.toElementTypeID(umlExtension, this._uMLElementTypes.getElementTypeConfiguration(umlExtension.getMetaclass())));
        baseType_1.getSpecializedTypes().add(this._uMLElementTypes.getElementTypeConfiguration(umlExtension.getMetaclass()));
        baseType_1.setHint(this._uMLElementTypes.getElementTypeConfiguration(umlExtension.getMetaclass()).getHint());
        baseType_1.setName(this._identifiers.toElementTypeName(umlExtension, this._uMLElementTypes.getElementTypeConfiguration(umlExtension.getMetaclass())));
        IconEntry icon = this.getIconEntry(umlExtension.getStereotype());
        IconEntry _xifexpression = null;
        boolean _notEquals = (!Objects.equal(icon, null));
        if (_notEquals) {
          _xifexpression = icon;
        } else {
          _xifexpression = this._uMLElementTypes.getIconEntry(umlExtension.getMetaclass());
        }
        baseType_1.setIconEntry(_xifexpression);
        final ElementTypeConfiguration addedBaseType = ConfigurationSetRule.addElementType(baseType_1);
        it.getSpecializedTypes().add(addedBaseType);
      }
    }
    it.getSpecializedTypes().add(supertype);
    it.setHint(supertype.getHint());
    it.setName(this._identifiers.toElementTypeName(umlExtension, supertype));
    final IElementType elemTypeFromRegistry = ElementTypeRegistry.getInstance().getType(it.getIdentifier());
    if ((elemTypeFromRegistry instanceof ConfiguredHintedSpecializationElementType)) {
      final ElementTypeConfiguration elemTypeConfigFromRegistry = ((ConfiguredHintedSpecializationElementType)elemTypeFromRegistry).getConfiguration();
      if ((elemTypeConfigFromRegistry instanceof SpecializationTypeConfiguration)) {
        final AbstractEditHelperAdviceConfiguration helperAdviceFromRegistry = ((SpecializationTypeConfiguration) elemTypeConfigFromRegistry).getEditHelperAdviceConfiguration();
        boolean _notEquals_1 = (!Objects.equal(helperAdviceFromRegistry, null));
        if (_notEquals_1) {
          it.setEditHelperAdviceConfiguration(helperAdviceFromRegistry);
        }
      }
    }
    IconEntry icon_1 = this.getIconEntry(umlExtension.getStereotype());
    IconEntry _xifexpression_1 = null;
    boolean _notEquals_2 = (!Objects.equal(icon_1, null));
    if (_notEquals_2) {
      _xifexpression_1 = icon_1;
    } else {
      _xifexpression_1 = this._uMLElementTypes.getIconEntry(umlExtension.getMetaclass());
    }
    it.setIconEntry(_xifexpression_1);
    boolean _hasSemanticSupertype_1 = this._uMLElementTypes.hasSemanticSupertype(supertype);
    boolean _not = (!_hasSemanticSupertype_1);
    if (_not) {
      it.setMatcherConfiguration(this.toMatcherConfiguration(umlExtension, supertype));
    }
  }
  
  private StereotypeApplicationMatcherConfiguration toMatcherConfiguration(final ImpliedExtension umlExtension, final ElementTypeConfiguration supertype) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlExtension, supertype);
    final StereotypeApplicationMatcherConfiguration _result;
    synchronized (_createCache_toMatcherConfiguration) {
      if (_createCache_toMatcherConfiguration.containsKey(_cacheKey)) {
        return _createCache_toMatcherConfiguration.get(_cacheKey);
      }
      StereotypeApplicationMatcherConfiguration _createStereotypeApplicationMatcherConfiguration = ElementTypeRule.stereotypeApplicationMatcherConfigurationFactory.createStereotypeApplicationMatcherConfiguration();
      _result = _createStereotypeApplicationMatcherConfiguration;
      _createCache_toMatcherConfiguration.put(_cacheKey, _result);
    }
    _init_toMatcherConfiguration(_result, umlExtension, supertype);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, StereotypeApplicationMatcherConfiguration> _createCache_toMatcherConfiguration = CollectionLiterals.newHashMap();
  
  private void _init_toMatcherConfiguration(final StereotypeApplicationMatcherConfiguration it, final ImpliedExtension umlExtension, final ElementTypeConfiguration supertype) {
    it.getStereotypesQualifiedNames().add(umlExtension.getStereotype().getQualifiedName());
  }
  
  private IconEntry getIconEntry(final Stereotype stereotype) {
    IconEntry _xblockexpression = null;
    {
      final Function1<Image, Boolean> _function = new Function1<Image, Boolean>() {
        @Override
        public Boolean apply(final Image it) {
          boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(it.getLocation());
          return Boolean.valueOf((!_isNullOrEmpty));
        }
      };
      final Image image = IterableExtensions.<Image>findFirst(stereotype.getIcons(), _function);
      IconEntry _xifexpression = null;
      boolean _notEquals = (!Objects.equal(image, null));
      if (_notEquals) {
        IconEntry _xblockexpression_1 = null;
        {
          final URI uri = URI.createURI(image.getLocation(), true);
          IconEntry _xifexpression_1 = null;
          boolean _notEquals_1 = (!Objects.equal(uri, null));
          if (_notEquals_1) {
            IconEntry _createIconEntry = ElementTypeRule.elementtypesconfigurationsFactory.createIconEntry();
            final Procedure1<IconEntry> _function_1 = new Procedure1<IconEntry>() {
              @Override
              public void apply(final IconEntry it) {
                boolean _isPlatform = uri.isPlatform();
                if (_isPlatform) {
                  it.setBundleId(uri.segment(1));
                  final Function1<String, CharSequence> _function = new Function1<String, CharSequence>() {
                    @Override
                    public CharSequence apply(final String it) {
                      return URI.decode(it);
                    }
                  };
                  String _join = IterableExtensions.<String>join(IterableExtensions.<String>drop(uri.segmentsList(), 2), "/", _function);
                  String _plus = ("/" + _join);
                  it.setIconPath(_plus);
                } else {
                  boolean _isRelative = uri.isRelative();
                  if (_isRelative) {
                    it.setBundleId(ElementTypeRule.this.containingProject(stereotype).getName());
                    String _decode = URI.decode(uri.toString());
                    String _plus_1 = ("/" + _decode);
                    it.setIconPath(_plus_1);
                  } else {
                    it.setIconPath(uri.toString());
                  }
                }
              }
            };
            _xifexpression_1 = ObjectExtensions.<IconEntry>operator_doubleArrow(_createIconEntry, _function_1);
          }
          _xblockexpression_1 = _xifexpression_1;
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private IProject containingProject(final EObject object) {
    return ResourcesPlugin.getWorkspace().getRoot().getProject(object.eResource().getURI().segment(2));
  }
}
