/**
 * Copyright (c) 2014, 2015, 2023 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Christian W. Damus - Initial API and implementation
 * Ansgar Radermacher - bug 582492, move to com.google.inject
 */
package org.eclipse.papyrus.uml.profile.types.generator;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Pattern;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.gmf.runtime.emf.type.core.ElementTypeRegistry;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.gmf.runtime.emf.type.core.IMetamodelType;
import org.eclipse.gmf.runtime.emf.type.core.ISpecializationType;
import org.eclipse.papyrus.infra.types.ElementTypeConfiguration;
import org.eclipse.papyrus.infra.types.ElementTypeSetConfiguration;
import org.eclipse.papyrus.infra.types.ElementTypesConfigurationsFactory;
import org.eclipse.papyrus.infra.types.IconEntry;
import org.eclipse.papyrus.infra.types.MetamodelTypeConfiguration;
import org.eclipse.papyrus.infra.types.SpecializationTypeConfiguration;
import org.eclipse.papyrus.infra.types.core.registries.ElementTypeSetConfigurationRegistry;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Utility extensions for working with and generating objects for the base UML element types specialized by the profile.
 */
@Singleton
@SuppressWarnings("all")
public class UMLElementTypes {
  private static final Pattern VISUAL_ID_PATTERN = Pattern.compile("\\d{4}");

  @Extension
  private static ElementTypesConfigurationsFactory elementtypesconfigurationsFactory = ElementTypesConfigurationsFactory.eINSTANCE;

  @Inject
  @Extension
  private Identifiers _identifiers;

  @Inject
  @Extension
  private UML _uML;

  public String getElementTypeID(final org.eclipse.uml2.uml.Class metaclass) {
    String _name = metaclass.getName();
    return ("org.eclipse.papyrus.uml." + _name);
  }

  public ElementTypeConfiguration getElementTypeConfiguration(final org.eclipse.uml2.uml.Class metaclass) {
    final Function1<ElementTypeConfiguration, Boolean> _function = (ElementTypeConfiguration it) -> {
      String _identifier = it.getIdentifier();
      String _elementTypeID = this.getElementTypeID(metaclass);
      return Boolean.valueOf(Objects.equal(_identifier, _elementTypeID));
    };
    return IterableExtensions.<ElementTypeConfiguration>findFirst(this.getBaseUMLElementTypeSet().getElementTypeConfigurations(), _function);
  }

  public ElementTypeSetConfiguration getBaseUMLElementTypeSet() {
    return ElementTypeSetConfigurationRegistry.getInstance().getElementTypeSetConfigurations().get(this._identifiers.getContextId()).get(this._identifiers.getUmlElementTypesSet());
  }

  public Iterable<IElementType> getBaseUMLElementTypes() {
    final Function1<ElementTypeConfiguration, IElementType> _function = (ElementTypeConfiguration it) -> {
      return ElementTypeRegistry.getInstance().getType(it.getIdentifier());
    };
    return IterableExtensions.<IElementType>filterNull(ListExtensions.<ElementTypeConfiguration, IElementType>map(this.getBaseUMLElementTypeSet().getElementTypeConfigurations(), _function));
  }

  public EClass getEClass(final org.eclipse.uml2.uml.Class metaclass) {
    EClassifier _eClassifier = UMLPackage.eINSTANCE.getEClassifier(metaclass.getName());
    return ((EClass) _eClassifier);
  }

  public IconEntry getIconEntry(final org.eclipse.uml2.uml.Class metaclass) {
    ElementTypeConfiguration _elementTypeConfiguration = this.getElementTypeConfiguration(metaclass);
    IconEntry _iconEntry = null;
    if (_elementTypeConfiguration!=null) {
      _iconEntry=_elementTypeConfiguration.getIconEntry();
    }
    return this.copy(_iconEntry);
  }

  private IconEntry copy(final IconEntry prototype) {
    IconEntry _createIconEntry = UMLElementTypes.elementtypesconfigurationsFactory.createIconEntry();
    final Procedure1<IconEntry> _function = (IconEntry it) -> {
      it.setBundleId(prototype.getBundleId());
      it.setIconPath(prototype.getIconPath());
    };
    return ObjectExtensions.<IconEntry>operator_doubleArrow(_createIconEntry, _function);
  }

  public ElementTypeSetConfiguration getBaseElementTypeSet() {
    ElementTypeSetConfiguration _elvis = null;
    ElementTypeSetConfiguration _baseElementTypesSetConfiguration = this._identifiers.getBaseElementTypesSetConfiguration();
    if (_baseElementTypesSetConfiguration != null) {
      _elvis = _baseElementTypesSetConfiguration;
    } else {
      ElementTypeSetConfiguration _get = ElementTypeSetConfigurationRegistry.getInstance().getElementTypeSetConfigurations().get(this._identifiers.getContextId()).get(this._identifiers.getBaseElementTypesSet());
      _elvis = _get;
    }
    return _elvis;
  }

  public Iterable<ElementTypeConfiguration> getBaseElementTypes() {
    final Function1<ElementTypeConfiguration, Boolean> _function = (ElementTypeConfiguration it) -> {
      return Boolean.valueOf(this.validType(it));
    };
    return IterableExtensions.<ElementTypeConfiguration>filter(this.getBaseElementTypeSet().getElementTypeConfigurations(), _function);
  }

  public boolean validType(final ElementTypeConfiguration elementType) {
    EClass _metaclass = this.getMetaclass(elementType);
    return (!Objects.equal(_metaclass, null));
  }

  private boolean isDiagramSpecific() {
    ElementTypeSetConfiguration _baseElementTypeSet = this.getBaseElementTypeSet();
    ElementTypeSetConfiguration _baseUMLElementTypeSet = this.getBaseUMLElementTypeSet();
    return (!Objects.equal(_baseElementTypeSet, _baseUMLElementTypeSet));
  }

  public boolean isDiagramSpecific(final ElementTypeConfiguration type) {
    return this.isVisualID(type.getHint());
  }

  public boolean hasSemanticSupertype(final ElementTypeConfiguration type) {
    return (this.isDiagramSpecific(type) && (!this._identifiers.isSuppressSemanticSuperElementTypes()));
  }

  private boolean isVisualID(final String string) {
    return ((!StringExtensions.isNullOrEmpty(string)) && (((((UMLElementTypes.VISUAL_ID_PATTERN.matcher(string).matches() || 
      string.contains("Diagram")) || 
      string.contains("Shape")) || 
      string.contains("Edge")) || 
      string.contains("Label")) || 
      string.contains("Compartment")));
  }

  public Iterable<? extends ElementTypeConfiguration> getDiagramSpecificElementTypes(final org.eclipse.uml2.uml.Class metaclass) {
    Iterable<? extends ElementTypeConfiguration> _xifexpression = null;
    boolean _isDiagramSpecific = this.isDiagramSpecific();
    boolean _not = (!_isDiagramSpecific);
    if (_not) {
      final Function1<ElementTypeConfiguration, Boolean> _function = (ElementTypeConfiguration it) -> {
        return Boolean.valueOf((this.validType(it) && Objects.equal(it.getIdentifier(), this.getElementTypeID(metaclass))));
      };
      _xifexpression = IterableExtensions.<ElementTypeConfiguration>filter(this.getBaseUMLElementTypeSet().getElementTypeConfigurations(), _function);
    } else {
      final Function1<SpecializationTypeConfiguration, Boolean> _function_1 = (SpecializationTypeConfiguration it) -> {
        return Boolean.valueOf((this.validType(it) && this.containsId(it.getSpecializedTypes(), this.getElementTypeID(metaclass))));
      };
      _xifexpression = IterableExtensions.<SpecializationTypeConfiguration>filter(Iterables.<SpecializationTypeConfiguration>filter(this.getBaseElementTypeSet().getElementTypeConfigurations(), SpecializationTypeConfiguration.class), _function_1);
    }
    return _xifexpression;
  }

  private boolean containsId(final List<ElementTypeConfiguration> elementTypeConfigurations, final String id) {
    for (final ElementTypeConfiguration elementTypeConfiguration : elementTypeConfigurations) {
      boolean _equals = elementTypeConfiguration.getIdentifier().equals(id);
      if (_equals) {
        return true;
      }
    }
    return false;
  }

  private boolean canContain(final IElementType containerType, final EClass containedEClass) {
    final Function1<EReference, Boolean> _function = (EReference it) -> {
      return Boolean.valueOf(it.getEReferenceType().isSuperTypeOf(containedEClass));
    };
    return IterableExtensions.<EReference>exists(containerType.getEClass().getEAllContainments(), _function);
  }

  private boolean canContain(final ElementTypeConfiguration containerType, final EClass containedEClass) {
    IElementType _type = ElementTypeRegistry.getInstance().getType(containerType.getIdentifier());
    boolean _canContain = false;
    if (_type!=null) {
      _canContain=this.canContain(_type, containedEClass);
    }
    return _canContain;
  }

  protected boolean _canContainType(final ElementTypeConfiguration containerType, final ElementTypeConfiguration containedTypeConfiguration) {
    return false;
  }

  protected boolean _canContainType(final ElementTypeConfiguration containerType, final MetamodelTypeConfiguration containedTypeConfiguration) {
    return this.canContain(containerType, containedTypeConfiguration.getEClass());
  }

  protected boolean _canContainType(final ElementTypeConfiguration containerType, final SpecializationTypeConfiguration containedTypeConfiguration) {
    final Function1<ElementTypeConfiguration, Boolean> _function = (ElementTypeConfiguration supertype) -> {
      return Boolean.valueOf(this.canContain(containerType, ElementTypeRegistry.getInstance().getType(supertype.getIdentifier()).getEClass()));
    };
    return IterableExtensions.<ElementTypeConfiguration>exists(containedTypeConfiguration.getSpecializedTypes(), _function);
  }

  public boolean isRelationship(final IElementType elementType) {
    return (Objects.equal(elementType.getEClass(), null) || this._uML.isRelationship(elementType.getEClass()));
  }

  public boolean isRelationship(final ElementTypeConfiguration elementType) {
    return (Objects.equal(this.getMetaclass(elementType), null) || this._uML.isRelationship(this.getMetaclass(elementType)));
  }

  private EClass _getMetaclass(final ElementTypeConfiguration elementType) {
    return null;
  }

  private EClass _getMetaclass(final SpecializationTypeConfiguration elementType) {
    EClass _xblockexpression = null;
    {
      IElementType registered = ElementTypeRegistry.getInstance().getType(elementType.getIdentifier());
      EClass _xifexpression = null;
      boolean _notEquals = (!Objects.equal(registered, null));
      if (_notEquals) {
        _xifexpression = this.getMetaclass(registered);
      } else {
        final Function1<ElementTypeConfiguration, IElementType> _function = (ElementTypeConfiguration it) -> {
          return ElementTypeRegistry.getInstance().getType(it.getIdentifier());
        };
        IElementType _head = IterableExtensions.<IElementType>head(IterableExtensions.<IElementType>filterNull(ListExtensions.<ElementTypeConfiguration, IElementType>map(elementType.getSpecializedTypes(), _function)));
        EClass _eClass = null;
        if (_head!=null) {
          _eClass=_head.getEClass();
        }
        _xifexpression = _eClass;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  private EClass _getMetaclass(final MetamodelTypeConfiguration elementType) {
    return elementType.getEClass();
  }

  private EClass _getMetaclass(final IElementType elementType) {
    return null;
  }

  private EClass _getMetaclass(final IMetamodelType elementType) {
    return elementType.getEClass();
  }

  private EClass _getMetaclass(final ISpecializationType elementType) {
    IMetamodelType _metamodelType = elementType.getMetamodelType();
    EClass _eClass = null;
    if (_metamodelType!=null) {
      _eClass=_metamodelType.getEClass();
    }
    return _eClass;
  }

  private boolean canSourceTo(final IElementType sourceType, final EClass relationshipEClass) {
    final Function1<EReference, Boolean> _function = (EReference it) -> {
      return Boolean.valueOf(it.getEReferenceType().isSuperTypeOf(sourceType.getEClass()));
    };
    return IterableExtensions.<EReference>exists(this._uML.getSourceReferences(relationshipEClass), _function);
  }

  private boolean canSourceTo(final ElementTypeConfiguration sourceType, final EClass relationshipEClass) {
    ElementTypeRegistry _instance = ElementTypeRegistry.getInstance();
    String _identifier = null;
    if (sourceType!=null) {
      _identifier=sourceType.getIdentifier();
    }
    IElementType _type = _instance.getType(_identifier);
    boolean _canSourceTo = false;
    if (_type!=null) {
      _canSourceTo=this.canSourceTo(_type, relationshipEClass);
    }
    return _canSourceTo;
  }

  protected boolean _canSourceToType(final ElementTypeConfiguration sourceType, final ElementTypeConfiguration relationshipTypeConfiguration) {
    return false;
  }

  protected boolean _canSourceToType(final ElementTypeConfiguration sourceType, final MetamodelTypeConfiguration relationshipTypeConfiguration) {
    return this.canSourceTo(sourceType, relationshipTypeConfiguration.getEClass());
  }

  protected boolean _canSourceToType(final ElementTypeConfiguration sourceType, final SpecializationTypeConfiguration relationshipTypeConfiguration) {
    final Function1<ElementTypeConfiguration, Boolean> _function = (ElementTypeConfiguration supertypeConfiguration) -> {
      boolean _xblockexpression = false;
      {
        final IElementType supertype = ElementTypeRegistry.getInstance().getType(supertypeConfiguration.getIdentifier());
        _xblockexpression = (((!Objects.equal(supertype, null)) && this._uML.isRelationship(supertype.getEClass())) && this.canSourceTo(sourceType, supertype.getEClass()));
      }
      return Boolean.valueOf(_xblockexpression);
    };
    return IterableExtensions.<ElementTypeConfiguration>exists(relationshipTypeConfiguration.getSpecializedTypes(), _function);
  }

  private boolean canTargetFrom(final IElementType targetType, final EClass relationshipEClass) {
    final Function1<EReference, Boolean> _function = (EReference it) -> {
      return Boolean.valueOf(it.getEReferenceType().isSuperTypeOf(targetType.getEClass()));
    };
    return IterableExtensions.<EReference>exists(this._uML.getTargetReferences(relationshipEClass), _function);
  }

  private boolean canTargetFrom(final ElementTypeConfiguration targetType, final EClass relationshipEClass) {
    ElementTypeRegistry _instance = ElementTypeRegistry.getInstance();
    String _identifier = null;
    if (targetType!=null) {
      _identifier=targetType.getIdentifier();
    }
    IElementType _type = _instance.getType(_identifier);
    boolean _canTargetFrom = false;
    if (_type!=null) {
      _canTargetFrom=this.canTargetFrom(_type, relationshipEClass);
    }
    return _canTargetFrom;
  }

  protected boolean _canTargetFromType(final ElementTypeConfiguration targetType, final ElementTypeConfiguration relationshipTypeConfiguration) {
    return false;
  }

  protected boolean _canTargetFromType(final ElementTypeConfiguration targetType, final MetamodelTypeConfiguration relationshipTypeConfiguration) {
    return this.canTargetFrom(targetType, relationshipTypeConfiguration.getEClass());
  }

  protected boolean _canTargetFromType(final ElementTypeConfiguration targetType, final SpecializationTypeConfiguration relationshipTypeConfiguration) {
    final Function1<ElementTypeConfiguration, Boolean> _function = (ElementTypeConfiguration supertypeConfiguration) -> {
      boolean _xblockexpression = false;
      {
        final IElementType supertype = ElementTypeRegistry.getInstance().getType(supertypeConfiguration.getIdentifier());
        _xblockexpression = (((!Objects.equal(supertype, null)) && this._uML.isRelationship(supertype.getEClass())) && this.canTargetFrom(targetType, supertype.getEClass()));
      }
      return Boolean.valueOf(_xblockexpression);
    };
    return IterableExtensions.<ElementTypeConfiguration>exists(relationshipTypeConfiguration.getSpecializedTypes(), _function);
  }

  public boolean canContainType(final ElementTypeConfiguration containerType, final ElementTypeConfiguration containedTypeConfiguration) {
    if (containedTypeConfiguration instanceof MetamodelTypeConfiguration) {
      return _canContainType(containerType, (MetamodelTypeConfiguration)containedTypeConfiguration);
    } else if (containedTypeConfiguration instanceof SpecializationTypeConfiguration) {
      return _canContainType(containerType, (SpecializationTypeConfiguration)containedTypeConfiguration);
    } else if (containedTypeConfiguration != null) {
      return _canContainType(containerType, containedTypeConfiguration);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(containerType, containedTypeConfiguration).toString());
    }
  }

  private EClass getMetaclass(final Object elementType) {
    if (elementType instanceof MetamodelTypeConfiguration) {
      return _getMetaclass((MetamodelTypeConfiguration)elementType);
    } else if (elementType instanceof SpecializationTypeConfiguration) {
      return _getMetaclass((SpecializationTypeConfiguration)elementType);
    } else if (elementType instanceof ElementTypeConfiguration) {
      return _getMetaclass((ElementTypeConfiguration)elementType);
    } else if (elementType instanceof IMetamodelType) {
      return _getMetaclass((IMetamodelType)elementType);
    } else if (elementType instanceof ISpecializationType) {
      return _getMetaclass((ISpecializationType)elementType);
    } else if (elementType instanceof IElementType) {
      return _getMetaclass((IElementType)elementType);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(elementType).toString());
    }
  }

  public boolean canSourceToType(final ElementTypeConfiguration sourceType, final ElementTypeConfiguration relationshipTypeConfiguration) {
    if (relationshipTypeConfiguration instanceof MetamodelTypeConfiguration) {
      return _canSourceToType(sourceType, (MetamodelTypeConfiguration)relationshipTypeConfiguration);
    } else if (relationshipTypeConfiguration instanceof SpecializationTypeConfiguration) {
      return _canSourceToType(sourceType, (SpecializationTypeConfiguration)relationshipTypeConfiguration);
    } else if (relationshipTypeConfiguration != null) {
      return _canSourceToType(sourceType, relationshipTypeConfiguration);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(sourceType, relationshipTypeConfiguration).toString());
    }
  }

  public boolean canTargetFromType(final ElementTypeConfiguration targetType, final ElementTypeConfiguration relationshipTypeConfiguration) {
    if (relationshipTypeConfiguration instanceof MetamodelTypeConfiguration) {
      return _canTargetFromType(targetType, (MetamodelTypeConfiguration)relationshipTypeConfiguration);
    } else if (relationshipTypeConfiguration instanceof SpecializationTypeConfiguration) {
      return _canTargetFromType(targetType, (SpecializationTypeConfiguration)relationshipTypeConfiguration);
    } else if (relationshipTypeConfiguration != null) {
      return _canTargetFromType(targetType, relationshipTypeConfiguration);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(targetType, relationshipTypeConfiguration).toString());
    }
  }
}
